/****************************************************************************\
*                                                                            *
* Figure 11                                                                  *
*                                                                            *
* WinSets, Guy Eddon, 1993                                                   *
*                                                                            *
* CARDS.C - This module handles all actions involving the cards.             *
*                                                                            *
\****************************************************************************/

#include "winsets.h"

const char win_four[FOUR_CARD_GAME][8] = { "G Y R B", "B Y R G", "GR BY",
   "RY GB", "GR YB", "YG RB", "GY RB", "GB RY", "B GR Y", "R YG B", "Y BG R",
   "G BY R", "Y RB G", "B YR G", "R GYB", "G BRY", "G RYB", "R BGY", "Y BGR",
   "Y RBG", "B YGR", "B GRY", "G R BY", "B G RY", "Y R BG", "Y B RG",
   "R B YG", "Y G BR", "BYRG", "RGYB" };

const char win_five[FIVE_CARD_GAME][10] = { "R G Y B P", "R G Y BP",
   "P G Y BR", "B Y G PR", "RBYPG", "B RPYG", "Y PRBG", "G PYBR", "P YBGR",
   "R YBGP", "BR G PY", "YR P GB", "YR B GP", "PB R GY", "RG Y BP",
   "RP G BY", "GP R YB", "GY B PR", "YG P BR", "PG Y BR", "GB R PY",
   "RG B YP", "BG Y PR", "RY G PB", "BY P RG", "B Y PG R", "P B GR Y",
   "R B PY G", "P Y BG R", "R P YB G", "R P YG B", "P B YR G", "PG YRB",
   "BG YRP", "BY GRP", "RP GYB", "YP GRB", "GY BRP", "RB GYP", "PB GRY",
   "GR YBP", "YR GBP", "Y G RBP", "R P GYB", "Y B RGP", "G P YRB", "B P GRY",
   "B R PYG", "R G YBP", "Y R GBP", "P Y GRB", "G B YRP" };

BOOL is_card_selected(int x, int y, RECT *card_rect)
   {
   POINT pt; // POINT data structure

   // Current mouse position
   pt.x = x;
   pt.y = y;

   // Return
   return PTINRECT(card_rect, pt);
   }

void get_card_coord(int card_number, RECT *rect, int ball_diam, int ledge,
   RECT *card_rect)
   {
   int CARD_WIDTH, CARD_HEIGHT;
   int ball_rad;

   ball_diam = ((ball_diam * 2) / 3);
   ball_rad = (ball_diam / 2);

   CARD_WIDTH  = (3 * ball_diam);
   CARD_HEIGHT = (int)((CARD_BALLS_TALL * ball_diam)
      + ((((CARD_BALLS_TALL + 1) * ball_rad) / 2.0) + 0.5));

   ledge = (int)(ledge * 1.5);

   card_rect->right = ((rect->right - ((card_number - 1)
      * (CARD_WIDTH + ledge))) - (card_number * ledge));
   card_rect->top = (rect->bottom - (CARD_HEIGHT + (1 * ledge)));
   card_rect->left = ((rect->right - (card_number * (CARD_WIDTH + ledge)))
      - (card_number * ledge));
   card_rect->bottom = (rect->bottom - (ledge * 1));
   }

void get_reverse_card_coord(RECT *rect, int ball_diam, int ledge,
   RECT *card_rect)
   {
   int save_top;

   get_card_coord(1, rect, ball_diam, ledge, card_rect);

   save_top = card_rect->top;

   card_rect->top = (card_rect->top - (card_rect->bottom - card_rect->top)
      - (ball_diam / 4));
   card_rect->bottom = (save_top - (ball_diam / 4));
   }

void draw_reverse_card(HDC hdc, RECT *rect, int ball_diam, int ledge,
   HBRUSH hGrayBrush)
   {
   HBRUSH hOldBrush;
   RECT card_rect;

   get_reverse_card_coord(rect, ball_diam, ledge, &card_rect);

   hOldBrush = SelectObject(hdc, hGrayBrush);

   RoundRect(hdc, card_rect.left, card_rect.top, card_rect.right,
      card_rect.bottom, ball_diam / 2, ball_diam / 2);

   SelectObject(hdc, hOldBrush);
   }

BOOL clicked_on_reverse_card(int x, int y, int ball_diam, int ledge,
   RECT *rect)
   {
   RECT card_rect;

   get_reverse_card_coord(rect, ball_diam, ledge, &card_rect);

   return is_card_selected(x, y, &card_rect);
   }

void draw_card_finished(HWND hwnd, RECT *win_rect, int finished_card,
   int ball_diam, int edge_ledge, current_cards *visible)
   {
   HBRUSH hHatchBrush;
   HRGN hRoundRectRgn;
   RECT card_rect;
   HDC  hdc;

   switch(finished_card)
      {
      case 1:
         visible->one_card_finished = TRUE;
         break;

      case 2:
         visible->two_card_finished = TRUE;
         break;

      case 3:
         visible->thr_card_finished = TRUE;
         break;
      }

   hdc = GetDC(hwnd);

   get_card_coord(finished_card, win_rect, ball_diam, edge_ledge,
      &card_rect);

   hRoundRectRgn = CreateRoundRectRgn(card_rect.left+1, card_rect.top+1,
      card_rect.right, card_rect.bottom, ball_diam / 2, ball_diam / 2);
   SelectObject(hdc, hRoundRectRgn);

   hHatchBrush = CreateHatchBrush(HS_DIAGCROSS, GRAY);

   SetBkMode(hdc, TRANSPARENT);

   FillRgn(hdc, hRoundRectRgn, hHatchBrush);

   DeleteObject(hHatchBrush);

   ReleaseDC(hwnd, hdc);

   DeleteObject(hRoundRectRgn);
   }

void do_card_complete(HWND hwnd, char *current_card, int num_circles,
   current_cards *visible, RECT *rect, int ball_diam, int edge_ledge,
   int *current_score, int num_moves_left, HMENU hMainMenu)
   {
   char current_ball_configuration[MAX_BALLS + 1];
   static BOOL one_card = FALSE;
   static BOOL two_card = FALSE;
   static BOOL thr_card = FALSE;
   BOOL is_configuration_correct = FALSE;

   make_string_from_bowl(current_ball_configuration, num_circles);

   if(num_circles == 8)
      {
      switch(*current_card)
         {
         case 1:
            is_configuration_correct = compare_card_to_balls(
               win_four[visible->one_card], current_ball_configuration);
            break;

         case 2:
            is_configuration_correct = compare_card_to_balls(
               win_four[visible->two_card], current_ball_configuration);
            break;

         case 3:
            is_configuration_correct = compare_card_to_balls(
               win_four[visible->thr_card], current_ball_configuration);
            break;
         }
      }

   if(num_circles == 10)
      {
      switch(*current_card)
         {
         case 1:
            is_configuration_correct = compare_card_to_balls(
               win_five[visible->one_card], current_ball_configuration);
            break;

         case 2:
            is_configuration_correct = compare_card_to_balls(
               win_five[visible->two_card], current_ball_configuration);
            break;

         case 3:
            is_configuration_correct = compare_card_to_balls(
               win_five[visible->thr_card], current_ball_configuration);
            break;
         }
      }

   if(is_configuration_correct == TRUE)
      {
      HDC hdc;
      char menu_buffer[NORMAL_BUFFER];

      EnableMenuItem(hMainMenu, IDM_UNDO, MF_GRAYED);

      hdc = GetDC(hwnd);

      draw_card_unhighlight(hwnd, rect, *current_card, ball_diam,
         edge_ledge);

      draw_card_finished(hwnd, rect, *current_card, ball_diam, edge_ledge,
         visible);

      (*current_score) += SCORE_FINISHED_CARD;

      if(!(num_circles == 8 && num_moves_left == 1
         && is_configuration_correct == TRUE))
         *current_card = (char)NULL;

      wsprintf(menu_buffer, "Score: %d", *current_score);
      draw_status_bar_text(hdc, menu_buffer, rect, status_bar_height, FALSE,
         THIRD_STATUS_BAR_BOX);

      ReleaseDC(hwnd, hdc);
      }

   if(num_circles == 8 && num_moves_left == 1)
      {
      char temp_value = 0;

      if((one_card == FALSE && two_card == FALSE && thr_card == FALSE
         && is_configuration_correct == TRUE) || one_card == TRUE
         || two_card == TRUE || thr_card == TRUE)
         {
         switch(*current_card)
            {
            case 1:
               one_card = TRUE;
               if(two_card == FALSE)
                  {
                  temp_value = 2;
                  do_card_complete(hwnd, &temp_value, num_circles, visible,
                     rect, ball_diam, edge_ledge, current_score,
                     num_moves_left, hMainMenu);
                  }
               if(two_card == TRUE && thr_card == FALSE)
                  {
                  temp_value = 3;
                  do_card_complete(hwnd, &temp_value, num_circles, visible,
                     rect, ball_diam, edge_ledge, current_score,
                     num_moves_left, hMainMenu);
                  }
               break;

            case 2:
               two_card = TRUE;
               if(one_card == FALSE)
                  {
                  temp_value = 1;
                  do_card_complete(hwnd, &temp_value, num_circles, visible,
                     rect, ball_diam, edge_ledge, current_score,
                     num_moves_left, hMainMenu);
                  }
               if(one_card == TRUE && thr_card == FALSE)
                  {
                  temp_value = 3;
                  do_card_complete(hwnd, &temp_value, num_circles, visible,
                     rect, ball_diam, edge_ledge, current_score,
                     num_moves_left, hMainMenu);
                  }
               break;

            case 3:
               thr_card = TRUE;
               if(one_card == FALSE)
                  {
                  temp_value = 1;
                  do_card_complete(hwnd, &temp_value, num_circles, visible,
                     rect, ball_diam, edge_ledge, current_score,
                     num_moves_left, hMainMenu);
                  }
               if(one_card == TRUE && two_card == FALSE)
                  {
                  temp_value = 2;
                  do_card_complete(hwnd, &temp_value, num_circles, visible,
                     rect, ball_diam, edge_ledge, current_score,
                     num_moves_left, hMainMenu);
                  }
               break;
            }

         one_card = FALSE;
         two_card = FALSE;
         thr_card = FALSE;
         *current_card = (char)NULL;
         }
      }
   }

void draw_card_unhighlight(HWND hwnd, RECT *win_rect, int previous_card,
   int ball_diam, int edge_ledge)
   {
   HPEN hpen, holdpen;
   RECT card_rect;
   HDC  hdc;
   int pen_width;

   hdc = GetDC(hwnd);

   get_card_coord(previous_card, win_rect, ball_diam, edge_ledge,
      &card_rect);

   pen_width = ((card_rect.right - card_rect.left) / HIGHLIGHT_BORDER);

   if(pen_width < 2)
      pen_width = 2;

   hpen = CreatePen(PS_INSIDEFRAME, pen_width, DK_GREEN);

   holdpen = SelectObject(hdc, hpen);
   SelectObject(hdc, GetStockObject(NULL_BRUSH));

   pen_width--;

   RoundRect(hdc,
      card_rect.left   - pen_width,
      card_rect.top    - pen_width,
      card_rect.right  + pen_width,
      card_rect.bottom + pen_width,
      ball_diam / 2, ball_diam / 2);

   SelectObject(hdc, holdpen);
   DeleteObject(hpen);

   RoundRect(hdc, card_rect.left, card_rect.top, card_rect.right,
      card_rect.bottom, ball_diam / 2, ball_diam / 2);

   ReleaseDC(hwnd, hdc);
   }

void draw_card_highlight(HWND hwnd, int new_card, RECT *win_rect,
   int ball_diam, int edge_ledge)
   {
   HPEN hpen, holdpen;
   RECT card_rect;
   HDC  hdc;
   int pen_width;

   if(new_card == (int)NULL)
      return;

   hdc = GetDC(hwnd);

   get_card_coord(new_card, win_rect, ball_diam, edge_ledge, &card_rect);

   pen_width = ((card_rect.right - card_rect.left) / HIGHLIGHT_BORDER);

   if(pen_width < 2)
      pen_width = 2;

   hpen = CreatePen(PS_INSIDEFRAME, pen_width, BLACK);
   holdpen = SelectObject(hdc, hpen);

   SelectObject(hdc, GetStockObject(NULL_BRUSH));

   pen_width--;

   RoundRect(hdc,
      card_rect.left   - pen_width,
      card_rect.top    - pen_width,
      card_rect.right  + pen_width,
      card_rect.bottom + pen_width,
      ball_diam / 2, ball_diam / 2);

   SelectObject(hdc, holdpen);
   DeleteObject(hpen);

   ReleaseDC(hwnd, hdc);
   }

void draw_cards(HDC hdc, RECT *rect, int ball_diam, int ledge, HBRUSH hBrush)
   {
   RECT card_rect;
   int count;

   if(hBrush == NULL)
      SelectObject(hdc, GetStockObject(WHITE_BRUSH));
   else
      SelectObject(hdc, hBrush);

   for(count = 1; count <= NUM_CARDS; count++)
      {
      get_card_coord(count, rect, ball_diam, ledge, &card_rect);

      RoundRect(hdc, card_rect.left, card_rect.top, card_rect.right,
         card_rect.bottom, ball_diam / 2, ball_diam / 2);
      }

   if(hBrush != NULL)
      SelectObject(hdc, GetStockObject(WHITE_BRUSH));
   }

void draw_card(HDC hdc, RECT *rect, int ball_diam, int ledge, int card_num)
   {
   RECT card_rect;

   SelectObject(hdc, GetStockObject(WHITE_BRUSH));

   get_card_coord(card_num, rect, ball_diam, ledge, &card_rect);

   RoundRect(hdc, card_rect.left, card_rect.top, card_rect.right,
      card_rect.bottom, ball_diam / 2, ball_diam / 2);
   }

void draw_card_configuration(PAINTSTRUCT *ps, HDC hdc, int card_number,
   const char *configuration, RECT *rect, int ball_diam, int ledge,
   int num_circles)
   {
   RECT card_rect;

   get_card_coord(card_number, rect, ball_diam, ledge, &card_rect);

   // Sophisticated window repainting in action!
   if(ps == NULL || INTRECT(&ps->rcPaint, &card_rect))
      draw_balls_on_card(hdc, &card_rect, configuration, ball_diam / 2,
         num_circles);
   }

void draw_balls_on_card(HDC hdc, RECT *card_rect, const char *configuration,
   int ball_rad, int num_circles)
   {
   int count, token_length, current_ball_in_token, StartFlag = TRUE,
      num_tokens, current_token = 1;
   char *blank;
   const char *current_char = configuration;

   num_tokens = num_chr_in_str(configuration, (char)32);

   num_tokens++; /* for last token */

   for(count = 0; count < (signed)strlen(configuration); count++)
      {
      if(current_ball_in_token == token_length || StartFlag)
         {
         if(configuration[count] == (char)32)
            {
            plot_ball(hdc, configuration[count], (int)NULL, (int)NULL,
               card_rect, ball_rad, current_token, num_tokens, num_circles);
            current_char++;
            current_token++;
            continue;
            }

         StartFlag = FALSE;

         blank = strchr(current_char, (char)32);

         if(blank)
            token_length = blank - current_char;
         else
            token_length = strlen(current_char);

         current_ball_in_token = 0;
         }

      plot_ball(hdc, configuration[count], current_ball_in_token + 1,
         token_length, card_rect, ball_rad, current_token, num_tokens,
         num_circles);

      current_ball_in_token++;

      current_char++;
      }
   }

void plot_ball(HDC hdc, const char char_color, int current_ball_in_token,
   int token_length, RECT *card_rect, int ball_rad, int current_token,
   int number_of_tokens, int num_circles)
   {
   HBRUSH hbr_color, hbr_old;
   static POINT Prev;
   float Y_Spacing_Index = (float)1;
   int pen_width;

   ball_rad = ((card_rect->bottom - card_rect->top) / (num_circles+2));

   if(current_ball_in_token == 1)
      Prev.x = card_rect->left;

   if(current_token == 1 && current_ball_in_token == 1)
      Prev.y = card_rect->top;

   if(num_circles == 8)
      switch(number_of_tokens)
         {
         case 1:
            Y_Spacing_Index = (float)2;
            break;
        
         case 2:
            Y_Spacing_Index = (float)1;
            if(token_length == 2)
               Y_Spacing_Index = (float)(0.5);
            if(token_length == 3)
               Y_Spacing_Index = (float)1;
            break;
        
         case 3:
            Y_Spacing_Index = (float)1;
            break;

         case 4:
            Y_Spacing_Index = (float)2;
            break;
         }

   if(num_circles == 10)
      switch(number_of_tokens)
         {
         case 1:
            Y_Spacing_Index = (float)2;
            break;

         case 2:
            Y_Spacing_Index = (float)(0.5);
            break;

         case 3:
            Y_Spacing_Index = (float)1;
            if(token_length == 2)
               Y_Spacing_Index = (float)(0.5);
            break;

         case 4:
            Y_Spacing_Index = (float)1;
            break;

         case 5:
            Y_Spacing_Index = (float)2;
            break;
         }

   switch(char_color)
      {
      case 'B': /* BLUE */
         hbr_color = CreateSolidBrush(BLUE);
         break;

      case 'G': /* GREEN */
         hbr_color = CreateSolidBrush(GREEN);
         break;

      case 'R': /* RED */
         hbr_color = CreateSolidBrush(RED);
         break;

      case 'Y': /* YELLOW */
         hbr_color = CreateSolidBrush(YELLOW);
         break;

      case 'P': /* PINK */
         hbr_color = CreateSolidBrush(PINK);
         break;

      case (char)32: /* BLACK */
         {
         pen_width = ball_rad / 3;
         hbr_color = CreatePen(PS_SOLID, pen_width, BLACK);
         }
         break;
      }

   hbr_old = SelectObject(hdc, hbr_color);

   switch(token_length)
      {
      case 1:
         draw_ball(hdc, ball_rad,
            (Prev.x + ((card_rect->right - card_rect->left) / 2)),
            (int)(Prev.y + ball_rad + (ball_rad / Y_Spacing_Index)));
         Prev.y += (int)((2 * ball_rad) + (ball_rad / Y_Spacing_Index));
         break;

      case 2:
         draw_ball(hdc, ball_rad,
            (Prev.x + ((card_rect->right - card_rect->left) / 4)),
            (int)(Prev.y + ball_rad + (ball_rad / Y_Spacing_Index)));
         if(current_ball_in_token == 2)
            {
            Prev.y += (int)((2 * ball_rad) + (ball_rad / Y_Spacing_Index));
            if(num_circles == 8 && number_of_tokens == 2)
               Prev.y += (int)(ball_rad / (Y_Spacing_Index * 2));
            if(num_circles == 10 && number_of_tokens == 3)
               Prev.y += (int)(ball_rad / (Y_Spacing_Index * 2));
            }
         Prev.x += ((card_rect->right - card_rect->left) / 2);
         break;

      case 3:
         draw_ball(hdc, ball_rad,
            (Prev.x + ((card_rect->right - card_rect->left) / 4)),
            (int)(Prev.y + ball_rad + (ball_rad / Y_Spacing_Index)));
         if(current_ball_in_token == 2 || current_ball_in_token == 3)
            Prev.y += (int)((2 * ball_rad) + (ball_rad / Y_Spacing_Index));
         if(current_ball_in_token == 2)
            Prev.x = card_rect->left
               - ((card_rect->right - card_rect->left) / 4);
         Prev.x += ((card_rect->right - card_rect->left) / 2);
         break;

      case 4:
         if(num_circles == 8)
            {
            /* Straight row */

            draw_ball(hdc, ball_rad,
               (Prev.x + ((card_rect->right - card_rect->left) / 2)),
               (int)(Prev.y + ball_rad + (ball_rad / Y_Spacing_Index)));
            Prev.y += (int)((2 * ball_rad) + (ball_rad / Y_Spacing_Index));
            }

         if(num_circles == 10)
            {
            /* Square */

            draw_ball(hdc, ball_rad,
               (Prev.x + ((card_rect->right - card_rect->left) / 4)),
               (int)(Prev.y + ball_rad + (ball_rad / Y_Spacing_Index)));
            Prev.x += ((card_rect->right - card_rect->left) / 2);
            if(current_ball_in_token == 2 || current_ball_in_token == 4)
               {
               Prev.y +=
                  (int)((2 * ball_rad) + (ball_rad / Y_Spacing_Index));
               Prev.x = card_rect->left;
               }
            }
         break;

      case 5:
         draw_ball(hdc, ball_rad,
            (Prev.x + ((card_rect->right - card_rect->left) / 2)),
            (int)(Prev.y + ball_rad + (ball_rad / Y_Spacing_Index)));
         Prev.y += (int)((2 * ball_rad) + (ball_rad / Y_Spacing_Index));
         break;

      case 0:
         {
         int increment = (int)((ball_rad / Y_Spacing_Index) / 2);
         int ledge = ((card_rect->right - card_rect->left) / 8);

         draw_line(hdc, card_rect->left + ledge, Prev.y + increment,
            card_rect->right - ledge, Prev.y + increment);
         }
         break;
      }

   SelectObject(hdc, hbr_old);
   DeleteObject(hbr_color);
   }

void draw_reverse_card_inverted(HDC hdc, RECT *rect, int ball_diam,
   int ledge)
   {
   RECT   card_rect;
   HBRUSH hGreenBrush;
   HPEN   hLtGreenPen, hOldPen;
   int    pen_width, save_card_left, diameter, add;

   pen_width = ball_diam / 4;

   get_reverse_card_coord(rect, ball_diam, ledge, &card_rect);

   hGreenBrush = CreateSolidBrush(DK_GREEN);
   hLtGreenPen = CreatePen(PS_SOLID, pen_width, GREEN);

   SelectObject(hdc, hGreenBrush);

   RoundRect(hdc, card_rect.left, card_rect.top, card_rect.right,
      card_rect.bottom, ball_diam / 2, ball_diam / 2);

   SelectObject(hdc, GetStockObject(NULL_BRUSH));

   DeleteObject(hGreenBrush);

   hOldPen = SelectObject(hdc, hLtGreenPen);

   save_card_left = card_rect.left;

   card_rect.left += ((card_rect.right - save_card_left) / 7);
   card_rect.right -= ((card_rect.right - save_card_left) / 7);

   diameter = (card_rect.right - card_rect.left);
   add = (((card_rect.bottom - card_rect.top) - diameter) / 2);

   card_rect.top += add;
   card_rect.bottom -= add;

   Ellipse(hdc, card_rect.left, card_rect.top, card_rect.right,
      card_rect.bottom);

   SelectObject(hdc, hOldPen);

   DeleteObject(hLtGreenPen);
   }

