/****************************************************************************\
*                                                                            *
* Figure 6                                                                   *
*                                                                            *
* WinSets,   Guy Eddon, 1993                                                 *
*                                                                            *
* MESSAGES.C - This module contains all of the messages handler functions.   *
*                                                                            *
\****************************************************************************/

#include "winsets.h"

// Handles WM_TIMER
BOOL winsets_timer(HWND hwnd, RECT *rect, int *time_left)
   {
   char menu_buffer[NORMAL_BUFFER] = { 0 };
   HDC hdc;

   if(IsIconic(hwnd) == FALSE)
      {
      if(*time_left == -1)
         return 0;

      hdc = GetDC(hwnd);
      fill_status_bar(hdc, *time_left, rect, status_bar_height, FALSE,
         TIME_BOX);
      ReleaseDC(hwnd, hdc);

      (*time_left)--;
      }

   return 0;
   }

// Handles WM_CLOSE
BOOL winsets_close(HWND hwnd, HBRUSH hGrayBrush, HBRUSH hHolderBrush)
   {
   // Delete the ball bitmaps
   DeleteSprite(&ball_pink);
   DeleteSprite(&ball_green);
   DeleteSprite(&ball_blue);
   DeleteSprite(&ball_red);
   DeleteSprite(&ball_yellow);

   // Release the timer
   KillTimer(hwnd, (UINT)NULL);

   // Delete some other bitmaps
   DeleteObject(hGrayBrush);
   DeleteObject(hHolderBrush);

   // Shut down the window
   DestroyWindow(hwnd);

   return 0;
   }

// Handles WM_DESTROY
BOOL winsets_destroy(HWND hwnd)
   {
   // Close WinHelp
   WinHelp(hwnd, NULL, HELP_QUIT, 0L);

   // Delete background brush
#ifdef WIN32
   DeleteObject((HBRUSH)SetClassWord(hwnd, GCL_HBRBACKGROUND,
      (WORD)GetStockObject(WHITE_BRUSH)));
#else
   DeleteObject(SetClassWord(hwnd, GCW_HBRBACKGROUND,
      GetStockObject(WHITE_BRUSH)));
#endif

   // Exit message loop in WinMain
   PostQuitMessage(0);

   return 0;
   }

// Handles WM_SETFOCUS
BOOL winsets_setfocus(void)
   {
   ShowCursor(TRUE);
   return 0;
   }

// Handles WM_KILLFOCUS
BOOL winsets_killfocus(void)
   {
   ShowCursor(FALSE);
   return 0;
   }

// Handles WM_KEYDOWN
BOOL winsets_keydown(HWND hwnd, WPARAM wParam, int ball_rad, unsigned radius,
   int num_circles, int ledge, float x_ellipse_factor,
   float y_ellipse_factor, RECT *rect, int ball_diam, char selected_card)
   {
   POINT point;
   RECT card_rect;
   BOOL change = FALSE;
   static int previous_message = WM_LBUTTONUP;
   static int current_hole = -1;
   static int current_card = -1;

   GetCursorPos(&point);
   ScreenToClient(hwnd, &point);

   switch(wParam)
      {
      case VK_UP:
         current_card = -1;
         if(current_hole == -1)
            current_hole = 0;
         else
            if(current_hole > 0)
               current_hole--;
            else
               current_hole = num_circles-1;
         get_hole_coord(current_hole, ball_rad, radius, num_circles, ledge,
            x_ellipse_factor, y_ellipse_factor, &point);
         transform_coord(&point, radius, x_ellipse_factor, y_ellipse_factor,
            ball_rad);
         change = TRUE;
         break;

      case VK_DOWN:
         current_card = -1;
         if(current_hole < num_circles-1)
            current_hole++;
         else
            current_hole = 0;
         get_hole_coord(current_hole, ball_rad, radius, num_circles, ledge,
            x_ellipse_factor, y_ellipse_factor, &point);
         transform_coord(&point, radius, x_ellipse_factor, y_ellipse_factor,
            ball_rad);
         change = TRUE;
         break;

      case VK_LEFT:
         current_hole = -1;
         if(current_card < NUM_CARDS)
            current_card++;
         else
            current_card = 0;
         if(current_card == 0)
            get_reverse_card_coord(rect, ball_diam, ledge, &card_rect);
         else
            get_card_coord(current_card, rect, ball_diam, ledge, &card_rect);
         point.x = (card_rect.left + card_rect.right) / 2;
         point.y = (card_rect.top + card_rect.bottom) / 2;
         change = TRUE;
         break;

      case VK_RIGHT:
         current_hole = -1;
         if(current_card == -1)
            current_card = 0;
         else
            if(current_card > 0)
               current_card--;
            else
               current_card = NUM_CARDS;
         if(current_card == 0)
            get_reverse_card_coord(rect, ball_diam, ledge, &card_rect);
         else
            get_card_coord(current_card, rect, ball_diam, ledge, &card_rect);
         point.x = (card_rect.left + card_rect.right) / 2;
         point.y = (card_rect.top + card_rect.bottom) / 2;
         change = TRUE;
         break;

      case VK_ESCAPE:
         if(num_circles == 10)
            if(ball_pink.bSelected)
               {
               FinishSpriteAnimation(hwnd, NO_HOLE, ball_rad, radius,
                  num_circles, ledge, &ball_pink, &point, x_ellipse_factor,
                  y_ellipse_factor, rect, FALSE, FALSE);
               ball_pink.bSelected = FALSE;
               previous_message = WM_LBUTTONUP;
               }

         if(ball_blue.bSelected)
            {
            FinishSpriteAnimation(hwnd, NO_HOLE, ball_rad, radius,
               num_circles, ledge, &ball_blue, &point, x_ellipse_factor,
               y_ellipse_factor, rect, FALSE, FALSE);
            ball_blue.bSelected = FALSE;
            previous_message = WM_LBUTTONUP;
            }

         if(ball_green.bSelected)
            {
            FinishSpriteAnimation(hwnd, NO_HOLE, ball_rad, radius,
               num_circles, ledge, &ball_green, &point, x_ellipse_factor,
               y_ellipse_factor, rect, FALSE, FALSE);
            ball_green.bSelected = FALSE;
            previous_message = WM_LBUTTONUP;
            }

         if(ball_red.bSelected)
            {
            FinishSpriteAnimation(hwnd, NO_HOLE, ball_rad, radius,
               num_circles, ledge, &ball_red, &point, x_ellipse_factor,
               y_ellipse_factor, rect, FALSE, FALSE);
            ball_red.bSelected = FALSE;
            previous_message = WM_LBUTTONUP;
            }

         if(ball_yellow.bSelected)
            {
            FinishSpriteAnimation(hwnd, NO_HOLE, ball_rad, radius,
               num_circles, ledge, &ball_yellow, &point, x_ellipse_factor,
               y_ellipse_factor, rect, FALSE, FALSE);
            ball_yellow.bSelected = FALSE;
            previous_message = WM_LBUTTONUP;
            }
         break;

      case VK_RETURN:
      case VK_SPACE:
         {
         int is_hole = 0, hole_number;

         for(hole_number = 0; hole_number < num_circles; hole_number++)
            if(is_point_in_hole(hwnd, hole_number, &point, ball_rad, radius,
               num_circles, ledge, x_ellipse_factor, y_ellipse_factor) != 0
               && is_a_ball_in_this_hole(hole_number, num_circles)
               && selected_card != (char)NULL)
               {
               is_hole = 1;
               break;
               }

         if(is_hole == 1 || previous_message == WM_LBUTTONDOWN)
            {
            if(previous_message == WM_LBUTTONDOWN)
               {
               SendMessage(hwnd, WM_LBUTTONUP, MK_LBUTTON,
                  MAKELONG(point.x, point.y));
               previous_message = WM_LBUTTONUP;
               }
            else
               {
               SendMessage(hwnd, WM_LBUTTONDOWN, MK_LBUTTON,
                  MAKELONG(point.x, point.y));
               previous_message = WM_LBUTTONDOWN;
               }
            }
         else
            {
            RECT new_rect;

            GetClientRect(hwnd, &new_rect);

            if(PTINRECT(&new_rect, point))
               {
               SendMessage(hwnd, WM_LBUTTONUP, MK_LBUTTON,
                  MAKELONG(point.x, point.y));
               SendMessage(hwnd, WM_LBUTTONDOWN, MK_LBUTTON,
                  MAKELONG(point.x, point.y));
               }
            }
         }
         break;
      }

   if(change == TRUE)
      {
      ClientToScreen(hwnd, &point);
      SetCursorPos(point.x, point.y);
      }

   return 0;
   }

// Handles WM_LBUTTONUP
BOOL winsets_lbuttonup(HWND hwnd, LPARAM lParam, int *num_moves_left,
   int num_circles, int ball_rad, int radius, int ledge, char *selected_card,
   RECT *rect, float x_ellipse_factor, float y_ellipse_factor,
   current_cards *visible, int ball_diam, HINSTANCE ghInst,
   char *global_card_array_count, char *card_array, int *current_score,
   BOOL *has_box_been_up, HMENU hMainMenu,
   BOOL *sizing, BOOL *paint_cards, int *time_left, BOOL *new_game_flag,
   BOOL *paint_reverse_card, HBRUSH hGrayBrush)
   {
   FARPROC lpfnScoreDlgProc;
   POINT point;
   int old_num_moves_left = *num_moves_left;

   if(*num_moves_left == 0)
      return 0;

   point.x = (signed short)LOWORD(lParam);
   point.y = (signed short)HIWORD(lParam);

   if(num_circles == 10)
      if(ball_pink.bSelected)
         {
         ball_pink.bSelected = FALSE;
         if(drop_ball(hwnd, &point, ball_rad, radius, num_circles, ledge,
            &ball_pink, selected_card, visible, rect, ball_diam,
            x_ellipse_factor, y_ellipse_factor, current_score,
            *num_moves_left, hMainMenu))
            (*num_moves_left)--;
         }

   if(ball_green.bSelected)
      {
      ball_green.bSelected = FALSE;
      if(drop_ball(hwnd, &point, ball_rad, radius, num_circles, ledge,
         &ball_green, selected_card, visible, rect, ball_diam,
         x_ellipse_factor, y_ellipse_factor, current_score, *num_moves_left,
         hMainMenu))
         (*num_moves_left)--;
      }

   if(ball_red.bSelected)
      {
      ball_red.bSelected = FALSE;
      if(drop_ball(hwnd, &point, ball_rad, radius, num_circles, ledge,
         &ball_red, selected_card, visible, rect, ball_diam,
         x_ellipse_factor, y_ellipse_factor, current_score, *num_moves_left,
         hMainMenu))
         (*num_moves_left)--;
      }

   if(ball_yellow.bSelected)
      {
      ball_yellow.bSelected = FALSE;
      if(drop_ball(hwnd, &point, ball_rad, radius, num_circles, ledge,
         &ball_yellow, selected_card, visible, rect, ball_diam,
         x_ellipse_factor, y_ellipse_factor, current_score, *num_moves_left,
         hMainMenu))
         (*num_moves_left)--;
      }

   if(ball_blue.bSelected)
      {
      ball_blue.bSelected = FALSE;
      if(drop_ball(hwnd, &point, ball_rad, radius, num_circles, ledge,
         &ball_blue, selected_card, visible, rect, ball_diam,
         x_ellipse_factor, y_ellipse_factor, current_score, *num_moves_left,
         hMainMenu))
         (*num_moves_left)--;
      }

   if(((visible->one_card_finished == TRUE
      && visible->two_card_finished == TRUE
      && visible->thr_card_finished == TRUE)
      || *num_moves_left == 0) && *has_box_been_up == FALSE)
      {
      HDC hdc;

      hdc = GetDC(hwnd);

      display_moves_left(hdc, rect, *num_moves_left, FALSE, ghInst);

      ReleaseDC(hwnd, hdc);

      EnableMenuItem(hMainMenu, IDM_UNDO, MF_GRAYED);

      draw_card_unhighlight(hwnd, rect, *selected_card, ball_diam, ledge);

      *selected_card = (char)NULL;

      lpfnScoreDlgProc = MakeProcInstance(ScoreDlgProc, ghInst);
      DialogBox(ghInst, "ScoreBox", hwnd, lpfnScoreDlgProc);
      FreeProcInstance(lpfnScoreDlgProc);

      *has_box_been_up = TRUE;

      hdc = GetDC(hwnd);

      *paint_reverse_card = TRUE;
      draw_reverse_card(hdc, rect, ball_diam, ledge, hGrayBrush);

      ReleaseDC(hwnd, hdc);

      if(num_circles == 10
         && *global_card_array_count >= FIVE_CARD_GAME - NUM_CARDS)
         {
         char buffer[NORMAL_BUFFER];

         wsprintf(buffer, "Game Over - Final Score = %d.", *current_score);
         MessageBeep(MB_ICONEXCLAMATION);
         MessageBox(hwnd, buffer, szAppName, MB_OK|MB_ICONEXCLAMATION);
         winsets_command_new(hwnd, num_circles, selected_card, ball_rad,
            radius, ledge, visible, x_ellipse_factor, y_ellipse_factor,
            num_moves_left, sizing, global_card_array_count, card_array,
            paint_cards, time_left, current_score, new_game_flag, hMainMenu,
            paint_reverse_card);
         }

      if(num_circles == 8
         && *global_card_array_count >= FOUR_CARD_GAME - NUM_CARDS)
         {
         char buffer[NORMAL_BUFFER];

         wsprintf(buffer, "Game Over - Final Score = %d.", *current_score);
         MessageBeep(MB_ICONEXCLAMATION);
         MessageBox(hwnd, buffer, szAppName, MB_OK|MB_ICONEXCLAMATION);
         winsets_command_new(hwnd, num_circles, selected_card, ball_rad,
            radius, ledge, visible, x_ellipse_factor, y_ellipse_factor,
            num_moves_left, sizing, global_card_array_count, card_array,
            paint_cards, time_left, current_score, new_game_flag, hMainMenu,
            paint_reverse_card);
         }
      }

   if(old_num_moves_left != *num_moves_left)
      {
      HDC hdc;

      hdc = GetDC(hwnd);

      display_moves_left(hdc, rect, *num_moves_left, FALSE, ghInst);

      ReleaseDC(hwnd, hdc);
      }

   return 0;
   }

// Handles WM_MOUSEMOVE
BOOL winsets_mousemove(HWND hwnd, LPARAM lParam, RECT *rect, int num_circles)
   {
   int x, y;

   x = (signed short)LOWORD(lParam);
   y = (signed short)HIWORD(lParam);

   if(num_circles == 10)
      if(ball_pink.bSelected)
         MoveSprite(hwnd, x, y, &ball_pink, rect);

   if(ball_green.bSelected)
      MoveSprite(hwnd, x, y, &ball_green, rect);

   if(ball_red.bSelected)
      MoveSprite(hwnd, x, y, &ball_red, rect);

   if(ball_yellow.bSelected)
      MoveSprite(hwnd, x, y, &ball_yellow, rect);

   if(ball_blue.bSelected)
      MoveSprite(hwnd, x, y, &ball_blue, rect);

   return 0;
   }

// Handles WM_LBUTTONDOWN
BOOL winsets_lbuttondown(HWND hwnd, LPARAM lParam, RECT *rect, int ball_diam,
   int ledge, BOOL *paint_cards, int num_circles, int *num_moves_left,
   current_cards *visible, char *selected_card, HINSTANCE ghInst,
   char *global_card_array_count, char *card_array, int *current_score,
   int *time_left, BOOL *has_box_been_up, int ball_rad, unsigned radius,
   float x_ellipse_factor, float y_ellipse_factor, BOOL *new_game_flag,
   HMENU hMainMenu, BOOL *paint_reverse_card)
   {
   RECT card_rect;
   int count;

   if(((visible->one_card_finished == TRUE
      && visible->two_card_finished == TRUE
      && visible->thr_card_finished == TRUE)
      || *num_moves_left == 0)
      && clicked_on_reverse_card(LOWORD(lParam), HIWORD(lParam), ball_diam,
         ledge, rect))
      {
      HDC hdc;

      hdc = GetDC(hwnd);

      *paint_cards = TRUE;

      *has_box_been_up = FALSE;

      *time_left = SECONDS_TO_FINISH;

      winsets_timer(hwnd, rect, time_left);

      if(num_circles == 8)
         *num_moves_left = 3;
      if(num_circles == 10)
         *num_moves_left = 4;

      *paint_reverse_card = FALSE;

      draw_reverse_card_inverted(hdc, rect, ball_diam, ledge);
         
      if(*new_game_flag == TRUE)
         {
         if(num_circles == 10)
            AnimateSprite(hwnd, &ball_pink, ball_rad, radius, num_circles,
               ledge, x_ellipse_factor, y_ellipse_factor, rect);

         AnimateSprite(hwnd, &ball_blue, ball_rad, radius, num_circles,
            ledge, x_ellipse_factor, y_ellipse_factor, rect);
         AnimateSprite(hwnd, &ball_red, ball_rad, radius, num_circles, ledge,
            x_ellipse_factor, y_ellipse_factor, rect);
         AnimateSprite(hwnd, &ball_yellow, ball_rad, radius, num_circles,
            ledge, x_ellipse_factor, y_ellipse_factor, rect);
         AnimateSprite(hwnd, &ball_green, ball_rad, radius, num_circles,
            ledge, x_ellipse_factor, y_ellipse_factor, rect);

         *new_game_flag = FALSE;
         }

      KillTimer(hwnd, (UINT)NULL);
      SetTimer(hwnd, (UINT)NULL, 1000, NULL);

      if(num_circles == 10)
         {
         if(visible->one_card_finished == TRUE)
            {
            visible->one_card = get_random_card(FIVE_CARD_GAME,
               global_card_array_count, card_array);
            draw_card(hdc, rect, ball_diam, ledge, 1);
            draw_card_configuration(NULL, hdc, 1,
               win_five[visible->one_card], rect, ball_diam, ledge,
               num_circles);
            }
         if(visible->two_card_finished == TRUE)
            {
            visible->two_card = get_random_card(FIVE_CARD_GAME,
               global_card_array_count, card_array);
            draw_card(hdc, rect, ball_diam, ledge, 2);
            draw_card_configuration(NULL, hdc, 2,
               win_five[visible->two_card], rect, ball_diam, ledge,
               num_circles);
            }
         if(visible->thr_card_finished == TRUE)
            {
            visible->thr_card = get_random_card(FIVE_CARD_GAME,
               global_card_array_count, card_array);
            draw_card(hdc, rect, ball_diam, ledge, 3);
            draw_card_configuration(NULL, hdc, 3,
               win_five[visible->thr_card], rect, ball_diam, ledge,
               num_circles);
            }
         }

      if(num_circles == 8)
         {
         if(visible->one_card_finished == TRUE)
            {
            visible->one_card = get_random_card(FOUR_CARD_GAME,
               global_card_array_count, card_array);
            draw_card(hdc, rect, ball_diam, ledge, 1);
            draw_card_configuration(NULL, hdc, 1,
               win_four[visible->one_card], rect, ball_diam, ledge,
               num_circles);
            }
         if(visible->two_card_finished == TRUE)
            {
            visible->two_card = get_random_card(FOUR_CARD_GAME,
               global_card_array_count, card_array);
            draw_card(hdc, rect, ball_diam, ledge, 2);
            draw_card_configuration(NULL, hdc, 2,
               win_four[visible->two_card], rect, ball_diam, ledge,
               num_circles);
            }
         if(visible->thr_card_finished == TRUE)
            {
            visible->thr_card = get_random_card(FOUR_CARD_GAME,
               global_card_array_count, card_array);
            draw_card(hdc, rect, ball_diam, ledge, 3);
            draw_card_configuration(NULL, hdc, 3,
               win_four[visible->thr_card], rect, ball_diam, ledge,
               num_circles);
            }
         }

      visible->one_card_finished = FALSE;
      visible->two_card_finished = FALSE;
      visible->thr_card_finished = FALSE;

      display_moves_left(hdc, rect, *num_moves_left, FALSE, ghInst);

      ReleaseDC(hwnd, hdc);

      return 0;
      }

   if(*num_moves_left == 0)
      return 0;

   for(count = 1; count <= NUM_CARDS; count++)
      {
      switch(count)
         {
         case 1:
            if(visible->one_card_finished == TRUE)
               continue;
            break;

         case 2:
            if(visible->two_card_finished == TRUE)
               continue;
            break;

         case 3:
            if(visible->thr_card_finished == TRUE)
               continue;
            break;
         }

      get_card_coord(count, rect, ball_diam, ledge, &card_rect);

      if(is_card_selected(LOWORD(lParam), HIWORD(lParam), &card_rect) != 0)
         {
         if(count == *selected_card)
            {
            draw_card_unhighlight(hwnd, rect, count, ball_diam, ledge);
            *selected_card = (char)NULL;
            }
         else
            {
            if(*selected_card != (char)NULL)
               {
               draw_card_unhighlight(hwnd, rect, *selected_card, ball_diam,
                  ledge);
               }

             /* or TRUE for inverted */
            draw_card_highlight(hwnd, count, rect, ball_diam, ledge);
            *selected_card = (char)count;
            do_card_complete(hwnd, selected_card, num_circles, visible, rect,
               ball_diam, ledge, current_score, *num_moves_left, hMainMenu);
            }

         return 0;
         }
      }

   if(*selected_card == (char)NULL)
      return 0;

   if(*selected_card != (char)NULL)
      {
      if(num_circles == 10)
         if(ball_pink.bSelected = IsSpriteSelected(LOWORD(lParam),
            HIWORD(lParam), &ball_pink, ball_rad))
            {
            BeginSpriteAnimation(hwnd, LOWORD(lParam), HIWORD(lParam),
               &ball_pink);
            return 0;
            }

      if(ball_green.bSelected = IsSpriteSelected(LOWORD(lParam),
         HIWORD(lParam), &ball_green, ball_rad))
         { 
         BeginSpriteAnimation(hwnd, LOWORD(lParam), HIWORD(lParam),
            &ball_green);
         return 0;
         }

      if(ball_red.bSelected = IsSpriteSelected(LOWORD(lParam),
         HIWORD(lParam), &ball_red, ball_rad))
         {
         BeginSpriteAnimation(hwnd, LOWORD(lParam), HIWORD(lParam),
            &ball_red);
         return 0;
         }

      if(ball_yellow.bSelected = IsSpriteSelected(LOWORD(lParam),
         HIWORD(lParam), &ball_yellow, ball_rad))
         {
         BeginSpriteAnimation(hwnd, LOWORD(lParam), HIWORD(lParam),
            &ball_yellow);
         return 0;
         }

      if(ball_blue.bSelected = IsSpriteSelected(LOWORD(lParam),
         HIWORD(lParam), &ball_blue, ball_rad))
         {
         BeginSpriteAnimation(hwnd, LOWORD(lParam), HIWORD(lParam),
            &ball_blue);
         return 0;
         }
      }

   return 0;
   }

// Handles WM_PAINT
BOOL winsets_paint(HWND hwnd, RECT *rect, int ball_diam, int ledge,
   HBRUSH hGrayBrush, int num_moves_left, float x_ellipse_factor,
   float y_ellipse_factor, int num_circles, current_cards *visible,
   int radius, int ball_rad, char selected_card, BOOL *sizing,
   HINSTANCE ghInst, int time_left, int current_score, BOOL paint_cards,
   HBRUSH hHolderBrush, BOOL new_game_flag, BOOL paint_reverse_card)
   {
   PAINTSTRUCT ps;
   HRGN hInsideRgn, hOutsideRgn, hCombinedRgn;
   HDC hdc;

   hdc = BeginPaint(hwnd, &ps);

   if(status_bar_flag)
      {
      display_moves_left(hdc, rect, num_moves_left, TRUE, ghInst);

      fill_status_bar(hdc, ((time_left < 0) ? 0 : time_left), rect,
         status_bar_height, FALSE, TIME_BOX);
      fill_status_bar(hdc, current_score, rect, status_bar_height, FALSE,
         SCORE_BOX);
      }

   hOutsideRgn = CreateEllipticRgn(ledge, ledge, (int)((float)(radius * 2)
      * (x_ellipse_factor)) - ledge, (int)((float)(radius * 2)
      * (y_ellipse_factor+PERSPECTIVE_SHADING)) - ledge);

   // Sophisticated window repainting in action!
   if(RectInRegion(hOutsideRgn, &ps.rcPaint))
      {
      hInsideRgn = CreateEllipticRgn(ledge, ledge,
         (int)((float)(radius * 2) * x_ellipse_factor) - ledge,
         (int)((float)(radius * 2) * y_ellipse_factor) - ledge);
      hCombinedRgn = CreateRectRgnIndirect(rect);
      CombineRgn(hCombinedRgn, hOutsideRgn, hInsideRgn, RGN_DIFF);
      FillRgn(hdc, hCombinedRgn, GetStockObject(GRAY_BRUSH));

      DeleteObject(hInsideRgn);
      DeleteObject(hOutsideRgn);
      DeleteObject(hCombinedRgn);

      SelectObject(hdc, GetStockObject(NULL_BRUSH));
      Ellipse(hdc, ledge, ledge, (int)((float)(radius * 2)
         * (x_ellipse_factor)) - ledge, (int)((float)(radius * 2)
         * (y_ellipse_factor+PERSPECTIVE_SHADING)) - ledge);

      SelectObject(hdc, GetStockObject(LTGRAY_BRUSH));
      Ellipse(hdc, ledge, ledge, (int)((float)(radius * 2)
         * x_ellipse_factor) - ledge, (int)((float)(radius * 2)
         * y_ellipse_factor) - ledge);
      draw_holes(hdc, GetStockObject(LTGRAY_BRUSH), num_circles,
         ball_rad, radius, ledge, x_ellipse_factor, y_ellipse_factor);
      }
   else
      DeleteObject(hOutsideRgn);

   SaveDC(hdc);

   if(*sizing == TRUE)
      { 
      if(num_circles == 10)
         GetSprite(hdc, &ball_pink);

      GetSprite(hdc, &ball_blue);
      GetSprite(hdc, &ball_green);
      GetSprite(hdc, &ball_red);
      GetSprite(hdc, &ball_yellow);
      *sizing = FALSE;
      }

   if(new_game_flag == FALSE)
      {
      if(num_circles == 10)
         DrawSprite(hdc, &ball_pink);

      DrawSprite(hdc, &ball_blue);
      DrawSprite(hdc, &ball_green);
      DrawSprite(hdc, &ball_red);
      DrawSprite(hdc, &ball_yellow);
      }

   RestoreDC(hdc, -1);

   if(paint_reverse_card == TRUE)
      draw_reverse_card(hdc, rect, ball_diam, ledge, hGrayBrush);
   else
      draw_reverse_card_inverted(hdc, rect, ball_diam, ledge);

   if(paint_cards)
      {
      draw_cards(hdc, rect, ball_diam, ledge, NULL);

      if(num_circles == 8)
         {
         draw_card_configuration(&ps, hdc, 1, win_four[visible->one_card],
            rect, ball_diam, ledge, num_circles);
         draw_card_configuration(&ps, hdc, 2, win_four[visible->two_card],
            rect, ball_diam, ledge, num_circles);
         draw_card_configuration(&ps, hdc, 3, win_four[visible->thr_card],
            rect, ball_diam, ledge, num_circles);
         }

      if(num_circles == 10)
         {
         draw_card_configuration(&ps, hdc, 1, win_five[visible->one_card],
            rect, ball_diam, ledge, num_circles);
         draw_card_configuration(&ps, hdc, 2, win_five[visible->two_card],
            rect, ball_diam, ledge, num_circles);
         draw_card_configuration(&ps, hdc, 3, win_five[visible->thr_card],
            rect, ball_diam, ledge, num_circles);
         }

      draw_card_highlight(hwnd, selected_card, rect, ball_diam, ledge);

      if(visible->one_card_finished == TRUE)
         draw_card_finished(hwnd, rect, 1, ball_diam, ledge, visible);
      if(visible->two_card_finished == TRUE)
         draw_card_finished(hwnd, rect, 2, ball_diam, ledge, visible);
      if(visible->thr_card_finished == TRUE)
         draw_card_finished(hwnd, rect, 3, ball_diam, ledge, visible);
      }
   else
      draw_cards(hdc, rect, ball_diam, ledge, hHolderBrush);

   EndPaint(hwnd, &ps);

   return 0;
   }

// Handles WM_COMMAND
BOOL winsets_command(HWND hwnd, WPARAM wParam, char *selected_card,
   int *num_circles, int ball_rad, int radius, int ledge,
   float *x_ellipse_factor, float *y_ellipse_factor, current_cards *visible,
   BOOL *paint_cards, int *num_moves_left, BOOL *sizing, HMENU hMainMenu,
   HMENU hOptionsMenu, HINSTANCE ghInst, char *global_card_array_count,
   char *card_array, int *time_left, int *current_score, BOOL *new_game_flag,
   BOOL *sound_flag, RECT *win_rect, BOOL *paint_reverse_card)
   {

#ifdef WIN32
   wParam = LOWORD(wParam);
#endif

   switch(wParam)
      {
      case IDM_NEW:
         winsets_command_new(hwnd, *num_circles, selected_card, ball_rad,
            radius, ledge, visible, *x_ellipse_factor, *y_ellipse_factor,
            num_moves_left, sizing, global_card_array_count, card_array,
            paint_cards, time_left, current_score, new_game_flag, hMainMenu,
            paint_reverse_card);
         break;

      case IDM_UNDO:
         winsets_command_undo(hwnd, *num_circles, ball_rad, radius, ledge,
            *x_ellipse_factor, *y_ellipse_factor, num_moves_left, ghInst,
            hMainMenu, win_rect);
         break;

      case IDM_FOUR:
         winsets_command_four(hwnd, num_circles, selected_card, ball_rad,
            radius, ledge, visible, *x_ellipse_factor, *y_ellipse_factor,
            num_moves_left, sizing, paint_cards, hMainMenu, hOptionsMenu,
            global_card_array_count, card_array, time_left, current_score,
            new_game_flag, paint_reverse_card);
         break;

      case IDM_FIVE:
         winsets_command_five(hwnd, num_circles, selected_card, ball_rad,
            radius, ledge, visible, *x_ellipse_factor, *y_ellipse_factor,
            num_moves_left, sizing, paint_cards, hMainMenu, hOptionsMenu,
            global_card_array_count, card_array, time_left, current_score,
            new_game_flag, paint_reverse_card);
         break;

      case IDM_EXIT:
         SendMessage(hwnd, WM_CLOSE, 0, 0L);
         break;

      case IDM_SOUND:
         if(*sound_flag == TRUE)
            {
            CheckMenuItem(hMainMenu, IDM_SOUND, MF_UNCHECKED);
            *sound_flag = FALSE;
            WritePrivateProfileString(szAppName, "Sound", "0", fqfn);
            }
         else
            {
            CheckMenuItem(hMainMenu, IDM_SOUND, MF_CHECKED);
            *sound_flag = TRUE;
            WritePrivateProfileString(szAppName, "Sound", "1", fqfn);
            }
         break;


      case IDM_STATUS_BAR:
         if(status_bar_flag == TRUE)
            {
            CheckMenuItem(hMainMenu, IDM_STATUS_BAR, MF_UNCHECKED);
            status_bar_flag = FALSE;
            WritePrivateProfileString(szAppName, "StatusBar", "0", fqfn);
            }
         else
            {
            CheckMenuItem(hMainMenu, IDM_STATUS_BAR, MF_CHECKED);
            status_bar_flag = TRUE;
            WritePrivateProfileString(szAppName, "StatusBar", "1", fqfn);
            }
         SendMessage(hwnd, WM_SIZE, (WPARAM)NULL, (LPARAM)NULL);
         InvalidateRect(hwnd, NULL, TRUE);
         UpdateWindow(hwnd);
         break;

      case IDM_CONTENTS:
         WinHelp(hwnd, "WINSETS.HLP", HELP_CONTENTS, 0L);
         break;

      case IDM_SEARCH:
         WinHelp(hwnd, "WINSETS.HLP", HELP_PARTIALKEY, (DWORD)(LPSTR)"");
         break;

      case IDM_HELP_ON_HELP:
         WinHelp(hwnd, NULL, HELP_HELPONHELP, 0L);
         break;

      case IDM_ABOUT:
         {
         FARPROC lpfnAboutDlgProc;

         lpfnAboutDlgProc = MakeProcInstance(AboutDlgProc, ghInst);
         DialogBox(ghInst, "AboutBox", hwnd, lpfnAboutDlgProc);
         FreeProcInstance(lpfnAboutDlgProc);
         break;
         }
      }

   return 0;
   }

// Handles WM_SIZE
BOOL winsets_size(HWND hwnd, RECT *rect, BOOL *sizing, unsigned *radius,
   int *ledge, int *ball_rad, int *ball_diam, int num_circles,
   float x_ellipse_factor, float y_ellipse_factor)
   {
   HDC hdc;
   HRGN hTestRgn;
   RECT card_rect, reverse_card;

   *sizing = TRUE;

   GetClientRect(hwnd, rect);

   // Make sure we have enough room to draw a status bar if the user wants...
   if(status_bar_flag)
      rect->bottom -= status_bar_height;

   // Used to scale all the objects in the entire window
   *radius = min(((rect->right - rect->left) / 2),
      ((rect->bottom - rect->top) / 2));

   // Test to make sure radius is not going to be too big...
TryAgain:

   *ledge = (*radius) / 30;
   *ball_rad   = (*radius) / 14;
   *ball_diam  = (*radius) / 7;

   hTestRgn = CreateEllipticRgn(*ledge, *ledge,
      (int)((float)(*radius * 2) * (x_ellipse_factor)) - *ledge,
      (int)((float)(*radius * 2) * (y_ellipse_factor+PERSPECTIVE_SHADING))
      - *ledge);

   get_card_coord(3, rect, *ball_diam, *ledge, &card_rect);
   get_reverse_card_coord(rect, *ball_diam, *ledge, &reverse_card);

   if(PtInRegion(hTestRgn, card_rect.left-5, card_rect.top-5)
      || PtInRegion(hTestRgn, reverse_card.left-2, reverse_card.top-2))
      {
      DeleteObject(hTestRgn);
      (*radius) -= 4; // Determines how quickly window resizes
      goto TryAgain;
      }

   DeleteObject(hTestRgn);

   hdc = GetDC(hwnd);

   if(num_circles == 10)
      InitSpriteInfo(hdc, ball_pink.hbmImg_color, *ball_rad, *radius,
         num_circles, *ledge, &ball_pink, x_ellipse_factor,
         y_ellipse_factor);

   InitSpriteInfo(hdc, ball_green.hbmImg_color, *ball_rad, *radius,
      num_circles, *ledge, &ball_green, x_ellipse_factor, y_ellipse_factor);
   InitSpriteInfo(hdc, ball_red.hbmImg_color, *ball_rad, *radius,
      num_circles, *ledge, &ball_red, x_ellipse_factor, y_ellipse_factor);
   InitSpriteInfo(hdc, ball_yellow.hbmImg_color, *ball_rad, *radius,
      num_circles, *ledge, &ball_yellow, x_ellipse_factor, y_ellipse_factor);
   InitSpriteInfo(hdc, ball_blue.hbmImg_color, *ball_rad, *radius,
      num_circles, *ledge, &ball_blue, x_ellipse_factor, y_ellipse_factor);

   ReleaseDC(hwnd, hdc);

   return 0;
   }

// Handles WM_CREATE
BOOL winsets_create(HWND hwnd, int *num_circles, current_cards *visible,
   HBRUSH *hGrayBrush, HINSTANCE ghInst, char *global_card_array_count,
   char *card_array, HBRUSH *hHolderBrush,
   float *x_ellipse_factor, float *y_ellipse_factor, BOOL *sound_flag,
   int *status_bar_height, BOOL *status_bar_flag)
   {
   TEXTMETRIC tm;
   HBITMAP hGrayBitmap, hHolderBitmap;
   HDC hdc;

   /* Get a random seed based on the clock second */
   random_seed();

    /* Throw away first random value              */
   (void)random_number(DUMMY);

   visible->one_card_finished = TRUE;
   visible->two_card_finished = TRUE;
   visible->thr_card_finished = TRUE;

   hdc = GetDC(hwnd);

   GetTextMetrics(hdc, &tm);

   *status_bar_height = tm.tmHeight+6; // kludge

   ReleaseDC(hwnd, hdc);

   *num_circles = GetPrivateProfileInt(szAppName, "Holes", 8, fqfn);
   *sound_flag = GetPrivateProfileInt(szAppName, "Sound", TRUE, fqfn);
   *status_bar_flag = GetPrivateProfileInt(szAppName, "StatusBar", TRUE,
      fqfn);

   *x_ellipse_factor = (float)1.0;
   *y_ellipse_factor = (float)0.8;

   if(*num_circles == 10)
      {
      CheckMenuItem(GetMenu(hwnd), IDM_FOUR, MF_UNCHECKED);
      CheckMenuItem(GetMenu(hwnd), IDM_FIVE, MF_CHECKED);
      }

   if(*sound_flag == FALSE)
      CheckMenuItem(GetMenu(hwnd), IDM_SOUND, MF_UNCHECKED);

   if(*status_bar_flag == FALSE)
      CheckMenuItem(GetMenu(hwnd), IDM_STATUS_BAR, MF_UNCHECKED);

   initialize_ball(&ball_green,  TRUE,  *num_circles);
   initialize_ball(&ball_blue,   FALSE, *num_circles);
   initialize_ball(&ball_yellow, FALSE, *num_circles);
   initialize_ball(&ball_red,    FALSE, *num_circles);
   initialize_ball(&ball_pink,   FALSE, *num_circles);

   ball_red.hbmImg_color    = LoadBitmap(ghInst, (LPSTR)"Object_Red");
   ball_yellow.hbmImg_color = LoadBitmap(ghInst, (LPSTR)"Object_Yellow");
   ball_blue.hbmImg_color   = LoadBitmap(ghInst, (LPSTR)"Object_Blue");
   ball_green.hbmImg_color  = LoadBitmap(ghInst, (LPSTR)"Object_Green");
   ball_pink.hbmImg_color   = LoadBitmap(ghInst, (LPSTR)"Object_Pink");

   hGrayBitmap = LoadBitmap(ghInst, (LPSTR)"Mono_Gray");
   *hGrayBrush = CreatePatternBrush(hGrayBitmap);
   DeleteObject(hGrayBitmap);

   hHolderBitmap = LoadBitmap(ghInst, (LPSTR)"Holder_Green");
   *hHolderBrush = CreatePatternBrush(hHolderBitmap);
   DeleteObject(hHolderBitmap);
   
   new_game(global_card_array_count, card_array);

   while(SetTimer(hwnd, (UINT)NULL, 1000, NULL) == 0)
      {
      MessageBeep(MB_ICONSTOP);
      if(MessageBox(hwnd,
         "No timers are currently available!  \
         Please close some other application and then try again.",
         szAppName, MB_ICONHAND|MB_RETRYCANCEL) == IDCANCEL)
         return -1;
      }

   return 0;
   }

// Handles WM_MENUSELECT
BOOL winsets_menuselect(HWND hwnd, LPARAM lParam, WPARAM wParam, RECT *rect,
   HMENU hGameMenu, HMENU hHelpMenu, HMENU hViewMenu, BOOL *paint_status_bar,
   int *nMenuID, int num_moves_left, HINSTANCE ghInst)
   {

#ifdef WIN32
   WORD flags = HIWORD(wParam);
   int item = (flags & MF_POPUP) ? 0 : LOWORD(wParam);
   HMENU hMenu = (HMENU)lParam;
   HMENU hMenuPopUp = (flags & MF_POPUP) ?
      GetSubMenu(hMenu, LOWORD(wParam)) : 0;
#else
   WORD flags = LOWORD(lParam);
   int item = (flags & MF_POPUP) ? 0 : (int)(wParam);
   HMENU hMenu = (HMENU)HIWORD(lParam);
   HMENU hMenuPopUp = (flags & MF_POPUP) ? (HMENU)(wParam) : 0;
#endif

   if(!status_bar_flag)
      return 0;

   *paint_status_bar = TRUE;

   if(flags == 0xFFFF && hMenu == NULL)
      {
      HDC hdc;
      char menu_buffer[NORMAL_BUFFER] = { 0 };

      *nMenuID = IDM_MOVES_LEFT;

      LoadString(ghInst, *nMenuID, menu_buffer, sizeof(menu_buffer));
      wsprintf(menu_buffer, menu_buffer, num_moves_left);

      hdc = GetDC(hwnd);

      draw_status_bar_text(hdc, menu_buffer, rect, status_bar_height, FALSE,
         FIRST_STATUS_BAR_BOX);

      ReleaseDC(hwnd, hdc);

      *paint_status_bar = FALSE;
      return 0;
      }
   else
      if(flags & MF_SYSMENU)
         if(!(flags & MF_POPUP))
            *nMenuID = item;
         else
            *nMenuID = IDM_SYSMENU;
      else
         if((flags & MF_POPUP))
            {   
            if(hMenuPopUp == hGameMenu)
               *nMenuID = IDM_GAME;
            else
               if(hMenuPopUp == hHelpMenu)
                  *nMenuID = IDM_HELP;
               else
                  if(hMenuPopUp == hViewMenu)
                     *nMenuID = IDM_OPTIONS;
            }
         else
            *nMenuID = item;
   return 0;
   }

// Handles WM_ENTERIDLE
BOOL winsets_enteridle(HWND hwnd, RECT *rect, int nMenuID,
   BOOL *paint_status_bar, HINSTANCE ghInst)
   {
   if(!status_bar_flag)
      return 0;

   if(*paint_status_bar == TRUE)
      {
      HDC hdc;
      char menu_buffer[NORMAL_BUFFER] = { 0 };

      hdc = GetDC(hwnd);

      LoadString(ghInst, nMenuID, menu_buffer, sizeof(menu_buffer));
      draw_status_bar_text(hdc, menu_buffer, rect, status_bar_height, FALSE,
         FIRST_STATUS_BAR_BOX);
      *paint_status_bar = FALSE;

      ReleaseDC(hwnd, hdc);
      }

   return 0;
   }

// Handles WM_GETMINMAXINFO
BOOL winsets_getminmaxinfo(LPARAM lParam)
   {
   MINMAXINFO FAR *lpmmi;

   lpmmi = (MINMAXINFO FAR *)lParam;
   lpmmi->ptMinTrackSize.x = X_MINIMUM_WINDOW_SIZE;
   lpmmi->ptMinTrackSize.y = Y_MINIMUM_WINDOW_SIZE;

   return 0;
   }

// Handles WM_INITDIALOG
BOOL winsets_initdialog(HWND hDlg)
   {
   RECT aRt, win_rect;
   POINT point;

   GetWindowRect(hDlg, &aRt);
   GetWindowRect(GetWindow(hDlg, GW_OWNER), &win_rect);
   OffsetRect(&aRt, -aRt.left, -aRt.top);

   point.x = ((((((win_rect.right - win_rect.left) - aRt.right) / 2)
      + win_rect.left) + 4) & ~7);
   point.y = ((((win_rect.bottom - win_rect.top) - aRt.bottom) / 2)
      + win_rect.top);

   MoveWindow(hDlg, point.x, point.y, aRt.right, aRt.bottom, 0);

   return TRUE;
   }

