/****************************************************************************\
*                                                                            *
* Figure 1                                                                   *
*                                                                            *
* WinSets,  Guy Eddon, 1993                                                  *
*                                                                            *
* SPRITES.C - This module handles all the sprite dragging and animation.     *
*                                                                            *
\****************************************************************************/

#include "winsets.h"

// Sprite's info structure
SPRITE ball_pink;
SPRITE ball_red;
SPRITE ball_green;
SPRITE ball_blue;
SPRITE ball_yellow;

void InitSpriteInfo(HDC hdc, HBITMAP hbm, int ball_rad, unsigned radius,
   int num_circles, int ledge, SPRITE *ball_color, float x_ellipse_factor,
   float y_ellipse_factor)
   {
   HDC     hdcMem, hdcBlit;  // Handles to window and memory dcs
   HBITMAP hbmPrev, hbmBlit; // Handles to bitmaps
   BITMAP  bm;               // BITMAP data structure
   POINT   pt;

   hdcMem = CreateCompatibleDC(hdc);
   hdcBlit = CreateCompatibleDC(hdc);

   // Get width and height of bitmap
   GetObject(hbm, sizeof(BITMAP), (LPSTR)&bm);

   get_hole_coord(ball_color->current_hole, ball_rad, radius, num_circles,
      ledge, x_ellipse_factor, y_ellipse_factor, &pt);

   ball_color->xPrev = pt.x;
   ball_color->yPrev = pt.y;

   // Initialize sprite's info and store rect for updating
   ball_color->bmX = (int)(radius*x_ellipse_factor)
      - ball_color->xPrev - ball_rad;
   ball_color->bmY = (int)((radius*y_ellipse_factor) - ball_color->yPrev
      - (PERSPECTIVE_BALL_VIEW * ball_rad));
   ball_color->bmWidth  = ball_rad * 2;
   ball_color->bmHeight = ball_rad * 2;

   if(ball_color->hbmImage)
      DeleteObject(ball_color->hbmImage);

   hbmBlit = CreateCompatibleBitmap(hdc, ball_rad * 2, ball_rad * 2);

   hbmPrev = SelectObject(hdcMem, hbmBlit);
   SelectObject(hdcBlit, hbm);

   if(ball_rad > 10)
#ifdef WIN32
      SetStretchBltMode(hdcMem, HALFTONE);
#else
      SetStretchBltMode(hdcMem, STRETCH_DELETESCANS);
#endif

   StretchBlt(hdcMem, 0, 0, ball_rad * 2, ball_rad * 2, hdcBlit, 0, 0,
      bm.bmWidth, bm.bmHeight, SRCCOPY);

   ball_color->hbmImage = hbmBlit;

   // Tidy up
   SelectObject(hdcMem, hbmPrev);
   SelectObject(hdcBlit, hbmPrev);
   DeleteDC(hdcMem);
   DeleteDC(hdcBlit);
   }

void DeleteSprite(SPRITE *ball_color)
   {
   if(ball_color->hbmImg_color)
      DeleteObject(ball_color->hbmImg_color);
   if(ball_color->hbmImage)
      DeleteObject(ball_color->hbmImage);
   if(ball_color->hbmBkg)
      DeleteObject(ball_color->hbmBkg);
   }

void AnimateSprite(HWND hWnd, SPRITE *ball_color, int ball_rad,
   unsigned radius, int num_circles, int ledge, float x_ellipse_factor,
   float y_ellipse_factor, RECT *win_rect)
   {
   POINT point = { -2*ball_rad, -2*ball_rad };

   ball_color->xPrev = ball_color->bmX;
   ball_color->yPrev = ball_color->bmY;

   FinishSpriteAnimation(hWnd, NO_HOLE, ball_rad, radius, num_circles, ledge,
      ball_color, &point, x_ellipse_factor, y_ellipse_factor, win_rect, TRUE,
      TRUE);
   }

void DrawSprite(HDC hdc, SPRITE *ball_color)
   {
   COLORREF rgbBk;
   HDC hdcMem, hdcCache, hdcMask;
   HBITMAP hbmPrev, hbmCache, hbmPrevCache, hbmMask, hbmPrevMask;

   hdcMem = CreateCompatibleDC(hdc);
   hdcMask = CreateCompatibleDC(hdc);
   hdcCache = CreateCompatibleDC(hdc);

   // Set masking color to black
   rgbBk = SetBkColor(hdcMem, RGB(0, 0, 0));

   // Create bitmaps
   hbmCache = CreateCompatibleBitmap(hdc, ball_color->bmWidth,
      ball_color->bmHeight);
   hbmMask = CreateCompatibleBitmap(hdcMask, ball_color->bmWidth,
      ball_color->bmHeight);

   // Select bitmaps
   hbmPrev = SelectObject(hdcMem, ball_color->hbmImage);
   hbmPrevCache = SelectObject(hdcCache, hbmCache);
   hbmPrevMask = SelectObject(hdcMask, hbmMask);

   // Create the bitmap mask
   BitBlt(hdcMask, 0, 0, ball_color->bmWidth, ball_color->bmHeight,
      hdcMem, 0, 0, SRCCOPY);

   // Draw the sprite in the window
   BitBlt(hdcCache, 0, 0, ball_color->bmWidth, ball_color->bmHeight, hdc,
      ball_color->bmX, ball_color->bmY, SRCCOPY);
   BitBlt(hdcCache, 0, 0, ball_color->bmWidth, ball_color->bmHeight, hdcMask,
      0, 0, SRCAND);
   BitBlt(hdcCache, 0, 0, ball_color->bmWidth, ball_color->bmHeight, hdcMem,
      0, 0, SRCPAINT);
   BitBlt(hdc, ball_color->bmX, ball_color->bmY, ball_color->bmWidth,
      ball_color->bmHeight, hdcCache, 0, 0, SRCCOPY);

   // Tidy up
   SelectObject(hdcMem, hbmPrev);
   SelectObject(hdcCache, hbmPrevCache);
   SelectObject(hdcMask, hbmPrevMask);
   SetBkColor(hdcMem, rgbBk);

   // Free resources
   DeleteObject(hbmMask);
   DeleteObject(hbmCache);

   // Delete DCs
   DeleteDC(hdcMem);
   DeleteDC(hdcMask);
   DeleteDC(hdcCache);
   }

void GetSprite(HDC hdc, SPRITE *ball_color)
   {
   HDC hdcMem;
   HBITMAP hbmPrev, hbmNew;

   hdcMem = CreateCompatibleDC(hdc);

   if(ball_color->hbmBkg)
      DeleteObject(ball_color->hbmBkg);

   // Create and select a new bitmap to store our background
   hbmNew  = CreateCompatibleBitmap(hdc, ball_color->bmWidth,
      ball_color->bmHeight);
   hbmPrev = SelectObject(hdcMem, hbmNew);

   // Get the background from the screen
   BitBlt(hdcMem, 0, 0, ball_color->bmWidth, ball_color->bmHeight, hdc,
      ball_color->bmX, ball_color->bmY, SRCCOPY);

   ball_color->hbmBkg = hbmNew;

   // Tidy up
   SelectObject(hdcMem, hbmPrev);
   DeleteDC(hdcMem);
   }

BOOL IsSpriteSelected(int nX, int nY, SPRITE *ball_color, int ball_rad)
   {
   RECT  rect;    // RECT data structure
   HRGN  hrgn;
   BOOL ret_val;

   // Current bitmap position
   rect.left   = (int)(ball_color->bmX - (0.5*ball_rad));
   rect.top    = (int)(ball_color->bmY - (0.5*ball_rad));
   rect.right  =
      (int)(ball_color->bmX + ball_color->bmWidth  + (0.5*ball_rad));
   rect.bottom =
      (int)(ball_color->bmY + ball_color->bmHeight + (0.5*ball_rad));

   hrgn = CreateEllipticRgnIndirect(&rect);

   ret_val = PtInRegion(hrgn, nX, nY);

   DeleteObject(hrgn);

   // Return TRUE if pt in rect of sprite
   return ret_val;
   }

void BeginSpriteAnimation(HWND hWnd, int nX, int nY, SPRITE *ball_color)
   {
   // Get all mouse messages
   SetCapture(hWnd);

   // Save previous mouse position
   ball_color->xPrev = nX;
   ball_color->yPrev = nY;
   }

void MoveSprite(HWND hWnd, int nX, int nY, SPRITE *ball_color,
   RECT *win_rect)
   {
   COLORREF rgbBk;
   HDC hdc, hdcMem, hdcNewBkg, hdcOldBkg, hdcMask, hdcCache;
   HBITMAP hbmNew, hbmNPrev, hbmOPrev, hbmPrev, hbmTemp, hbmMask,
      hbmPrevMask, hbmCache, hbmPrevCache;
   int dx, dy;

   // Calculate delta x and delta y
   dx = ball_color->xPrev - nX;
   dy = ball_color->yPrev - nY;

   // Save previous mouse position
   ball_color->xPrev = nX;
   ball_color->yPrev = nY;

   // Update sprite's position
   ball_color->bmX -= dx;
   ball_color->bmY -= dy;

   // Get window DC
   hdc = GetDC(hWnd);

   // Ensure that sprite does not drag over the status bar
   ExcludeClipRect(hdc, win_rect->left, win_rect->bottom-1, win_rect->right,
      win_rect->bottom+status_bar_height);

   hdcMem = CreateCompatibleDC(hdc);
   hdcMask = CreateCompatibleDC(hdc);
   hdcCache = CreateCompatibleDC(hdc);
   hdcNewBkg = CreateCompatibleDC(hdc);
   hdcOldBkg = CreateCompatibleDC(hdc);

   // Set the transparent color to black
   rgbBk = SetBkColor(hdcMem, RGB(0, 0, 0));

   // Create a temp bitmap for our new background
   hbmNew = CreateCompatibleBitmap(hdc, ball_color->bmWidth,
      ball_color->bmHeight);
   hbmMask = CreateCompatibleBitmap(hdcMask, ball_color->bmWidth,
      ball_color->bmHeight);
   hbmCache = CreateCompatibleBitmap(hdc, ball_color->bmWidth,
      ball_color->bmHeight);

   // Select our bitmaps
   hbmPrev = SelectObject(hdcMem, ball_color->hbmImage);
   hbmNPrev = SelectObject(hdcNewBkg, hbmNew);
   hbmOPrev = SelectObject(hdcOldBkg, ball_color->hbmBkg);
   hbmPrevMask = SelectObject(hdcMask, hbmMask);
   hbmPrevCache = SelectObject(hdcCache, hbmCache);

   // Create the bitmap mask
   BitBlt(hdcMask, 0, 0, ball_color->bmWidth, ball_color->bmHeight,
      hdcMem, 0, 0, SRCCOPY);

   // Copy screen to new background
   BitBlt(hdcNewBkg, 0, 0, ball_color->bmWidth, ball_color->bmHeight,
      hdc, ball_color->bmX, ball_color->bmY, SRCCOPY);

   // Replace part of new bkg with old background
   BitBlt(hdcNewBkg, dx, dy, ball_color->bmWidth, ball_color->bmHeight,
      hdcOldBkg, 0, 0, SRCCOPY);

   // Copy sprite to old background
   BitBlt(hdcOldBkg, -dx, -dy, ball_color->bmWidth, ball_color->bmHeight,
      hdcMask, 0, 0, SRCAND);
   BitBlt(hdcOldBkg, -dx, -dy, ball_color->bmWidth, ball_color->bmHeight,
      hdcMem, 0, 0, SRCPAINT);

   // Copy sprite to screen without flicker
   BitBlt(hdcCache, 0, 0, ball_color->bmWidth, ball_color->bmHeight,
      hdcNewBkg, 0, 0, SRCCOPY);
   BitBlt(hdcCache, 0, 0, ball_color->bmWidth, ball_color->bmHeight, hdcMask,
      0, 0, SRCAND);
   BitBlt(hdcCache, 0, 0, ball_color->bmWidth, ball_color->bmHeight, hdcMem,
      0, 0, SRCPAINT);
   BitBlt(hdc, ball_color->bmX, ball_color->bmY, ball_color->bmWidth,
      ball_color->bmHeight, hdcCache, 0, 0, SRCCOPY);

   // Copy old background to screen
   BitBlt(hdc, ball_color->bmX+dx, ball_color->bmY+dy, ball_color->bmWidth,
      ball_color->bmHeight, hdcOldBkg, 0, 0, SRCCOPY);

   // Tidy up
   SelectObject(hdcMem, hbmPrev);
   SelectObject(hdcNewBkg, hbmNPrev);
   SelectObject(hdcOldBkg, hbmOPrev);
   SelectObject(hdcMask, hbmPrevMask);
   SelectObject(hdcCache, hbmPrevCache);
   SetBkColor(hdcMem, rgbBk);

   // Swap old with new background
   hbmTemp = ball_color->hbmBkg;
   ball_color->hbmBkg = hbmNew;
   hbmNew = hbmTemp;

   // Free resources
   DeleteObject(hbmNew);
   DeleteObject(hbmMask);
   DeleteObject(hbmCache);

   // Tidy up some more
   DeleteDC(hdcMem);
   DeleteDC(hdcMask);
   DeleteDC(hdcCache);
   DeleteDC(hdcNewBkg);
   DeleteDC(hdcOldBkg);

   ReleaseDC(hWnd, hdc);
   }

void FinishSpriteAnimation(HWND hWnd, int hole_number, int ball_rad,
   unsigned radius, int num_circles, int ledge, SPRITE *ball_color,
   POINT *point, float x_ellipse_factor, float y_ellipse_factor,
   RECT *win_rect, BOOL make_noise_flag, BOOL speed_flag)
   {
   POINT   pt;
   int     save_hole_number = hole_number;
   int     ball_step_factor = 10;

#ifdef WIN32
   ball_step_factor = 11;
#else
   if(dwFlags & WF_CPU086 || dwFlags & WF_CPU186)
      ball_step_factor = 30;
   else
      if(dwFlags & WF_CPU286)
         ball_step_factor = 25;
      else
         if(dwFlags & WF_CPU386)
            ball_step_factor = 20;
         else
            if(dwFlags & WF_CPU486)
               ball_step_factor = 11;
            else
               ball_step_factor = 5; /* For PENTIUM !!! */
#endif 

   if(speed_flag != TRUE)
      ball_step_factor += 10;

   if(hole_number == NO_HOLE)
      {
      float slope, b, x_test;

#ifdef WIN32
      double length;
#else
      long double length;
#endif

      int optimize;

      hole_number = ball_color->current_hole;

      get_hole_coord(hole_number, ball_rad, radius, num_circles, ledge,
         x_ellipse_factor, y_ellipse_factor, &pt);

      if(speed_flag == TRUE)
         {
         pt.x = (int)(radius*x_ellipse_factor) - pt.x - ball_rad;
         pt.y = (int)((radius*y_ellipse_factor) - pt.y
            - (PERSPECTIVE_BALL_VIEW * ball_rad));
         }
      else
         {
         pt.x = (int)(radius*x_ellipse_factor) - pt.x;
         pt.y = (int)((radius*y_ellipse_factor) - pt.y
            - (PERSPECTIVE_BALL_VIEW * 0.5*ball_rad));
         }

      // Important!  Otherwise sprite will occasionally jump!
      if(point->x == pt.x)
         point->x--;

      if(point->x < pt.x)
         {
/* x1 = point->x,    y1 = point->y    = current location   */
/* x2 = pt.x,        y2 = pt.y        = hole going towards */

         slope = ((pt.y - point->y) / (float)(max((pt.x - point->x), 1.0)));
         b = (point->y - (slope * point->x));

#ifdef WIN32
         length = max((sqrt(pow(pt.x - point->x, 2)
            + pow(pt.y - point->y, 2)) / ball_step_factor), 1.0);
#else
         length = max((sqrtl(powl(pt.x - point->x, 2)
            + powl(pt.y - point->y, 2)) / ball_step_factor), 1.0);
#endif

         optimize = pt.x - point->x;

         for(x_test = (float)point->x; x_test < pt.x;
            x_test += (float)(optimize / length))
            MoveSprite(hWnd, (int)x_test, (int)((slope * x_test) + b),
               ball_color, win_rect);
         }
      else
         {
/* x1 = pt.x,        y1 = pt.y     = hole going towards */
/* x2 = point->x,    y2 = point->y = current location   */

         slope = ((point->y - pt.y) / (float)(max((point->x - pt.x), 1.0)));
         b = (point->y - (slope * point->x));

#ifdef WIN32
         length = max((sqrt(pow(point->x - pt.x, 2)
            + pow(point->y - pt.y, 2)) / ball_step_factor), 1.0);
#else
         length = max((sqrtl(powl(point->x - pt.x, 2)
            + powl(point->y - pt.y, 2)) / ball_step_factor), 1.0);
#endif

         optimize = point->x - pt.x;

         for(x_test = (float)point->x; x_test > pt.x;
            x_test -= (float)(optimize / length))
            MoveSprite(hWnd, (int)x_test, (int)((slope * x_test) + b),
               ball_color, win_rect);
         }
      }

   // Set previous_hole for UNDO purposes elsewhere...
   if(save_hole_number != NO_HOLE)
      {
      ball_color->previous_hole = ball_color->current_hole;
      ball_color->current_hole = hole_number;
      }

   // Reset previous mouse position
   ball_color->xPrev = ball_color->bmX;
   ball_color->yPrev = ball_color->bmY;

   get_hole_coord(hole_number, ball_rad, radius, num_circles, ledge,
      x_ellipse_factor, y_ellipse_factor, &pt);

   MoveSprite(hWnd, (int)(radius*x_ellipse_factor)-pt.x-ball_rad,
      (int)((radius*y_ellipse_factor)-pt.y-(PERSPECTIVE_BALL_VIEW
      *ball_rad)), ball_color, win_rect);

   // Release mouse capture
   ReleaseCapture();

   // Make ball drop sound
   if((save_hole_number != NO_HOLE || make_noise_flag == TRUE)
      && sound_flag == TRUE)
      MessageBeep(MB_OK);
   }

