/*
 Ŀ
  Module....: ERRORSYS.prg                                                
  Author....: Brian Marasca                                               
  Copyright.: Princeton MICRAN Associates, Inc.                           
  Date......: August 1993                                                 
 Ĵ
  Notes.....: General ObjectDB Error Handler.                             
  Arguments.: NONE                                                        
  Return....: NONE                                                        
 
*/

/*****************************************************************************
   Include file headers.
*****************************************************************************/

#include "fileio.ch"
#include "error.ch"
#include "odbdemo.ch"
#include "objectdb.ch"

/*****************************************************************************
   Manifest constants.
*****************************************************************************/

#define NTRIM(n)  ( LTrim(Str(n)) )
#define ERRLOG    "errorlog.txt"
#define CRLF      chr(13)+chr(10)

/*****************************************************************************
   Automatically called at startup.
*****************************************************************************/

FUNCTION errorSys
  errorBlock ( { |e| errorHandler ( e ) } )

  RETURN nil

/*****************************************************************************
   FUNCTION: errorHandler()
*****************************************************************************/

STATIC FUNCTION errorHandler ( e )
  STATIC lCrapOut := FALSE
  LOCAL cCap := ""
  LOCAL cMessage 
  LOCAL nAction 
  LOCAL aOpt 
  LOCAL xRet 
  LOCAL nHan 
  LOCAL i 

  /*****
  * Trap recursion.
  *****/

  IF lCrapOut
     alert( "Error in error handler ..." )
     QUIT
  ENDIF

  lCrapOut := TRUE

  /*****
  * Errors we can handle without user input. 
  *****/

  DO CASE
  CASE e:genCode == EG_ZERODIV
     xRet := 0
  CASE e:genCode == EG_OPEN .AND. e:osCode == 32 .AND. e:canDefault  
     NetErr( TRUE )
     xRet := FALSE
  CASE e:genCode == EG_APPENDLOCK .AND. e:canDefault 
     NetErr( TRUE )
     xRet := FALSE
  CASE e:subSystem == "ObjectDB"
     cCap := "ObjectDB "
     DO CASE
     CASE e:subCode == E_KEYMISMATCH
        e:args[1]:build()
        xRet := TRUE
     CASE e:subCode == E_PRETRIGFAIL
        DO CASE
        CASE e:args[1]:name == "NOWAY"
           tellUser( "No Way", "No one may delete Brian Marasca" )
           xRet := FALSE
        ENDCASE
     ENDCASE
  ENDCASE

  /*****
  * Error not handled, user response required.
  *****/

  IF xRet == NIL
     cMessage := errorMessage( e )
     aOpt := errOptions( e )
   
     cCap += iif( e:severity > ES_WARNING, "Error", "Warning" )
   
     IF ! empty( e:osCode )
        cMessage += " (DOS Error " + NTRIM( e:osCode ) + ")"
     ENDIF
   
     nAction := msgBox( cCap, cMessage, aOpt )
     xRet := iif( nAction > 0, aOpt[nAction] == " Retry ", FALSE )

     /*****
     * Let's log the error to a disk file ...
     *****/
   
     IF file( ERRLOG )
        nHan := fopen( ERRLOG, FO_WRITE )
     ELSE
        nHan := fcreate( ERRLOG, FC_NORMAL )
     ENDIF
   
     IF nHan > 0
        fseek( nHan, 0, FS_END )
        fwrite( nHan, CRLF + dtoc ( date () ) + " " + time () + CRLF )
        fwrite( nHan, cMessage + CRLF )
   
        IF ! empty( e:osCode )
           fwrite( nHan, "DOS Error " + NTRIM( e:osCode ) + CRLF )
        ENDIF
   
        i := 2
        WHILE ! empty( methodName( i ) )
           fwrite( nHan, "Called from " + trim( methodName( i ) ) + "(" + NTRIM( procLine( i ) ) + ") " + CRLF )
           i++
        END
   
        fclose( nHan )
     ENDIF

     /*****
     * It's been real ...
     *****/
   
     IF nAction == 0 .OR. aOpt[nAction] == "  Quit  "
        errorLevel( 1 )
        SetBlink( TRUE )
        SetCursor( 1 )
        QUIT
     ENDIF
  ENDIF

  lCrapOut := FALSE

  RETURN xRet

/*****************************************************************************
   FUNCTION: errorMessage()
*****************************************************************************/

STATIC FUNCTION errorMessage( e )
  LOCAL cMessage := ""

  /*****
  * Translate some of the more obscure ObjectDB error messages
  * for the benefit of the user ...
  *****/

  IF e:subSystem == "ObjectDB"
     DO CASE
     CASE e:subCode == E_HASCHANGED
        e:description := "Someone else has already changed the record"
     CASE e:subCode == E_RESTRICTED
        e:description := "This record is in use in another table"
     CASE e:subCode == E_ENTINTEGRITY
        e:description := "This operation would create a duplicate or empty key"
     CASE e:subCode == E_NOREFERENT
        e:description := "One or more fields do not match their parent records"
     CASE e:subCode == E_NOTUNIQUE
        e:description := "This operation would create a duplicate key"
     CASE e:subCode == E_ALREADYON
        e:description := "You are already logged on at another workstation"
     CASE e:subCode == E_CONTEXTEMPTY
        e:description := "There is nothing to update"
     ENDCASE
  ELSE
     IF valType( e:subsystem ) == "C" 
        cMessage += e:subsystem()
     ELSE
        cMessage += "???"
     END
   
     IF valType( e:subCode ) == "N" 
        cMessage += ( "/" + NTRIM( e:subCode ) )
     ELSE
        cMessage += "/???"
     END
  ENDIF

  IF valType( e:description ) == "C" 
     cMessage += "  " + e:description 
  ENDIF

  DO CASE
  CASE ! empty( e:filename ) 
     cMessage += ": " + e:filename
  CASE ! empty( e:operation ) 
     cMessage += ": " + e:operation 
  ENDCASE

  RETURN cMessage

/*****************************************************************************
   FUNCTION: errOptions()
*****************************************************************************/

STATIC FUNCTION errOptions( e )
  LOCAL aRet := {}

  IF ! e:canDefault
     aadd( aRet, "  Quit  " )
  ENDIF
  
  IF e:canRetry
     aadd( aRet, " Retry " )
  ENDIF
  
  IF e:canDefault
     aadd( aRet, " Default " )
  ENDIF

  RETURN aRet
   





