/*

  DMPS: The Doom PostScript Map Generator, Version SWB 1.3a
  ---------------------------------------------------------
  Original program for Unix and copyright (c) James Bonfield, March 3, 1994.
  Modified for PC by Gerhard Karnik, March 29, 1994.
  Extensive modifications & enhancements by Steve W Brewer, May 2, 1994.

  Description
  -----------
  This program produces PostScript maps of Doom levels, extracted from the WAD
  file.
  
  Thick lines represent solid walls on the map.
  Black circles represent objects.  The letters inside represent what kind of
  object it is.
  White circles with arrows represent the enemy. The direction of the arrow is
  the direction the creature is initially facing.
  Grey lines represent secret areas.

  Usage
  -----
  
  DMPS [options]
  
    /D                 Enable debugging.
    /W wadfile[.WAD]   Wadfile to read from.  (Default: DOOM.WAD)
    /L level           Level to draw.  (Default: E1M1)
    /S1                Draw things present at skills 1 & 2.
    /S3                Draw things present at skill 3.
    /S4                Draw things present at skills 4 & 5.
    /MUL               Draw things present in multiple player mode.
    /NO                Don't draw objects.
    /NM                Don't draw monsters.
    /NS                Don't draw player start positions.
    /?                 Display command line options.

  The PostScript will then be displayed to stdout.

  Compiled using Borland C++ 3.1 using 'bcc -mh'

*/

#define L_MOD_JB   "March 3, 1994"
#define L_MOD_GK   "March 29, 1994"
#define L_MOD_SWB  "May 2, 1994"
#define VERSION    "SWB 1.3a"

#define LI_IMPASS 0x01                  /* Linedef attribute bits */
#define LI_SECRET 0x20

#define TH_SKILL1 0x01                  /* Thing type bits */
#define TH_SKILL2 0x01
#define TH_SKILL3 0x02
#define TH_SKILL4 0x04
#define TH_MULTI  0x10

#define PAPER_X      (72 * 8.5)         /* Paper width  */
#define PAPER_Y      (72 * 11 )         /* Paper height */
#define PAPER_BORDER (72 * 0.5)         /* Paper margin */

#include <io.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <share.h>
#include <fcntl.h>
#include <dos.h>
#include <alloc.h>

unsigned nread;                        /* throw away variable for _dos_read() */

typedef long int4;
typedef short int2;

struct wad_header {
  int4 magic;
  int4 dir_size;
  int4 dir_off;
  };

struct directory {
  int4 start;
  int4 length;
  char name[8];
  };

typedef
  struct linedef_ {
    int2 from_vertex;
    int2 to_vertex;
    int2 attrib;
    int2 type;
    int2 trigger;
    int2 sidedef1;
    int2 sidedef2;
    }
  linedef;

typedef
  struct vertex_ {
    int2 x;
    int2 y;
    }
  vertex;

typedef
  struct sidedef_ {
    int2 x;
    int2 y;
    char wall_above[8];
    char wall_below[8];
    char wall_part[8];
    int2 sector;
    }
  sidedef;

typedef
  struct ssector_ {
    int2 num;
    int2 seg;
    }
  ssector;

typedef
  struct sector_ {
    int2 bot_pos;
    int2 top_pos;
    char bot_texture[8];
    char top_texture[8];
    int2 brightness;
    int2 special;
    int2 trigger;
    }
  sector;

typedef
  struct thing_ {
    int2 x;
    int2 y;
    int2 angle;
    int2 type;
    int2 attrib;
    }
  thing;

typedef int2 blockmap;

struct directory *open_wad(char *file, int *fd, long *size);
long get_index(struct directory *d, long size, char *name, long st);

linedef  *read_linedefs(int fd, struct directory *d, long size, long start, long *num);
vertex   *read_vertexes(int fd, struct directory *d, long size, long start, long *num);
blockmap *read_blockmap(int fd, struct directory *d, long size, long start, long *num);
sidedef  *read_sidedefs(int fd, struct directory *d, long size, long start, long *num);
sector   *read_sectors (int fd, struct directory *d, long size, long start, long *num);
thing    *read_things  (int fd, struct directory *d, long size, long start, long *num);

void doit(int fd, struct directory *dir, long size, char *wadfile, char *lev_name,
  long objs, long mons, long strt, long lev1, long lev3, long lev4, long mult);

void dump_dir(struct directory *d, long size) {
  int i;

  for(i=0; i<size; i++) {
    printf("Item=%5d, Start=%8ld, Len=%8ld, Name=%.8s\n", i, d[i].start, d[i].length, d[i].name);
  }
}

int main(int argc, char **argv) {
  struct directory *d;
  int fd;
  long size, c;
  long objs = 1, mons = 1, strt = 1, lev1 = 0, lev3 = 0, lev4 = 0, mult = 0, dump = 0, help = 0;
  char *level = "E1M1";
  char *wadfile = "doom.wad";
  int i;

  for(i=1; i<=argc; i++) {
    if(!strcmpi(argv[i], "/D")   || !strcmpi(argv[i], "-D"))   dump = 1;
    if(!strcmpi(argv[i], "/W")   || !strcmpi(argv[i], "-W"))   wadfile = argv[(i++)+1];
    if(!strcmpi(argv[i], "/L")   || !strcmpi(argv[i], "-L"))   level = strupr(argv[(i++)+1]);
    if(!strcmpi(argv[i], "/S1")  || !strcmpi(argv[i], "-S1"))  lev1 = 1;
    if(!strcmpi(argv[i], "/S3")  || !strcmpi(argv[i], "-S3"))  lev3 = 1;
    if(!strcmpi(argv[i], "/S4")  || !strcmpi(argv[i], "-S4"))  lev4 = 1;
    if(!strcmpi(argv[i], "/MUL") || !strcmpi(argv[i], "-MUL")) mult = 1;
    if(!strcmpi(argv[i], "/NO")  || !strcmpi(argv[i], "-NO"))  objs = 0;
    if(!strcmpi(argv[i], "/NM")  || !strcmpi(argv[i], "-NM"))  mons = 0;
    if(!strcmpi(argv[i], "/NS")  || !strcmpi(argv[i], "-NS"))  strt = 0;
    if(!strcmpi(argv[i], "/?")   || !strcmpi(argv[i], "-?"))   help = 1;
  }

  /*if (NULL == strchr(wadfile, 46)) strcat(wadfile, ".WAD");*/

  if (help) {
    fprintf(stderr, "\nDMPS: The Doom PostScript Map Generator, Version %s\n", VERSION);
    fprintf(stderr, "---------------------------------------------------------\n");
    fprintf(stderr, "Produces PostScript map of Doom level on stdout.\n\n");
    fprintf(stderr, "DMPS [options]\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "  /W wadfile.WAD     Wadfile to read from.  (Default: DOOM.WAD)\n");
    fprintf(stderr, "  /L level           Level to draw.  (Default: E1M1)\n");
    fprintf(stderr, "  /S1                Draw things present at skills 1 & 2.\n");
    fprintf(stderr, "  /S3                Draw things present at skill 3.\n");
    fprintf(stderr, "  /S4                Draw things present at skills 4 & 5.\n");
    fprintf(stderr, "  /MUL               Draw things present in multi-player mode.\n");
    fprintf(stderr, "  /NO                Don't draw objects.\n");
    fprintf(stderr, "  /NM                Don't draw monsters.\n");
    fprintf(stderr, "  /NS                Don't draw player start positions.\n");
    fprintf(stderr, "  /?                 Display command-line options.\n\n");
    fprintf(stderr, "Original program for Unix and copyright (c) James Bonfield, %s.\n", L_MOD_JB);
    fprintf(stderr, "Modified for PC by Gerhard Karnik, %s.\n", L_MOD_GK);
    fprintf(stderr, "Extensive modifications & enhancements by Steve W Brewer, %s.\n", L_MOD_SWB);
    return 1;
  }

  if (NULL == (d = open_wad(wadfile, &fd, &size))) {
    fprintf(stderr, "Type DMPS /? for help.\n");
    return 1;
  }

  if(dump) {
    dump_dir(d, size);
    return 0;
  }

  doit(fd, d, size, wadfile, level, objs, mons, strt, lev1, lev3, lev4, mult);

  return 0;
}

void doit(int fd, struct directory *dir, long size, char *wadfile, char *lev_name,
  long objs, long mons, long strt, long lev1, long lev3, long lev4, long mult) {

  linedef *linedefs;
  vertex *vertexes;
  blockmap *blocks;
  sidedef *sidedefs;
  sector *sectors;
  thing *things;
  long numline, numvert, numblock, numside, numsect, numthing;
  long lev_index;
  long xorigin, yorigin, xsize, ysize;
  int2 sector;
  double xscale, yscale, xscalelnd, yscalelnd, xscaleprt, yscaleprt;
  long landscape = 0;
  long found = 0;
  long i, j;
  char nametag[2];
        
  /* find level index */

  lev_index = get_index(dir, size, lev_name, 0);

  if(lev_index == -1) {
    fprintf(stderr, "Unknown level: %s\n", lev_name);
    exit(1);
  }

  /* load relevent arrays for this level */

  linedefs = read_linedefs(fd, dir, size, lev_index, &numline );
  vertexes = read_vertexes(fd, dir, size, lev_index, &numvert );
  blocks   = read_blockmap(fd, dir, size, lev_index, &numblock);
  sidedefs = read_sidedefs(fd, dir, size, lev_index, &numside );
  sectors  = read_sectors (fd, dir, size, lev_index, &numsect );
  things   = read_things  (fd, dir, size, lev_index, &numthing);

  /* calculate scaling info */

  xorigin = blocks[0];
  yorigin = blocks[1];
  xsize = blocks[2] * 0x80;
  ysize = blocks[3] * 0x80;

  xscalelnd = (double) (PAPER_Y - 2*PAPER_BORDER)/xsize;
  yscalelnd = (double) (PAPER_X - 2*PAPER_BORDER)/ysize;

  if (xscalelnd > yscalelnd)
    xscalelnd = yscalelnd;
   else
    yscalelnd = xscalelnd;

  xscaleprt = (double) (PAPER_X - 2*PAPER_BORDER)/xsize;
  yscaleprt = (double) (PAPER_Y - 2*PAPER_BORDER)/ysize;

  if (xscaleprt > yscaleprt) 
    xscaleprt = yscaleprt;
   else
    yscaleprt = xscaleprt;

  if (xscaleprt > xscalelnd) {
    xscale = xscaleprt;
    yscale = xscaleprt;
    }
   else {
    xscale = xscalelnd;
    yscale = xscalelnd;
    landscape = 1;
  }

  /* output postscript header */

  printf("%%!\n");
  printf("%%\n");
  printf("%%  Doom Map: %s Episode %c Mission %c\n", strupr(wadfile), lev_name[1], lev_name[3]);
  printf("%%  ------------------------------------------\n");
  printf("%%  Created by DMPS: The Doom PostScript Map Generator, Version %s\n", VERSION);
  printf("%%\n");
  printf("%%  Original program for Unix and copyright (c) James Bonfield, %s.\n", L_MOD_JB);
  printf("%%  Modified for MS-DOS by Gerhard Karnik, %s.\n", L_MOD_GK);
  printf("%%  Extensive modifications & enhancements by Steve W Brewer, %s.\n", L_MOD_SWB);
  printf("%%\n");
  printf("%%  Options in effect:\n");
  printf("%%  ------------------\n");
  if (lev1) printf("%%    Draw things present at skills 1 & 2.\n");
  if (lev3) printf("%%    Draw things present at skill 3.\n");
  if (lev4) printf("%%    Draw things present at skills 4 & 5.\n");
  if (mult) printf("%%    Draw things present in multi-player mode.\n");
  if (objs) printf("%%    Draw objects.\n");
  if (mons) printf("%%    Draw monsters.\n");
  if (strt) printf("%%    Draw player start positions.\n");
  printf("%%\n");
  printf("currentscreen 3 1 roll pop pop 120 45 3 -1 roll setscreen\n");
  printf("newpath\n");
  printf("1 setlinecap\n");
  printf("%f %f translate\n", PAPER_BORDER, landscape ? (PAPER_Y - PAPER_BORDER) : PAPER_BORDER);
  if (landscape) printf("-90 rotate\n");
  printf("%f %f scale\n", xscale, yscale);
  printf("%ld %ld translate\n", -xorigin, -yorigin);
  printf("%f setlinewidth\n", (double) .5 / xscale);
  printf("\n");
  printf("/l {\n");
  printf("    setgray setlinewidth moveto lineto stroke\n");
  printf("} def\n");
  printf("\n");
  printf("/a {%% Draw Object (Arguments: tag xcenter ycenter)\n");
  printf("    gsave\n");
  printf("    translate\n");
  printf("    20 20 scale\n");
  printf("    /tag exch def\n");
  printf("    1 setgray\n");
  printf("    0 setlinewidth\n");
  printf("    0 0 1.01 0 360 arc fill\n");
  printf("    0 setgray\n");
  printf("    0 0 1 0 360 arc fill\n");
  printf("    1 setgray\n");
  printf("    0 0 moveto\n");
  printf("    /Helvetica-Bold findfont [1.2 0 0 1.7 0 0] makefont setfont\n");
  printf("    -.1 0 tag dup stringwidth pop neg 2 div .105 add -.55 rmoveto ashow\n");
  printf("    grestore\n");
  printf("} def\n");
  printf("\n");
  printf("/m {%% Draw Monster (Arguments: tag radius angle xcenter ycenter)\n");
  printf("    gsave\n");
  printf("    translate\n");
  printf("    /angle exch 90 sub def\n");
  printf("    dup scale\n");
  printf("    /tag exch def\n");
  printf("    1 setgray\n");
  printf("    0 setlinewidth\n");
  printf("    0 0 1.01 0 360 arc fill\n");
  printf("    0 setgray\n");
  printf("    0 0 1 0 360 arc fill\n");
  printf("    1 setgray\n");
  printf("    0 0 .95 0 360 arc fill\n");
  printf("    0 setgray\n");
  printf("    0 0 .9 0 360 arc fill\n");
  printf("    1 setgray\n");
  printf("    0 0 moveto\n");
  printf("    /Helvetica-Bold findfont [1 0 0 1.6 0 0] makefont setfont\n");
  printf("    -.1 0 tag dup stringwidth pop neg 2 div .10 add -.55 rmoveto ashow\n");
  printf("    0 0 moveto\n");
  printf("    angle rotate\n");
  printf("    0 setgray\n");
  printf("    newpath\n");
  printf("    0 1.3 moveto\n");
  printf("    0 0 .97 80 100 arc\n");
  printf("    0 1.3 lineto\n");
  printf("    fill\n");
  printf("    grestore\n");
  printf("} def\n");

  /* Display the walls. Secret passages are done in grey. */

  for (i = 0; i < numline; i++) {
    sector = sidedefs[linedefs[i].sidedef1].sector;

    printf("%d %d %d %d ",
      vertexes[linedefs[i].to_vertex].x,
      vertexes[linedefs[i].to_vertex].y,
      vertexes[linedefs[i].from_vertex].x,
      vertexes[linedefs[i].from_vertex].y);

    if (linedefs[i].attrib & LI_IMPASS) {
      printf("%f ", 1.2/xscale); }
     else {
      printf("%f ", 0.5/xscale);
    }

    if (linedefs[i].attrib & LI_SECRET || sectors[sector].special == 9) {
      printf(".8 l\n"); } 
     else {
      printf("0 l\n");
    }
  }

  for(i=0; i<numthing; i++) {

    /* filter out unwanted stuff */

    if(!(((things[i].attrib & 0x1) && lev1)
      || ((things[i].attrib & 0x2) && lev3)
      || ((things[i].attrib & 0x4) && lev4))) continue;

    if(!(!(things[i].attrib & 0x10) || mult)) continue;

    /* Draw monsters */

    if (mons) {
      found=1;
      
      switch(things[i].type) {
        case    7:            /* Spider Demon          */
          printf("(SP) 128");
          break;
        case    9:            /* Former Human Sergeant */
          printf("(SR)  20");
          break;
        case   16:            /* Cyber-Demon           */
          printf("(CD)  40");
          break;
        case   58:            /* Spectre               */
          printf("(SE)  30");
          break;
        case 3001:            /* Imp                   */
          printf("(IM)  20");
          break;
        case 3002:            /* Demon                 */
          printf("(DM)  30");
          break;
        case 3003:            /* Baron of Hell         */
          printf("(BR)  24");
          break;
        case 3004:            /* Former Human          */
          printf("(HM)  20");
          break;
        case 3005:            /* Cacodemon             */
          printf("(TM)  31");
          break;
        case 3006:            /* Lost Soul             */
          printf("(LS)  16");
          break;
        default:
          found=0;
          break;
      }
    
      if (found) printf(" %d %d %d m\n", things[i].angle, things[i].x, things[i].y);
    }

    /* Draw player start positions */
    
    if (strt) {
      found=1;
      
      switch(things[i].type) {
        case 1:               /* Player 1 Start        */
          printf("(P1)");
          break;
        case 2:               /* Player 2 Start        */
          if (mult) printf("(P2)"); else found=0;
          break;
        case 3:               /* Player 3 Start        */
          if (mult) printf("(P3)"); else found=0;
          break;
        case 4:               /* Player 4 Start        */
          if (mult) printf("(P4)"); else found=0;
          break;
        case 11:              /* Deathmatch Start      */
	  if (mult) printf("(DS)"); else found=0;
          break;
        case 14:              /* Teleport Exit         */
          printf("(TX)");
          break;
        default:
          found=0;
          break;
      }
    
      if (found) printf(" 16 %d %d %d m\n", things[i].angle, things[i].x, things[i].y);
    }

    /* Draw objects */
    
    if(objs) {
      found=1;
      
      switch(things[i].type) {
        case 5:               /* Blue keycard          */
          printf("(BK)");
          break;
        case 6:               /* Yellow keycard        */
          printf("(YK)");
          break;
        case 13:              /* Red keycard           */
          printf("(RK)");
          break;
        case 38:              /* Red skullkey          */
          printf("(RK)");
          break;
        case 39:              /* Yellow skullkey       */
          printf("(YK)");
          break;
        case 40:              /* Blue skullkey         */
          printf("(BK)");
          break;
        case 2001:            /* Shotgun               */
          printf("(SG)");
          break;
        case 2002:            /* Chaingun              */
          printf("(CG)");
          break;
        case 2003:            /* Rocket launcher       */
          printf("(RL)");
          break;
        case 2004:            /* Plasma gun            */
          printf("(PG)");
          break;
        case 2005:            /* Chainsaw              */
          printf("(CH)");
          break;
        case 2006:            /* BFG9000               */
          printf("(BG)");
          break;
        case 8   :            /* Backpack              */
          printf("(BP)");
          break;
        case 17  :            /* Cell charge pack      */
          printf("(CL)");
          break;
        case 2007:            /* Ammo clip             */
          printf("(CP)");
          break;
        case 2008:            /* 4 shotgun shells      */
          printf("(SH)");
          break;
        case 2010:            /* 1 rocket              */
          printf("(RT)");
          break;
        case 2046:            /* Box of Rockets        */
          printf("(BT)");
          break;
        case 2047:            /* Cell charge           */
          printf("(CC)");
          break;
        case 2048:            /* Box of Ammo           */
          printf("(BB)");
          break;
        case 2049:            /* Box of Shells         */
          printf("(BH)");
          break;
        case 2011:            /* Stimpak               */
          printf("(SM)");
          break;
        case 2012:            /* Medikit               */
          printf("(MK)");
          break;
        case 2013:            /* Soulsphere            */
          printf("(SS)");
          break;
        case 2014:            /* Health bonus          */
          printf("(HL)");
          break;
        case 2015:            /* Armor bonus           */
          printf("(AM)");
          break;
        case 2018:            /* Green armor 100%      */
          printf("(GA)");
          break;
        case 2019:            /* Blue armor 200%       */
          printf("(BA)");
          break;
        case 2022:            /* Invulnerability       */
          printf("(IL)");
          break;
        case 2023:            /* Berserk Strength      */
          printf("(BR)");
          break;
        case 2024:            /* Invisibility          */
          printf("(IS)");
          break;
        case 2025:            /* Radiation suit        */
          printf("(RD)");
          break;
        case 2026:            /* Computer map          */
          printf("(MP)");
          break;
        case 2045:            /* Lite goggles          */
          printf("(LT)");
          break;
        default:
          found=0;
          break;
      }
    
      if (found) printf(" %d %d a\n", things[i].x, things[i].y);
    }
  }

  /*printf("/Times-Roman findfont %f scalefont setfont\n", 20 / xscale); */
  /*printf("0 setgray 0 %ld moveto (%s) show\n", yorigin, lev_name);     */

  puts("showpage");
  return;
}

struct directory *open_wad(char *file, int *fd, long *size) {
  struct wad_header h;
  struct directory *d;

  if (_dos_open(file, O_RDONLY, fd) != 0) {
    fprintf(stderr, "Could not open ");
    perror(strupr(file));
    return (struct directory *)0;
  }

  _dos_read(*fd, &h, sizeof(h), &nread);

  if(NULL == (d = (struct directory *)farcalloc(h.dir_size, sizeof(*d)))) {
    fprintf(stderr, "Error: Could not allocate %ld bytes.\n", h.dir_size * sizeof(*d));
    return (struct directory *) 0;
  }

  *size = h.dir_size;

  lseek(*fd, h.dir_off, SEEK_SET);
  _dos_read(*fd, d, *size * sizeof(*d), &nread);

  return d;
}

long get_index(struct directory *d, long size, char *name, long start) {
  long i;

  for(i=start; i<size; i++) {

    if(_fstrncmp(d[i].name, name, 8) == 0)
      return i;
  }

  return -1;
}

linedef *read_linedefs(int fd, struct directory *d, long size, long start, long *num) {
  long index;
  linedef *indexp;
  long i;

  index = get_index(d, size, "LINEDEFS", start);

  if(!(indexp = (linedef *) farmalloc(d[index].length))) {
    printf("Error: Could not allocate %ld bytes\n",d[index].length);
    exit(-1);
  }

  lseek(fd, d[index].start, SEEK_SET);
  _dos_read(fd, indexp, d[index].length, &nread);
  *num = d[index].length / sizeof(linedef);

  return indexp;
}

vertex *read_vertexes(int fd, struct directory *d, long size, long start, long *num) {
  long index;
  vertex *indexp;
  long i;

  index = get_index(d, size, "VERTEXES", start);

  if(!(indexp = (vertex *) farmalloc(d[index].length))) {
    printf("Error: Could not allocate %ld bytes\n",d[index].length);
    exit(-1);
  }

  lseek(fd, d[index].start, SEEK_SET);
  _dos_read(fd, indexp, d[index].length, &nread);
  *num = d[index].length / sizeof(vertex);

  return indexp;
}

blockmap *read_blockmap(int fd, struct directory *d, long size, long start, long *num) {
  long index;
  blockmap *indexp;
  long i;

  index = get_index(d, size, "BLOCKMAP", start);

  if(!(indexp = (blockmap *) farmalloc(d[index].length))) {
    printf("Error: Could not allocate %ld bytes\n",d[index].length);
    exit(-1);
  }

  lseek(fd, d[index].start, SEEK_SET);
  _dos_read(fd, indexp, d[index].length, &nread);
  *num = d[index].length / sizeof(blockmap);

  return indexp;
}

sidedef *read_sidedefs(int fd, struct directory *d, long size, long start, long *num) {
  long index;
  sidedef *indexp;
  long i;

  index = get_index(d, size, "SIDEDEFS", start);

  if(!(indexp = (sidedef *) farmalloc(d[index].length))) {
    printf("Error: Could not allocate %ld bytes\n",d[index].length);
    exit(-1);
 }

  lseek(fd, d[index].start, SEEK_SET);
  _dos_read(fd, indexp, d[index].length, &nread);
  *num = d[index].length / sizeof(sidedef);

  return indexp;
}

sector *read_sectors(int fd, struct directory *d, long size, long start, long *num) {
  long index;
  sector *indexp;
  long i;

  index = get_index(d, size, "SECTORS", start);

  if(!(indexp = (sector *) farmalloc(d[index].length))) {
    printf("Error: Could not allocate %ld bytes\n",d[index].length);
    exit(-1);
  }

  lseek(fd, d[index].start, SEEK_SET);
  _dos_read(fd, indexp, d[index].length, &nread);
  *num = d[index].length / sizeof(sector);

  return indexp;
}

thing *read_things(int fd, struct directory *d, long size, long start, long *num) {
  long index;
  thing *indexp;
  long i;

  index = get_index(d, size, "THINGS", start);

  if(!(indexp = (thing *) farmalloc(d[index].length))) {
    printf("Error: Could not allocate %ld bytes\n",d[index].length);
    exit(-1);
  }

  lseek(fd, d[index].start, SEEK_SET);
  _dos_read(fd, indexp, d[index].length, &nread);
  *num = d[index].length / sizeof(thing);

  return indexp;
}
