{
 TD> I've seen it done in many places, but I haven't seen any info on
 TD> how it's done:  What is the basic algorithm for fading from one
 TD> palette to another.

        Many people do palette fading incorrectly.  The correct
way to do it would be to set up a relation such as:

        Palette_Element     Calculated_Element
        ---------------  =  ------------------
         Max_Intensity      Current_Intensity

Where Palette_Element is a single element in our master DAC
table, Max_Intensity is the maximum allowable intensity level for
our scale, Current_Intensity is a number between 0 and
Max_Intensity which represents the level we want, and
Calculated_Element is the new value for the element of our DAC
table.  But since we want the Calculated_Element, we re-write it
as this equation:

        Calculated_Element = Palette_Element * Current_Intensity
                             -----------------------------------
                                         Max_Intensity

The above equation will allow us to fade a given palette set to
black or from black to a given palette set.  To fade out an entire
palette set, you would need to calculate the above for the red,
green, and blue components of each color in the 256 element DAC
table.
        Fading from one palette set to another palette set is
very similar.  What you must do is fade one palette set to black
while simultaneously fade from black to another palette set and
add the two values.  The equation for this is:

       CE = ((PE1 * (MI - CI)) + (PE2 * CI)) / MI

Where CE is the calculated element, PE1 and PE2 are corresponding
palette elements from palette 1 and 2, MI is the maximum
intensity in our scale, and CI is the current intensity we want
(num between 0 and MI).
}

Unit Palette;
{ Programmed By David Dahl @ FidoNet 1:272/38 }
(* PUBLIC DOMAIN *)
Interface
  Type PaletteRec = Record
                          Red   : Byte;
                          Green : Byte;
                          Blue  : Byte;
                    End;
       PaletteArray = Array [0..255] of PaletteRec;

  Procedure SetPalette (Var PaletteIn : PaletteArray);
  Procedure FadeFromPaletteToBlack (Var PaletteIn : PaletteArray);
  Procedure FadeFromBlackToPalette (Var PaletteIn : PaletteArray);
  Procedure FadeFromPalette1ToPalette2 (Var Palette1 : PaletteArray;
                                        Var Palette2 : PaletteArray);
Implementation
Procedure SetPalette (Var PaletteIn : PaletteArray); Assembler;
Asm
   { Get Address of PaletteIn }
   LDS SI, PaletteIn
   CLD

   { Tell VGA To Start With First Palette Element }
   XOR AX, AX
   MOV DX, $3C8
   OUT DX, AL

   { Wait For Retrace }
   MOV DX, $3DA
   @VRWait1:
     IN AL, DX
     AND AL, 8
   JZ @VRWait1

   { Set First Half Of Palette }
   MOV DX, $3C9
   MOV CX, 128 * 3
   @PALLOOP1:
     LODSB  { DON'T use "REP OUTSB" since some VGA cards can't handle it }
     OUT DX, AL
   LOOP @PALLOOP1

   { Wait For Retrace }
   PUSH DX
   MOV DX, $3DA
   @VRWait2:
     IN AL, DX
     AND AL, 8
   JZ @VRWait2
   POP DX

   { Set Last Half Of Palette }
   MOV CX, 128 * 3
   @PALLOOP2:
     LODSB
     OUT DX, AL
   LOOP @PALLOOP2
End;

Procedure FadeFromPaletteToBlack (Var PaletteIn : PaletteArray);
Var WorkPalette : PaletteArray;
    Counter     : Integer;
    Intensity   : Integer;
Begin
     For Intensity := 31 downto 0 do
     Begin
       For Counter := 0 to 255 do
       Begin
          WorkPalette[Counter].Red   :=
                   (PaletteIn[Counter].Red   * Intensity) DIV 32;
          WorkPalette[Counter].Green :=
                   (PaletteIn[Counter].Green * Intensity) DIV 32;
          WorkPalette[Counter].Blue  :=
                   (PaletteIn[Counter].Blue  * Intensity) DIV 32;
       End;
       SetPalette (WorkPalette);
     End;
End;

Procedure FadeFromBlackToPalette (Var PaletteIn : PaletteArray);
Var WorkPalette : PaletteArray;
    Counter     : Integer;
    Intensity   : Integer;
Begin
     For Intensity := 1 to 32 do
     Begin
       For Counter := 0 to 255 do
       Begin
          WorkPalette[Counter].Red   :=
                   (PaletteIn[Counter].Red   * Intensity) DIV 32;
          WorkPalette[Counter].Green :=
                   (PaletteIn[Counter].Green * Intensity) DIV 32;
          WorkPalette[Counter].Blue  :=
                   (PaletteIn[Counter].Blue  * Intensity) DIV 32;
       End;
       SetPalette (WorkPalette);
     End;
End;

Procedure FadeFromPalette1ToPalette2 (Var Palette1 : PaletteArray;
                                      Var Palette2 : PaletteArray);
Var WorkPalette : PaletteArray;
    Counter     : Integer;
    CrossFade   : Integer;
Begin
     For CrossFade := 0 to 32 do
     Begin
       For Counter := 0 to 255 do
       Begin
         WorkPalette[Counter].Red   :=
             ((Palette1[Counter].Red   * (32 - CrossFade)) +
              (Palette2[Counter].Red   * CrossFade)) DIV 32;
         WorkPalette[Counter].Green :=
             ((Palette1[Counter].Green * (32 - CrossFade)) +
              (Palette2[Counter].Green * CrossFade)) DIV 32;
         WorkPalette[Counter].Blue  :=
             ((Palette1[Counter].Blue  * (32 - CrossFade)) +
              (Palette2[Counter].Blue  * CrossFade)) DIV 32;
       End;
       SetPalette (WorkPalette);
     End;
End;
End.

{
--- 
 * Origin: PDN HQ (914)344-0350 HST/V32T 343-7540 HST/V32b (1:272/38)
}
