/* ***************************************************************

An AutoCAD ADS program to remove "extra entities"

This module contains the functions to run the dialog box version

Jon Fleming    CIS 70334,2443    July 19, 1994

Public Domain; Please give me credit if you use this or any
significant portion of it

Revision history:

July 19, 1994  Version 1.0: Initial release

October 23, 1994 Version 1.1: Modified EER_CORE.C to handle negative line
    endpoints properly, and to avoid checking for entities to remove when
    no entities were selected in the "candidates for removal" selection set.
    Modified EER_CORE.C and EEREM.H to include version number report when
    DoEntityRemoval starts.

*********************************************************** */

#include <string.h>
#include  "adslib.h"
#include "adsdlg.h"
#include "eerem.h"

/* Function prototypes */

int ddeerem ();
void CALLB OKCallBack(ads_callback_packet *);
void CALLB SelectCallBack(ads_callback_packet *);

/* Coordinates for the dialog box position on the screen */

static int DialogX = -1, DialogY = -1;

/* Indicators for whether to consider layer, linetype, and/or color */

extern int CheckLayer, CheckLType, CheckColor;

/* Tolerance within which numbers are considered equal (at least, that's
   approximately what this is; see below for more detail)  */

extern ads_real Tolerance;

/* Character versions of the above, for the dialog box */

char ToleranceStr[13], CheckLayerStr[2], CheckLTypeStr[2], CheckColorStr[2];



/* Function for doing extra entity removal from the command line through
   a dialog box.

   Return value:

   T / RSRLST if the DCL file and dialog box definiton were found

   NIL / RSERR otherwise
*/

int ddeerem () {

   ads_name SelSet;

   ads_hdlg DialogHandle;

   int DialogStatus = 10, DCL_ID, stat = RSRSLT;
   
   short EntitiesNotSelected = 1;

   int LastTolChar;

   char ZeroChar[] = "0";

   /* Set up the default values of the tiles */

   ads_rtos(Tolerance, 2, 12, ToleranceStr);
   /* Strip trailing zeros */
   LastTolChar = strlen(ToleranceStr) - 1;
   while (ToleranceStr[LastTolChar] == ZeroChar[0]) {
      ToleranceStr[LastTolChar] = 0;
      LastTolChar = --LastTolChar;
   }


   if (CheckLayer) {
      strcpy(CheckLayerStr, "1");
   }
   else {
      strcpy(CheckLayerStr, "0");
   }
   if (CheckLType) {
      strcpy(CheckLTypeStr, "1");
   }
   else {
      strcpy(CheckLTypeStr, "0");
   }
   if (CheckColor) {
      strcpy(CheckColorStr, "1");
   }
   else {
      strcpy(CheckColorStr, "0");
   }

   /* Open the dialog box.  The loop allows us to pop back up after picking
      entities for checking */

   if (ads_load_dialog("EEREM.DCL", &DCL_ID) == RTNORM) {
      while (DialogStatus > 1) {
         if (ads_new_positioned_dialog("eerem", DCL_ID, NULLCB, DialogX, DialogY, &DialogHandle) == RTNORM) {
            
            /* Disable the OK button until entities are selected.
               EntitiesNotSelected should be a SHORT with a value of
               0 to indicate entities have been selected, 1 to indicate
               entities have not yet been selected, and NO OTHER value */

            ads_mode_tile(DialogHandle, "accept", EntitiesNotSelected);

            /* Give the "Select Entities" button the focus unless entieis
               have been selected (in which case give the OK button the
               focus */

            if (EntitiesNotSelected == 0) {
               ads_mode_tile(DialogHandle, "accept", 2);
            }
            else {
               ads_mode_tile(DialogHandle, "SelectEnts", 2);
            }

            /* Initialize the tiles */

            ads_set_tile(DialogHandle, "Tolerance", ToleranceStr);

            ads_set_tile(DialogHandle, "CheckLayer", CheckLayerStr);
            ads_set_tile(DialogHandle, "CheckLType", CheckLTypeStr);
            ads_set_tile(DialogHandle, "CheckColor", CheckColorStr);


            /* Set up callbacks for the buttons */

            ads_action_tile(DialogHandle, "accept", OKCallBack);
            ads_action_tile(DialogHandle, "SelectEnts", SelectCallBack);

            /* Display the dialog */

            ads_start_dialog(DialogHandle, &DialogStatus);

            /* If the "Select Entities" button was pushed, select some
               entities and leave DialogStatus at 2 to re-pop the dialog
               box.  If the suser really selected entities, set flag so
               the OK button will be enabled */

            if (DialogStatus == 2
                  && ads_ssget(NULL, NULL, NULL, NULL, SelSet) == RTNORM) {
                     EntitiesNotSelected = 0;
            }
         }
         else {
            ads_printf("\nDDEEREM error: can't find EEREM dialog in EEREM.DCL\n");
            stat = RSERR;
         }
      }

      ads_unload_dialog(DCL_ID);
   }
   else {
      ads_printf("\nDDEEREM error: can't find EEREM.DCL\n");
      stat = RSERR;
   }

   /* If the user pressed OK ... */

   if (DialogStatus == 1) {

      /* Do the extra entity removal */

      DoEntityRemoval (SelSet);

   }

   /* If a selection set was created, free the memory */

   if (EntitiesNotSelected == 0) {
      ads_ssfree(SelSet);
   }

   ads_retvoid ();
   return (stat);
}

/* Callback function for the OK button */

static void CALLB OKCallBack(ads_callback_packet *CBPacket) {

   /* Get the values of the check boxes into the appropriate variables */

   ads_get_tile(CBPacket->dialog, "CheckLayer", CheckLayerStr, 2);
   ads_get_tile(CBPacket->dialog, "CheckLType", CheckLTypeStr, 2);
   ads_get_tile(CBPacket->dialog, "CheckColor", CheckColorStr, 2);

   CheckLayer = atoi(CheckLayerStr);
   CheckLType = atoi(CheckLTypeStr);
   CheckColor = atoi(CheckColorStr);

   /* Get the value of the tolerance */

   ads_get_tile(CBPacket->dialog, "Tolerance", ToleranceStr, 13);
   Tolerance = atof(ToleranceStr);

   ads_done_positioned_dialog(CBPacket->dialog, 1, &DialogX, &DialogY);
}

/* Callback function for the "Select Entities" button */

static void CALLB SelectCallBack(ads_callback_packet *CBPacket) {

   /* Save the current tile values */

   ads_get_tile(CBPacket->dialog, "CheckLayer", CheckLayerStr, 2);
   ads_get_tile(CBPacket->dialog, "CheckLType", CheckLTypeStr, 2);
   ads_get_tile(CBPacket->dialog, "CheckColor", CheckColorStr, 2);
   ads_get_tile(CBPacket->dialog, "Tolerance", ToleranceStr, 13);

   /* Pop down with a status of 2 to let the user select */

   ads_done_positioned_dialog(CBPacket->dialog, 2, &DialogX, &DialogY);
}
