/* ptrace.c -- Implement ptrace()
   Copyright (c) 1994 by Eberhard Mattes

This file is part of emx.

emx is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

emx is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with emx; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

As special exception, emx.dll can be distributed without source code
unless it has been changed.  If you modify emx.dll, this exception
no longer applies and you must remove this paragraph from all source
files for emx.dll.  */


#define INCL_DOSPROCESS
#define INCL_DOSSESMGR
#define INCL_DOSEXCEPTIONS
#define INCL_DOSERRORS
#include <os2emx.h>
#include "clib.h"
#include <sys/signal.h>
#include <sys/ptrace.h>
#include <sys/user.h>
#include "reg.h"
#include <sys/errno.h>
#include "emxdll.h"

#define USEROFF(F) offsetof (struct user, F)

struct reg_table
{
  ULONG addr;
  void *debug;
  BYTE len;
};

struct co_regs
{
  USHORT cw;
  USHORT unused1;
  USHORT sw;
  USHORT unused2;
  USHORT tw;
  USHORT unused3;
  ULONG fip;
  USHORT fcs;
  USHORT fop;
  ULONG foo;
  USHORT fos;
  USHORT unused4;
  BYTE fst[8][10];
};

static BYTE float_regs;
static uDB_t dbgbuf;
static ULONG child_syscall;
static ULONG child_layout;
static ULONG child_session;
static ULONG run_tid;
static BYTE auto_switch;
static ULONG run_pid;
static struct co_regs dbgco;
static BYTE brk_flag;
static BYTE more;
static BYTE second;

/* Table for converting user struct offsets into pointers to fields of
   dbgbuf.  It also contains the lengths of the fields. */

static struct reg_table const ptrace_regs[] =
{
  {USEROFF (u_regs[R_GS]),   &dbgbuf.GS,     2},
  {USEROFF (u_regs[R_FS]),   &dbgbuf.FS,     2},
  {USEROFF (u_regs[R_ES]),   &dbgbuf.ES,     2},
  {USEROFF (u_regs[R_DS]),   &dbgbuf.DS,     2},
  {USEROFF (u_regs[R_EDI]),  &dbgbuf.EDI,    4},
  {USEROFF (u_regs[R_ESI]),  &dbgbuf.ESI,    4},
  {USEROFF (u_regs[R_EBP]),  &dbgbuf.EBP,    4},
  {USEROFF (u_regs[R_ESP]),  &dbgbuf.ESP,    4},
  {USEROFF (u_regs[R_EBX]),  &dbgbuf.EBX,    4},
  {USEROFF (u_regs[R_EDX]),  &dbgbuf.EDX,    4},
  {USEROFF (u_regs[R_ECX]),  &dbgbuf.ECX,    4},
  {USEROFF (u_regs[R_EAX]),  &dbgbuf.EAX,    4},
  {USEROFF (u_regs[R_EIP]),  &dbgbuf.EIP,    4},
  {USEROFF (u_regs[R_CS]),   &dbgbuf.CS,     2},
  {USEROFF (u_regs[R_EFL]),  &dbgbuf.EFlags, 4},
  {USEROFF (u_regs[R_UESP]), &dbgbuf.ESP,    4},
  {USEROFF (u_regs[R_SS]),   &dbgbuf.SS,     2}
};


static const struct reg_table *user_addr (ULONG addr)
{
  int i;

  for (i = 0; i < sizeof (ptrace_regs) / sizeof (ptrace_regs[0]); ++i)
    if (ptrace_regs[i].addr == addr)
      return (&ptrace_regs[i]);
  return (NULL);
}


ULONG debug (void)
{
  ULONG rc;

  rc = DosDebug (&dbgbuf);
  return (rc == 0 ? 0 : set_error (rc));
}


ULONG debug_read_byte (ULONG addr, BYTE *dst)
{
  ULONG rc;

  dbgbuf.Cmd  = DBG_C_ReadMem;
  dbgbuf.Addr = addr;
  rc = debug ();
  if (rc != 0) return (rc);
  *dst = (BYTE)dbgbuf.Value;
  return (0);
}


ULONG debug_read_word (ULONG addr, ULONG *dst)
{
  ULONG rc, w;

  dbgbuf.Cmd  = DBG_C_ReadMem;
  dbgbuf.Addr = addr;
  rc = debug ();
  if (rc != 0) return (rc);
  w = dbgbuf.Value & 0xffff;
  dbgbuf.Cmd  = DBG_C_ReadMem;
  dbgbuf.Addr = addr + 2;
  rc = debug ();
  if (rc != 0) return (rc);
  w |= (dbgbuf.Value & 0xffff) << 16;
  *dst = w;
  return (0);
}


void do_auto_switch (void)
{
  DosSelectSession (child_session);
  auto_switch = FALSE;
}


static int debug_poke16 (ULONG addr, ULONG value)
{
  ULONG rc;

  dbgbuf.Cmd   = DBG_C_WriteMem;
  dbgbuf.Addr  = addr;
  dbgbuf.Value = value;
  rc = debug ();
  if (rc == 0 && dbgbuf.Cmd != DBG_N_Success)
    rc = EIO;
  return (rc);
}


/* Read the registers.  TID is the thread ID (0 means the active
  thread).  Return errno. */

static ULONG read_reg (ULONG tid)
{
  ULONG rc;

  dbgbuf.Cmd = DBG_C_ReadReg;
  dbgbuf.Tid = tid;
  rc = debug ();
  if (rc == 0 && dbgbuf.Cmd != DBG_N_Success)
    rc = EIO;
  return (rc);
}


/* Write the registers.  TID is the thread ID (which must be
   non-zero).  Return errno. */

static ULONG write_reg (ULONG tid)
{
  ULONG rc;

  dbgbuf.Cmd = DBG_C_WriteReg;
  dbgbuf.Tid = tid;
  rc = debug ();
  if (rc == 0 && dbgbuf.Cmd != DBG_N_Success)
    rc = EIO;
  return (rc);
}


/* Set a watchpoint.  Return errno. */

static ULONG set_watch (ULONG addr, ULONG len, ULONG type)
{
  ULONG rc;

  dbgbuf.Cmd   = DBG_C_SetWatch;
  dbgbuf.Addr  = addr;
  dbgbuf.Len   = len;
  dbgbuf.Index = 0;             /* Reserved */
  dbgbuf.Value = type | DBG_W_Local;
  dbgbuf.Pid   = run_pid;
  rc = debug ();
  return (rc);
}


static ULONG terminate (void)
{
  ULONG rc;

  dbgbuf.Cmd = DBG_C_Term;
  rc = debug ();
  return (rc);
}


static ULONG get_fpstate (void)
{
  ULONG rc;

  dbgbuf.Cmd    = DBG_C_ReadCoRegs;
  dbgbuf.Tid    = 0;            /* Thread: active thread */
  dbgbuf.Value  = DBG_CO_387;   /* Coprocessor type */
  dbgbuf.Buffer = (ULONG)&dbgco;
  dbgbuf.Len    = sizeof (dbgco);
  dbgbuf.Index  = 0;            /* Reserved */
  rc = debug ();
  if (rc == 0 && dbgbuf.Cmd != DBG_N_Success)
    rc = EIO;
  return (rc);
}


/* Return true if EIP points to a CALL instruction.  Moreover, true is
   returned if an error occurs (to be on the safe side for
   automatically switching to the child session). */

static int callp (void)
{
  ULONG eip, rc;
  BYTE b;

  if (read_reg (0) != 0)
    return (TRUE);
  eip = dbgbuf.EIP;
  do
    {
      rc = debug_read_byte (eip++, &b);
      if (rc != 0) return (TRUE);
    } while ((b & 0xe7) == 0x26 /* Segment override prefix */
             || b == 0x64       /* FS prefix */
             || b == 0x65       /* GS prefix */
             || b == 0x66       /* Operand size prefix */
             || b == 0x67);     /* Address size prefix */
  if (b == 0xe8 || b == 0x9a)   /* CALL near label, CALL far label*/
    return (TRUE);
  if (b == 0xff)
    {
      rc = debug_read_byte (eip++, &b);
      if (rc != 0) return (TRUE);
      /* Note: This was off by one bit in child.asm! */
      if ((b & 0x38) == 0x10)   /* CALL near reg/mem */
        return (TRUE);
      if ((b & 0x38) == 0x18)   /* CALL far reg/mem */
        return (TRUE);
    }
  return (FALSE);
}


#define N_STOP                  (-1)
#define N_CONTINUE_STOP         (-2)
#define N_CONTINUE_SEARCH       (-3)
#define N_RESUME                (-4)


static int n_exception (void)
{
  int rc;
  ULONG report, info, w;

  switch (dbgbuf.Value)
    {
    case DBG_X_PRE_FIRST_CHANCE: /* pre first chance */
    case DBG_X_STACK_INVALID:   /* invalid stack */
      /* The exception number is in dbgbuf.Buffer. */
      if (dbgbuf.Buffer == XCPT_BREAKPOINT)
        {
          brk_flag = TRUE; more = FALSE;
          debug_set_wait (run_pid, 0x7f | (SIGTRAP << 8), FALSE);
          return (N_CONTINUE_STOP);
        }
      else if (dbgbuf.Buffer == XCPT_SINGLE_STEP)
        {
          more = FALSE;
          debug_set_wait (run_pid, 0x7f | (SIGTRAP << 8), FALSE);
          return (N_CONTINUE_STOP);
        }
      break;

    case DBG_X_FIRST_CHANCE:    /* first chance */
      report = dbgbuf.Buffer;   /* Address of report buffer */
      rc = debug_read_word (report, &w);
      if (rc == 0)
        switch (w)
          {
          case XCPT_GUARD_PAGE_VIOLATION:
          case XCPT_PROCESS_TERMINATE:
            second = TRUE;
            return (N_CONTINUE_SEARCH);

          case XCPT_SIGNAL:
            more = FALSE;
            info = offsetof (EXCEPTIONREPORTRECORD, ExceptionInfo) + 0;
            rc = debug_read_word (report + info, &w);
            if (rc == 0)
              switch (w)
                {
                case XCPT_SIGNAL_INTR:
                  debug_set_wait (run_pid, 0x7f | (SIGINT << 8), FALSE);
                  return (N_CONTINUE_STOP);

                case XCPT_SIGNAL_BREAK:
                  debug_set_wait (run_pid, 0x7f | (SIGBREAK << 8), FALSE);
                  return (N_CONTINUE_STOP);

                case XCPT_SIGNAL_KILLPROC:
                  debug_set_wait (run_pid, 0x7f | (SIGTERM << 8), FALSE);
                  return (N_CONTINUE_STOP);
                }
            break;
          }
      break;

    case DBG_X_LAST_CHANCE:     /* last chance */

      /* dbgbuf.Buffer does not point to the exception report record
         (documentation error or implementation error). */

      if (second)
        {
          second = FALSE;
          return (N_CONTINUE_SEARCH);
        }
      break;
    }

  more = FALSE;
  debug_set_wait (run_pid, 0x7f | (SIGSEGV << 8), FALSE);
  return (N_CONTINUE_STOP);
}


/* Handle the current notification and return N_STOP, N_CONTINUE_STOP,
   N_CONTINUE_SEARCH, N_RESUME, or errno (non-negative). */

static int notification (void)
{
  int rc;

  switch (dbgbuf.Cmd)
    {
    case DBG_N_Success:

      /* The request was completed successfully.  Run or step again
         unless `more' has been set to FALSE. */

      if (more)
        return (N_RESUME);

      /* Running or stepping the debuggee is completed.  Read the
         registers.  Skip over INT3 if a breakpoint was hit. */

      rc = read_reg (0);
      if (rc == 0 && brk_flag)
        {
          /* Skip the INT3 instruction for compatibility with emx.exe
             and GDB. */

          dbgbuf.EIP += 1;      /* Skip INT3 */
          rc = write_reg (dbgbuf.Tid);
        }
      return (rc);              /* ptrace() successful (usually) */

    case DBG_N_Error:

      /* An error occured. */

      oprintf ("DosDebug error: 0x%.8x\r\n", (unsigned)dbgbuf.Value);
      quit (255);

    case DBG_N_ProcTerm:

      /* Process terminated.  Let wait() return the termination
         code. */

      debug_set_wait (run_pid, (dbgbuf.Value & 0xff) << 8, TRUE);
      more = FALSE;
      rc = terminate ();
      return (rc);              /* ptrace() successful (usually) */

    case DBG_N_Exception:

      /* An exception occured. */

      return (n_exception ());

    case DBG_N_ModuleLoad:
    case DBG_N_ModuleFree:
    case DBG_N_ThreadTerm:
      return (N_CONTINUE_STOP);
          
    case DBG_N_ThreadCreate:
      run_tid = dbgbuf.Tid;
      return (N_CONTINUE_STOP);

    case DBG_N_Watchpoint:

      /* Watchpoint hit.  Watchpoints are currently used only for
         skipping over the call to emx_syscall when single-stepping.
         Stop the debuggee and let wait() indicate SIGTRAP. */

      more = FALSE;
      debug_set_wait (run_tid, 0x7f | (SIGTRAP << 8), FALSE);
      return (N_STOP);

    case DBG_N_NewProc:
    case DBG_N_AliasFree:
    default:

      /* These notifications are not expected to occur. */

      oprintf ("Unexpected DosDebug notification: 0x%.8x\r\n",
               (unsigned)dbgbuf.Cmd);
      quit (255);
    }
}


/* Perform a DBG_C_Go or DBG_C_SStep command. */

static ULONG run (ULONG cmd)
{
  ULONG rc;
  int next;

  if (auto_switch && cmd == DBG_C_Go)
    do_auto_switch ();
  more = TRUE; brk_flag = FALSE; second = FALSE; float_regs = FALSE;
  next = N_RESUME;
  for (;;)
    {
      switch (next)
        {
        case N_RESUME:
          if (auto_switch && cmd == DBG_C_SStep && callp ())
            do_auto_switch ();
          dbgbuf.Cmd = cmd;
          dbgbuf.Pid = run_pid;
          dbgbuf.Tid = 0;       /* All threads */
          rc = debug ();
          if (rc != 0) return (rc);
          break;

        case N_STOP:
          dbgbuf.Cmd = DBG_C_Stop;
          dbgbuf.Pid = run_pid;
          rc = debug ();
          if (rc != 0) return (rc);
          break;

        case N_CONTINUE_STOP:
        case N_CONTINUE_SEARCH:
          dbgbuf.Value = (next == N_CONTINUE_STOP
                          ? XCPT_CONTINUE_STOP : XCPT_CONTINUE_SEARCH);
          dbgbuf.Cmd   = DBG_C_Continue;
          dbgbuf.Pid   = run_pid;
          dbgbuf.Tid   = 1;
          rc = debug ();
          if (rc != 0) return (rc);
          break;

        default:
          /* errno value */
          return ((ULONG)next);
        }
      next = notification ();
    }
}


int do_ptrace (ULONG request, ULONG pid, ULONG addr, ULONG data, ULONG *errnop)
{
  ULONG rc, w;
  const struct reg_table *rp;

  dbgbuf.Pid = run_pid = pid;
  switch (request)
    {
    case PTRACE_EXIT:
      float_regs = FALSE;       /* Floating point status is now invalid */
      rc = terminate ();
      *errnop = rc;
      if (rc != 0) return (-1);
      debug_set_wait (run_pid, 0, TRUE);
      return (0);

    case PTRACE_PEEKTEXT:
    case PTRACE_PEEKDATA:
      rc = debug_read_word (addr, &w);
      *errnop = rc;
      if (rc != 0) return (-1);
      return (w);

    case PTRACE_POKETEXT:
    case PTRACE_POKEDATA:
      rc = debug_poke16 (addr, (ULONG)data & 0xffff);
      if (rc == 0)
        rc = debug_poke16 (addr +2, (ULONG)data >> 16);
      *errnop = rc;
      return (rc == 0 ? 0 : -1);

    case PTRACE_PEEKUSER:
      if (addr == USEROFF (u_ar0))
        w = USEROFF (u_regs) + KERNEL_U_ADDR;
      else if (addr == USEROFF (u_fpvalid))
        {
          if (!float_regs && get_fpstate () == 0)
            float_regs = TRUE;
          w = (float_regs ? 0xff : 0);
        }
      else if (addr == USEROFF (u_fpstate.status))
        w = 0;                  /* ... */
      else if (addr >= USEROFF (u_fpstate.state)
               && addr <= USEROFF (u_fpstate.state) + sizeof (dbgco) - 4)
        {
          if (!float_regs)
            {
              rc = get_fpstate ();
              if (rc != 0)
                {
                  *errnop = rc;
                  return (-1);
                }
              float_regs = TRUE;
            }
          memcpy (&w, (char *)&dbgco + addr - USEROFF (u_fpstate.state), 4);
        }
      else if ((rp = user_addr (addr)) != NULL)
        {
          w = 0;
          memcpy (&w, rp->debug, rp->len);
        }
      else
        {
          *errnop = EINVAL;
          return (-1);
        }
      *errnop = 0;
      return (w);

    case PTRACE_POKEUSER:
      rp = user_addr (addr);
      if (rp == NULL)
        rc = EIO;
      else
        rc = read_reg (0);
      if (rc == 0)
        {
          memcpy (rp->debug, &data, rp->len);
          rc = write_reg (dbgbuf.Tid);
        }
      *errnop = rc;
      return (rc == 0 ? 0 : -1);

    case PTRACE_RESUME:
      rc = run (DBG_C_Go);
      *errnop = rc;
      return (rc == 0 ? 0 : -1);

    case PTRACE_STEP:
      if (child_syscall != 0 && child_syscall == dbgbuf.EIP)
        {
          /* Avoid stepping into emx_syscall. */
          rc = set_watch (child_syscall + 5, 1, DBG_W_Execute);
          if (rc == 0)
            rc = run (DBG_C_Go);
        }
      else
        rc = run (DBG_C_SStep);
      *errnop = rc;
      return (rc == 0 ? 0 : -1);

    case PTRACE_SESSION:
      if (!debug_same_sess)
        switch (data)
          {
          case 0:               /* Switch to debugger */
            DosSelectSession (0);
            auto_switch = FALSE;
            break;

          case 1:               /* Switch to child */
            DosSelectSession (child_session);
            auto_switch = FALSE;
            break;

          case 2:               /* Automatic switch to child */
            auto_switch = TRUE;
            break;

          default:
            /* Succeed for undefined values. */
            break;
          }
      *errnop = 0;
      return (0);

    case PTRACE_TRACEME:
    default:
      *errnop = EINVAL;
      return (-1);
    }
}


/* Prepare a child process for debugging. */

ULONG spawn_debug (ULONG pid, ULONG sid)
{
  ULONG rc, w;
  BYTE b;

  run_tid = 0;
  float_regs = FALSE;
  child_syscall = 0;
  child_layout = 0;
  child_session = sid;
  dbgbuf.Pid = pid;
  dbgbuf.Tid = 0;               /* reserved */
  dbgbuf.Cmd = DBG_C_Connect;
  dbgbuf.Value = DBG_L_386;     /* level */
  rc = debug ();
  if (rc != 0) return (rc);
  if (dbgbuf.Cmd != DBG_N_Success)
    return (EINVAL);
  dbgbuf.Cmd = DBG_C_ReadReg;
  dbgbuf.Tid = 0;               /* Active thread */
  rc = debug ();
  if (rc != 0) return (rc);
  if (dbgbuf.Cmd != DBG_N_Success)
    return (EINVAL);
  if (debug_read_byte (ENTRY_POINT+0, &b) == 0 && b == 0x68     /* PUSH n */
      && debug_read_byte (ENTRY_POINT+5, &b) == 0 && b == 0xe8  /* CALL */
      && debug_read_byte (ENTRY_POINT+10, &b) == 0 && b == 0xeb /* JMP */
      && debug_read_byte (ENTRY_POINT+12, &b) == 0 && b == 0xe8 /* CALL */
      && debug_read_word (ENTRY_POINT+1, &w) == 0)
    {
      child_layout = w;
      child_syscall = ENTRY_POINT + 12;
    }
  return (0);
}
