/* select.c -- Implement select()
   Copyright (c) 1993-1994 by Eberhard Mattes

This file is part of emx.

emx is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

emx is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with emx; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

As special exception, emx.dll can be distributed without source code
unless it has been changed.  If you modify emx.dll, this exception
no longer applies and you must remove this paragraph from all source
files for emx.dll.  */


#define INCL_DOSDEVIOCTL
#define INCL_DOSSEMAPHORES
#define INCL_DOSPROCESS
#define INCL_DOSMISC
#define INCL_DOSERRORS
#include <os2emx.h>
#include <sys/emx.h>
#include <sys/types.h>
#include <sys/termio.h>
#include <sys/time.h>
#include <sys/errno.h>
#include "emxdll.h"
#include "files.h"
#include "tcpip.h"
#include "select.h"
#include "clib.h"

static HEV npipe_sem;
static BYTE npipe_sem_open;

HEV socket_sem;
HEV socket_done_sem;
HEV socket_start_sem;
BYTE select_socket_done;
static BYTE socket_sem_open;


static int select_add_read (struct select_data *d, int fd)
{
  ULONG rc, ht;

  if (fd >= handle_count)
    return (EBADF);
  ht = handle_flags[fd];
  if (!(ht & HF_OPEN))
    return (EBADF);
  if (ht & HF_NPIPE)
    {
      if (!d->sem_npipe_flag)
        {
          if (!npipe_sem_open)
            {
              if (create_event_sem (&npipe_sem, DC_SEM_SHARED) != 0)
                return (EINVAL);
              npipe_sem_open = TRUE;
            }
          if (reset_event_sem (npipe_sem) != 0)
            return (EINVAL);
          d->list[d->sem_count].hsemCur = (HSEM)npipe_sem;
          d->list[d->sem_count].ulUser = 0;
          ++d->sem_count;
          d->sem_npipe_flag = TRUE;
        }
      rc = DosSetNPipeSem (fd, (HSEM)npipe_sem, 0);
      if (rc != 0)
        {
          error (rc, "DosSetNPipeSem");
          return (EACCES);
        }
    }
  else if (ht & HF_CON)
    {
      if (IS_VALID_FILE (fd) && !(GET_FILE (fd)->c_lflag & IDEFAULT)
          && !d->sem_kbd_flag)
        {
          if (reset_event_sem (kbd_sem_new) != 0)
            return (EINVAL);
          d->list[d->sem_count].hsemCur = (HSEM)kbd_sem_new;
          d->list[d->sem_count].ulUser = 0;
          ++d->sem_count;
          d->sem_kbd_flag = TRUE;
        }
    }
  else if (ht & HF_SOCKET)
    {
      if (d->socket_count >= SELECT_MAX_SOCKETS || !IS_VALID_FILE (fd))
        return (EBADF);
      d->sockets[d->socket_count] = GET_FILE (fd)->x.socket.handle;
      d->socketh[d->socket_count] = fd;
      ++d->socket_count; ++d->socket_nread;
    }
  else
    {
      /* Ignore other types of handles */
    }
  return (0);
}


static int select_add_write (struct select_data *d, int fd)
{
  ULONG ht;

  if (fd >= handle_count)
    return (EBADF);
  ht = handle_flags[fd];
  if (!(ht & HF_OPEN))
    return (EBADF);
  if (ht & HF_SOCKET)
    {
      if (d->socket_count >= SELECT_MAX_SOCKETS || !IS_VALID_FILE (fd))
        return (EBADF);
      d->sockets[d->socket_count] = GET_FILE (fd)->x.socket.handle;
      d->socketh[d->socket_count] = fd;
      ++d->socket_count; ++d->socket_nwrite;
    }
  else
    {
      /* Ignore other types of handles */
    }
  return (0);
}


static int select_add_except (struct select_data *d, int fd)
{
  ULONG ht;

  if (fd >= handle_count)
    return (EBADF);
  ht = handle_flags[fd];
  if (!(ht & HF_OPEN))
    return (EBADF);
  if (ht & HF_SOCKET)
    {
      if (d->socket_count >= SELECT_MAX_SOCKETS || !IS_VALID_FILE (fd))
        return (EBADF);
      d->sockets[d->socket_count] = GET_FILE (fd)->x.socket.handle;
      d->socketh[d->socket_count] = fd;
      ++d->socket_count; ++d->socket_nexcept;
    }
  else
    {
      /* Ignore other types of handles */
    }
  return (0);
}


static int select_init (struct select_data *d, struct _select *args)
{
  int fd, e;
  ULONG rc;
  struct timeval *tv;

  d->timeout = SEM_INDEFINITE_WAIT;
  tv = args->timeout;
  if (tv != NULL)
    {
      if (tv->tv_sec < 0 || tv->tv_sec >= 4294967 || tv->tv_usec < 0)
        return (EINVAL);
      d->timeout = tv->tv_sec * 1000;
      d->timeout += tv->tv_usec / 1000; /* TODO: overflow? rounding? */
    }
  if (args->readfds != NULL)
    for (fd = 0; fd < args->nfds; ++fd)
      if (FD_ISSET (fd, args->readfds))
        {
          e = select_add_read (d, fd);
          if (e != 0)
            return (e);
        }
  if (args->writefds != NULL)
    for (fd = 0; fd < args->nfds; ++fd)
      if (FD_ISSET (fd, args->writefds))
        {
          e = select_add_write (d, fd);
          if (e != 0)
            return (e);
        }
  if (args->exceptfds != NULL)
    for (fd = 0; fd < args->nfds; ++fd)
      if (FD_ISSET (fd, args->exceptfds))
        {
          e = select_add_except (d, fd);
          if (e != 0)
            return (e);
        }

  /* If socket handles are involved and the time-out is non-zero, add
     a semaphore which will be posted when a socket handle becomes
     ready. */

  if (d->socket_count != 0 && d->timeout != 0)
    {
      if (!socket_sem_open)
        {
          if (create_event_sem (&socket_sem, DC_SEM_SHARED) != 0
              || create_event_sem (&socket_done_sem, 0) != 0
              || create_event_sem (&socket_start_sem, 0) != 0)
            return (EINVAL);
          socket_sem_open = TRUE;
        }
      if (reset_event_sem (socket_sem) != 0
          || reset_event_sem (socket_done_sem) != 0
          || reset_event_sem (socket_start_sem) != 0)
        return (EINVAL);
      d->list[d->sem_count].hsemCur = (HSEM)socket_sem;
      d->list[d->sem_count].ulUser = 0;
      ++d->sem_count;
    }

  d->list[d->sem_count].hsemCur = (HSEM)signal_sem;
  d->list[d->sem_count].ulUser = 1;
  ++d->sem_count;
  rc = DosCreateMuxWaitSem (NULL, &d->sem_mux, d->sem_count, d->list,
                            DCMW_WAIT_ANY);
  if (rc != 0)
    {
      error (rc, "DosCreateMuxWaitSem");
      return (EINVAL);
    }
  d->sem_mux_flag = TRUE;
  return (0);
}


static void select_check_read (struct select_data *d, int fd)
{
  ULONG rc, ht;

  ht = handle_flags[fd];
  if (!(ht & HF_OPEN))
    return;
  if (ht & HF_NPIPE)
    {
      ULONG buffer, nread, state;
      AVAILDATA avail;

      rc = DosPeekNPipe (fd, &buffer, 0, &nread, &avail, &state);
      if (rc != 0)
        {
          /* Ignore */
        }
      else if (avail.cbpipe != 0)
        {
          FD_SET (fd, &d->rbits);
          ++d->ready_count;
        }
    }
  else if (ht & HF_CON)
    {
      if (IS_VALID_FILE (fd) && !(GET_FILE (fd)->c_lflag & IDEFAULT)
          && termio_avail (fd) != 0)
        {
          FD_SET (fd, &d->rbits);
          ++d->ready_count;
        }
    }
  else if (ht & HF_ASYNC)
    {
      if (async_avail (fd) != 0)
        {
          FD_SET (fd, &d->rbits);
          ++d->ready_count;
        }
    }
}


static int select_poll (struct select_data *d, struct _select *args)
{
  int fd;

  FD_ZERO (&d->rbits);
  FD_ZERO (&d->wbits);
  FD_ZERO (&d->ebits);

  if (args->readfds != 0)
    {
      for (fd = 0; fd < args->nfds; ++fd)
        if (FD_ISSET (fd, args->readfds))
          select_check_read (d, fd);
    }

  if (d->socket_count != 0)
    {
      int n, e;

      n = tcpip_select_poll (d, &e);
      if (n < 0)
        return (e);
    }

  if (d->ready_count != 0)
    {
      if (args->readfds != 0)
        *args->readfds = d->rbits;
      if (args->writefds != 0)
        *args->writefds = d->wbits;
      if (args->exceptfds != 0)
        *args->exceptfds = d->ebits;
      d->return_value = d->ready_count;
    }
  return (0);
}


/* Wait until a semaphore is posted (pipe ready, keyboard data
   available, socket ready, signal) or until the timeout elapses,
   whichever comes first.  Return -1 on timeout.  Otherwise return 0
   or errno. */

static int select_wait (struct select_data *d)
{
  ULONG rc, user;
  TID socket_tid;

  if (d->timeout == 0)
    return (-1);                /* Timed out */

  if (d->socket_count != 0)
    {
      select_socket_done = FALSE;
      rc = DosCreateThread (&socket_tid, tcpip_select_thread,
                            (ULONG)d, 0, 0x4000);
      if (rc != 0)
        {
          error (rc, "DosCreateThread");
          return (EINVAL);
        }

      /* Wait until tcpip_select_thread() has installed its exception
         handler. */
      do
        {
          rc = DosWaitEventSem (socket_start_sem, SEM_INDEFINITE_WAIT);
        } while (rc == ERROR_INTERRUPT);
      if (rc != 0)
        error (rc, "DosWaitEventSem");
    }

  rc = DosWaitMuxWaitSem (d->sem_mux, d->timeout, &user);

  if (d->socket_count != 0)
    {
      ULONG rc2;

      /* Fortunately, a thread blocked in the select() function of IBM
         TCP/IP for OS/2 can be killed.  At least under OS/2 3.0.
         Unfortunately, DosKillThread is reported to not work
         correctly under OS/2 2.1 and older. */

      select_socket_done = TRUE;
      if (!dont_doskillthread)
        DosKillThread (socket_tid);
      do
        {
          rc2 = DosWaitEventSem (socket_done_sem, SEM_INDEFINITE_WAIT);
        } while (rc2 == ERROR_INTERRUPT);
      if (rc2 != 0)
        error (rc2, "DosWaitEventSem");
    }

  if (rc == ERROR_INTERRUPT || sig_flag)
    return (EINTR);
  if (rc == ERROR_TIMEOUT)
    return (-1);
  if (rc != 0)
    {
      error (rc, "DosWaitMuxWaitSem");
      return (EINVAL);
    }
  return (0);
}


static void select_cleanup (struct select_data *d)
{
  if (d->sem_mux_flag)
    DosCloseMuxWaitSem (d->sem_mux);
}


int do_select (struct _select *args, int *errnop)
{
  struct select_data d;
  int e;

  sig_block_start ();
  d.td = get_thread ();
  d.sem_npipe_flag = FALSE; d.sem_kbd_flag = FALSE; d.sem_mux_flag = FALSE;
  d.sem_count = 0; d.ready_count = 0; d.return_value = 0;
  d.socket_count = 0; d.socket_nread = d.socket_nwrite = d.socket_nexcept = 0;
  e = select_init (&d, args);
  if (e == 0)
    {
      e = select_poll (&d, args);
      if (e == 0 && d.return_value == 0) /* No handles ready */
        {
          e = select_wait (&d);
          if (e == -1)
            e = 0;
          else if (e == 0)
            e = select_poll (&d, args);
        }
    }
  select_cleanup (&d);
  sig_block_end ();
  *errnop = e;
  return (e == 0 ? d.return_value : -1);
}
