 /*
 * Copyright (C) 1992, 1995 Stephen A. Wood.  All rights reserved.
 *
 * This file is part of PAF C-Tools.
 *  
 * These programs are free software; you can redistribute them and/or modify
 * them under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * The GNU General Public License can be found in the file LICENSE.
 * 
 * The author of PAF C-Tools is Stephen A. Wood.  He may be reached on internet
 * at the address saw@cegaf.gov, or via mail at 328 Dominion Drive,
 * Newport News, VA 23602.
 * 
 * ------------
 * 
 * This file contains subroutines used by the various PAF C-Tools programs.
 *
 * These routines are kind of restrictive as they only easily support a single
 * PAF database open at a time.  Instead of requiring the application
 * programmer to make an open call for each of the files in a PAF database,
 * we should set up a system where a single call is made and a handle for
 * the opened database is returned.  This handle can then be passed to the
 * various database manipulation routines.
 *
 * Last modified:
 *    1/31/95 Add path name argument to open calls so that the PAF database
 *            can be found in other than the current directory.
 */

#include <stdio.h>
#include "pafsubs.h"

PAF_FILE paf_name = {"name2.dat", 21};
PAF_FILE paf_indiv = {"indiv2.dat", 92};
PAF_FILE paf_marr = {"marr2.dat", 28};
PAF_FILE paf_notes = {"notes2.dat", 256};
PAF_FILE paf_namadd = {"namadd2.dat", 224};

void get_paf_rec(RECORD_PTR recno,PAF_FILE *fil,void *buf);
void write_paf_rec(RECORD_PTR recno,PAF_FILE *fil,void *buf);
int paf_open(char *path, char c,PAF_FILE *fil);

int paf_open_name(char *path, char c){return(paf_open(path, c,&paf_name));}
int paf_open_indiv(char *path, char c){return(paf_open(path, c,&paf_indiv));}
int paf_open_marr(char *path, char c){return(paf_open(path, c,&paf_marr));}
int paf_open_notes(char *path, char c)
{
  NOTE_REC pad;

  if(paf_open(path, c, &paf_notes)) return 1;
  /* First record is pointer to header of free list */
  /* No notes are ever put in the first record */
  get_note_rec(1,&pad);
  paf_notes.freehead = pad.next_pad;
  paf_notes.freetail = get_tail(paf_notes.freehead,&paf_notes);
/*  fprintf(stderr,"Freehead=%d, Freetail=%d\n",paf_notes.freehead
	  ,paf_notes.freetail);
*/
  return 0;
}
int paf_open_namadd(char *path, char c){return(paf_open(path, c,&paf_namadd));}


int paf_open(char *path, char c, PAF_FILE *fil)
{
  char cbuf[11];
  char mode[4];
  char *file_name;
  char *p;

  mode[0] = 'r';
  if(c == 'w'){
    mode[1] = '+';
    mode[2] = 'b';
    mode[3] = 0;
  } else {
    mode[1] = 'b';
    mode[2] = 0;
  }
  {                             /* Put the path in front of the file name */
    char x;
    file_name = (char *) malloc(strlen(path)+strlen(fil->fname)+3);
    strcpy(file_name,path);
    if(strlen(file_name) > 0) {
      x = file_name[strlen(file_name)-1];
      if(x != '\\' && x != '/') {
	strcat(file_name,"/");
      }
      strcat(file_name,fil->fname);
    } else
      strcpy(file_name,fil->fname);
  }
  if((fil->ptr = fopen(file_name,mode)) == NULL){
    fprintf(stderr, "Cannot open %s.\n",file_name);
    return 1;
  }
  (void) fgetc(fil->ptr);
  (void) fread(cbuf,1,10,fil->ptr);
  cbuf[11] = '\0';
  fil->nrec = atoi(cbuf);
  fil->currec = -1;
  if(c=='w'){
    (void) fread(cbuf,1,10,fil->ptr);
    cbuf[11] = '\0';
    fil->freehead = atoi(cbuf);
    fil->freetail = get_tail(fil->freehead,fil);
  }
/*  fprintf(stderr,"%d Records, Freehead=%d, Freetail=%d\n",fil->nrec
	  ,fil->freehead,fil->freetail);
*/
  return 0;
}


PAF_FILE *get_paf_filed(enum paffiles type)
{
  switch(type)
    {
    case NAME:
      return(&paf_name);
    case INDIV:
      return(&paf_indiv);
    case MARR:
      return(&paf_marr);
    case NOTE:
      return(&paf_notes);
    case NAMADD:
      return(&paf_namadd);
    }
  return(0);
}
void get_paf_rec(RECORD_PTR recno,PAF_FILE *fil,void *buf)
{
  long int offset;
	int i;
  
  offset = (long int) recno*fil->reclen;
  (void) fseek(fil->ptr,offset,SEEK_SET);
  i = fread(buf,1,fil->reclen,fil->ptr);
  if(i<=0)printf("Error in read %d\n",recno);
  fil->currec = recno + 1;
  return;
}
  
void get_name_rec(RECORD_PTR recno,NAME_REC *namrec)
{
  char namraw[sizeof(namrec->left)+sizeof(namrec->name)+sizeof(namrec->right)];

  get_paf_rec(recno,&paf_name,namraw);
  memcpy(&namrec->left,namraw,sizeof(namrec->left));
  memcpy(namrec->name,&namraw[2],sizeof(namrec->name));
  memcpy(&namrec->right,&namraw[19],sizeof(namrec->right));
}
void get_indiv_rec(unsigned short int rin,INDIV_REC *indrec)
{
  char indraw[92];

  get_paf_rec(rin,&paf_indiv,indraw);
  memcpy(&indrec->names[0],&indraw[0],10);
  indrec->sex = indraw[10];
  memcpy(&indrec->birthdate,&indraw[11],48);
  memcpy(indrec->ldsinfo,&indraw[59],15); /* Just save LDS, but don't parse */
  memcpy(&indrec->oldersib,&indraw[74],18);
}
void get_marr_rec(unsigned short int mrin,MARR_REC *marec)
{
  char marraw[28];

  get_paf_rec(mrin,&paf_marr,marraw);
  memcpy(&marec->husband,&marraw[0],6);
  memcpy(&marec->marriagedate,&marraw[6],17);
  memcpy(&marec->mnext_husband,&marraw[23],4);
  marec->divorce = marraw[27];
}
void get_note_rec(unsigned short int nbp,NOTE_REC *noterec)
{
  char notearray[256];

  get_paf_rec(nbp,&paf_notes,notearray);
  memcpy(&noterec->next_pad,notearray,2);
  memcpy(noterec->notelines,&notearray[2],254);
}
void get_namadd_rec(RECORD_PTR recno, NAMADD_REC *namaddrec)
{
  char raw[224];

  get_paf_rec(recno,&paf_namadd,raw);
  memcpy(namaddrec->name,raw,41);
  memcpy(namaddrec->addr1,&raw[41],41);
  memcpy(namaddrec->addr2,&raw[82],41);
  memcpy(namaddrec->addr3,&raw[123],41);
  memcpy(namaddrec->phone,&raw[164],26);
  memcpy(namaddrec->stake,&raw[190],26);
  memcpy(namaddrec->unit,&raw[216],8);
  return;
}
void write_paf_rec(RECORD_PTR recno,PAF_FILE *fil,void *buf)
{
  long int offset;
  
  offset = (long int) recno*fil->reclen;
  (void) fseek(fil->ptr,offset,SEEK_SET);
  (void) fwrite(buf,1,fil->reclen,fil->ptr);
  fil->currec = recno + 1;
  return;
}
void write_indiv_rec(unsigned short int rin,INDIV_REC *indrec)
{
  char indraw[92];

  memcpy(&indraw[0],&indrec->names[0],10);
  indraw[10] = indrec->sex;
  memcpy(&indraw[11],&indrec->birthdate,48);
  memcpy(&indraw[59],indrec->ldsinfo,15);
  memcpy(&indraw[74],&indrec->oldersib,18);
  write_paf_rec(rin,&paf_indiv,indraw);
}

void add_to_freelist(RECORD_PTR newfree, PAF_FILE *fil)
{
  long int offset;
  RECORD_PTR i;

  if(fil->freetail !=0) {
    offset = (long int) fil->freetail*fil->reclen;
    (void) fseek(fil->ptr,offset,SEEK_SET);
    (void) fread(&i,1,2,fil->ptr);
    if(i != 0) {printf("TROUBLE\n");exit(1);}
    (void) fseek(fil->ptr,offset,SEEK_SET);
    (void) fwrite(&newfree,1,2,fil->ptr);
  } else {
    fil->freehead = newfree;
/* Assume it is a notes  file  here */
    {
      long int off;
      off = fil->reclen;
      (void) fseek(fil->ptr,off,SEEK_SET);
      (void) fwrite(&newfree,1,2,fil->ptr);
    }
  }
  fil->freetail = get_tail(newfree,fil);

}
RECORD_PTR get_tail(RECORD_PTR first, PAF_FILE *fil)
{
  RECORD_PTR next,last;
  long int offset;

  next = first;
  last = 0;
  while(next != 0){
    last = next;
    offset = (long int) next*fil->reclen;
    (void) fseek(fil->ptr,offset,SEEK_SET);
    (void) fread(&next,1,2,fil->ptr);
  }
  return(last);
}

RECORD_PTR get_next(RECORD_PTR recno, PAF_FILE *fil)
{
  RECORD_PTR next;
  long int offset;

  offset = (long int) recno*fil->reclen;
  (void) fseek(fil->ptr,offset,SEEK_SET);
  (void) fread(&next,1,2,fil->ptr);
  return(next);
}

void write_note_rec(unsigned short int nbp,NOTE_REC *noterec)
{
  char notearray[256];

  memcpy(notearray,&noterec->next_pad,2);
  memcpy(&notearray[2],noterec->notelines,254);
  write_paf_rec(nbp,&paf_notes,notearray);
}


/*
 * Return pointer to the first record on the free list
 * Reset the free list pointer to point to the next.
 * Only works for notes files.
 */
RECORD_PTR get_free_rec(PAF_FILE *fil)
{
  long int offset;
  RECORD_PTR next,freerec;
  char rec_count[11];
  
  if((freerec = fil->freehead) != 0){
    offset = (long int) freerec*fil->reclen;
    (void) fseek(fil->ptr,offset,SEEK_SET);
    (void) fread(&next,1,2,fil->ptr);
    fil->freehead = next;
/* Assume it is a notes  file  here */
    {
      long int off;
      off = fil->reclen;
      (void) fseek(fil->ptr,off,SEEK_SET);
      (void) fwrite(&next,1,2,fil->ptr);
    }
    if(next == 0) fil->freetail = 0;
    else {
      next = 0;
      (void) fseek(fil->ptr,offset,SEEK_SET);
      (void) fwrite(&next,1,2,fil->ptr);
    }
  }else{        /* No free records, add one to end of file. */
    freerec = ++fil->nrec;
    offset = (long int) freerec*fil->reclen;
    next = 0;
    (void) fseek(fil->ptr,(long int) 0,SEEK_SET);
    sprintf(rec_count," %-9d",fil->nrec);
    printf("Change rec_count to %d\n",fil->nrec);
    if(!fwrite(rec_count,1,10,fil->ptr))
	printf("Error change rec_count to %d\n",fil->nrec);
    (void) fseek(fil->ptr,offset,SEEK_SET);
    (void) fwrite(&next,1,2,fil->ptr); /* Zero out pointer just in case */
  }
  return(freerec);
}
    
