// Windows Progress Bar Control

#include <windows.h>
#include "progress.h"

extern "C" {
  long far pascal _export ProgressBarProc(HWND, UINT, WPARAM, LPARAM);
}

/*-----------------------------------------------------------------------------
This function registers the PROGRESSBAR class. It is called automatically when
the program starts up.

The external variables _hPrev and _hInstance duplicate the arguments
hPrevInstance and hInstance, which are passed to WinMain(). If the startup
code does not supply these external variables, you must pass the arguments to
this function and call it explicitly before invoking any PROGRESSBAR control.
-----------------------------------------------------------------------------*/

extern HINSTANCE _hPrev, _hInstance;

static void register_progress_bar(void)
{
  if (!_hPrev)
  {
    WNDCLASS w;
    w.cbClsExtra = 0;
    w.cbWndExtra = 4*sizeof(WORD);
    w.hbrBackground = (HBRUSH) COLOR_WINDOW + 1;
    w.hCursor = LoadCursor(NULL, IDC_ARROW);
    w.hIcon = NULL;
    w.hInstance = _hInstance;
    w.lpfnWndProc = ProgressBarProc;
    w.lpszClassName = "PROGRESSBAR";
    w.lpszMenuName = NULL;
    w.style = 0;
    RegisterClass(&w);
  }
}

#pragma startup register_progress_bar

/*-----------------------------------------------------------------------------
This function receives all messages directed to the control.
-----------------------------------------------------------------------------*/

long far pascal _export ProgressBarProc(HWND handle, UINT message,
  WPARAM wParam, LPARAM lParam)
{
  WORD height = GetWindowWord(handle, 0);
  WORD width = GetWindowWord(handle, 2);
  WORD completed = GetWindowWord(handle, 4);
  WORD total = GetWindowWord(handle, 6);
  switch (message)
  {
    case WM_CREATE:
    {
      SetWindowWord(handle, 0, ((CREATESTRUCT far *) lParam)->cy);  // height
      SetWindowWord(handle, 2, ((CREATESTRUCT far *) lParam)->cx);  // width
      SetWindowWord(handle, 6, 100); // default value of total
      return 0;
    }
    case WM_PAINT:
    {
      PAINTSTRUCT paint;
      HDC dc = BeginPaint(handle, &paint);
      HPEN pen = CreatePen(PS_SOLID, 1, GetSysColor(COLOR_WINDOWTEXT));
      HPEN old_pen = SelectObject(dc, pen);
      HBRUSH brush = CreateSolidBrush(GetSysColor(COLOR_WINDOW));
      HBRUSH old_brush = SelectObject(dc, brush);
      RECT r;
      char text[5];
      // draw border
      Rectangle(dc, 0, 0, width-1, height-1);
      // display completion percentage
      SetTextColor(dc, GetSysColor(COLOR_WINDOWTEXT));
      SetBkColor(dc, GetSysColor(COLOR_WINDOW));
      r.left = r.top = 1;
      r.bottom = height - 2;
      r.right = width - 2;
      wsprintf(text, "%d%%", (100L * (DWORD) completed) / (DWORD) total);
      DrawText(dc, text, lstrlen(text), &r, DT_CENTER | DT_VCENTER | DT_SINGLELINE);
      // invert progress portion
      r.right = 1 + ((DWORD) (width - 3) * (DWORD) completed) / (DWORD) total;
      if (r.right > r.left)
        InvertRect(dc, &r);
      SelectObject(dc, old_pen);
      DeleteObject(pen);
      SelectObject(dc, old_brush);
      DeleteObject(brush);
      EndPaint(handle, &paint);
      return 0;
    }
    case PB_SETCOMPLETED:
    {
      RECT r;
      r.top = 1;
      r.bottom = height - 2;
      SetWindowWord(handle, 4, wParam); // completed
      if ((100L * (DWORD) wParam) / (DWORD) total != (100L * (DWORD) completed) / (DWORD) total)
      {
        HDC dc = GetDC(handle);
        unsigned text_width = (WORD) GetTextExtent(dc, "100%", 4);
        r.left = 1 + (width - 2) / 2 - (text_width + 1) / 2;
        r.right = 1 + (width - 2) / 2 + (text_width + 1) / 2;
        InvalidateRect(handle, &r, TRUE);
        ReleaseDC(handle, dc);
      }
      r.left = 1 + ((DWORD) (width - 3) * (DWORD) completed) / (DWORD) total;
      r.right = 1 + ((DWORD) (width - 3) * (DWORD) wParam) / (DWORD) total;
      if (r.left != r.right)
      {
        if (r.left > r.right)
        {
          int x = r.left;
          r.left = r.right;
          r.right = x;
        }
        InvalidateRect(handle, &r, TRUE);
      }
      return 0;
    }
    case PB_GETCOMPLETED:
      return completed;
    case PB_SETTOTAL:
      SetWindowWord(handle, 6, wParam); // total
      SetWindowWord(handle, 4, 0); // completed
      InvalidateRect(handle, NULL, TRUE);
      return 0;
    case PB_GETTOTAL:
      return total;
  }
  return DefWindowProc(handle, message, wParam, lParam);
}

