//****************************************************************************
// File:
// 
//     mainwnd.cpp
//
// Purpose:
//
//     Implementation file for the CMainWindow class.
//
// Functions:
//
//     CMainWindow::CMainWindow           - Constructor for the main window.
//     CMainWindow::DoDataExchange        - Exchanges data between class
//                                            members and window controls.
//     CMainWindow::OnInitDialog          - Handles initial setup of main
//                                            window dialog.
//     CMainWindow::OnSysColorChange      - Used by CTL3D32.DLL.
//     CMainWindow::OnClickedAbout        - Displays the "About" dialog box.
//     CMainWindow::OnClickedDoSort       - Calls the sorting DLL.
//     CMainWindow::OnClickedClearResults - Clears results from previous sorts.
//     CMainWindow::OnClickedCancelSort   - Cancels currently active sorts.
//     RegisterThreadsFinished            - Keeps track of finished sorting
//                                            threads.
//     ShowResults                        - Displays the sort results.
//     ShowElapsedTime                    - Displays sort elapsed time.
//
//
// Development Team:
//
//     Joel Krist
//
// Written by Microsoft Product Support Services, Languages Developer Support
// Copyright (c) 1993 Microsoft Corporation. All rights reserved.
//****************************************************************************

#include "stdafx.h"
#include "sortdemo.h"
#include "aboutbox.h"

#define _BUILDING_EXE_
#include "mainwnd.h"
#undef _BUILDING_EXE_

#include "sortdll.h"
#include "ctl3d.h"
#include "resource.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

// pMainWnd is initialized in CSortdemoApp::InitInstance in sortdemo.cpp
extern CMainWindow * pMainWnd;

// Handle to the main dialog window
HWND hwndMainWindow;

// Delare handles to the buttons which need to be enabled and disabled
// from functions which are not CMainWindow members. These are initialized in
// CMainWindow::OnInitDialog()
HWND hwndDoSortButton;
HWND hwndClearResultsButton;
HWND hwndCancelSortButton;
HWND hwndExitButton;

// These ints are used to keep track of when all sorting threads are finished
int nThreadsStarted, nThreadsFinished;

//***********************************************************************
// Function:
//
//     CMainWindow::CMainWindow()
//
// Purpose:
//
//     Constructs a CMainWindow object and takes care of initializing
//     member data to appropriate values.
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

CMainWindow::CMainWindow(CWnd* pParent /*=NULL*/)
      : CDialog(CMainWindow::IDD, pParent)
{
    //{{AFX_DATA_INIT(CMainWindow)
    m_nElements = 20;
    m_nListOrder = 1;
    m_fDoBubbleSort = FALSE;
    m_fDoExchangeSort = FALSE;
    m_fDoHeapSort = FALSE;
    m_fDoInsertionSort = FALSE;
    m_fDoQuickSort = FALSE;
    m_fDoShellSort = FALSE;
    m_nWhenToDisplayResults = 1;
    //}}AFX_DATA_INIT
}

//***********************************************************************
// Function:
//
//     CMainWindow::DoDataExchange() 
//
// Purpose:
//
//     Handles the exchange of data between CMainWindow members and
//     window controls.
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

void CMainWindow::DoDataExchange(CDataExchange* pDX)
{
    CDialog::DoDataExchange(pDX);
    //{{AFX_DATA_MAP(CMainWindow)
    DDX_Control(pDX, IDCANCEL, m_ExitButton);
    DDX_Control(pDX, IDC_CLEARRESULTS, m_ClearResultsButton);
    DDX_Control(pDX, IDC_DOSORT, m_DoSortButton);
    DDX_Control(pDX, IDC_BUBBLETYPE, m_BubbleControl);
    DDX_Control(pDX, IDC_CANCELSORT, m_CancelSortButton);
    DDX_Text(pDX, IDC_NUMBEROFELEMENTS, m_nElements);
    DDV_MinMaxInt(pDX, m_nElements, 2, 20000);
    DDX_Radio(pDX, IDC_ORDERED, m_nListOrder);
    DDX_Check(pDX, IDC_BUBBLETYPE, m_fDoBubbleSort);
    DDX_Check(pDX, IDC_EXCHANGETYPE, m_fDoExchangeSort);
    DDX_Check(pDX, IDC_HEAPTYPE, m_fDoHeapSort);
    DDX_Check(pDX, IDC_INSERTIONTYPE, m_fDoInsertionSort);
    DDX_Check(pDX, IDC_QUICKTYPE, m_fDoQuickSort);
    DDX_Check(pDX, IDC_SHELLTYPE, m_fDoShellSort);
    DDX_Radio(pDX, IDC_RESULTSAFTER, m_nWhenToDisplayResults);
    //}}AFX_DATA_MAP
}

// Declare the message map for CMainWindow
BEGIN_MESSAGE_MAP(CMainWindow, CDialog)
    //{{AFX_MSG_MAP(CMainWindow)
    ON_BN_CLICKED(IDC_ABOUT, OnClickedAbout)
    ON_BN_CLICKED(IDC_DOSORT, OnClickedDoSort)
    ON_BN_CLICKED(IDC_CLEARRESULTS, OnClickedClearResults)
    ON_BN_CLICKED(IDC_CANCELSORT, OnClickedCancelSort)
    ON_WM_SYSCOLORCHANGE()
    //}}AFX_MSG_MAP
END_MESSAGE_MAP()


//***********************************************************************
// Function:
//
//     CMainWindow::OnInitDialog()
//
// Purpose:
//
//     Takes care of centering the main dialog window on the desktop and
//     initializing the global handles to the Sort, Cancel Sort, and
//     Clear Results buttons. Also initializes the global main window
//     handle used in ShowResults() and ShowElapsedTime().
//
// Parameters:
//
//     None.
//          
// Returns:
//
//     A BOOL value of TRUE indicating that the focus should be set to
//     the first control in the dialog box.
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

BOOL CMainWindow::OnInitDialog()
{    
    // Call the base class implementation
    CDialog::OnInitDialog();
    
    // Center the main window on the desktop
    CenterWindow();

    // Initialize button handles
    hwndDoSortButton = m_DoSortButton.m_hWnd;
    hwndClearResultsButton = m_ClearResultsButton.m_hWnd;
    hwndCancelSortButton = m_CancelSortButton.m_hWnd;
    hwndExitButton = m_ExitButton.m_hWnd;

    // Initialize the main window handle
    hwndMainWindow = pMainWnd->GetSafeHwnd();
    
    // Specify that the focus should be set to the first control
    return TRUE;
}

//***********************************************************************
// Function:
//
//     CMainWindow::OnSysColorChange()
//
// Purpose:
//
//     Used by CTL3D32.DLL to enable 3-D controls.
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

void CMainWindow::OnSysColorChange()
{
    // Needed by the CTL3D32 DLL
    Ctl3dColorChange();
}

//***********************************************************************
// Function:
//
//     CMainWindow::OnClickedAbout()
//
// Purpose:
//     
//     Displays the About dialog box when the user clicks the About
//     button.
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

void CMainWindow::OnClickedAbout()
{
    CAboutDlg dlg( this );
    
    dlg.DoModal();  
}

//***********************************************************************
// Function:
//
//     CMainWindow::OnClickedDoSort()
//
// Purpose:
//
//     Calls the DLL to do the sorts requested by the user. 
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
// 11/16/93 Added code to check the number of threads started   JKK
//***********************************************************************

void CMainWindow::OnClickedDoSort()
{
    int nTmpMask;
    int nNumberOfSortsSelected = 0;

    // Initialize the mask used to represent the sorts selected by the user
    m_nSelectedSortMask = 0;

    // Move control values to member variables
    if (!UpdateData())  
        return;

    // Create a simple mask to represent the sorts the user selected
    m_nSelectedSortMask = ( m_fDoBubbleSort | ( m_fDoExchangeSort << 1 ) | ( m_fDoHeapSort << 2 ) |
                 ( m_fDoInsertionSort << 3 ) | ( m_fDoQuickSort << 4 ) | ( m_fDoShellSort << 5 ) );
                    
    if( m_nSelectedSortMask )  
    {
        // The user selected at least 1 type of sort so enable and disable
        // the correct buttons
        m_DoSortButton.EnableWindow( FALSE );
        m_ClearResultsButton.EnableWindow( FALSE );
        m_CancelSortButton.EnableWindow( TRUE );
        m_ExitButton.EnableWindow( FALSE );

        // Initialize the variables used by RegisterThreadsFinished()
        nThreadsFinished = nThreadsStarted = 0;

        // Use a simple bit-counter to figure out how many sort types the user selected
        nTmpMask = m_nSelectedSortMask;
        while( nTmpMask )
            nTmpMask &= (nTmpMask - 1), nNumberOfSortsSelected++;

        // Call the DLL to do the sorts
        nThreadsStarted = DoSorts( m_nSelectedSortMask, m_nListOrder, m_nElements, m_nWhenToDisplayResults, pMainWnd->m_hWnd );

        if( nThreadsStarted != nNumberOfSortsSelected )
            MessageBox( "Unable to Start All Threads!", "SORTDEMO", MB_OK );
        
        if( !nThreadsStarted )
        {
            MessageBox( "Unable to Start Any Threads, Exiting!", "SORTDEMO", MB_OK | MB_ICONSTOP );
            ::PostQuitMessage(0);
        }
    }
    else // The user did not select at least one sort type
    {
        MessageBox( "Please select at least one type of sort.", "SortDemo", MB_ICONEXCLAMATION  );
        
        // Set the focus to the Bubble sort check box
        m_BubbleControl.SetFocus();
    }
}

//***********************************************************************
// Function:
//
//     CMainWindow::OnClickedClearResults()
//
// Purpose:
//
//     Clears the results displayed from previous sorts. 
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

void CMainWindow::OnClickedClearResults()
{
    // The user clicked on the Clear Results button
    SetDlgItemText( IDC_BUBBLESWAPS, "" );
    SetDlgItemText( IDC_BUBBLECOMPARES, "" );
    SetDlgItemText( IDC_BUBBLETIME, "" );
    SetDlgItemText( IDC_EXCHANGESWAPS, "" );
    SetDlgItemText( IDC_EXCHANGECOMPARES, "" );
    SetDlgItemText( IDC_EXCHANGETIME, "" );
    SetDlgItemText( IDC_HEAPSWAPS, "" );
    SetDlgItemText( IDC_HEAPCOMPARES, "" );
    SetDlgItemText( IDC_HEAPTIME, "" );
    SetDlgItemText( IDC_INSERTIONSWAPS, "" );
    SetDlgItemText( IDC_INSERTIONCOMPARES, "" );
    SetDlgItemText( IDC_INSERTIONTIME, "" );
    SetDlgItemText( IDC_QUICKSWAPS, "" );
    SetDlgItemText( IDC_QUICKCOMPARES, "" );
    SetDlgItemText( IDC_QUICKTIME, "" );
    SetDlgItemText( IDC_SHELLSWAPS, "" );
    SetDlgItemText( IDC_SHELLCOMPARES, "" );
    SetDlgItemText( IDC_SHELLTIME, "" );
}

//***********************************************************************
// Function:
//
//     CMainWindow::OnClickedCancelSort()
//
// Purpose:
//
//     Terminates the currently executing sort threads.
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

void CMainWindow::OnClickedCancelSort()
{
    // The user clicked on the Cancel Sorts button. Set the global flag
    // checked by the sorting threads
    fStopSortingNow = TRUE;
    
    // Enable and disable the correct buttons. Need to set nThreadsFinished
    // equal to nThreadsStarted - 1 because RegisterThreadsFinished() will
    // increment nThreadsFinished upon entry.
    nThreadsFinished = nThreadsStarted - 1;
    RegisterThreadsFinished();
}

//***********************************************************************
// Function:
//
//     RegisterThreadsFinished()
//
// Purpose:
//
//     Keeps track of how many of the started threads have finished.
//     When all threads are finished takes care of enabling and disabling
//     the correct buttons and calling CleanUp() in the DLL.
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
// 11/10/93 Added call to CleanUp                               JKK
//***********************************************************************

__declspec( dllexport ) void RegisterThreadsFinished()
{
    // Increment the finished thread counter
    nThreadsFinished++;

    // Check to see if all started threads have finished
    if( nThreadsFinished == nThreadsStarted )
    {
        // Call CleanUp() in the DLL to free up memory
        // allocated for the master list.
        CleanUp();

        // Enable and disable the correct buttons
        EnableWindow( hwndDoSortButton, TRUE );
        EnableWindow( hwndClearResultsButton, TRUE );
        EnableWindow( hwndCancelSortButton, FALSE );
        EnableWindow( hwndExitButton, TRUE );
    }
}
 
//***********************************************************************
// Function:
//
//     ShowResults()
//
// Purpose:
//
//     Displays the number of swaps and comparisons done by each type of
//     sort.
//
// Parameters:
//
//     SortType     - The sort type the results apply to
//     nSwaps       - The number of swaps made by the sort
//     nComparisons - The number of comparisons made by the sort
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
// 11/10/93 Moved handling of critical object from threads      JKK
//***********************************************************************

__declspec( dllexport ) void ShowResults( int SortType, int nSwaps, int nComparisons )
{
    int SwapResultsID, CompareResultsID;
    char lpszSwaps[20];
    char lpszCompares[20];

    // Format the results to be displayed
    sprintf(lpszSwaps, "%d", nSwaps);
    sprintf(lpszCompares, "%d", nComparisons);
    
    // Get the correct static text control IDs for the sort type
    switch( SortType )
    {
        case BUBBLE:

            SwapResultsID = IDC_BUBBLESWAPS;
            CompareResultsID = IDC_BUBBLECOMPARES;
            break;

        case EXCHANGE:

            SwapResultsID = IDC_EXCHANGESWAPS;
            CompareResultsID = IDC_EXCHANGECOMPARES;
            break;

        case HEAP:

            SwapResultsID = IDC_HEAPSWAPS;
            CompareResultsID = IDC_HEAPCOMPARES;
            break;

        case INSERTION:

            SwapResultsID = IDC_INSERTIONSWAPS;
            CompareResultsID = IDC_INSERTIONCOMPARES;
            break;

        case QUICK:

            SwapResultsID = IDC_QUICKSWAPS;
            CompareResultsID = IDC_QUICKCOMPARES;
            break;

        case SHELL:

            SwapResultsID = IDC_SHELLSWAPS;
            CompareResultsID = IDC_SHELLCOMPARES;
            break;

        default: ;
    }

    // Grab the critical section object
    EnterCriticalSection( &GlobalCriticalSection );
    // Set the static text controls to the current values
    SetDlgItemText(hwndMainWindow, SwapResultsID, lpszSwaps);
    SetDlgItemText(hwndMainWindow, CompareResultsID, lpszCompares);
    // Release the critical section object    
    LeaveCriticalSection( &GlobalCriticalSection );
}

//***********************************************************************
// Function:
//
//     ShowElapsedTime()
//
// Purpose:
//
//     Displays the elapsed time of the specified sort.
//
// Parameters:
//
//     SortType - The sort type the time value applies to
//     Time     - The elapsed time in milliseconds.
//
// Comments:
//
//     The elapsed time will vary greatly depending on whether the
//     user has chosen to have the number of swaps and comparisons
//     displayed while the sort is in progress or after the sort is
//     complete.
//
//     Time will be equal to CLEARTIME when the curently displayed elapsed
//     time value is being cleared. This is done by each of the sorting
//     threads before they begin sorting. 
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
// 11/10/93 Moved handling of critical object from threads      JKK
//***********************************************************************

__declspec( dllexport ) void ShowElapsedTime( int nSortType, DWORD dwTime )
{
    char lpszTime[20] = "";
    int nTimeResultsID;
    DWORD dwMinutes, dwSeconds, dwMilliseconds;
    
    dwMinutes = dwSeconds = dwMilliseconds = 0;

    // If we're not clearing the current displayed elapsed time, generate a formatted time string
    if( dwTime != CLEARTIME )
    {
        dwMilliseconds = dwTime % 1000;

        dwSeconds = dwTime / 1000;

        if( dwSeconds > 59 )
        {
            dwMinutes = dwSeconds / 60;
            dwSeconds %= 60;
        }

        sprintf( lpszTime, "%u.%02u.%03u", dwMinutes, dwSeconds, dwMilliseconds );
    }

    // Get the correct static control ID based on the sort type
    switch( nSortType )
    {
        case BUBBLE:

            nTimeResultsID = IDC_BUBBLETIME;
            break;

        case EXCHANGE:

            nTimeResultsID = IDC_EXCHANGETIME;
            break;

        case HEAP:

            nTimeResultsID = IDC_HEAPTIME;
            break;

        case INSERTION:

            nTimeResultsID = IDC_INSERTIONTIME;
            break;

        case QUICK:

            nTimeResultsID = IDC_QUICKTIME;
            break;

        case SHELL:

            nTimeResultsID = IDC_SHELLTIME;
            break;

        default: ;
    }

    // Grab the critical section object
    EnterCriticalSection( &GlobalCriticalSection );
    // Update the time value for the sort
    SetDlgItemText(hwndMainWindow, nTimeResultsID, lpszTime);
    // Release the critical section object
    LeaveCriticalSection( &GlobalCriticalSection );
}
