//****************************************************************************
// File:
//     
//     sortdll.cpp
//
// Purpose:
//
//     Implementation file for the sorting DLL.
//
// Functions:
//
//     DoSorts       - Starts the correct sorting threads based upon user
//                       input.
//     GenerateList  - Generates the master list of elements to be sorted.
//     CleanUp       - Deletes the master list of elements to be sorted.
//     QuickSort     - Handles initialization and calls DoQuickSort.
//     DoQuickSort   - Implements the QuickSort algorithm.
//     BubbleSort    - Implements the BubbleSort algorithm.
//     InsertSort    - Implements the InsertionSort algorithm.
//     ExchangeSort  - Implements the ExchangeSort algorithm.
//     HeapSort      - Implements the HeapSort algorithm.
//     PercolateUp   - Used by HeapSort to convert the sort list to a heap
//                       with the largest element at the top. 
//     PercolateDown - Used by HeapSort to convert a reversed heap to a sorted
//                       list. 
//     ShellSort     - Implements the ShellSort algorithm.
//
// Development Team:
//
//     Joel Krist
//
// Written by Microsoft Product Support Services, Languages Developer Support
// Copyright (c) 1993 Microsoft Corporation. All rights reserved.
//****************************************************************************

#include "stdafx.h"
#include <windows.h>
#include <stdlib.h>
#include <process.h>
#include <time.h>
#include "sortdemo.h"
#include "mainwnd.h"

#define _BUILDING_DLL_
#include "sortdll.h"

// Thread-local data
__declspec( thread ) static int *pWork;   // Used to simplify the QuickSort recursion
__declspec( thread ) int nSwaps;          // Used to track the number of swaps made
__declspec( thread ) int nComparisons;    // Used to track the number of comparisons

// Global data
ThreadParamBlock GlobalParameterBlock;    // Used to pass the list to sort and number
                                          //   of elements in the list to the threads.
int fShowResultsDuring;                   // Used to tell the threads when to display
                                          //   their results.
HWND hwndParent;                          // HWND used to allow message boxes from
                                          //   threads to be displayed on top of the
                                          //   main dialog window.

// Array of pointers to sorting functions used with _beginthread() in DoSorts()
void (*pfnSortThreads[])(void *) = { BubbleSort,
                                     ExchangeSort,
                                     HeapSort,
                                     InsertSort,
                                     QuickSort,
                                     ShellSort };

//***********************************************************************
// Function: DoSorts()
//
// Purpose:
//
//     Handles creating the critical section object used to control
//     access to the display and takes care of starting the correct
//     sorting threads.
//
// Parameters:
//
//     nSortMask             - The mask representing the sort types
//                             selected by the user.
//     nListOrder            - The ordering of the list to be sorted.
//     nElements             - The number of elements in the list to
//                             be sorted.
//     fWhenToDisplayResults - An integer flag representing when the
//                             threads should display the number of
//                             swaps and comparisons done. 0 means
//                             to display during the sort, 1 means
//                             to display when the sort is finished.
//     hwndMainWnd           - Used to allow message boxes from threads
//                             to be displayed on top of the main dialog
//                             window.
//
// Returns:
//
//     A BOOL representing whether or not the sorting threads were
//     successfully started. FALSE means failure, TRUE means
//     success.  
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
// 11/10/93 Changed to use BOOL for return type                 JKK
// 11/16/93 Changed to return number of threads started         JKK
// 11/16/93 Added the hwndMainWnd param to argument list        JKK
//***********************************************************************

INTERFACE int DoSorts( int nSelectedSortMask, int nListOrder, int nElements, int nWhenToDisplayResults, HWND hwndMainWnd )
{
    int loop;
    int nThreadsStarted = 0;

    // Seed the pseudo-random number generator used later when generating a
    // random list to be sorted
    srand( (unsigned)time( NULL ) );

    // Determine when the threads should display their results
    fShowResultsDuring = nWhenToDisplayResults;

    // Initialize the flag used to tell the threads to stop
    fStopSortingNow = FALSE;

    // Initialize the HWND used when displaying message boxes from the threads
    hwndParent = hwndMainWnd;

    // Initialize the critical section object to control access to the display
    InitializeCriticalSection( &GlobalCriticalSection);
    
    // Create the master list to be sorted
    if( !GenerateList( nListOrder, nElements ) )
        return 0;
    
    GlobalParameterBlock.nNumberOfElements = nElements;
        
    // Start the correct sort threads based upon the SortMask parameter
    for( loop = NUMSORTS - 1; loop >= 0; loop-- )
    {
        if( nSelectedSortMask & SORTMASK )
            if( _beginthread( *pfnSortThreads[ loop ], 0, &GlobalParameterBlock ) != -1 )
                nThreadsStarted++;
    
        nSelectedSortMask = nSelectedSortMask << 1;    
    }

    return nThreadsStarted;
}

//***********************************************************************
// Function:
//
//     GenerateList()
//
// Purpose:
//
//     Allocates memory for and initializes the global master sort list.
//
// Parameters:
//
//     nListOrder - The order to be used for the generated list.
//     nElements  - The number of elements in the list.
//
// Returns:
//
//     A BOOL representing whether or not the master list was
//     successfully allocated. FALSE means failure, TRUE means
//     success.  
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
// 11/10/93 Changed to use BOOL for return type                 JKK
//***********************************************************************

BOOL GenerateList( int nListOrder, int nElements )
{
    int i;

    // Allocate memory for the global master list
    if((GlobalParameterBlock.pListToSort = new int [nElements]) == NULL)
    {
        MessageBox( hwndParent, "Unable to Generate Master Sort List", "SORTDLL", MB_OK | MB_ICONSTOP );
        return FALSE;
    }    

    // Initialize the master list
    switch (nListOrder)
    {
        case ORDERED:
            for (i = 0; i < nElements; i++)
                GlobalParameterBlock.pListToSort[i] = i;
            break;
        case RANDOM:
            for (i = 0; i < nElements; i++)
                GlobalParameterBlock.pListToSort[i] = rand();
            break;
        case REVERSE:
            for (i = 0; i < nElements; i++)
                GlobalParameterBlock.pListToSort[i] = nElements - i;
            break;
        default:
            MessageBox( hwndParent, "Invalid List Order Specified", "SORTDLL", MB_OK | MB_ICONSTOP );
            return 1;
    }

    return TRUE;
}

//***********************************************************************
// Function:
//
//     CleanUp()
//
// Purpose:
//
//     Frees the memory allocated for the master sort list.
//
// Parameters:
//
//     None.
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 11/10/93 Created                                             JKK
//***********************************************************************

INTERFACE void CleanUp( void )
{
    // Delete the master sort list allocated in GenerateList
    delete GlobalParameterBlock.pListToSort;
}

//***********************************************************************
// Function:
//
//     QuickSort()
//
// Purpose:
//
//     Allocates the thread local copy of the list to be sorted and
//     calls DoQuickSort which actually implements the QuickSort 
//     algorithm.
//
//
// Parameters:
//
//     void * - Not used but is required for thread entry point functions
//              for threads started with _beginthread().
//
// Returns:
//
//     void
//
// Comments:
//
//     QuickSort works by picking a "pivot" element, then moving
//     every element that is bigger to one side of the pivot, and every
//     element that is smaller to the other side. QuickSort is then called
//     recursively with the two subdivisions created by the pivot. Once
//     the number of elements in a subdivision reaches two, the recursive
//     calls end and the array is sorted.
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

void QuickSort(void *)
{
    int *pList;
    int cbLength;

    DWORD dwStart, dwFinish, dwDuration;

    // Get the number of elements to be sorted
    cbLength = GlobalParameterBlock.nNumberOfElements;
    
    // Allocate and initialize a thread local copy of the list to be sorted
    if( (pList = new int [cbLength * sizeof( int )]) == NULL )
    {
        RegisterThreadsFinished();
        MessageBox( hwndParent, "Unable to Allocate Thread Local List!", "QuickSort", MB_OK | MB_ICONSTOP );
        _endthread();
    }

    memcpy( pList, GlobalParameterBlock.pListToSort, cbLength * sizeof( int ) );

    // Make list accessible to DoQuickSort()
    pWork = pList;      
    nComparisons = nSwaps = 0;

    // Clear the current displayed elapsed time
    ShowElapsedTime( QUICK, CLEARTIME );

    // Get the starting time
    dwStart = GetTickCount();
    
    // Do the sort
    DoQuickSort(0, cbLength - 1);

    // Get the finish time, calculate the elapsed time in milliseconds
    dwFinish = GetTickCount();
    dwDuration = dwFinish - dwStart;
    
    if( !fStopSortingNow )
    {
        ShowResults( QUICK, nSwaps, nComparisons );
        ShowElapsedTime( QUICK, dwDuration );
    }

    // Delete the local list
    delete( pList );

    //Let the calling application know that this thread is finished
    RegisterThreadsFinished();

    // End this thread
    _endthread();
}

//***********************************************************************
// Function:
//
//     DoQuickSort()
//
// Purpose:
//
//     Implements the QuickSort sorting algorithm.
//
// Parameters:
//
//     nLeft  - The start of the subfile to sort.
//     nRight - The end of the subfile to sort.
//
// Returns:
//
//     void
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

static void DoQuickSort(int nLeft, int nRight)
{
    int nPivot, nleft, nright, nTemp;
    
    if ( (nRight > nLeft) && !fStopSortingNow )    // Terminating condition for recursion
    {
        nPivot = pWork[nRight];
        nleft = nLeft-1;
        nright = nRight;

        while ( (nright > nleft) && !fStopSortingNow )
        {
            do
            {
                nleft++;
                nComparisons++;
                
                if( fShowResultsDuring )
                    ShowResults( QUICK, nSwaps, nComparisons );

            } while ( (pWork[nleft] < nPivot) && !fStopSortingNow );

            do
            {
                nright--;
                nComparisons++;
                
                if( fShowResultsDuring )
                    ShowResults( QUICK, nSwaps, nComparisons );

            } while ((pWork[nright] > nPivot) && (nright >= 0) && !fStopSortingNow );

            // Swap the values that are out of place with
            // respect to the pivot value
            nSwaps++;
                                        
            if( fShowResultsDuring )
                ShowResults( QUICK, nSwaps, nComparisons );
                        
            nTemp = pWork[nleft];
            pWork[nleft] = pWork[nright];
            pWork[nright] = nTemp;
        }

        nSwaps++;

        if( fShowResultsDuring )
            ShowResults( QUICK, nSwaps, nComparisons );

        nTemp = pWork[nright];
        pWork[nright] = pWork[nleft];
        pWork[nleft] = nPivot;
        pWork[nRight] = nTemp;

        // Sort the subfiles on either side of the partition
        DoQuickSort(nLeft, nleft - 1);
        DoQuickSort(nleft + 1, nRight);
    }
}

//***********************************************************************
// Function:
//
//     BubbleSort()
//
// Purpose:
//
//     Implements the BubbleSort sorting algorithm.
//
// Parameters:
//
//     void * - Not used but is required for thread entry point functions
//              for threads started with _beginthread().
//
// Returns:
//
//     void
//
// Comments:
//
//     BubbleSort cycles through the elements, comparing adjacent
//     elements and swapping pairs that are out of order. It continues
//     to do this until no out-of-order pairs are found.
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

void BubbleSort(void *)
{
    int nTemp;
    int * pList;
    int cbLength;
    DWORD dwStart, dwFinish, dwDuration;

    // Get the number of elements to be sorted
    cbLength = GlobalParameterBlock.nNumberOfElements;
    
    // Allocate and initialize a thread local copy of the list to be sorted
    if( (pList = new int [cbLength * sizeof( int )]) == NULL )
    {
        MessageBox( hwndParent, "Unable to Allocate Thread Local List!", "BubbleSort", MB_OK | MB_ICONSTOP );
        _endthread();
    }
    
    memcpy( pList, GlobalParameterBlock.pListToSort, cbLength * sizeof( int ) );

    nComparisons = nSwaps = 0;

    // Clear the current displayed elapsed time
    ShowElapsedTime( BUBBLE, CLEARTIME );

    // Get the starting time
    dwStart = GetTickCount();
    
    for (int i = 0; i < cbLength; i++)
        if( fStopSortingNow )
            break;
        else
        {
            for (int j = 0; j < cbLength - 1; j++)
                if( fStopSortingNow )
                    break;
                else
                {
                    nComparisons++;                       

                    if( fShowResultsDuring )
                        ShowResults( BUBBLE, nSwaps, nComparisons );

                    if (pList[j] > pList[j+1])
                    {
                        nSwaps++;

                        if( fShowResultsDuring )
                            ShowResults( BUBBLE, nSwaps, nComparisons );

                        nTemp = pList[j];
                        pList[j] = pList[j+1];
                        pList[j+1] = nTemp;
                    }
                }
        }

    // Get the finish time, calculate the elapsed time in milliseconds
    dwFinish = GetTickCount();
    dwDuration = dwFinish - dwStart;

    if( !fStopSortingNow )
    {
        // Display the final results    
        ShowResults( BUBBLE, nSwaps, nComparisons );
        ShowElapsedTime( BUBBLE, dwDuration );
    }

    // Delete the local list
    delete pList;

    // Let the calling app know this thread is finished
    RegisterThreadsFinished();

    // End this thread
    _endthread();
}

//***********************************************************************
// Function:
//
//     InsertSort()
//
// Purpose:
//
//     Implements the InsertionSort sorting algorithm.
//
// Parameters:
//
//     void * - Not used but is required for thread entry point functions
//              for threads started with _beginthread().
//
// Returns:
//
//     void
//
// Comments:
//
//     InsertionSort compares each element with all the preceding
//     elements. When the appropriate place for the new element is found,
//     the element is inserted and all the other elements are moved down
//     one place.
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

void InsertSort(void *)
{
    int i, j, nVal;
    int *pList;
    int cbLength;
    DWORD dwStart, dwFinish, dwDuration;

    // Get the number of elements to be sorted
    cbLength = GlobalParameterBlock.nNumberOfElements;
    
    // Allocate and initialize a thread local copy of the list to be sorted
    if( (pList = new int [cbLength * sizeof( int )]) == NULL )
    {
        MessageBox( hwndParent, "Unable to Allocate Thread Local List!", "InsertSort", MB_OK | MB_ICONSTOP );
        _endthread();
    }
    
    memcpy( pList, GlobalParameterBlock.pListToSort, cbLength * sizeof( int ) );

    nComparisons = nSwaps = 0;

    // Clear the current displayed elapsed time
    ShowElapsedTime( INSERTION, CLEARTIME );

    // Get the starting time
    dwStart = GetTickCount();

    for (i = 1; i < cbLength; i++)
        if( fStopSortingNow )
            break;
        else        
        {
            nSwaps++;
            nComparisons++;

            if( fShowResultsDuring )
                ShowResults( INSERTION, nSwaps, nComparisons );

            nVal = pList[i];
            j = i;

            while ((j > 0) && (pList[j-1] > nVal) && !fStopSortingNow )
            {
                nSwaps++;
                nComparisons++;
            
                if( fShowResultsDuring )
                    ShowResults( INSERTION, nSwaps, nComparisons );

                pList[j] = pList[j - 1];
                j--;
            }

            pList[j] = nVal;
        }

    // Get the finish time, calculate the elapsed time in milliseconds
    dwFinish = GetTickCount();
    dwDuration = dwFinish - dwStart;

    if( !fStopSortingNow )
    {
        // Display the final results    
        ShowResults( INSERTION, nSwaps, nComparisons );
        ShowElapsedTime( INSERTION, dwDuration );
    }

    // Delete the local list
    delete pList;

    // Let the calling app know this thread is finished
    RegisterThreadsFinished();

    // End this thread
    _endthread();
}

//***********************************************************************
// Function:
//
//     ExchangeSort()
//
// Purpose:
//
//     Implements the ExchangeSort sorting algorithm.
//
// Parameters:
//
//     void * - Not used but is required for thread entry point functions
//              for threads started with _beginthread().
//
// Returns:
//
//     void
//
// Comments:
//
//     The ExchangeSort compares each element, starting with the first,
//     with every following element. If any of the following elements is
//     smaller than the current element, it is exchanged with the current
//     element and the process is repeated for the next element.
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

void ExchangeSort(void *)
{
    int iRowCur, iRowMin, iRowNext, tmp;
    int *pList;
    int cbLength;
    DWORD dwStart, dwFinish, dwDuration;

    // Get the number of elements to be sorted
    cbLength = GlobalParameterBlock.nNumberOfElements;
    
    // Allocate and initialize a thread local copy of the list to be sorted
    if( (pList = new int [cbLength * sizeof( int )]) == NULL )
    {
        MessageBox( hwndParent, "Unable to Allocate Thread Local List!", "ExchangeSort", MB_OK | MB_ICONSTOP );
        _endthread();
    }
    
    memcpy( pList, GlobalParameterBlock.pListToSort, cbLength * sizeof( int ) );

    nComparisons = nSwaps = 0;

    // Clear the current displayed elapsed time
    ShowElapsedTime( EXCHANGE, CLEARTIME );

    // Get the starting time
    dwStart = GetTickCount();

    for( iRowCur = 0; iRowCur < cbLength; iRowCur++ )
        if( fStopSortingNow )
            break;
        else        
        {
            iRowMin = iRowCur;
            
            for( iRowNext = iRowCur; iRowNext < cbLength; iRowNext++ )
                if( fStopSortingNow )
                    break;
                else        
                {
                    nComparisons++;
            
                    if( fShowResultsDuring )
                        ShowResults( EXCHANGE, nSwaps, nComparisons );

                    if( pList[iRowNext] < pList[iRowMin] )                
                        iRowMin = iRowNext;
                }

            // If an element is smaller than the current element, swap those two
            // array elements.
            if( iRowMin > iRowCur )
            {
                tmp = pList[iRowCur];
                pList[iRowCur] = pList[iRowMin];
                pList[iRowMin] = tmp;
                nSwaps++;

                if( fShowResultsDuring )
                    ShowResults( EXCHANGE, nSwaps, nComparisons );
            }

            if( fShowResultsDuring )
                ShowResults( EXCHANGE, nSwaps, nComparisons );
        }

    // Get the finish time, calculate the elapsed time in milliseconds
    dwFinish = GetTickCount();
    dwDuration = dwFinish - dwStart;

    if( !fStopSortingNow )
    {
        // Display the final results    
        ShowResults( EXCHANGE, nSwaps, nComparisons );
        ShowElapsedTime( EXCHANGE, dwDuration );
    }
    
    // Delete the local list
    delete( pList );

    // Let the calling app know this thread is finished
    RegisterThreadsFinished();
    
    // End this thread
    _endthread();
}

//***********************************************************************
// Function:
//
//     HeapSort()
//
// Purpose:
//
//     Implements the HeapSort sorting algorithm.
//
// Parameters:
//
//     void * - Not used but is required for thread entry point functions
//              for threads started with _beginthread().
//
// Returns:
//
//     void
//
// Comments:
//
//     HeapSort (also called TreeSort) works by calling PercolateUp and
//     PercolateDown. PercolateUp organizes the elements into a "heap"
//     or "tree," which has the properties shown below:
/*
                              element[1]
                            /            \
                 element[2]                element[3]
                /          \              /          \
          element[4]     element[5]   element[6]    element[7]
          /        \     /        \   /        \    /        \
         ...      ...   ...      ... ...      ...  ...      ...
*/
//     Each "parent node" is greater than each of its "child nodes"; for
//     example, element[1] is greater than element[2] or element[3];
//     element[4] is greater than element[5] or element[6], and so forth.
//     Therefore, once the first loop in HeapSort is finished, the
//     largest element is in element[1].
//
//     The second loop rebuilds the heap (with PercolateDown), but starts
//     at the top and works down, moving the largest elements to the
//     bottom. This has the effect of moving the smallest elements to the
//     top and sorting the heap.
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

void HeapSort(void *)
{
    int i, tmp;
    int *pList;
    int cbLength;
    DWORD dwStart, dwFinish, dwDuration;

    // Get the number of elements to be sorted
    cbLength = GlobalParameterBlock.nNumberOfElements;
    
    // Allocate and initialize a thread local copy of the list to be sorted
    if( (pList = new int [cbLength * sizeof( int )]) == NULL )
    {
        MessageBox( hwndParent, "Unable to Allocate Thread Local List!", "HeapSort", MB_OK | MB_ICONSTOP );
        _endthread();
    }
    
    memcpy( pList, GlobalParameterBlock.pListToSort, cbLength * sizeof( int ) );

    nComparisons = nSwaps = 0;

    // Clear the current displayed elapsed time
    ShowElapsedTime( HEAP, CLEARTIME );

    // Get the starting time
    dwStart = GetTickCount();

    for( i = 1; i < cbLength; i++ )
        if( fStopSortingNow )
            break;
        else
            PercolateUp( i, pList );

    for( i = cbLength - 1; i > 0; i-- )
        if( fStopSortingNow )
            break;
        else
        {
            tmp = pList[0];
            pList[0] = pList[i];
            pList[i] = tmp;
            nSwaps++;

            if( fShowResultsDuring )
                ShowResults( HEAP, nSwaps, nComparisons );
        
            PercolateDown( i - 1, pList );
        }

    // Get the finish time, calculate the elapsed time in milliseconds
    dwFinish = GetTickCount();
    dwDuration = dwFinish - dwStart;

    if( !fStopSortingNow )
    {
        // Display the final results    
        ShowResults( HEAP, nSwaps, nComparisons );
        ShowElapsedTime( HEAP, dwDuration );
    }

    // Delete the local list
    delete( pList );

    // Let the calling app know this thread is finished
    RegisterThreadsFinished();

    // End this thread
    _endthread();
}

//***********************************************************************
// Function:
//
//     PercolateUp()
//
// Purpose:
//
//     Converts elements into a "heap" with the largest element at the
//     top (see the diagram above).
//
// Parameters:
//
//     iMaxLevel - Specifies the list element being moved.
//     pList     - A pointer to the list to be sorted.
//
// Returns:
//
//     void
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

void PercolateUp( int iMaxLevel, int * pList )
{
    int i = iMaxLevel, iParent, tmp;

    // Move the value in pList[iMaxLevel] up the heap until it has
    // reached its proper node (that is, until it is greater than either
    // of its child nodes, or until it has reached 1, the top of the heap).

    while( i && !fStopSortingNow )
    {
        iParent = i / 2;    // Get the subscript for the parent node
        nComparisons++;

        if( fShowResultsDuring )
            ShowResults( HEAP, nSwaps, nComparisons );

        if( pList[i] > pList[iParent] )
        {
            // The value at the current node is bigger than the value at
            // its parent node, so swap these two array elements.
            //
            tmp = pList[iParent];
            pList[iParent] = pList[i];
            pList[i] = tmp;
            i = iParent;
            nSwaps++;

            if( fShowResultsDuring )
                ShowResults( HEAP, nSwaps, nComparisons );
        }
        else
            // Otherwise, the element has reached its proper place in the
            // heap, so exit this procedure.
            break;
    }
}

//***********************************************************************
// Function:
//
//     PercolateDown()
//
// Purpose:
//
//     Converts elements to a "heap" with the largest elements at the
//     bottom. When this is done to a reversed heap (largest elements at
//     top), it has the effect of sorting the elements.
//
// Parameters:
//
//     iMaxLevel - Specifies the list element being moved.
//     pList     - A pointer to the list to be sorted.
//
// Returns:
//
//     void
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

void PercolateDown( int iMaxLevel, int * pList )
{
    int iChild, i = 0, tmp;

    // Move the value in pList[0] down the heap until it has reached
    // its proper node (that is, until it is less than its parent node
    // or until it has reached iMaxLevel, the bottom of the current heap).

    while( TRUE && !fStopSortingNow )
    {
        // Get the subscript for the child node.
        iChild = 2 * i;

        // Reached the bottom of the heap, so exit this procedure.
        if( iChild > iMaxLevel )
            break;

        // If there are two child nodes, find out which one is bigger.
        if( iChild + 1 <= iMaxLevel )
        {
            nComparisons++;
            if( pList[iChild + 1] > pList[iChild] )
                iChild++;
        }

        nComparisons++;

        if( fShowResultsDuring )
            ShowResults( HEAP, nSwaps, nComparisons );

        if( pList[i] < pList[iChild] )
        {
            // Move the value down since it is still not bigger than
            // either one of its children.
            tmp = pList[i];
            pList[i] = pList[iChild];
            pList[iChild] = tmp;
            nSwaps++;

            if( fShowResultsDuring )
                ShowResults( HEAP, nSwaps, nComparisons );

            i = iChild;
        }
        else
            // Otherwise, pList has been restored to a heap from 1 to
            // iMaxLevel, so exit.
            break;
    }
}

//***********************************************************************
// Function:
//
//     ShellSort()
//
// Purpose:
//
//     Implements the ShellSort sorting algorithm.
//
// Parameters:
//
//     void * - Not used but is required for thread entry point functions
//              for threads started with _beginthread().
//
// Returns:
//
//     void
//
// Comments:
//
//     ShellSort is similar to the BubbleSort. However, it begins by
//     comparing elements that are far apart (separated by the value of
//     the iOffset variable, which is initially half the distance between
//     the first and last element), then comparing elements that are
//     closer together. When iOffset is one, the last iteration is merely
//     a bubble sort.
//
// History:
//
//   Date   Comment                                           Initials
// ======== ================================================= ========
// 10/12/93 Created                                             JKK
//***********************************************************************

void ShellSort(void *)
{
    int iOffset, iSwitch, iLimit, iRow, tmp;
    int *pList;
    int cbLength;
    DWORD dwStart, dwFinish, dwDuration;

    // Get the number of elements to be sorted
    cbLength = GlobalParameterBlock.nNumberOfElements;
    
    // Allocate and initialize a thread local copy of the list to be sorted
    if( (pList = new int [cbLength * sizeof( int )]) == NULL )
    {
        MessageBox( hwndParent, "Unable to Allocate Thread Local List!", "ShellSort", MB_OK | MB_ICONSTOP );
        _endthread();
    }
    
    memcpy( pList, GlobalParameterBlock.pListToSort, cbLength * sizeof( int ) );

    nComparisons = nSwaps = 0;

    // Set comparison offset to half the number of elements.
    iOffset = cbLength / 2;

    // Clear the current displayed elapsed time
    ShowElapsedTime( SHELL, CLEARTIME );

    // Get the starting time
    dwStart = GetTickCount();

    while( iOffset && !fStopSortingNow )
    {
        // Loop until offset gets to zero.
        iLimit =  cbLength - iOffset - 1 ;
        do
        {
            iSwitch = FALSE;     // Assume no switches at this offset.

            // Compare elements and switch ones out of order.
            for( iRow = 0; iRow <= iLimit; iRow++ )
            {
                if( fStopSortingNow )
                    break;
                else
                { 
                    nComparisons++;
                    if( pList[iRow] > pList[iRow + iOffset] )
                    {
                        tmp = pList[iRow];
                        pList[iRow] = pList[iRow + iOffset];
                        pList[iRow + iOffset] = tmp;

                        nSwaps++;
                        iSwitch = iRow;
                    }

                    if( fShowResultsDuring )
                        ShowResults( SHELL, nSwaps, nComparisons );
                }
            }

            // Sort on next pass only to where last switch was made.
            iLimit = iSwitch - iOffset;
        } while( iSwitch && !fStopSortingNow );

        // No switches at last offset, try one half as big.
        iOffset = iOffset / 2;
    }

    // Get the finish time, calculate the elapsed time in milliseconds
    dwFinish = GetTickCount();
    dwDuration = dwFinish - dwStart;

    if( !fStopSortingNow )
    {
        // Display the final results           
        ShowResults( SHELL, nSwaps, nComparisons );
        ShowElapsedTime( SHELL, dwDuration );
    }

    // Delete the local copy of the list
    delete( pList );

    // Let the calling app know this thread is finished
    RegisterThreadsFinished();
        
    // Terminate this thread
    _endthread();
}
