/**************************************************************************
*  File:        DEMO.CPP      Copyright (c) 1994-1995 by Matthew Hildebrand
*
*  Purpose:     Demonstrate TCG 1.03's features.
**************************************************************************/

#include <alloc.h>
#include <conio.h>
#include <dos.h>
#include <mem.h>
#include <stdio.h>
#include <stdlib.h>
#include "..\include\tcg.h"
#include "..\include\tcgmouse.h"
#include "..\include\varfont.h"
#include "..\include\vcoord.h"

#define FONT_FILENAME       "BIGTEXT.FNT"
#define QUIT_IF_ESCAPE      if (getKey() == 27)  \
                              exit(EXIT_SUCCESS)

enum
{
  Center, BottomCenter
};

void setup(void);
void printMessage(char *string, int where=BottomCenter);
void lineDemo(void);
void putPixelDemo(void);
void drawRectDemo(void);
void filledRectDemo(void);
void ellipseDemo(void);
void filledEllipseDemo(void);
void circleDemo(void);
void filledCircleDemo(void);
void putImageDemo(void);
void fontDemo(void);
void viewportDemo(void);
void scaleBitmapDemo(void);
void mouseDemo(void);
void signOff(void);
int getKey(void);

VariableFont font;
void *tcgLogo;



/**************************************************************************
*  Function:    main
*
*  Purpose:     Program entry point.
*
*  Entry:       argc = Number of parameters - 1.
*               argv = Array containing the actual parameters.
*
*  Exit:        N/A
**************************************************************************/

void main(void)
{
  //*** Set things up
  setup();

  //*** Run the demos
  lineDemo();
  putPixelDemo();
  drawRectDemo();
  filledRectDemo();
  ellipseDemo();
  filledEllipseDemo();
  circleDemo();
  filledCircleDemo();
  fontDemo();
  viewportDemo();
  mouseDemo();

  //*** Get a key before quitting
  getKey();
}



/**************************************************************************
*  Function:    setup
*
*  Purpose:     Load drivers and fonts, and do other setup things.
*
*  Entry:       N/A
*
*  Exit:        If an error occurs, the program will exit from within this
*               function.
**************************************************************************/

void setup(void)
{
  char *filename;

  //*** Load a suitable graphics driver
  filename = detectGraph();
  if (filename == NULL)
  {
    printf("No truecolour video mode detected; aborting.\n\n"
           "Try installing a VESA BIOS extensions TSR, such as UNIVESA\n"
           "(in TCG's VESABE directory) or the TSR (if any) which came\n"
           "with your video card.  VESA BIOS extensions version 1.1 or\n"
           "higher are required.\n\n");
    exit(EXIT_FAILURE);
  }
  switch (loadGraphDriver(filename))
  {
    case TCG_SUCCESS:                       // success
      break;
    case TCG_OPEN_ERR:                      // file not found
      printf("Error opening driver file %s; aborting.\n\n"
             "Please ensure that the TCGDRIVERS environment variable has been set to\n"
             "point to the TCG drivers directory; refer to TCGDEMO.DOC for details.\n\n",
             filename);
      exit(EXIT_FAILURE);
    case TCG_FORMAT_ERR:                    // not a TCG driver
      printf("Format error processing file %s; aborting.\n\n", filename);
      exit(EXIT_FAILURE);
    case TCG_ALLOC_ERR:                     // out of memory
      printf("Insufficient memory; aborting.\n\n");
      exit(EXIT_FAILURE);
    case TCG_FILE_ERR:                      // file read error
      printf("Error reading file %s; aborting.\n\n", filename);
      exit(EXIT_FAILURE);
    default:                                // who knows?
      printf("Unknown error loading driver %s; aborting.\n\n", filename);
      exit(EXIT_FAILURE);
  }
  atexit(unloadGraphDriver);                // unload driver on exit

  //*** Switch to graphics mode
  if (!initGraphics())                      // initialize graphics mode
  {
    printf("Unable to initialize graphics hardware; aborting.\n\n");
    exit(EXIT_FAILURE);
  }

  //*** Load the font
  if (font.load(FONT_FILENAME) == 0)
  {
    deInitGraphics();
    printf("Error loading font file %s; aborting.\n\n"
           "Please ensure that the file specified is spelled correctly, and that the\n"
           "TCGFONTS environment variable has been set to point to the TCG fonts\n"
           "directory; refer to TCGDEMO.DOC for more information.\n\n",
           FONT_FILENAME);
    exit(EXIT_FAILURE);
  }

  //*** Seed the random number generator
  randomize();

  //*** Sign off before exit
  atexit(signOff);                          // display sign-off message
  atexit(deInitGraphics);                   // restore text mode
}



/**************************************************************************
*  Function:    printMessage
*
*  Purpose:     Print a one-line message centered on the screen.
*
*  Entry:       string = Message to display
*               where = Flag indicating where to display message.  Can be
*                       Center or BottomCenter.
*
*  Exit:        N/A
**************************************************************************/

void printMessage(char *string, int where)
{
  switch (where)
  {
    case Center:
      font.put(((OUTVIEWPORTLRX-OUTVIEWPORTULX+1-font.width(string)) / 2),
              ((OUTVIEWPORTLRY-OUTVIEWPORTULY+1-font.maxHeight()) / 2),
               string);
      break;
    case BottomCenter:
      font.put(((OUTVIEWPORTLRX-OUTVIEWPORTULX+1-font.width(string)) / 2),
               OUTVIEWPORTLRY-font.maxHeight(),
               string);
      break;
    default:
      ;
  }
}



/**************************************************************************
*  Function:    lineDemo
*
*  Purpose:     Demonstrate line().
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

void lineDemo(void)
{
  static char message[] = "TCG can draw lines...";
  int maxY, count;

  //*** Set up the message
  printMessage(message);                    // print it
  maxY = OUTMAXY - font.maxHeight() - 4;    // get usable limits

  //*** Draw lines until a key is struck
  while (!kbhit())
  {
    for (count=0; count<100; count++)
      line(random(OUTMAXX+1), random(maxY),
           random(OUTMAXX+1), random(maxY),
           random(256), random(256), random(256));
  }
  QUIT_IF_ESCAPE;                           // quit if ESC pressed
}



/**************************************************************************
*  Function:    putPixelDemo
*
*  Purpose:     Demonstrate putPixel().
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

void putPixelDemo(void)
{
  static char message[] = "pixels...";
  int maxY, count;

  //*** Set up the message
  clearGraphics(0, 0, 0);                   // clear screen to black
  printMessage(message);                    // print it
  maxY = OUTMAXY - font.maxHeight() - 4;    // get usable limits

  //*** Draw pixels until a key is struck
  while (!kbhit())
  {
    for (count=0; count<2000; count++)
      putPixel(random(OUTMAXX+1), random(maxY),
      	       random(256), random(256), random(256));
  }
  QUIT_IF_ESCAPE;                           // quit if ESC pressed
}



/**************************************************************************
*  Function:    drawRectDemo
*
*  Purpose:     Demonstrate drawRect().
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

void drawRectDemo(void)
{
  static char message[] = "rectangles...";
  int maxY, count;

  //*** Set up the message
  clearGraphics(0, 0, 0);                   // clear screen to black
  printMessage(message);                    // print it
  maxY = OUTMAXY - font.maxHeight() - 4;    // get usable limits

  //*** Draw rectangles until a key is struck
  while (!kbhit())
  {
    for (count=0; count<100; count++)
      drawRect(random(OUTMAXX+1), random(maxY),
               random(OUTMAXX+1), random(maxY),
               random(256), random(256), random(256));
  }
  QUIT_IF_ESCAPE;                           // quit if ESC pressed
}



/**************************************************************************
*  Function:    filledRectDemo
*
*  Purpose:     Demonstrate filledRectDemo().
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

void filledRectDemo(void)
{
  static char message[] = "filled rectangles...";
  int maxY, count;

  //*** Set up the message
  clearGraphics(0, 0, 0);                   // clear screen to black
  printMessage(message);                    // print it
  maxY = OUTMAXY - font.maxHeight() - 4;    // get usable limits

  //*** Draw filled rectangles until a key is struck
  while (!kbhit())
  {
    for (count=0; count<20; count++)
      filledRect(random(OUTMAXX+1), random(maxY),
                 random(OUTMAXX+1), random(maxY),
                 random(256), random(256), random(256));
  }
  QUIT_IF_ESCAPE;                           // quit if ESC pressed
}



/**************************************************************************
*  Function:    ellipseDemo
*
*  Purpose:     Demonstrate ellipse().
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

void ellipseDemo(void)
{
  static char message[] = "ellipses...";
  int maxY, count;

  //*** Set up the message
  clearGraphics(0, 0, 0);                   // clear screen to black
  printMessage(message);                    // print it
  maxY = OUTMAXY - font.maxHeight() - 4;    // get usable limits

  //*** Draw ellipses until a key is struck
  setOutputViewport(0, 0, SCREENMAXX, maxY);  // set the viewport
  while (!kbhit())
  {
    for (count=0; count<100; count++)
      ellipse(random(OUTMAXX+1), random(OUTMAXY+1),
              random((OUTMAXX+1)/8)*2+10, random((OUTMAXY+1)/8)*2+10,
              random(256), random(256), random(256));
  }
  QUIT_IF_ESCAPE;                           // quit if ESC pressed
  setOutputViewport(0, 0, SCREENMAXX, SCREENMAXY);  // reset viewport
}



/**************************************************************************
*  Function:    filledEllipseDemo
*
*  Purpose:     Demonstrate filledEllipse().
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

void filledEllipseDemo(void)
{
  static char message[] = "filled ellipses...";
  int maxY, count;

  //*** Set up the message
  clearGraphics(0, 0, 0);                   // clear screen to black
  printMessage(message);                    // print it
  maxY = OUTMAXY - font.maxHeight() - 4;    // get usable limits

  //*** Draw filled ellipses until a key is struck
  setOutputViewport(0, 0, SCREENMAXX, maxY);  // set the viewport
  while (!kbhit())
  {
    for (count=0; count<30; count++)
      filledEllipse(random(OUTMAXX+1), random(OUTMAXY+1),
                    random((OUTMAXX+1)/8)*2+10, random((OUTMAXY+1)/8)*2+10,
                    random(256), random(256), random(256));
  }
  QUIT_IF_ESCAPE;                           // quit if ESC pressed
  setOutputViewport(0, 0, SCREENMAXX, SCREENMAXY);  // reset viewport
}



/**************************************************************************
*  Function:    circleDemo
*
*  Purpose:     Demonstrate circle().
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

void circleDemo(void)
{
  static char message[] = "circles...";
  int maxY, count;

  //*** Set up the message
  clearGraphics(0, 0, 0);                   // clear screen to black
  printMessage(message);                    // print it
  maxY = OUTMAXY - font.maxHeight() - 4;    // get usable limits

  //*** Draw circles until a key is struck
  setOutputViewport(0, 0, SCREENMAXX, maxY);  // set the viewport
  while (!kbhit())
  {
    for (count=0; count<100; count++)
      circle(random(OUTMAXX+1), random(OUTMAXY+1),
             random((OUTMAXX+1)/8)+10,
             random(256), random(256), random(256));
  }
  QUIT_IF_ESCAPE;                           // quit if ESC pressed
  setOutputViewport(0, 0, SCREENMAXX, SCREENMAXY);  // reset viewport
}



/**************************************************************************
*  Function:    filledCircleDemo
*
*  Purpose:     Demonstrate filledCircle().
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

void filledCircleDemo(void)
{
  static char message[] = "filled circles...";
  int maxY, count;

  //*** Set up the message
  clearGraphics(0, 0, 0);                   // clear screen to black
  printMessage(message);                    // print it
  maxY = OUTMAXY - font.maxHeight() - 4;    // get usable limits

  //*** Draw filled circles until a key is struck
  setOutputViewport(0, 0, SCREENMAXX, maxY);  // set the viewport
  while (!kbhit())
  {
    for (count=0; count<30; count++)
      filledCircle(random(OUTMAXX+1), random(OUTMAXY+1),
                   random((OUTMAXX+1)/8)+10,
                   random(256), random(256), random(256));
  }
  QUIT_IF_ESCAPE;                           // quit if ESC pressed
  setOutputViewport(0, 0, SCREENMAXX, SCREENMAXY);  // reset viewport
}



/**************************************************************************
*  Function:    fontDemo
*
*  Purpose:     Demonstrate fonts.
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

void fontDemo(void)
{
  static char message[] = "and, of course, text.";

  //*** Set up the message
  clearGraphics(0, 0, 0);                   // clear screen to black
  printMessage(message, Center);            // print it
  QUIT_IF_ESCAPE;                           // quit if ESC pressed
}



/**************************************************************************
*  Function:    viewportDemo
*
*  Purpose:     Demonstrate viewports.
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

void viewportDemo(void)
{
  static char message[] = "Output can be clipped.";
  VirtualCoord virtScreen;
  int x1, y1, x2, y2;

  //*** Build the image
  tcgLogo = makeVirtScreen(font.width("TCG"), font.height("TCG"));
  if (tcgLogo == NULL)
  {
    deInitGraphics();
    printf("Insufficient memory; aborting.\n\n");
    exit(EXIT_FAILURE);
  }
  setGraphicsOutputAddr(tcgLogo);
  clearGraphics(0, 0, 0);
  font.palette(1, 100, 0, 255);
  font.put(0, 0, "TCG");
  setGraphicsOutputAddr(NULL);

  //*** Set up the message
  font.palette(1, 255, 255, 255);           // update font colour
  clearGraphics(90, 90, 90);                // clear screen to grey
  printMessage(message);                    // print it

  //*** Configure the virtual coordinate system
  virtScreen.virtParams(4, 4);
  virtScreen.realParams(OUTMAXX, OUTMAXY);

  //*** Set the viewport to exclude the message area
  x1 = virtScreen.realX(1);
  y1 = virtScreen.realY(1);
  x2 = virtScreen.realX(3);
  y2 = virtScreen.realY(3);
  setOutputViewport(x1, y1, x2, y2);
  filledRect(x1, y1, x2, y2, 0, 0, 0);

  //*** Draw random things at random places.
  while (!kbhit())
  {
    switch (random(8))                      // choose a thing to draw
    {
      case 0:                               // line()
	x1 = random(OUTMAXX+1);
	y1 = random(OUTMAXY+1);
	x2 = random(OUTMAXX+1);
	y2 = random(OUTMAXY+1);
	if (clipLine(&x1, &y1, &x2, &y2))
          line(x1, y1, x2, y2, random(256), random(256), random(256));
	break;
      case 1:                               // ellipse()
        ellipse(random(OUTMAXX+1), random(OUTMAXY+1),
                random(OUTMAXX/3), random(OUTMAXY/3),
                random(256), random(256), random(256));
	break;
      case 2:                               // filledEllipse();
        filledEllipse(random(OUTMAXX+1), random(OUTMAXY+1),
                      random(OUTMAXX/3), random(OUTMAXY/3),
                      random(256), random(256), random(256));
	break;
      case 3:                               // circle()
	circle(random(OUTMAXX+1), random(OUTMAXY+1), random(OUTMAXX/3),
                random(256), random(256), random(256));
	break;
      case 4:                               // filledCircle()
	filledCircle(random(OUTMAXX+1), random(OUTMAXY+1), random(OUTMAXX/3),
                random(256), random(256), random(256));
	break;
      case 5:                               // putImageInv()
        putImageInv(random(OUTMAXX+1), random(OUTMAXY+1), tcgLogo);
	break;
      case 6:                               // filledRect()
	x1 = random(OUTMAXX+1);
	y1 = random(OUTMAXY+1);
	x2 = random(OUTMAXX+1);
	y2 = random(OUTMAXY+1);
	if (clipFilledRect(&x1, &y1, &x2, &y2))
          filledRect(x1, y1, x2, y2, random(256), random(256), random(256));
	break;
      case 7:
	drawRect(random(OUTMAXX+1), random(OUTMAXY+1), random(OUTMAXX+1),
                random(OUTMAXY+1), random(256), random(256), random(256));
	break;
      default:
	;
    }
  }
  QUIT_IF_ESCAPE;                           // quit if ESC pressed

  //*** Reset the viewport to cover the whole screen
  setOutputViewport(0, 0, SCREENMAXX, SCREENMAXY);
}



/**************************************************************************
*  Function:    mouseDemo
*
*  Purpose:     Demonstrate the use of the mouse.
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

void mouseDemo(void)
{
  int pointerNum;

  //*** Ensure that a mouse is installed
  if (!resetMouse())
  {
    deInitGraphics();
    printf("Microsoft or compatible mouse not detected.\n"
           "The demo for definable, device-independent pointers will not be run.\n\n"
           "Press a key to continue...");
    getKey();
    clrscr();
    exit(EXIT_SUCCESS);
  }

  //*** Initialize the mouse handler
  initNewMouse();                           // initialize handler
  setHorizLimitsMouse(0, OUTMAXX);          // set horizontal limits
  setVertLimitsMouse(0, OUTMAXY);           // set vertical limits
  setPosMouse(OUTMAXX/2, OUTMAXY/2);        // center pointer on-screen
  pointerNum = BIG_ARROW_POINTER;           // signal using BIG_ARROW_POINTER
  setupMousePointer(pointerNum);            // make BIG_ARROW_POINTER active
  showMouse();                              // show the pointer

  //*** Main loop
  while (!kbhit())
  {
    //*** Button pressed; change to BIG_TARGET_POINTER
    if (buttonMouse() && pointerNum!=BIG_TARGET_POINTER)
    {
      pointerNum = BIG_TARGET_POINTER;      // signal using BIG_TARGET_POINTER
      setupMousePointer(pointerNum);        // make BIG_TARGET_POINTER active
    }
    //*** Button released; change to BIG_ARROW_POINTER
    else if (!buttonMouse() && pointerNum==BIG_TARGET_POINTER)
    {
      pointerNum = BIG_ARROW_POINTER;       // signal using BIG_ARROW_POINTER
      setupMousePointer(pointerNum);        // make BIG_ARROW_POINTER active
    }
  }

  //*** Shut off mouse handler
  deInitNewMouse();
}



/**************************************************************************
*  Function:    signOff
*
*  Purpose:     Print a sign-off message.
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

void signOff(void)
{
  static char notice[] =
  {
    "Ŀ"
    " TrueColour Graphics 1.03 Demo  Copyright (c) 1994-1995 by Matthew Hildebrand "
    "͵"
    " Although this program is not particularly gripping, it demonstrates TCG's    "
    " speed and most of its many features.  TCG is especially suited to device-    "
    " independence, as can be seen by running this demo with different drivers,    "
    " though it may of course be used in single-mode programs as well.             "
    "                                                                              "
    " TGE has many features, including:                                            "
    "       Use of loadable graphics drivers and loadable fonts                   "
    "       A powerful set of graphical functions                                 "
    "       Support for viewports, or clipping regions                            "
    "       Support for virtual screens of varying sizes                          "
    "       Extensive mouse support, including definable pointers                 "
    "       Graphics output using COPY, AND, NOT, OR, and XOR                     "
    "       Support for 32k-, 64k-, and 16.8M-colour modes                        "
    "                                                                              "
    " TCG costs only $30 US funds or $40 Canadian funds; paid users are entitled   "
    " to free upgrades, technical support, complete source code, and eternal       "
    " happiness.  Refer to TCG.DOC for details.                                    "
    "͵"
    " Matthew Hildebrand, 4 College St., St. Catharines, Ontario, Canada, L2R 2W7  "
    ""
  };

  printf(notice);                           // print the message
}



/**************************************************************************
*  Function:    getKey
*
*  Purpose:     Get a character from the keyboard.  This function differs
*               from getch() in that it can correctly handle extended keys
*               such as the function keys.
*
*  Entry:       N/A
*
*  Exit:        Returns the keycode of the key pressed.
**************************************************************************/

int getKey(void)
{
  int ch;

  ch = getch();                             // get a byte
  if (!(ch & 0xFF))                         // is it zero?
    ch = getch() << 8;                      // yes, get another byte
  return (ch);                              // return the keycode
}
