/*
 * %W% %E% %U%  [EXTREL_1.2]
 *
 * VersaTrack orbit calculations are based on those that appear in Dr. Manfred
 * Bester's sattrack program (the Unix(tm) versions 1 and 2).
 *
 * The data from which the maps where generated come from "xsat", an
 * X-Windows program by David A. Curry (N9MSW).
 *
 * Site coordinates come from various sources, including a couple of
 * World Almanacs, and also from both of the programs mentioned above.
 *
 * The following are authors' applicable copyright notices:
 *
 *                                                                               
 * Copyright (c) 1992, 1993, 1994 Manfred Bester. All Rights Reserved.        
 *                                                                           
 * Permission to use, copy, modify, and distribute this software and its      
 * documentation for educational, research and non-profit purposes, without   
 * fee, and without a written agreement is hereby granted, provided that the  
 * above copyright notice and the following three paragraphs appear in all    
 * copies.                                                                    
 *                                                                              
 * Permission to incorporate this software into commercial products may be    
 * obtained from the author, Dr. Manfred Bester, 1636 M. L. King Jr. Way,     
 * Berkeley, CA 94709, USA.                                                   
 *                                                                             
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,  
 * SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF    
 * THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE AUTHOR HAS BEEN ADVISED   
 * OF THE POSSIBILITY OF SUCH DAMAGE.                                         
 *                                                                             
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT       
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A    
 * PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"       
 * BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,  
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.                                   
 *                                                                             
 *                                                                             
 * Copyright 1992 by David A. Curry                                            
 *                                                                             
 * Permission to use, copy, modify, distribute, and sell this software and its 
 * documentation for any purpose is hereby granted without fee, provided that  
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation.  The  
 * author makes no representations about the suitability of this software for  
 * any purpose.  It is provided "as is" without express or implied warranty.   
 *                                                                             
 * David A. Curry, N9MSW                                                       
 * Purdue University                                                           
 * Engineering Computer Network                                                
 * 1285 Electrical Engineering Building                                        
 * West Lafayette, IN 47907                                                    
 * davy@ecn.purdue.edu                                                         
 *                                                                             
 * VersaTrack Copyright (c) 1993, 1994 Siamack Navabpour. All Rights Reserved.
 *
 * Permission is hereby granted to copy, modify and distribute VersaTrack
 * in whole, or in part, for educational, non-profit and non-commercial use
 * only, free of charge or obligation, and without agreement, provided that
 * all copyrights and restrictions noted herein are observed and followed, and
 * additionally, that this and all other copyright notices listed herein
 * appear unaltered in all copies and in all derived work.
 *
 * This notice shall not in any way void or supersede any of the other authors
 * rights or privileges.
 *
 * VersaTrack IS PRESENTED FREE AND "AS IS", WITHOUT ANY WARRANTY OR SUPPORT.
 * YOU USE IT AT YOUR OWN RISK. The author(s) shall not be liable for any
 * direct, indirect, incidental, or consequential damage, loss of profits or
 * other tangible or intangible losses or benefits, arising out of or related
 * to its use. VersaTrack carries no warranty, explicit or implied, including
 * but not limited to those of merchantablity and fitness for a particular
 * purpose.
 *
 * Siamack Navabpour, 12342 Hunter's Chase Dr. Apt. 2114, Austin, TX 78729.
 * sia@bga.com or sia@realtime.com.
 */


#include <windows.h>
#include <stdio.h>
#include <math.h>
#include <sys/types.h>

#include "vstdefs.h"
#include "constant.h"
#include "vsttype.h"
#include "vstextrn.h"
#include "resource.h"

static char *direction[] = {
    "N", "NNE", "NE", "ENE", "E", "ESE", "SE", "SSE", "S",
    "SSW", "SW", "WSW", "W", "WNW", "NW", "NNW"
};

static char     *bufp;

struct itemtag {
    int item_id;
    int offset;
};

static struct itemtag Items[] = {
    IDC_RST_TITLE,    0,
    IDC_RST_DATE,     80,
    IDC_RST_RISE,     120,
    IDC_RST_SET,      150,
    IDC_RST_DURATION, 220,
    IDC_RST_MAXELEV,  180,
    IDC_RST_DIR,      200,
    -1, -1
};

static double   risetime, settime, maxelev, lasttime;
static char     risedir[4], setdir[4];
static select_t *selectp;
static track_t  *trackp;
static result_t *resultp;

static void
getdir(dir, str)
double dir;
char *str;
{
    double f;
    char **cp;
    
    str[0] = 0;
    for (f = 0, cp = direction; f < 360.0; f += (360.0/16.0), cp++)
        if (dir >= f && dir < (f + (360.0/16.0))) {
            strcpy(str, *cp);
            break;
        }
}

static void
bcopy(src, dst, n)
void *src, *dst;
register int n;
{
    register char *s,*d;
    s = (char *) src;
    d = (char *) dst;
    for (; n>0; n--)
        *d++ = *s++;
}

static void
rst_free()
{
    if (selectp)
        safeFree((void *) selectp);
    if (resultp)
        safeFree((void *) resultp);
    if (trackp)
        safeFree((void *) trackp);
    if (bufp)
        safeFree((void *) bufp);
    selectp = NULL;
    resultp = NULL;
    trackp = NULL;
    bufp = NULL;
}

static double predStopTime;

static BOOL
rst_next(sp, ttime)
select_t *sp;
double *ttime;
{
    double predTime, slowStepTime, fastStepTime;
    BOOL prevVisible, done;
    track_t *tp;
    result_t *rp;

    tp = sp->sl_tp;
    rp = sp->sl_rp;

    predTime = *ttime;

    if (predTime == -1.0)
        predTime = utctime();

    slowStepTime = tp->steptime;    /* 1.0 / tp->epochmeanmotion / 2000.0; */
    fastStepTime = slowStepTime * 10.0;

    prevVisible = done = FALSE;
    maxelev = -400.0;
    done = FALSE;

    do {
        if (predict(modelflag, sp, predTime) == 0)
            break;

        if (rp->r_elevation >= minElevation) {
            if (!prevVisible) {
                prevVisible = TRUE;
                risetime = predTime;
                getdir(rp->r_azimuth, risedir);
                maxelev = rp->r_elevation;
            }
            if (rp->r_elevation > maxelev)
                maxelev = rp->r_elevation;
        }
        else {
            if (prevVisible) {
                settime = predTime;
                getdir(rp->r_azimuth, setdir);
                done = TRUE;
            }
            prevVisible = FALSE;
        }

        predTime += (rp->r_elevation < VLOWELEV) ? fastStepTime : tp->steptime;

    } while (!done);

    *ttime = predTime;

    dateOnlyStr(risetime, tp->sitep,  sp->flags, 1, bufp+80);
    timeStr(risetime, tp->sitep,  sp->flags, 1, bufp+120);
    timeStr(settime, tp->sitep, sp->flags, 1, bufp+150);
    sprintf(bufp+180,"%-3.1lf", maxelev);
    sprintf(bufp+200,"%s  >>  %s",risedir, setdir);
    durStr(settime - risetime, bufp+220);

    return (predTime <= predStopTime ? TRUE : FALSE);
}

static LRESULT CALLBACK
RSTProc(HWND hwnd, UINT message, WPARAM wparam, LPARAM lparam)
{
    POINT *p;
    select_t *sp;
    struct itemtag *itp;

    switch (message) {
    case WM_INITDIALOG:
        p = DialogPos(Gwnd, hwnd);
        SetWindowPos(hwnd, 0, (int)p->x, (int)p->y, 0, 0,
            SWP_NOZORDER | SWP_NOSIZE | SWP_SHOWWINDOW);
        break;

    case RST_UPDATE:
        sp = (select_t *) wparam;
        for (itp = Items; itp->offset >= 0; itp++)
            SendDlgItemMessage(hwnd, itp->item_id, WM_SETTEXT, (WPARAM) 0,
                (LPARAM) (bufp+itp->offset) );
        return TRUE;

    case WM_COMMAND:
        if (LOWORD(wparam) == IDNEXT) {
            sp = (select_t *) GetWindowLong(hwnd, DWL_USER);
            ASSERT(sp);
            if (rst_next(sp, &lasttime))
                PostMessage(hwnd, RST_UPDATE, (WPARAM) 0, (LPARAM) sp);
            else
                usermsg("Next rise beyond specified duration.");
        }
        else if (LOWORD(wparam) == IDOK) {
            rst_free();
            rsthwnd = NULL;
            DestroyWindow(hwnd);
        }
        return TRUE;
        
    case WM_CLOSE:
        rst_free();
        DestroyWindow(hwnd);
        rsthwnd = NULL;
        break;
        
    case WM_DESTROY:
        rst_free();
        rsthwnd = NULL;
        break;

    case WM_CTLCOLORDLG:
    case WM_CTLCOLORSTATIC:
        ColorSet(wparam, CWHITE, 7);  /* was white and dark blue=5 */
        return (BOOL) hDrawBrush[7];  /* dark cyan = 6 */

    case WM_CTLCOLOREDIT:
        ColorSet(wparam, CBLACK, 8);   /* was white and light blue=12 */
        return (BOOL) hDrawBrush[8];    /* light grey = 8 */

    default:
        break;
    }
    return FALSE;
}


void
rst(sp)
select_t *sp;
{
    double predTime;
    track_t *tp;
    result_t *rp;

    if (rsthwnd || !sp)
        return;

    WaitCursor();
    if (!selectp)
        selectp = (select_t *) safeAlloc(sizeof(select_t));

    if (!trackp)
        trackp  = (track_t *)  safeAlloc(sizeof(track_t));

    if (!resultp)
        resultp = (result_t *) safeAlloc(sizeof(result_t));

    bcopy(sp,        selectp, sizeof(select_t));
    bcopy(sp->sl_tp, trackp,  sizeof(track_t));
    bcopy(sp->sl_rp, resultp, sizeof(result_t));

    sp = selectp;
    sp->sl_tp = trackp;
    tp = trackp;
    sp->sl_rp = resultp;
    rp = resultp;

    if (!tp->sitep || !tp->satp) {
        rst_free();
        NormCursor();
        return;
    }

    predTime = (tp->starttime == -1.0 ) ? utctime() : tp->starttime;

    predict_init(modelflag, sp, predTime);

    if (tp->satp->s_flags & SF_GEOSTAT) {
        NormCursor();
        sprintf(tmpbuf,"%s is Geostationary. Please use real-time Display.",
            tp->satp->s_name);
        usermsg(tmpbuf);
        rst_free();
        return;
    }

    if (!bufp)
        bufp = (char *) safeAlloc(260);

    sprintf(bufp, "%s  From  %-.31s,  %-.39s", tp->satp->s_name, tp->sitep->c_name,
        tp->sitep->c_locale);

    predStopTime = predTime + tp->duration;

    if (predStopTime < tp->stoptime)
        predStopTime = tp->stoptime;

    lasttime = predTime;

    (void) rst_next(sp, &lasttime);

    rsthwnd = CreateDialog(hInst, MAKEINTRESOURCE(IDD_RST), Gwnd, RSTProc);

    SetWindowLong(rsthwnd, DWL_USER, (long) sp);

    PostMessage(rsthwnd, RST_UPDATE, (WPARAM) 0, (LPARAM) sp);
    NormCursor();
}

