/*
 * %W% %E% %U%  [EXTREL_1.2]
 *
 * VersaTrack orbit calculations are based on those that appear in Dr. Manfred
 * Bester's sattrack program (the Unix(tm) versions 1 and 2).
 *
 * The data from which the maps where generated come from "xsat", an
 * X-Windows program by David A. Curry (N9MSW).
 *
 * Site coordinates come from various sources, including a couple of
 * World Almanacs, and also from both of the programs mentioned above.
 *
 * The following are authors' applicable copyright notices:
 *
 *                                                                               
 * Copyright (c) 1992, 1993, 1994 Manfred Bester. All Rights Reserved.        
 *                                                                           
 * Permission to use, copy, modify, and distribute this software and its      
 * documentation for educational, research and non-profit purposes, without   
 * fee, and without a written agreement is hereby granted, provided that the  
 * above copyright notice and the following three paragraphs appear in all    
 * copies.                                                                    
 *                                                                              
 * Permission to incorporate this software into commercial products may be    
 * obtained from the author, Dr. Manfred Bester, 1636 M. L. King Jr. Way,     
 * Berkeley, CA 94709, USA.                                                   
 *                                                                             
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,  
 * SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF    
 * THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE AUTHOR HAS BEEN ADVISED   
 * OF THE POSSIBILITY OF SUCH DAMAGE.                                         
 *                                                                             
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT       
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A    
 * PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"       
 * BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,  
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.                                   
 *                                                                             
 *                                                                             
 * Copyright 1992 by David A. Curry                                            
 *                                                                             
 * Permission to use, copy, modify, distribute, and sell this software and its 
 * documentation for any purpose is hereby granted without fee, provided that  
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation.  The  
 * author makes no representations about the suitability of this software for  
 * any purpose.  It is provided "as is" without express or implied warranty.   
 *                                                                             
 * David A. Curry, N9MSW                                                       
 * Purdue University                                                           
 * Engineering Computer Network                                                
 * 1285 Electrical Engineering Building                                        
 * West Lafayette, IN 47907                                                    
 * davy@ecn.purdue.edu                                                         
 *                                                                             
 * VersaTrack Copyright (c) 1993, 1994 Siamack Navabpour. All Rights Reserved.
 *
 * Permission is hereby granted to copy, modify and distribute VersaTrack
 * in whole, or in part, for educational, non-profit and non-commercial use
 * only, free of charge or obligation, and without agreement, provided that
 * all copyrights and restrictions noted herein are observed and followed, and
 * additionally, that this and all other copyright notices listed herein
 * appear unaltered in all copies and in all derived work.
 *
 * This notice shall not in any way void or supersede any of the other authors
 * rights or privileges.
 *
 * VersaTrack IS PRESENTED FREE AND "AS IS", WITHOUT ANY WARRANTY OR SUPPORT.
 * YOU USE IT AT YOUR OWN RISK. The author(s) shall not be liable for any
 * direct, indirect, incidental, or consequential damage, loss of profits or
 * other tangible or intangible losses or benefits, arising out of or related
 * to its use. VersaTrack carries no warranty, explicit or implied, including
 * but not limited to those of merchantablity and fitness for a particular
 * purpose.
 *
 * Siamack Navabpour, 12342 Hunter's Chase Dr. Apt. 2114, Austin, TX 78729.
 * sia@bga.com or sia@realtime.com.
 */


#include <windows.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "resource.h"
#include "vstdefs.h"
#include "constant.h"
#include "vsttype.h"
#include "vstextrn.h"


#undef NITEMS
#define NITEMS 13

static char *itemp;
static short itemid[NITEMS] = {
    IDC_SI01, IDC_SI02, IDC_SI03, IDC_SI04,
    IDC_SI05, IDC_SI06, IDC_SI07, IDC_SI08,
    IDC_SI09, IDC_SI10, IDC_SI11, IDC_SI12, IDC_SI13
};

/*
 * Show current satellite information.
 */

void
showSatellite(hwnd, satp)
HWND hwnd;
satellite_t *satp;
{
    mode_t *mp;
    
    if (satp == NULL) {
        usermsg("No Current Satellite");
        return;
    }

    itemp = (char *) (LPVOID) LocalAlloc(LMEM_FIXED, 40 * NITEMS);

    sprintf(itemp+(0*40),  "%-.28s", satp->s_name);
    if ((mp = satp->s_modep) != NULL) {
        if (mp->alias[0])
            sprintf(itemp+(0*40), "%-.28s (%s)", satp->s_name, mp->alias);
    }    
    itemp[40-1] = 0;
    sprintf(itemp+(1*40),  "%ld", satp->s_satnum);
    sprintf(itemp+(2*40),  "%-18.18s UTC", satp->s_epochstr);
    sprintf(itemp+(3*40),  "%lu", satp->s_elementset);
    sprintf(itemp+(4*40),  "%lf  d", satp->s_inclination);
    sprintf(itemp+(5*40),  "%lf  d", satp->s_raan);
    sprintf(itemp+(6*40),  "%lf", satp->s_eccentricity);
    sprintf(itemp+(7*40),  "%lf  d", satp->s_argperigee);
    sprintf(itemp+(8*40),  "%lf  d", satp->s_meananomaly);
    sprintf(itemp+(9*40), "%-.2lf  r/d",  satp->s_meanmotion);
    sprintf(itemp+(10*40), "%-2.3le  r/d**2",satp->s_decayrate);
    sprintf(itemp+(11*40), "%lu", satp->s_orbitnum);
    if (mp)
        sprintf(itemp+(12*40),"%-.4lf MHZ", mp->beacon * 1.0e-6);
    else
        strcpy(itemp+12*40, "N/A");

    DialogBox(NULL, MAKEINTRESOURCE(IDD_SATINFO), hwnd, ShowSatProc);

    LocalFree((HANDLE)itemp);
}


BOOL CALLBACK ShowSatProc(hwnd, message, wParam, lParam)
HWND hwnd;
UINT message;
WPARAM wParam;
LPARAM lParam;
{
    int i;
    POINT *p;
    
    switch (message) {
    case WM_INITDIALOG:
        p = DialogPos(Gwnd, hwnd);
        for (i=0; i<NITEMS; i++)
            SendDlgItemMessage(hwnd, itemid[i], WM_SETTEXT, (WPARAM)0,
                    (LPARAM) itemp+(i*40));
        SetWindowPos(hwnd, 0, (int)p->x, (int)p->y, 0, 0,
            SWP_NOZORDER | SWP_NOSIZE | SWP_SHOWWINDOW);
        break;
    case WM_COMMAND:
        if (LOWORD(wParam) == IDOK) {
            EndDialog(hwnd, wParam);
            return TRUE;
        }
        break;

    case WM_CTLCOLOREDIT:
        ColorSet(wParam, CBLACK, 8);
        return (BOOL) hDrawBrush[8];
    }
    return FALSE;
}

/*
 * List all satellites.
 */
void
listSatellites(hwnd)
HWND hwnd;
{
    register satellite_t **sipp,*sip;
    char buf[80];
    int n;

    if (!satInfo) return;
    
    for (n=0,sipp = satInfo; *sipp ; sipp++, n++) {
	sip = *sipp;
	sprintf(buf, "%.48s   (%d)", sip->s_name, sip->s_satnum); 
	sip->s_lbid = (short) SendMessage(hwnd, LB_ADDSTRING, (WPARAM)0,
                                          (LPARAM)buf);
    }
    sprintf(buf, "List of Satellites  ( %d )", n);
    SendDlgItemMessage(GetParent(hwnd), IDC_LISTOFSATS, WM_SETTEXT,
        (WPARAM) 0, (LPARAM) buf);
}

/* The following comment is left here for "historical reasons". */
/******************************************************************************/
/*                                                                            */
/*  Title       : readnorad.c                                                 */
/*  Author      : Manfred Bester, DL5KR                                       */
/*  Date        : 19Jun91                                                     */
/*  Last change : 10Mar93                                                     */
/*                                                                            *//*                                                                            */
/*                                                                            */
/*  Data for each satellite consist of three lines in the following format:   */
/*  0        1         2         3         4         5         6         7
/*  1234567890123456789012345678901234567890123456789012345678901234567890    */
/*  XXXXXXXXXXX                                                               */
/*  1 NNNNNU NNNNNAAA NNNNN.NNNNNNNN +.NNNNNNNN +NNNNN-N +NNNNN-N N NNNNN     */
/*  2 NNNNN NNN.NNNN NNN.NNNN NNNNNNN NNN.NNNN NNN.NNNN NN.NNNNNNNNNNNNNN     */
/*                                                                            */
/*  XXXXXXXXXXX                                                               */
/*  1 AAAAAU 00  0  0 BBBBB.BBBBBBBB  .CCCCCCCC  00000-0  00000-0 0  DDDZ     */
/*  2 AAAAA EEE.EEEE FFF.FFFF GGGGGGG HHH.HHHH III.IIII JJ.JJJJJJJJKKKKKZ     */
/*                                                                            */
/*  KEY:                                                                      */
/*                                                                            */
/*  X = Satellite name                                                        */
/*  A = Catalog number                       G = Eccentricity                 */
/*  B = Epoch time                           H = Argument of perigee          */
/*  C = Decay rate                           I = Mean anomaly                 */
/*  D = Number of the element set            J = Mean motion                  */
/*  E = Inclination                          K = Orbit number                 */
/*  F = RAAN                                 Z = Check sum                    */
/*                                                                            */
/*  Line 0 is an eleven-character name. Lines 1 and 2 are the standard        */
/*  two-line orbital element set format identical to that used by NASA and    */
/*  NORAD. The format description is as follows:                              */
/*                                                                            */
/*  Line 0:                                                                   */
/*  Column     Description                                                    */
/*   01-11     Satellite name                                                 */
/*                                                                            */
/*  Line 1:                                                                   */
/*   01-01     Line number of element set                                     */
/*   03-07     Satellite number                                               *//*   08-08     Classification (U = unclassified)                              */
/*   10-11     International designator (last two digits of launch year)      */
/*   12-14     International designator (launch number of the year)           */
/*   15-17     International designator (piece of launch)                     */
/*   19-20     Epoch year (last two digits of year)                           */
/*   21-32     Epoch (Julian day and fractional portion of the day)           */
/*   34-43     First time derivative of the mean motion                       */
/*             or ballistic coefficient (depending on ephemeris type)         */
/*   45-52     Second time derivative of mean motion (decimal point assumed;  */
/*             blank if n/a)                                                  */
/*   54-61     BSTAR drag term if GP4 general perturbation theory was used;   */
/*             otherwise, radiation pressure coefficient (decimal point       */
/*             assumed)                                                       */
/*   63-63     Ephemeris type                                                 */
/*   65-68     Element set number                                             */
/*   69-69     Check sum (modulo 10)                                          */
/*             (letters, blanks, periods, plus sign = 0; minus sign = 1)      */
/*                                                                            */
/*  Line 2:                                                                   */
/*   01-01     Line number of element set                                     */
/*   03-07     Satellite number                                               */
/*   09-16     Inclination [deg]                                              */
/*   18-25     Right Ascension of the ascending node [deg]                    */
/*   27-33     Eccentricity (decimal point assumed)                           */
/*   35-42     Argument of perigee [deg]                                      */
/*   44-51     Mean anomaly [deg]                                             */
/*   53-63     Mean motion [rev/d]                                            */
/*   64-68     Orbit (revolution number) at epoch [rev]                       */
/*   69-69     Check sum (modulo 10)                                          */
/*                                                                            */
/*  All other columns are blank or fixed.                                     */
/*                                                                            */
/*  Copyright 1992, 1993 Manfred Bester, DL5KR --- All Rights Reserved        */
/*                                                                            */
/******************************************************************************/

satellite_t *
findSatbyName(char *buf)
{
    int i, h, l, d;

    if (!satInfo)
        return NULL;

    for (l = 0, h = NSats -1; h >= l; ) {
        i = (h + l) / 2;
        if ((d = cistrcmp(buf, satInfo[i]->s_name)) == 0)
            return satInfo[i];
        if (d > 0)
            l = i+1;
        else
            h = i-1;
    }
    return (satellite_t *) 0;
}


satellite_t *
findSatbyID(id)
register int id;
{
    if (!satInfo || (id < 0) || (id > NSats))
        return NULL;
        
    return satInfo[id];
}

/*
 * Display information about the current site.
 */
 
 
void
showSite(hwnd, sitep)
HWND hwnd;
site_t *sitep;
{
    if ( !sitep ) {
    	usermsg("No Site selected.");
	    return;
    }

    sprintf(tmpbuf, "%-.31s", sitep->c_name);
    tmpbuf[31] = 0;
    sprintf(tmpbuf+40, "%-.39s", sitep->c_locale);
    tmpbuf[79] = 0;
    sprintf(tmpbuf+80, "%-.4f deg %c", fabs(sitep->c_lat * CRD),
    	sitep->c_lat < 0.0 ? 'S' : 'N');
    sprintf(tmpbuf+100, "%-.4f deg %c", fabs(sitep->c_lng * CRD),
	sitep->c_lng > 0.0 ? 'W' : 'E');
    sprintf(tmpbuf+120, "%-.0f m Above MSL", sitep->c_alt / CMKM);
    sprintf(tmpbuf+140, "%4.4s  ( %+.1f hrs )", sitep->c_tzname, sitep->c_timezone*24.0);
    
    (void) DialogBox(NULL, MAKEINTRESOURCE(IDD_SITEINFO), hwnd, ShowSiteProc);
}


BOOL CALLBACK ShowSiteProc(hwnd, message, wParam, lParam)
HWND hwnd;
UINT message;
WPARAM wParam;
LPARAM lParam;
{
    POINT *p;
    static struct a {
        char *str;
        int  id;
    } items[6] = { tmpbuf, IDC_LOCNAME, tmpbuf+40, IDC_LOCSTATE,
        tmpbuf+80, IDC_LOCLAT, tmpbuf+100, IDC_LOCLON,
        tmpbuf+120, IDC_LOCELV, tmpbuf+140, IDC_LOCTZONE,
    };
    int i;
    
    switch(message) {
        case WM_INITDIALOG:
            p = DialogPos(Gwnd, hwnd);
            for (i=0; i<6; i++) 
                SendDlgItemMessage(hwnd, (int)items[i].id,
                    (UINT)WM_SETTEXT, (WPARAM) 0, (LPARAM) items[i].str );
            SetWindowPos(hwnd, 0, (int)p->x, (int)p->y, 0, 0,
                SWP_NOZORDER | SWP_NOSIZE | SWP_SHOWWINDOW);
            break;

        case WM_COMMAND:
            if (LOWORD(wParam) == IDOK)
                EndDialog(hwnd, wParam);
            return TRUE;

        case WM_CTLCOLOREDIT:
            ColorSet(wParam, CBLACK, 8);
            return (BOOL)hDrawBrush[8];
    }
    return FALSE;
}

/*
 * List all known cities.
 */
void
listSites(hwnd)
HWND hwnd; 
{
    site_t **cp;
    int n;

    if (!siteInfo)
        return;
    for (n=0,cp = siteInfo; *cp ; cp++,n++) {
    	sprintf(tmpbuf, "%-.31s, %-.39s", (*cp)->c_name, (*cp)->c_locale);
            (*cp)->c_lbid = SendMessage(hwnd, LB_ADDSTRING,
            (WPARAM)0, (LPARAM)tmpbuf);
    }
    sprintf(tmpbuf,"List of Sites  ( %d )", n);
    SendDlgItemMessage(GetParent(hwnd), IDC_LISTOFSITES, WM_SETTEXT,
        (WPARAM) 0, (LPARAM) tmpbuf);
}
 
site_t *
findSitebyName(cname)
char *cname;
{
    char *cp, *clocale;
    register int h,l,i,d;

    if (!siteInfo)
        return NULL;

    for (clocale = NULL, cp = cname; *cp; cp++) {
        if (*cp == ',') {
            clocale = cp+1;
            *cp = 0;
            break;
        }
    }

    for (l=0, h = NSites -1; h >= l; ) {
        i = (l + h) / 2;
		d = cistrcmp(cname, siteInfo[i]->c_name);
		if (d == 0) {
		    if (clocale) {
	           	if (clocale && !(d=cistrcmp(clocale, siteInfo[i]->c_locale))) {
		            *(clocale-1) = ',';
	    	        return siteInfo[i];
    	    	}
        	}
	    	else
            	return siteInfo[i];
		}
        if (d > 0)
            l = i + 1;
        else
            h = i - 1;
    } while ( l <= h );

    if (clocale)
        *(clocale-1) = ',';

    return (site_t *)NULL;
}

site_t *
findSitebyID(id)
register int id;
{
    register site_t **ci;

    if (!siteInfo)
        return NULL;

    for (ci = siteInfo; *ci ; ci++)
        if ((*ci)->c_lbid == id)
            return *ci;

    return (site_t *) NULL;
}
