/* ============ */
/* hsort.c	*/
/* ============ */
#include <defcodes.h>
#include <assert.h>
/* ------------------- */
/* FUNCTION PROTOTYPES */
/* ------------------- */
# undef F
# if defined(__STDC__) || defined(__PROTO__)
#	define	F( P )	P
# else
#	define	F( P )	()
# endif
/* INDENT OFF */
static	void	BuildHeap F((size_t));
static	void	Heapify F((size_t, size_t));
static	void	Heapsort F((size_t));
extern	void	hsort F((void *, size_t, size_t,
			int (*)(const void *, const void *)));

static	void	Swap F((char *, char *));

static	int	(*Cmpr) F((const void *, const void *));
# undef F

/* INDENT ON */

extern	long	SwapCtr;

static	size_t	DataWidth;
static	char	  *MyBase;

#define ADDR(i) ((ULONG)((ULONG)(i)*(ULONG)DataWidth))
#define DEC(A)	(A -= DataWidth)
#define INC(A)	(A += DataWidth)
/* ==================================================================== */
/* hsort - sort a set of elements by the heapsort method		*/
/* ==================================================================== */
# if defined(__STDC__) || defined(__PROTO__)
void
hsort(void *Base, size_t Num, size_t Width,
    int(*Compare) (const void *, const void *))
# else
void
hsort(Base, Num, Width, Compare)
void   *Base;
size_t	Num;
size_t	Width;
int	(*Compare)();
# endif
{
    Cmpr = Compare;
    DataWidth = Width;
    MyBase = Base;

    Heapsort(Num);
}
/* ==================================================================== */
/* BuildHeap - makes sort target array a heap				*/
/* ==================================================================== */
# if defined(__STDC__) || defined(__PROTO__)
static void
BuildHeap(size_t  Num)
# else
static void
BuildHeap(Num)
size_t	Num;
# endif
{
    size_t  CtDwnCtr;

    for (CtDwnCtr = (Num >> 1); CtDwnCtr >= 1; --CtDwnCtr)
    {
	Heapify(CtDwnCtr, Num);
    }
}
# if defined(RECURSIVE_HEAP)
/* ==================================================================== */
/* Heapify - maintains the heap property for targeted array		*/
/* ==================================================================== */
# if defined(__STDC__) || defined(__PROTO__)
static void
Heapify(size_t Left, size_t Right)
# else
static void
Heapify(Left, Right)
size_t	Left;
size_t	Right;
# endif
{
    ULONG   Child;
    char    *ChildAdr, *LeftAdr, *NextAdr;

    Child = (Left << 1);

    ChildAdr = MyBase + ADDR(Child-1);
    LeftAdr  = MyBase + ADDR(Left-1);

    if (Child <= (ULONG) Right)
    {
	if (Child < Right)
	{
	    NextAdr = ChildAdr + DataWidth;

	    if (0 < (*Cmpr)(NextAdr, ChildAdr))
	    {
		++Child, ChildAdr = NextAdr;
	    }
	}

	if (0 < (*Cmpr)(ChildAdr, LeftAdr))
	{
	    Swap(ChildAdr, LeftAdr);

	    Heapify((size_t) Child, Right);
	}
    }
}
# else
/* ==================================================================== */
/* Heapify - maintains the heap property for targeted array		*/
/* ==================================================================== */
# if defined(__STDC__) || defined(__PROTO__)
static void
Heapify(size_t Left, size_t Right)
# else
static void
Heapify(Left, Right)
size_t	Left;
size_t	Right;
# endif
{
    ULONG   Child, Parent;
    char    *ChildAdr, *NextAdr, *ParentAdr;

    ChildAdr = MyBase + ADDR(Left-1);
    for (Parent = Left; (Child = (Parent << 1)) <= Right; Parent = Child)
    {
	ParentAdr = ChildAdr;
	ChildAdr  = MyBase + ADDR(Child-1);

	if ((Child < Right))
	{
	    NextAdr = ChildAdr + DataWidth;

	    if (0 < Cmpr(NextAdr, ChildAdr))
	    {
		++Child, ChildAdr = NextAdr;
	    }
	}

	if (0 > Cmpr(ParentAdr, ChildAdr))
	{
	    Swap(ChildAdr, ParentAdr);
	}
	else				/* Done if Nothing Swapped */
	{
	    break;
	}
    }
}
# endif
/* ==================================================================== */
/* Heapsort - implements the heapsort algorithm of Williams & Floyd	*/
/* ==================================================================== */
# if defined(__STDC__) || defined(__PROTO__)
static void
Heapsort(size_t  Num)
# else
static void
Heapsort(Num)
size_t	Num;
# endif
{
    char    *EndAddr;		/* Last Address in Heap */
    size_t  CtDwnCtr;		/* Count Down Counter	*/

    /* ---------------------------------- */
    /* Phase 1 - Create the initial heap. */
    /* Upon return, the first element in  */
    /* the heap is the largest. 	  */
    /* ---------------------------------- */
    BuildHeap(Num);

    /* ---------------------------------------- */
    /* Phase 2 - Sort by exchanging the first	*/
    /* and last elements in the heap, reducing	*/
    /* the size of the heap by one to discard	*/
    /* discard the most-recent largest, and	*/
    /* then making a new heap.			*/
    /* ---------------------------------------- */
    EndAddr = MyBase + ADDR(Num-1);
    for (CtDwnCtr = Num; CtDwnCtr > 1; --CtDwnCtr)
    {
	Swap(MyBase, EndAddr);
	Heapify(1, CtDwnCtr-1);
	DEC(EndAddr);
    }
}
/* ==================================================================== */
/* Swap - exchanges two data elements each of DataWidth bytes		*/
/* ==================================================================== */
# if defined(__STDC__) || defined(__PROTO__)
PRIVATE	void
Swap(char  * Elem1, char  * Elem2)
# else
PRIVATE void
Swap(Elem1, Elem2)
char  *Elem1;
char  *Elem2;
# endif
# if !defined(MAX_SWAP)
#	define	MAX_SWAP	256
# endif
{
    extern long SwapCtr;
    extern size_t DataWidth;

    ++SwapCtr;

    /* ----------------------------- */
    /* Swap in Most Efficient Manner */
    /* ----------------------------- */
    switch (DataWidth)
    {
    case 1:
	{
	    char    OneXfer;
	    OneXfer = *Elem1;
	    *Elem1 = *Elem2;
	    *Elem2 = OneXfer;
	    break;
	}
    case 2:
	{
	    short   TwoXfer;
	    TwoXfer = *(short *) Elem1;
	    *(short *) Elem1 = *(short *) Elem2;
	    *(short *) Elem2 = TwoXfer;
	    break;
	}
    case 4:
	{
	    long    FourXfer;
	    FourXfer = *(long *) Elem1;
	    *(long *) Elem1 = *(long *) Elem2;
	    *(long *) Elem2 = FourXfer;
	    break;
	}
    default:
	{
	    char    TempBuf[MAX_SWAP];
	    size_t  CopySize, XferSize;

	    /* ---------------------------------- */
	    /* Copy as much as possible each pass */
	    /* ---------------------------------- */
	    for (XferSize = DataWidth; 0 < XferSize; XferSize -= CopySize)
	    {
		CopySize = (XferSize < sizeof(TempBuf)) ? XferSize :
		    sizeof(TempBuf);

		memcpy(TempBuf, Elem1, CopySize);
		memcpy(Elem1, Elem2, CopySize);
		memcpy(Elem2, TempBuf, CopySize);
		Elem1 += CopySize, Elem2 += CopySize;
	    }
	    break;
	}
    }
}
# if defined(TEST)
#include "tsrtdefs.h"
# if defined(SORT_NAME)
# else
#	define	SORT_NAME  hsort
# endif
void
main(int argc, char **argv)
{
    SORT_TEST_STRU  SortParms;

    SortParms.argc	= argc;
    SortParms.argv	= argv;
    SortParms.SortFun	= SORT_NAME;
    SortParms.OKCodes	= (char *) NULL;
    SortParms.ExecName	= DQ(SORT_NAME);
    SortParms.ExecLabel = "Heapsort";

    tstsort(&SortParms);
}
# endif
