/*
** Copyright (C) 1994-1995 by Dustin Puryear. All rights reserved.
*/

/*
** $Header: C:/PROJ/SML/RCS/del.c 1.4 1995/05/02 06:05:16 dpuryear Exp dpuryear $
**
** $Locker: dpuryear $
**
** $Log: del.c $
** Revision 1.4  1995/05/02 06:05:16  dpuryear
** *** empty log message ***
**
*/

#include "sml.h"

/*
** Remove a substring from a string:
**
** sml_ndel()
** sml_ssdel()
** sml_fssdel()
** sml_cdel()
** sml_csdel()
** sml_ldel()
** sml_tdel()
** sml_wsdel()
** sml_lwsdel()
** sml_twsdel()
*/

/*
** sml_ndel()           Remove a region from a string
**
** Input:
**      - str = string
**      - pos = position to start
**      - n = number of characters to remove
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_ndel(char *str, size_t pos, size_t n)
{
      char *base = str;

      sml_serr(SML_SUCCESS);

      if ( n == 0 )
            return (str);

      str = str + pos;
      memmove(str, str + n, strlen(str + n) + 1);

      return (base);
}

/*
** sml_ssdel()          Remove a substring from string
**
** Input:
**      - str = string
**      - del = substring to remove
**      - cs = case sensitivity
**              - SML_SEN  = case sensitive
**              - SML_ISEN = case insensitive
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_ssdel(char *str, char *del, SML_CASE cs)
{
      char *found;                  /* pointer to found substring */
      size_t len = strlen(del);     /* length of string to remove */

      if ( str == "" || del == "" )
      {
            sml_serr(SML_NOSS);
            return (NULL);
      }

      if ( (found = sml_psearch(str, del, cs)) != NULL )
      {
            memmove(found, found + len, strlen(found + len) + 1);
            sml_serr(SML_SUCCESS);
            return (str);
      }
      else
      {
            sml_serr(SML_NOSS);
            return (NULL);
      }
}

/*
** sml_fssdel()         Remove all found occurances of a substring
**
** Input:
**      - str = string
**      - del = substring to remove
**      - cs = case sensitivity
**              - SML_SEN  = case sensitive
**              - SML_ISEN = case insensitive
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_fssdel(char *str, char *del, SML_CASE cs)
{
      char *found;                  /* pointer to found substring */
      size_t len = strlen(del);     /* length of string to remove */

      if ( str == "" || del == "" )
      {
            sml_serr(SML_NOSS);
            return (NULL);
      }

      /* this function is basically sml_ssdel() with a while loop */

      if ( (found = sml_psearch(str, del, cs)) != NULL )
      {
            sml_serr(SML_SUCCESS);
            do
            {
                  memmove(found, found + len, strlen(found + len) + 1);
            } while ( (found = sml_psearch(str, del, cs)) != NULL );
            return (str);
      }
      else
      {
            sml_serr(SML_NOSS);
            return (NULL);
      }
}

/*
** sml_cdel()           Strip undesired character from string
**
** Input:
**      - str = string
**      - c = character to strip
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_cdel(char *str, char c)
{
      size_t front = 0;
      size_t back = 0;

      sml_serr(SML_SUCCESS);

      while ( str[front] != NUL )
      {
            if ( str[front] != c )
                  str[back++] = str[front++];
      }
      str[back] = NUL;

      return (str);
}

/*
** sml_fcdel()          Strip undesired characters from string
**
** Input:
**      - str = string
**      - del = string of characters to remove
**
** Output:
**      - pointer to passed string or NULL if error
*/

char * sml_fcdel(char *str, char *del)
{
      size_t front;
      size_t back;

      sml_serr(SML_SUCCESS);

      while ( *del != NUL )
      {
            front = 0;
            back = 0;

            while ( str[front] != NUL )
            {
                  if ( str[front] != *del )
                        str[back++] = str[front++];
            }
            str[back] = NUL;
            del++;
      }

      return (str);
}

/*
** sml_ldel()           Strip leading character from string
**
** Input:
**      - str = string
**      - c = character to strip
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_ldel(char *str, char c)
{
      size_t front = 0;

      sml_serr(SML_SUCCESS);

      /* get to first non-c character */
      while ( str[front] == c && str[front] != NUL )
            front++;

      if ( str[front] == NUL )
            str[0] = NUL;
      else
            memmove(str, &str[front], strlen(&str[front]) + 1);

      return (str);
}

/*
** sml_tdel()           Strip trailing character from string
**
** Input:
**      - str = string
**      - c = character to strip
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_tdel(char *str, char c)
{
      int end;    /* this number might be negative, so make it signed */

      sml_serr(SML_SUCCESS);

      if ( (end = strlen(str) - 1) < 0 )  /* go to end of string */
            return (str);

      /* go to last non-c character */
      while ( str[end] == c && end >= 0 )
            end--;
      str[end + 1] = NUL;

      return (str);
}

/*
** sml_wsdel()           Strip ALL whitespace from a string
**
** Input:
**      - str = string
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_wsdel(char *str)
{
      size_t front = 0;
      size_t back = 0;

      sml_serr(SML_SUCCESS);

      while ( str[front] != NUL )
      {
            if ( !isspace(str[front]) )
                  str[back++] = str[front++];
      }
      str[back] = NUL;

      return (str);
}

/*
** sml_lwsdel()         Strip all leading whitespace from string
**
** Input:
**      - str = string
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_lwsdel(char *str)
{
      size_t front = 0;

      sml_serr(SML_SUCCESS);

      /* find front of string */
      while ( isspace(str[front]) && str[front] != NUL )
            front++;

      if ( str[front] == NUL )
            str[0] = NUL;
      else
            memmove(str, &str[front], strlen(&str[front]) + 1);

      return (str);
}

/*
** sml_twsdel()         Strip all trailing whitespace from string
**
** Input:
**      - str = string
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_twsdel(char *str)
{
      int end;    /* this number might be negative, so make it signed */

      sml_serr(SML_SUCCESS);

      if ( (end = strlen(str) - 1) < 0 )  /* go to end of string */
            return (str);

      while ( isspace(str[end]) && end >= 0 )
            end--;
      str[end + 1] = NUL;

      return (str);
}
