/*
** Copyright (C) 1994-1995 by Dustin Puryear. All rights reserved.
*/

/*
** $Header: C:/PROJ/SML/RCS/replace.c 1.3 1995/04/28 00:58:02 dpuryear Exp dpuryear $
**
** $Locker: dpuryear $
**
** $Log: replace.c $
** Revision 1.3  1995/04/28 00:58:02  dpuryear
** Redid logic for inner loops of ssrep() style functions. Now, they use
** a little bit of logic to decide whether a hole needs to be opened or
** closed so that the replacement string will fit in. This should increase
** speed because I was using TWO blind memmove()'s before.
**
** Revision 1.2  1995/04/21 00:38:22  DPURYEAR
** Fixed a NASTY bug in ssrep() and fssrep() (the two are almost identical
** and used the same method for ss-replacement, thus the shared bug). The
** entire inner loop for the replacement was changed.
**
** Revision 1.1  1995/03/27 23:50:07  dpuryear
** Initial revision
**
*/

#include "sml.h"

extern SML_ERROR sml_err;

/*
** Substring replacement:
**
** sml_ssrep()
** sml_fssrep()
** sml_crep()
*/

/*
** sml_ssrep()          Substring replace
**
** Input:
**      - str = string
**      - ss = substring to replace
**      - rep = substring replacement
**      - cs = case sensitivity
**              - SML_SEN = case sensitive
**              - SML_ISEN = case insensitive
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_ssrep(char *str, char *ss, char *rep, SML_CASE cs)
{
        char *found;  /* found substring to replace */

        /* check for an empty string */
        if ( str == "" || rep == "" )
        {
                sml_serr(SML_NOSS);
                return (NULL);
        }
        else
                sml_serr(SML_SUCCESS);

        /* now, just find ss and replace it */

        if ( (found = sml_psearch(str, ss, cs)) != NULL )
        {
                /* note that the order of variables below IS important */

                size_t sslen = strlen(ss);      /* substring length    */
                size_t rplen = strlen(rep);     /* replacement length  */
                char   *loc = found + sslen;    /* location to move to */
                size_t len = strlen(loc) + 1;

                /* rep is < ss, so close the hole (where rep goes) a lil' */

                if ( rplen < sslen )
                        memmove(found + sslen - rplen, loc, len);

                /* rep is > ss, so open the hole a lil' */

                else if ( rplen > sslen )
                        memmove(found + rplen, loc, len);

                /* move replacement in */
                memmove(found, rep, strlen(rep));
        }
        else
        {
                sml_serr(SML_NOSS);
                return (NULL);
        }

        return (str);
}

/*
** sml_fssrep()         Replace all substrings
**
** Input:
**      - str = string
**      - ss = substring to replace
**      - rep = substring replacement
**      - cs = case sensitivity
**              - SML_SEN = case sensitive
**              - SML_ISEN = case insensitive
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_fssrep(char *str, char *ss, char *rep, SML_CASE cs)
{
        char   *found;  /* found substring to replace */

        /* check for an empty string */
        if ( str == "" || rep == "" )
        {
                sml_serr(SML_NOSS);
                return (NULL);
        }
        else
                sml_serr(SML_SUCCESS);

        /* now, just find ss and replace it */

        if ( (found = sml_psearch(str, ss, cs)) != NULL )
        {
                size_t sslen = strlen(ss);  /* substring length         */
                size_t rplen = strlen(rep); /* replacement length       */
                size_t len;                 /* length of string loc + 1 */
                char   *loc;                /* location to move to      */

                do
                {
                        loc = found + sslen;
                        len = strlen(loc) + 1;

                        /* rep is < ss, so close the hole (where rep goes)
                        ** a lil'
                        */

                        if ( rplen < sslen )
                                memmove(found + sslen - rplen, loc, len);

                        /* rep is > ss, so open the hole a lil' */

                        else if ( rplen > sslen )
                                memmove(found + rplen, loc, len);

                        /* move replacement in */
                        memmove(found, rep, strlen(rep));

                } while ( (found = sml_psearch(str, ss, cs)) != NULL );
        }
        else
        {
                sml_serr(SML_NOSS);
                return (NULL);
        }

        return (str);
}

/*
** sml_crep()           Character replace
**
** Input:
**      - str = string
**      - c = character to search
**      - rep = character to replace
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_crep(char *str, char c, char rep)
{
        /* simple loop: check if current char == c, if so, replace */
        while ( *str != NUL )
        {
                if ( *str == c )
                        *str = rep;
                str++;
        }

        return (str);
}
