/*
** Copyright (C) 1994-1995 by Dustin Puryear. All rights reserved.
*/

/*
** $Id: search.c 1.3 1995/05/02 06:05:16 dpuryear Exp dpuryear $
**
** $Log: search.c $
** Revision 1.3  1995/05/02 06:05:16  dpuryear
** *** empty log message ***
**
** Revision 1.2  1995/04/28 01:29:05  dpuryear
** Touch-ups for v2.0 release.
**
** Revision 1.1  1995/03/27 23:50:07  dpuryear
** Initial revision
**
** Revision 1.1  1995/03/27 22:13:14  dpuryear
** Initial revision
**
*/

#include "sml.h"

/* actual search routines... by Raymond Gardner */
static void bmhi_init(char *pattern);
static char *bmhi_search(char *string, int stringlen);
static void bmh_init(char *pattern);
static char *bmh_search(char *string, int stringlen);

extern void bmhi_cleanup(void);

/*
** Search routines:
**
** sml_psearch()
** sml_esearch()
** sml_rend()
*/

/*
** sml_psearch()        String search with pointer return
**
** Input:
**      - str1 = string to be searched
**              - NULL to find next occurrance of str2 in previous str1
**      - str2 = sub-string to search for
**      - cs = case sensitivity
**              - SML_SEN = case sensitive
**              - SML_ISEN = case insensitive
**
** Output:
**      - pointer to found sub-string or NULL if error
**
** Notes:
**      - limitation: pattern length + string length must be less than 32767
*/

extern char * sml_psearch(char *str1, char *str2, SML_CASE cs)
{
      static char *p_str = NULL;    /* static pointer to str1  */
      char *found;                  /* pointer to str2 in str1 */

      sml_serr(SML_SUCCESS);

      if ( str1 != NULL ) /* if new search string, repoint p_str */
            p_str = str1;

      if ( cs == SML_SEN )
      {
            bmh_init(str2);
            found = bmh_search(p_str, strlen(p_str));
      }
      else
      {
            /* note - bmhi_init() alters sml_err to indicate memory failure */
            bmhi_init(str2);

            if ( sml_rerr() == SML_NOMEM )
                  found = NULL;
            else
                  found = bmhi_search(p_str, strlen(p_str));
      }

      if ( found != NULL )
            p_str = found + strlen(str2);
      else
      {
            if ( sml_rerr() != SML_NOMEM )
                  sml_serr(SML_NOSS);
      }

      return (found);
}

/*
** sml_esearch()        String search with element return
**
** Input:
**      - str1 = string to be searched
**              - NULL to find next occurrance of str2 in previous str1
**      - str2 = sub-string to search for
**      - cs = case sensitivity
**              - SML_SEN = case sensitive
**              - SML_ISEN = case insensitive
**
** Output:
**      - offset of found substring or negative value if error
**
** Notes:
**      - limitation: pattern length + string length must be less than 32767
*/

extern int sml_esearch(char *str1, char *str2, SML_CASE cs)
{
      static char *p_str = NULL;    /* permanent pointer to str1       */
      char *found;                  /* pointer to str2 in str1         */
      int e;                        /* element found starts at in str1 */

      if ( str2 == NULL )
      {
            sml_serr(SML_BADARG);
            return (NULL);
      }
      else
            sml_serr(SML_SUCCESS);

      /* if new search string, repoint p_str */
      if ( str1 != NULL )
            p_str = str1;

      if ( cs == SML_SEN )
      {
            bmh_init(str2);
            found = bmh_search(p_str, strlen(p_str));
      }
      else
      {
            /* note - bmhi_init() alters sml_err to indicate memory failure */
            bmhi_init(str2);

            if ( sml_rerr() == SML_NOMEM )
                  found = NULL;
            else
                  found = bmhi_search(p_str, strlen(p_str));
      }

      if ( found != NULL )
      {
            e = found - p_str;
            p_str = found + strlen(str2);
      }
      else
      {
            e = -1;

            /* bmh?_search() just couldn't find the ss */
            if ( sml_rerr() == SML_SUCCESS )
                  sml_serr(SML_NOSS);
      }

      return (e);
}

/*
** sml_rend()           Return pointer to NUL in string
**
** Input:
**      - str = string to be searched
**
** Output:
**      - pointer to NUL in string or NULL if error
*/

extern char * sml_rend(char *str)
{
      /* check arguments */

      if ( str == NULL || str == "" )
      {
            sml_serr(SML_BADARG);
            return (NULL);
      }
      else
            sml_serr(SML_SUCCESS);

      /* now, just find NUL */

      while ( *str != NUL )
            str++;

      return (str);
}

/********************************
** static
*/

/*
**  Case-Insensitive Boyer-Moore-Horspool pattern match
**
**  Public Domain version by Thad Smith 7/21/1992,
**  based on a 7/92 public domain BMH version by Raymond Gardner.
**
**  This program is written in ANSI C and inherits the compilers
**  ability (or lack thereof) to support non-"C" locales by use of
**  toupper() and tolower() to perform case conversions.
**  Limitation: pattern length + string length must be less than 32767.
**
**  10/21/93 rdg  Fixed bugs found by Jeff Dunlop
*/

typedef unsigned char uchar;

#define SML_LARGE 32767                 /* flag for last character match    */

static int ipatlen; /* # chars in pattern               */
static int iskip[UCHAR_MAX+1]; /* skip-ahead count for test chars  */
static int iskip2; /* skip-ahead after non-match with
                                        ** matching final character         */
static uchar *ipat = NULL; /* uppercase copy of pattern        */

/*
** bmhi_init() is called prior to bmhi_search() to calculate the
** skip array for the given pattern.
** Error: exit(1) is called if no memory is available.
*/

static void bmhi_init(char *pattern)
{
      int i, lastpatchar;
      ipatlen = strlen(pattern);

      /* Make uppercase copy of pattern */

      ipat = realloc((void *) ipat, ipatlen);
      if (!ipat)
            sml_serr(SML_NOMEM); /* dp  12/94 */
      for ( i = 0; i < ipatlen; i++)
            ipat[i] = toupper(pattern[i]);

      /* initialize skip array */

      for ( i = 0; i <= UCHAR_MAX; ++i ) /* rdg 10/93 */
            iskip[i] = ipatlen;
      for ( i = 0; i < ipatlen - 1; ++i )
      {
            iskip[ ipat[i] ] = ipatlen - i - 1;
            iskip[tolower(ipat[i])] = ipatlen - i - 1;
      }
      lastpatchar = ipat[ipatlen - 1];
      iskip[ lastpatchar ] = SML_LARGE;
      iskip[tolower(lastpatchar)] = SML_LARGE;
      iskip2 = ipatlen; /* Horspool's fixed second shift */
      for (i = 0; i < ipatlen - 1; ++i)
      {
            if ( ipat[i] == (uchar) lastpatchar )
                  iskip2 = ipatlen - i - 1;
      }
}

static char *bmhi_search(char *string, int stringlen)
{
      int i, j;
      char *s;

      i = ipatlen - 1 - stringlen;
      if (i >= 0)
            return NULL;
      string += stringlen;
      for ( ;; )
      {
            while ( (i += iskip[((uchar *)string)[i]]) < 0 )
                  ; /* mighty fast inner loop */
            if (i < (SML_LARGE - stringlen))
                  return NULL;
            i -= SML_LARGE;
            j = ipatlen - 1;
            s = (char *)string + (i - j);
            while ( --j >= 0 && toupper(s[j]) == ipat[j] )
                  ;
            if ( j < 0 ) /* rdg 10/93 */
            { /* dp  12/94 */
                  s[0] = s[0]; /* dp  12/94 */
                  return s; /* rdg 10/93 */
            }/* dp  12/94 */
            if ( (i += iskip2) >= 0 ) /* rdg 10/93 */
                  return NULL; /* rdg 10/93 */
      }
}

/*
** bmhi_cleanup()       free()'s ipat, which is dynamically allocated.
*/

void bmhi_cleanup(void)
{
      free(ipat);
}

/*
**  Case-sensitive Boyer-Moore-Horspool pattern match
**
**  public domain by Raymond Gardner 7/92
**
**  limitation: pattern length + string length must be less than 32767
**
**  10/21/93 rdg  Fixed bug found by Jeff Dunlop
*/

static int patlen;
static int skip[UCHAR_MAX+1]; /* rdg 10/93 */
static int skip2;
static uchar *pat;

static void bmh_init(char *pattern)
{
      int i, lastpatchar;

      pat = (uchar *)pattern;
      patlen = strlen(pattern);
      for (i = 0; i <= UCHAR_MAX; ++i) /* rdg 10/93 */
            skip[i] = patlen;
      for (i = 0; i < patlen; ++i)
            skip[pat[i]] = patlen - i - 1;
      lastpatchar = pat[patlen - 1];
      skip[lastpatchar] = SML_LARGE;
      skip2 = patlen; /* Horspool's fixed second shift */
      for (i = 0; i < patlen - 1; ++i)
      {
            if (pat[i] == (uchar) lastpatchar)
                  skip2 = patlen - i - 1;
      }
}

static char *bmh_search(char *string, int stringlen)
{
      int i, j;
      char *s;

      i = patlen - 1 - stringlen;
      if (i >= 0)
            return NULL;
      string += stringlen;
      for ( ;; )
      {
            while ( (i += skip[((uchar *)string)[i]]) < 0 )
                  ; /* mighty fast inner loop */
            if (i < (SML_LARGE - stringlen))
                  return NULL;
            i -= SML_LARGE;
            j = patlen - 1;
            s = (char *)string + (i - j);
            while (--j >= 0 && s[j] == pat[j])
                  ;
            if ( j < 0 ) /* rdg 10/93 */
            { /* dp  12/94 */
                  s[0] = s[0]; /* dp  12/94 */
                  return s; /* rdg 10/93 */
            }/* dp  12/94 */
            if ( (i += skip2) >= 0 ) /* rdg 10/93 */
                  return NULL; /* rdg 10/93 */
      }
}
