/*
** Copyright (C) 1994-1995 by Dustin Puryear. All rights reserved.
*/

/*
** $Header: C:/PROJ/SML/RCS/sml.h 1.8 1995/05/18 21:58:54 dpuryear Exp dpuryear $
**
** $Locker: dpuryear $
**
** $Log: sml.h $
** Revision 1.8  1995/05/18 21:58:54  dpuryear
** Changed sml_free(), sml_serr(), sml_rerr(), sml_overlay(), sml_fill(),
** sml_rrotate(), and sml_lrotate() to macros for speed.
**
** Revision 1.7  1995/05/04 05:22:54  dpuryear
** sml_free() was not properly changed to a macro - fixed.
**
** Revision 1.6  1995/05/02 06:43:55  dpuryear
** *** empty log message ***
**
** Revision 1.5  1995/05/02 06:09:14  dpuryear
** *** empty log message ***
**
** Revision 1.4  1995/05/02 06:07:48  dpuryear
** Revamp for v2.0
**
** Revision 1.3  1995/04/28 05:26:12  dpuryear
** Cosmetic touch-ups (readibility)
**
** Revision 1.2  1995/04/28 01:43:58  dpuryear
** Major revamp of the SML package for release as v2.0. Many functions have
** been added or removed. Comments in this file are MUCH improved.
**
** Revision 1.1  1995/03/27 23:54:41  dpuryear
** Initial revision
**
*/

/*
** SML, String Manipulation Library
**
**   This library contains a set of routines to deal with strings in ways
** not supported by the ANSI/ISO C standard library, although the routines
** only use standard functions.
**
** v2.0  Released as Freeware
**
** By Dustin Puryear (dpuryear@tyrell.net)
*/

#ifndef SML_H

 #if __cplusplus
  extern "C" {
 #endif

 #define SML_H   1
 #define NUL     '\0'

 #define SML_iMAX(a, b) ((a) < (b) ? b : a)
 #define SML_iMIN(a, b) ((a) < (b) ? a : b)

 #include <stddef.h>
 #include <stdarg.h>
 #include <string.h>
 #include <stdio.h>
 #include <stdlib.h>
 #include <ctype.h>
 #include <limits.h>

/*
** -------------------------------------------------------------------
** Enumerated types
** -------------------------------------------------------------------
*/

/* CASE sensitivity */

typedef enum _sml_case
{
        SML_SEN,              /* operation is case sen      */
        SML_ISEN              /* operation is not case sen  */
} SML_CASE;

/* ERROR values */

typedef enum _sml_error
{
        SML_SUCCESS,          /* op was successful               */
        SML_BADARG,           /* op was passed a bad argument    */
        SML_NOMEM,            /* not enough memory               */
        SML_BADPOS,           /* op passed bad position          */
        SML_NOSS,             /* op dealing with a string failed */
        SML_NOTOK             /* op dealing with a token failed  */
} SML_ERROR;

/*
** -------------------------------------------------------------------
** SML global variables
** -------------------------------------------------------------------
*/

extern SML_ERROR sml_err;    /* SML error condition */

/*
** -------------------------------------------------------------------
** Interface
** -------------------------------------------------------------------
*/

/*
** sml_init()           Initialize SML package
**
** Input:
**      - none
**
** Output:
**      - none
*/

extern void sml_init(void);

/*
** sml_rerr()           Return SML error condition
**
** Input:
**      - none
**
** Output:
**      - error condition caused by last SML function: sml_rerr() does not
**        alter the error condition
*/

#define sml_rerr() sml_err

/*
** sml_serr()           Set SML error condition
**
** Input:
**    - error condition: of type SML_ERROR
**
** Output:
**    - none
**
** Notes:
**    - This function is basically for me (author). I am rethinking how I want
**      to handle errors in the next release, so providing this function
**      means I won't have to change a lot of source. Just a line or two of
**      the macro/function.
*/

#define sml_serr(a) sml_err = a

/*
** sml_isalpha()        Is a string composed of all alpha
**
** Input:
**      - str = string
**
** Output:
**      - !0 if all alpha
**      - 0 otherwise
*/

extern int sml_isalpha(char *str);

/*
** sml_isalphanum()     Is a string composed of all alphanumerical or '_'
**
** Input:
**      - str = string
**
** Output:
**      - !0 if all alphanumerical or '_'
**      - 0 otherwise
*/

extern int sml_isalphanum(char *str);

/*
** sml_isdigit()          Is a string composed of all digits
**
** Input:
**      - str = string
**
** Output:
**      - !0 if all digits
**      - 0 otherwise
*/

extern int sml_isdigit(char *str);

/*
** sml_iswhite()        Is string all whitespace
**
** Input:
**      - str = string
**
** Output:
**      - !0 if all whitespace
**      - 0 otherwise
*/

extern int sml_iswhite(char *str);

/*
** sml_is()             Generic is tester
**
** Input:
**      - str = string
**      - is() = function to evaluate string (ie., isprint(), isxdigit())
**
** Output:
**      - !0 if all whitespace
**      - 0 otherwise
*/

extern int sml_is(char *str, int (*is)(int c));

/*
** sml_create()         Create a string with optional initialization/size
**
** Input:
**      - size = length of new string: 0 if size will be strlen(init_str)
**      - init_str = initializer string: NULL for no initializer string
**
** Output:
**      - pointer to new string or NULL if error
*/

extern char * sml_create(size_t size, char *init_str);

/*
** sml_csize()          Create empty string
**
** Input:
**      - n = size of string to create
**
** Output:
**      - pointer to new string or NULL if error
*/

extern char * sml_csize(size_t n);

/*
** sml_ccopy()          Create Copy
**
** Input:
**      - str = string to be copied
**
** Output:
**      - pointer to new string or NULL if error
*/

extern char * sml_ccopy(char *str);

/*
** sml_free()           Free a string created by SML
**
** Input:
**      - str = string
**
** Output:
**      - none
*/

#define sml_free(a) (sml_serr(SML_SUCCESS), free(a))

/*
** sml_mcat()          Multiple string concatenation (use old string)
**
** Input:
**      - str = string
**      - ... = string(s) to append
**      - NULL must be last argument passed
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_mcat(char *str, ...);

/*
** sml_nmcat()          Multiple string concatenation (create new string)
**
** Input:
**      - str = first string
**      - ... = following strings
**      - NULL must be last argument passed
**
** Output:
**      - pointer to new string or NULL if fail condition
*/

extern char * sml_nmcat(char *str, ...);

/*
** sml_nleft()          Return n characters from left of string ($LEFT)
**
** Input:
**      - str = string
**      - n = number of characters to return
**
** Output:
**      - pointer to new string or NULL if error
*/

extern char * sml_nleft(char *str, size_t n);

/*
** sml_nright()         Return n characters from right of string ($RIGHT)
**
** Input:
**      - str = string
**      - n = number of characters to return
**
** Output:
**      - pointer to new string or NULL error
*/

extern char * sml_nright(char *str, size_t n);

/*
** sml_nmid()           Return n characters from some position ($MID)
**
** Input:
**      - str = string
**      - pos = position to start
**      - n = number of characters to return
**
** Output:
**      - pointer to new string or NULL if error
*/

extern char * sml_nmid(char *str, size_t pos, size_t n);

/*
** sml_ndel()           Remove a region from a string
**
** Input:
**      - str = string
**      - pos = position to start
**      - n = number of characters to remove
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_ndel(char *str, size_t pos, size_t n);

/*
** sml_ssdel()          Remove a substring from string
**
** Input:
**      - str = string
**      - del = substring to remove
**      - cs = case sensitivity
**              - SML_SEN  = case sensitive
**              - SML_ISEN = case insensitive
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_ssdel(char *str, char *del, SML_CASE cs);

/*
** sml_fssdel()         Remove all found occurances of a substring
**
** Input:
**      - str = string
**      - del = substring to remove
**      - cs = case sensitivity
**              - SML_SEN  = case sensitive
**              - SML_ISEN = case insensitive
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_fssdel(char *str, char *del, SML_CASE cs);

/*
** sml_cdel()           Strip undesired character from string
**
** Input:
**      - str = string
**      - c = character to strip
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_cdel(char *str, char c);

/*
** sml_fcdel()          Strip undesired characters from string
**
** Input:
**      - str = string
**      - del = string of characters to remove
**
** Output:
**      - pointer to passed string or NULL if error
*/

char * sml_fcdel(char *str, char *del);

/*
** sml_ldel()           Strip leading character from string
**
** Input:
**      - str = string
**      - c = character to strip
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_ldel(char *str, char c);

/*
** sml_tdel()           Strip trailing character from string
**
** Input:
**      - str = string
**      - c = character to strip
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_tdel(char *str, char c);

/*
** sml_wsdel()           Strip ALL whitespace from a string
**
** Input:
**      - str = string
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_wsdel(char *str);

/*
** sml_lwsdel()         Strip all leading whitespace from string
**
** Input:
**      - str = string
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_lwsdel(char *str);

/*
** sml_twsdel()         Strip all trailing whitespace from string
**
** Input:
**      - str = string
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_twsdel(char *str);

/*
** sml_insert()         Insert a substring into some position
**
** Input:
**      - str = string
**      - sub = string to be inserted
**      - pos = position to insert
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_insert(char *str, char *sub, size_t pos);

/*
** sml_overlay()        Overlay a substring into a string
**
** Input:
**      - str = string
**      - sub = substring to overlay with
**      - pos = position to overlay
**
** Output:
**      - pointer to substitution string or NULL if error
*/

#define sml_overlay(str, sub, pos) (sml_serr(SML_SUCCESS), memmove(str \
      + pos, sub, strlen(sub)))

/*
** sml_toksplit()      Parse a string into an array of strings
**
** Input:
**      - str = string to be parsed
**      - toksep = tokens used to parse string
**
** Output:
**      - array of strings or NULL if error
**              - last element will be a NULL
*/

extern char ** sml_toksplit(char *str, char *tok);

/*
** sml_tokmerge()      Convert array of strings to a string
**
** Input:
**      - array = array of strings
**              - last element must be a NULL
**      - toksep = token(s) string to pad substrings (in string)
**              - empty string for no padding
**
** Output:
**      - pointer to new string or NULL if error
*/

extern char * sml_tokmerge(char **array, char *tok);

/*
** sml_toknext()        Jump to next token within string
**
** Input:
**      - str = string to be used
**              - NULL if you wish to use the previously passed string
**      - toksep - token separators
**
** Output:
**      - pointer to token (NUL if no more tokens) or NULL if error
*/

extern char * sml_toknext(char *str, char *toksep);

/*
** sml_tokprev()        Jump to previous token within string
**
** Input:
**      - str = string to be used
**              - NULL if you wish to use the previously passed string
**      - toksep - token separators
**
** Output:
**      - pointer to token or NULL if error or no more tokens
**
** Note:
**      - Will automatically start from eos and proceed to beginning.
*/

extern char * sml_tokprev(char *str, char *toksep);

/*
** sml_tokskip()        Returns pointer to n token within string
**
** Input:
**      - str = string
**      - toksep = token separators
**      - n = token pointer to return (1 is the first token)
**
** Output:
**      - pointer to token or NULL if error
*/

extern char * sml_tokskip(char *str, char *toksep, int n);

/*
** sml_tokcopy()        Return copy of n token from string
**
** Input:
**      - str = string to be searched
**      - toksep = token separators
**      - n = token pointer to return (1 is the first token)
**
** Output:
**      - pointer to token or NULL if error
*/

extern char * sml_tokcopy(char *str, char *toksep, int n);

/*
** sml_tokcnt()         Count number of tokens within string
**
** Input:
**      - str = string to be searched
**      - toksep = token separators
**
** Output:
**      - number of tokens within string
*/

extern int sml_tokcnt(char *str, char *toksep);

/*
** sml_ssrep()          Substring replace
**
** Input:
**      - str = string
**      - ss = substring to replace
**      - rep = substring replacement
**      - cs = case sensitivity
**              - SML_SEN = case sensitive
**              - SML_ISEN = case insensitive
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_ssrep(char *str, char *ss, char *rep, SML_CASE cs);

/*
** sml_fssrep()         Replace all substrings
**
** Input:
**      - str = string
**      - ss = substring to replace
**      - rep = substring replacement
**      - cs = case sensitivity
**              - SML_SEN = case sensitive
**              - SML_ISEN = case insensitive
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_fssrep(char *str, char *ss, char *rep, SML_CASE cs);

/*
** sml_crep()           Character replace
**
** Input:
**      - str = string
**      - c = character to search
**      - rep = character to replace
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_crep(char *str, char c, char rep);

/*
** sml_xchg()           String eXchange
**
** Input:
**      - str1 = string to be swapped with str2
**      - str2 = string to be swapped with str1
**
** Output:
**      - pointer to passed string or NULL if error
**
** Notes:
**      - make sure string(s) do not overstep memory with swap
*/

extern char * sml_xchg(char *str1, char *str2);

/*
** sml_fill()           Fill a string with a character
**
** Input:
**      - str = string
**      - c = character
**
** Output:
**      - pointer to passed string or NULL if error
*/

#define sml_fill(a, b) sml_nfill(a, b, 0, strlen(a))

/*
** sml_nfill()          Fill a region of a string with a character
**
** Input:
**      - str = string
**      - c = character
**      - pos = offset within string to start
**      - n = number of characters to fill
**
** Output:
**      - pointer to passed string or NULL if error
*/

#define sml_nfill(str, c, pos, n) (sml_serr(SML_SUCCESS), memset(str \
      + pos, c, n))

/*
** sml_2upper()         Upper the case of a string
**
** Input:
**      - str = string
**
** Output:
**      - pointer to passed string or NULL if error
*/

#define sml_2upper(a) sml_n2upper(a, 0, strlen(a))

/*
** sml_n2upper()        Upper the case of n characters within a string
**
** Input:
**      - str = string
**      - pos = offset within string to start
**      - n = number of characters to uppercase
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_n2upper(char *str, size_t pos, size_t n);

/*
** sml_2lower()         Lower the case of a string
**
** Input:
**      - str = string
**
** Output:
**      - pointer to passed string or NULL if error
*/

#define sml_2lower(a) sml_n2lower(a, 0, strlen(a))

/*
** sml_n2lower()        Lower the case of n characters within a string
**
** Input:
**      - str = string
**      - pos = offset within string to start
**      - n = number of characters to lowercase
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_n2lower(char *str, size_t pos, size_t n);

/*
** sml_rend()           Return pointer to NUL in string
**
** Input:
**      - str = string to be searched
**
** Output:
**      - pointer to NUL in string or NULL if error
*/

extern char * sml_rend(char *str);

/*
** It must be noted that the search functions are based on a snipped copy of
** Raymond Gardner's Boyer-Moore-Horspool pattern matching routines.
*/

/*
** sml_psearch()        String search with pointer return
**
** Input:
**      - str1 = string to be searched
**              - NULL to find next occurrance of str2 in previous str1
**      - str2 = sub-string to search for
**      - cs = case sensitivity
**              - SML_SEN = case sensitive
**              - SML_ISEN = case insensitive
**
** Output:
**      - pointer to found sub-string or NULL if error
**
** Notes:
**      - limitation: pattern length + string length must be less than 32767
*/

extern char * sml_psearch(char *str1, char *str2, SML_CASE cs);

/*
** sml_esearch()        String search with element return
**
** Input:
**      - str1 = string to be searched
**              - NULL to find next occurrance of str2 in previous str1
**      - str2 = sub-string to search for
**      - cs = case sensitivity
**              - SML_SEN = case sensitive
**              - SML_ISEN = case insensitive
**
** Output:
**      - offset of found substring or negative value if error
**
** Notes:
**      - limitation: pattern length + string length must be less than 32767
*/

extern int sml_esearch(char *str1, char *str2, SML_CASE cs);

/*
** sml_jleft()          Left justify string
**
** Input:
**      - str = string to be justified
**      - pad = character to pad right of string
**      - n = total memory available to string
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_jleft(char *str, char pad, size_t n);

/*
** sml_jright()         Right justify string
**
** Input:
**      - str = string to be justified
**      - pad = character to pad left of string
**      - n = total memory available to string
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_jright(char *str, char pad, size_t n);

/*
** sml_jcenter()        Center justify string
**
** Input:
**      - str = string to be justified
**      - pad = character to pad left and right of string
**      - n = total memory available to string
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_jcenter(char *str, char pad, size_t n);

/*
** sml_rrotate()        Rotate string right once
**
** Input:
**      - str = string
**
** Output:
**      - pointer to passed string or NULL if error
*/

#define sml_rrotate(a) sml_nrrotate(a, 1)

/*
** sml_nrrotate()       Rotate string right n times
**
** Input:
**      - str = string
**      - n = times to rotate characters
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_nrrotate(char *str, size_t n);

/*
** sml_lrotate()        Rotate string left once
**
** Input:
**      - str = string
**
** Output:
**      - pointer to passed string or NULL if error
*/

#define sml_lrotate(a) sml_nlrotate(a, 1)

/*
** sml_nlrotate()       Rotate string left n times
**
** Input:
**      - str = string
**      - n = times to rotate characters
**
** Output:
**      - pointer to passed string or NULL if error
*/

extern char * sml_nlrotate(char *str, size_t n);

 #if __cplusplus
  }
 #endif

#endif                  /* #ifndef SML_H         */
