/*
** Copyright (C) 1994-1995 by Dustin Puryear. All rights reserved.
*/

/*
** $Header: C:/PROJ/SML/RCS/tok.c 1.3 1995/05/02 06:05:16 dpuryear Exp dpuryear $
**
** $Locker: dpuryear $
**
** $Log: tok.c $
** Revision 1.3  1995/05/02 06:05:16  dpuryear
** *** empty log message ***
**
** Revision 1.2  1995/04/28 01:24:07  dpuryear
** Touch-ups for v2.0 release.
**
** Revision 1.1  1995/03/27 23:54:46  dpuryear
** Initial revision
**
*/

#include "sml.h"

/*
** String token functions:
**
** sml_toksplit()
** sml_tokmerge()
** sml_tokskip()
** sml_toknext()
** sml_tokprev()
** sml_tokcopy()
** sml_tokcnt()
*/

/*
** sml_toksplit()      Parse a string into an array of strings
**
** Input:
**      - str = string to be parsed
**      - toksep = tokens used to parse string
**
** Output:
**      - array of strings or NULL if error
**              - last element will be a NULL
*/

extern char ** sml_toksplit(char *str, char *toksep)
{
      char *cstr;       /* copy of passed string                  */
      char **array;     /* array of tokens being created          */
      char **barray;    /* memory location of array               */
      int size = 10;    /* current size of array in elements      */
      int cnt = 0;      /* current count - elements used          */

      sml_serr(SML_SUCCESS);

      /* make a copy so we can change it */
      if ( (cstr = sml_ccopy(str)) == NULL )
            return (NULL);

      /* array of tokens */
      if ( (array = (char **) malloc(sizeof(char *) * size)) == NULL )
      {
            sml_free(cstr);
            sml_serr(SML_NOMEM);
            return (NULL);
      }

      /* barray will always point to current base of array in memory */
      barray = array;

      array[0] = strtok(cstr, toksep);
      while ( array[cnt] != NULL )
      {
            cnt++;
            if ( cnt == size )
            {
                  /* end of array, make it bigger */
                  size += 10;
                  array = (char **) realloc(barray, sizeof(char *) * size);
                  if ( array == NULL )
                  {
                        /* error reallocating memory - kill mem and exit */
                        array = barray;
                        while ( cnt != 0 )
                              sml_free(array[--cnt]);
                        sml_serr(SML_NOMEM);
                        return (NULL);
                  }
                  else
                  {
                        /* realloc() worked, repoint and continue */
                        barray = array;
                        array += cnt - 1; /* offset = cnt - 1 */
                  }
            }
            array[cnt] = strtok(NULL, toksep);
      }

      /* free copy of string */
      sml_free(cstr);

      return (barray);
}

/*
** sml_tokmerge()      Convert array of strings to a string
**
** Input:
**      - array = array of strings
**              - last element must be a NULL
**      - toksep = token(s) string to pad substrings (in string)
**              - empty string for no padding
**
** Output:
**      - pointer to new string or NULL if error
*/

extern char * sml_tokmerge(char **array, char *toksep)
{
      size_t size;      /* size of string to create for merge of tokens */
      size_t e;         /* current element within token array           */
      char *buf;        /* buf - the string created                     */

      sml_serr(SML_SUCCESS);

      /* get total size for new string */
      for ( e = 0, size = 0; array[e] != NULL; e++ )
            size += strlen(array[e]);

      /* create the string and copy first element into it */
      if ( (buf = sml_create(size, array[0])) == NULL )
            return (NULL);

      /* now, copy the rest of the array into the string  */
      for ( e = 1; array[e] != NULL; e++ )
            sml_mcat(buf, toksep, array[e], NULL);

      return (buf);
}

/*
** sml_tokskip()        Returns pointer to n token within string
**
** Input:
**      - str = string
**      - toksep = token separators
**      - n = token pointer to return (1 is the first token)
**
** Output:
**      - pointer to token or NULL if error
*/

extern char * sml_tokskip(char *str, char *toksep, int n)
{
      sml_serr(SML_SUCCESS);

      /* skip leading token separators */
      str += strspn(str, toksep);
      /* we are now pointing at first token */
      n--;

      /* just skip tokens until we get to n token: we are already at n = 1 */
      while ( str != NULL && *str != NUL && n > 0 )
      {
            str = strpbrk(str, toksep);
            if ( str != NULL )
            {
                  str += strspn(str, toksep);
                  n--;
            }
      }

      return (str);
}

/*
** sml_toknext()        Jump to next token within string
**
** Input:
**      - str = string to be used
**              - NULL if you wish to use the previously passed string
**      - toksep - token separators
**
** Output:
**      - pointer to token (NUL if no more tokens) or NULL if error
*/

extern char * sml_toknext(char *str, char *toksep)
{
      static char *ptr = NULL;
      static int cnt;
      static int cur;

      sml_serr(SML_SUCCESS);

      /* check if a new string was passed - if not, use last string passed */
      if ( str != NULL )
      {
            ptr = str;
            cnt = sml_tokcnt(ptr, toksep);
            cur = 1;
      }
      else if ( ptr == NULL )       /* a string was never sent! */
      {
                  sml_serr(SML_BADARG);
                  return (NULL);
      }

      /* check if we are already at last token */
      if ( cur > cnt )
      {
            sml_serr(SML_NOTOK);
            return (NULL);
      }

      /* use sml_tokskip() to get to next token */
      return (sml_tokskip(ptr, toksep, cur++));
}

/*
** sml_tokprev()        Jump to previous token within string
**
** Input:
**      - str = string to be used
**              - NULL if you wish to use the previously passed string
**      - toksep - token separators
**
** Output:
**      - pointer to token or NULL if error or no more tokens
**
** Note:
**      - Will automatically start from eos and proceed to beginning.
*/

extern char * sml_tokprev(char *str, char *toksep)
{
      static char *ptr = NULL;
      static int cnt;

      sml_serr(SML_SUCCESS);

      /* check if a new string was passed - if not, use last string passed */

      if ( str != NULL )
      {
            ptr = str;
            cnt = sml_tokcnt(str, toksep);
      }
      else if ( ptr == NULL )       /* a string was never sent! */
      {
                  sml_serr(SML_BADARG);
                  return (NULL);
      }

      /* just use sml_tokskip() to get to next (umm.. previous) token */
      if ( cnt == 0 )
      {
            sml_serr(SML_NOTOK);
            return (NULL);
      }

      return (sml_tokskip(ptr, toksep, cnt--));
}

/*
** sml_tokcopy()        Return copy of n token from string
**
** Input:
**      - str = string to be searched
**      - toksep = token separators
**      - n = token pointer to return (1 is the first token)
**
** Output:
**      - pointer to token or NULL if error
*/

extern char * sml_tokcopy(char *str, char *toksep, int n)
{
      char *ptr;
      size_t len;

      sml_serr(SML_SUCCESS);

      /* skip leading token separators */

      str += strspn(str, toksep);
      n--;        /* we are now pointing at first token */

      while ( str != NULL && *str != NUL && n > 0 )
      {
            str = strpbrk(str, toksep);
            if ( str != NULL )
            {
                  str += strspn(str, toksep);
                  n--;
            }
      }

      /* we are now pointing at desired token - just copy it */
      if ( str != NULL )
      {
            ptr = strpbrk(str, toksep);   /* find next toksep */
            if ( ptr == NULL )
                  len = strlen(str);      /* n token is last token in str */
            else
                  len = ptr - str;        /* start of string - next toksep */

            /* now, do the actual copy */
            ptr = (char *) malloc(len + 1);
            if ( ptr != NULL )
            {
                  memcpy(ptr, str, len);
                  ptr[len] = NUL;
            }
      }

      return (ptr);
}

/*
** sml_tokcnt()         Count number of tokens within string
**
** Input:
**      - str = string to be searched
**      - toksep = token separators
**
** Output:
**      - number of tokens within string
*/

extern int sml_tokcnt(char *str, char *toksep)
{
      size_t cnt;

      sml_serr(SML_SUCCESS);

      /* skip leading token separators */
      str += strspn(str, toksep);

      /* just find tokens until we are out */
      cnt = 0;
      while ( *str != NUL )
      {
            cnt++;
            str = strpbrk(str, toksep);
            str += strspn(str, toksep);
      }

      return (cnt);
}
