/*
 * async.c -- asynchronous COM port I/O
 *
 * V. Abell
 *
 */

/*
 * Copyright 1994 Victor A. Abell, Lafayette, Indiana  47906.  All rights
 * reserved.
 *
 * Written by Victor A. Abell.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Victor A. Abell is not responsible for any consequences of the use of
 * this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to Victor A. Abell must
 *    appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#if	!defined(lint)

# if	defined(_BCC)
#pragma	warn	-use
# endif

static char copyright[] =
"@(#) Copyright 1994 Victor A. Abell.\nAll rights reserved.\n";
#endif

#include "touch2.h"
#include <time.h>

#define TMO	10		/* send timeout seconds */
#define BUFSZ	256		/* circular buffer size */
#define NRFULL	200		/* buffer near-full size */
#define NREMPTY	32		/* buffer near-empty size */

#define IER	1		/* Interrupt Enable Register offset */
#define	LCR	3		/* Line Control Register offset */
#define	MCR	4		/* Modem Control Register offset */
#define	LSR	5		/* Line Status Register offset */
#define	MSR	6		/* Modem Status Register offset */

static int  B_Hnd;		/* handshake flag */
static int  B_In;               /* Buf[] in pointer */
static int  B_Out;              /* Buf[] out pointer */
static char *Buf = NULL;	/* buffer */
int  AsynBfLen;			/* characters in Buf[] */
static int  IntLev;		/* port interrupt level */
static int  OpenSt = 0;		/* open status */
static int  OvrRun;		/* over-run status */

static int OldDiv;		/* old baud rate Divisor */
static int OldIER;		/* old Interrupt Enable Register */
static int OldIntM;		/* old Interrupt controller Mask */
static int OldLCR;		/* old Line Control Register */
static int OldMCR;		/* old Modem Control Register */
static
void (interrupt far *OldV)();	/* old Interrupt Vector */
static int Pad;			/* Port address */

static void AsynSend(char c, int *err_code);


/*
 * AsynClose() - close asynchronous COM port
 */

void
AsynClose()
{
	if (OpenSt) {
		_disable();
		outp(0x21, OldIntM);
		outp(Pad + LCR, (OldLCR | 0x80));
		outpw(Pad, OldDiv);
		outp(Pad + LCR, OldLCR);
		outp(Pad + IER, OldIER);
		outp(Pad + MCR, OldMCR);
		_dos_setvect(IntLev, OldV);
		_enable();
		OpenSt = 0;
	}
}


/*
 * AsynInp() - input one character
 */

void
AsynInp(char *c,		/* character receiver */
	int *error_code)	/* error reply: 0 = none
				 *		6 = no character available
 				 *		7 = buffer overflow
 				 *	       10 = com port not initialized */
{

/*
 * Check for errors: no character available;
 *		     buffer overflow; 
 *		     uninitialized com port.
 */ 
	if ( ! OpenSt) {
		*error_code = 10;
		return;
	}
	if ( ! AsynBfLen) {
		*error_code = 6;
		return;
	}
	if (OvrRun) {
		*error_code = 7;
		return;
	}
/*
 * Supply character and update buffer management controls.
 */
	*error_code = 0;
	*c = Buf[B_Out++];
	if (B_Out >= BUFSZ)
		B_Out = 0;
/*
 * Enable RTS if buffer length < NREMPTY.
 */
	if (--AsynBfLen < NREMPTY && B_Hnd) {
		outp(Pad + MCR, (8 + 2 + 1));
		B_Hnd = 0;
	}
}


/*
 * AsynISR() - input service request function
 */

static void interrupt far
AsynISR()
{
	if (OpenSt) {

	/*
	 * Get character and store in circular buffer.
	 * Do wrap-around of the pointer, as required.
	 */
		Buf[B_In++] = inp(Pad);
		if (B_In >= BUFSZ)
			B_In = 0;
	/*
	 * Count the characters in the buffer and check for possible
	 * over-run.  Disable RTS if the count is at the near-full threshold.
	 */
		if (++AsynBfLen > BUFSZ) {
			AsynBfLen = BUFSZ;
			OvrRun = 1;
		}
		if (AsynBfLen > NRFULL) {
			outp(Pad + MCR, 8 + 1);	/* Disable RTS,
						 * preserve RTS and OUT2 */
			B_Hnd = 1;
		}
	}
	outp(0x20, 0x20);			/* End of Interrupt to 8259 */
}


/*
 * AsynOpen() - open asynchronous COM port
 */

void
AsynOpen(int Cport,		/* COM port: 0 = COM1
				 *	     1 = COM2
				 *	     2 = COM3
				 *	     3 = COM4 */
	 int irq,		/* IRQ number: 0 = none */
	 int portaddr,		/* alternate port address: 0 = none */
         int baud,		/* baud rate: 300, 1200, 2400, or 9600 */
         int parity,		/* parity: 0 = none
				 *	   1 = even
				 *	   2 = odd */
         int stopbits,		/* stop bits - 1 or 2 */
         int numbits,		/* word length - 7 or 8 */
         int *error_code)	/* error code: -1 = no COM buffer space
				 *		0 = no error
				 *		1 = bad COM port
				 *		2 = bad baud rate
				 *		3 = bad parity
				 *		4 = bad stop bits
				 *		5 = bad word length */
{
	int bits, div, mask, par, startstop;

	*error_code = 0;
/*
 * Process bits per byte.
 */
	switch (numbits) {
	case 7:
		bits = 2;
		break;
	case 8:
		bits = 3;
		break;
	default:
		*error_code = 5;
		return;
	}
/*
 * Process start/stop bit count.
 */
	switch (stopbits) {
	case 1:
		startstop = 0<<2;
		break;
	case 2:
		startstop = 1<<2;
		break;
	default:
		*error_code = 4;
		return;
	}
/*
 * Process parity.
 */
	switch (parity) {
	case 0:			/* none */
		par = 000;
		break;
	case 1:			/* even */
		par = 030;
		break;
	case 2:			/* odd */
		par = 010;
		break;
	default:
		*error_code = 3;
		return;
	}
/*
 * Set baud rate divisor.
 */
	switch (baud) {
	case 300:
		div = 384;
		break;
	case 1200:
		div = 96;
		break;
	case 2400:
		div = 48;
		break;
	case 9600:
		div = 12;
		break;
	default:
		*error_code = 2;
		return;
	}
/*
 * Determine port address.
 */
	if (Cport < 0 || Cport > 3) {
		*error_code = 1;
		return;
	}
	if (portaddr != 0)
		Pad = portaddr;
	else if ((Pad = peek(0x40, Cport * 2)) == 0) {
		*error_code = 1;
		return;
	}
/*
 * Determine interrupt level.
 *
 * Defaults:
 *		Port	COM	IRQ	Int	Mask
 *		  0	 1	 4	0xc	0xef
 *		  1	 2	 3	0xb	0xf7
 *		  2	 3	 4	0xc	0xef
 *		  3	 4	 3	0xb	0xf7
 * Others:
 *		Port	COM	IRQ	Int	Mask
 *		 0-3	1-4	 5	0xd	0xdf
 *		 0-3	1-4 	 7	0xf	0x7f
 */
	if (irq != 0) {
		if (irq < 3 || irq > 7 || irq == 6) {
			*error_code = 9;
			return;
		}
		IntLev = 0xb - 3 + irq;
	} else {
		IntLev = 0xc - (Cport & 1);
		irq = 1 - (Cport & 1) + 3;
	}
	mask = 0xff ^ (1 << irq);
/*
 * Allocate buffer space.
 */
	if (Buf == NULL) {
		if ((Buf = (char *)malloc(BUFSZ)) == NULL) {
			*error_code = -1;
			return;
		}
	}
/*
 * Save current COM port information.
 */
	OldV = _dos_getvect(IntLev);
	OldIntM = inp(0x21);
	OldLCR = inp(Pad + LCR) & 0x7f;
	_disable();
	_dos_setvect(IntLev, AsynISR);
	OldMCR = inp(Pad + MCR);
	OldIER = inp(Pad + IER);
	outp(Pad + LCR, (OldLCR | 0x80));
	OldDiv = inpw(Pad);
/*
 * Set new port information:
 *
 *		baud rate divisor,
 *		bit length,
 *		start/stop bits.
 */

	outpw(Pad, div);
	outp(Pad + LCR, ((par | startstop | bits) & 0x7f));
/*
 * Reset errors and flush input.
 * Set 8259 interrupt mask.
 * Set OUT2, RTS, and DTR in modem control register.
 * Re-enable interrupts.
 */
	(void) inp(Pad + LSR);
	(void) inp(Pad);
	outp(0x21, (OldIntM & mask));
	outp(Pad + IER, 1);
	outp(Pad + MCR, (8 + 2 + 1));
	_enable();
/*
 * All done.
 * Set OpenSt status.
 * Reset buffer as a last gesture before returning.
 */
	OpenSt = 1;
	AsynRstBf();
}


/*
 * AsynRstBf() - reset input buffer
 */

void
AsynRstBf(void)
{
	AsynBfLen = 0;		/* buffer length = 0 */
	B_Hnd = 0;		/* buffer full handshake = false */
	B_In = B_Out = 0;	/* buffer in = out = 0 */
	OvrRun = 0;		/* OvrRun flag = false */
}


/*
 * AsynSend() - send one character
 */

void
AsynSend(char c,		/* character to send */
         int *error_code)	/* error reply: 0 = none
				 *		8 = time out error
				 *	       10 = com port not initialized */
{
	short i, j, k;
	time_t et;
/*
 * Check for errors: com port uninitialized.
 */
	if ( ! OpenSt) {
		*error_code = 10;
		return;
	}
/*
 * Wait TMO seconds for DSR and an empty transmit register.
 *
 * The DSR wait is optional.
 */
	for (et = (time_t)0;;) {
		if (DSRwait) {
			if ((inp(Pad + MSR) & 0x20) && (inp(Pad + LSR) & 0x20))
				break;
		} else if (inp(Pad + LSR) & 0x20)
			break;
		if (et == (time_t)0)
			et = time(NULL) + (time_t)TMO;
		else if (time(NULL) >= et) {
			*error_code = 8;
			return;
		}
		for (j = k = 0; k < 2; k++) {
			for (i = 1; i < 200; i++) {
				j += 1;
			}
		}
	}
/*
 * Send the character.
 */
	_disable();
	outp(Pad, c);
	_enable();
	*error_code = 0;
}


/*
 * AsynSndStr() - send string
 */

void
AsynSndStr(char *str,		/* string to send */
           int *error_code)	/* error code: 0 = none
				 *		8 = time out error
				 *	       10 = com port not initialized */
{
	int i, length;
/*
 * Send string, followed by a CR.
 */
	for (i = 0, length = strlen(str); i < length; i++) {
		AsynSend(str[i], error_code);
		if (*error_code)
			return;
	}
	AsynSend(CR, error_code);
}
