/*
 * comm.c - COM[12] interface
 *
 * V. Abell
 */

/*
 * Copyright 1994 Victor A. Abell, Lafayette, Indiana  47906.  All rights
 * reserved.
 *
 * Written by Victor A. Abell.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Victor A. Abell is not responsible for any consequences of the use of
 * this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to Victor A. Abell must
 *    appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#if	!defined(lint)

# if	defined(_BCC)
#pragma warn -use
# endif

static char copyright[] =
"@(#) Copyright 1994 Victor A. Abell.\nAll rights reserved.\n";
#endif

#include "touch2.h"

extern struct videoconfig Vc;

#define	BAUDLN		6
#define COL		20
#define DSRLN		10
#define EXITLN		20
#define	FRAMELN		12
#define	IRQLN		14
#define	PAINPLN		22
#define PARLN		16
#define	PORTADDRLN	4
#define PORTLN		8
#define	STSTLN		18

short Baud[] = { 9600, 2400, 1200, 300, 0 };

struct parity Par[] = {
	{ 0, "none" },
	{ 1, "even" },
	{ 2, "odd"  },
	{ 0, NULL },
};

short Baudx = 0;
short ComOpen = 0;
short DSRwait = 1;
short Frame = 8;
short Irq = 0;
short Parx = 0;
short Port = 0;
short PortAddr = 0;
short StSt = 1;


struct menu ComPar[] = {
	{  2, 20, "Communications parameters:" },
	{  0,  0, NULL },
};


static void DisBaud(void);
static void DisDSRwait(void);
static void DisFrame(void);
static void DisIRQ(void);
static void DisPar(void);
static void DisPort(void);
static void DisPortAddr(void);
static void DisStst(void);


/*
 * CloseCom() - close COM[12] port
 */

void
CloseCom()
{
	if (ComOpen) {
		AsynClose();
		ComOpen = 0;
	}
}


/*
 * DefCom() - define COM[12] parameters
 */

void
DefCom()
{
	char abuf[5], *cp;
	int ch, hd, hv, m;

	for (m = 0;;) {
		if (m == 0) {
			DispMenu(ComPar, NULL);
			DisBaud();
			DisDSRwait();
			DisFrame();
			DisIRQ();
			DisPar();
			DisPort();
			DisPortAddr();
			DisStst();
			gotoxy(COL, EXITLN);
			cputs("X - eXit");
			m = 1;
		}
		if ( ! kbhit()) {
			AsynRstBf();
			continue;
		}
		ch = getch();
		switch(ch) {

		case ESC:
		case 'x':
		case 'X':
			return;

		case 'a':
		case 'A':
			m = 0;
			if (GetInp(PAINPLN, 1, "Hexadecimal Port Address?",
				NULL, abuf, sizeof(abuf))
			== 0)
				break;
			for (cp = abuf, hd = hv = 0; *cp; cp++) {
				if ((hd = AsmHex(cp)) < 0)
					break;
				hv = (hv << 4) + hd;
			}
			if (hd < 0) {
				(void) WarnMsg(11, 20,
					"Please enter 1 to 4 hex characters.",
				0, 0, NULL, 0);
			} else
				PortAddr = hv;
			break;
		case 'b':
		case 'B':
			Baudx++;
			if (Baud[Baudx] == 0)
				Baudx = 0;
			DisBaud();
			break;

		case 'c':
		case 'C':
			if (++Port > 3)
				Port = 0;
			DisPort();
			if ( ! Irq)
				DisIRQ();
			if ( ! PortAddr)
				DisPortAddr();
			break;

		case 'd':
		case 'D':
			DSRwait = DSRwait ? 0 : 1;
			DisDSRwait();
			break;
		case 'f':
		case 'F':
			Frame = (Frame == 8) ? 7 : 8;
			DisFrame();
			break;

		case 'i':
		case 'I':
			if (Irq == 0 || Irq == 7)
				Irq = 3;
			else {
				Irq++;
				if (Irq == 6)
					Irq++;
			}
			DisIRQ();
			break;
		case 'p':
		case 'P':
			Parx++;
			if (Parx > 2)
				Parx = 0;
			DisPar();
			break;

		case 's':
		case 'S':
			StSt = (StSt == 1) ? 2 : 1;
			DisStst();
			break;

		case 0:
			ch = getch();
			/* fall through */
		default:
			putch(BELL);
		}
	}
}


/*
 * DisBaud() - display baud rate
 */

static void
DisBaud(void)
{
	ClearRow(BAUDLN, COL);
	gotoxy(COL, BAUDLN);
	cprintf("B - Baud rate: %d", Baud[Baudx]);
}


/*
 * DisDSRwait() - display wait for DSR on transmit status
 */

static void
DisDSRwait(void)
{
	ClearRow(DSRLN, COL);
	gotoxy(COL, DSRLN);
	cprintf("D - %s waiting for DSR (currently %s)",
		DSRwait ? "stop" : "start",
		DSRwait ? "waiting" : "not waiting");
}



/*
 * DisFrame() - display frame size
 */

static void
DisFrame(void)
{
	ClearRow(FRAMELN, COL);
	gotoxy(COL, FRAMELN);
	cprintf("F - Frame size: %d", Frame);
}


/*
 * DisIRQ() - display IRQ
 */

static void
DisIRQ(void)
{
	ClearRow(IRQLN, COL);
	gotoxy(COL, IRQLN);
	cprintf("I - IRQ: %d", (Irq != 0) ? Irq : (1 - (Port & 1) + 3));
}


/*
 * DisPar() - display parity
 */

static void
DisPar(void)
{
	ClearRow(PARLN, COL);
	gotoxy(COL, PARLN);
	cprintf("P - Parity: %s", Par[Parx].nm);
}


/*
 * DisPort() - display COM port
 */

static void
DisPort(void)
{
	ClearRow(PORTLN, COL);
	gotoxy(COL, PORTLN);
	cprintf("C - COM port: %d", Port + 1);
}


/*
 * DisPortAddr() - display port address
 */

static void
DisPortAddr(void)
{
	int pad;

	ClearRow(PORTADDRLN, COL);
	gotoxy(COL, PORTADDRLN);
	if (PortAddr)
		pad = PortAddr;
	else
		pad = peek(0x40, Port * 2);
	cputs("A - Port Address: ");
	if (pad)
		cprintf("0x%04x", pad);
	else
		cputs("(none)");
}


/*
 * DisStst() - display start/stop bit count
 */

static void
DisStst(void)
{
	ClearRow(STSTLN, COL);
	gotoxy(COL, STSTLN);
	cprintf("S - Start/Stop bits: %d", StSt);
}


/*
 * OpenCom() - open the COM[12] port
 */

void
OpenCom()
{
	int err;

	if (ComOpen)
		return;
	AsynOpen(Port, Irq, PortAddr, Baud[Baudx], Par[Parx].val, StSt,
		Frame, &err);
	if (err) {
		clrscr();
		gotoxy(25, 12);
		switch (err) {
		case -1:
			cputs("No space available for COM input buffer.");
			break;
		case 1:
			cprintf("COM%d is not available.", Port + 1);
			break;
		case 2:
			cprintf("%d baud is not supported.", Baud[Baudx]);
			break;
		case 3:
			cprintf("Can't set parity to \"%s\".", Par[Parx].nm);
			break;
		case 4:
			cprintf("Can't set start/stop bit size to %d.", StSt);
			break;
		case 5:
			cprintf("Can't set frame size to %d bits.", Frame);
			break;
		case 9:
			cprintf("Illegal IRQ number: %d", Irq);
			break;
		default:
			cprintf("Can't open COM%d port: error %d", Port+1,
				err);
		}
		PromptMsg("Press any key to exit.");
		(void) WaitAnyKey();
		TouchExit(1);
	}
	ComOpen = 1;
}


/*
 * WaitAnyKey() - wait for any key press
 */

int
WaitAnyKey()
{
	int ch;

	for (;;) {
		if (kbhit()) {
			if ((ch = getch()) == 0)
				ch = getch();
			return(ch);
		}
		AsynRstBf();
	}
}
