/*
 * conf.c - read configuration file
 *
 * V. Abell
 */

/*
 * Copyright 1994 Victor A. Abell, Lafayette, Indiana  47906.  All rights
 * reserved.
 *
 * Written by Victor A. Abell.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Victor A. Abell is not responsible for any consequences of the use of
 * this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to Victor A. Abell must
 *    appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#if	!defined(lint)

# if	defined(_BCC)
# pragma warn -use
# endif

static char copyright[] =
"@(#) Copyright 1994 Victor A. Abell.\nAll rights reserved.\n";
#endif

#include "touch2.h"
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

static int AsmPrStr(char **s, char *m, char *v);
static short FindClr(char *v);
static int SetBaudRate(char *b, char *v);
static int SetBkClr(char *b, char *v);
static int SetCksum(char *b, char *v);
static int SetComPort(char *b, char *v);
static int SetDSRwait(char *b, char *v);
static int SetDumpFmt(char *b, char *v);
static int SetEvent(char *b, char *v);
static int SetFrameSz(char *b, char *v);
static int SetGraphNm(char *b, char *v);
static int SetIgnore(char *b, char *v);
static int SetParity(char *b, char *v);
static int SetPcDisp(char *b, char *v);
static int SetPromptBkClr(char *b, char *v);
static int SetPromptTxtClr(char *b, char *v);
static int SetQDumpNm(char *b, char *v);
static int SetRDumpNm(char *b, char *v);
static int SetSSBits(char *b, char *v);
static int SetTmRange(char *b, char *v);
static int SetTtlIO(char *b, char *v);
static int SetTxtClr(char *b, char *v);
static int SetDispWarn(char *b, char *v);

/*
 * The Colors[] table is ordered by color index -- i.e., the color
 * index is implied from the position of an entry in the table.  So
 * don't change the order of the entries.
 */

char *Colors[] = {
	"black",		/* 0 */
	"blue",			/* 1 */
	"green",		/* 2 */
	"cyan",			/* 3 */
	"red",			/* 4 */
	"magenta",		/* 5 */
	"brown",		/* 6 */
	"white",		/* 7 */
	"gray", 		/* 8 */
	"light blue",		/* 9 */
	"light green",		/* 10 */
	"light cyan",		/* 11 */
	"light red",		/* 12 */
	"light magenta",	/* 13 */
	"yellow",		/* 14 */
	"bright white",		/* 15 */
	NULL
};

struct kcmd {
	char *nm;			/* keyword */
	int (*fun)();			/* processing function */
} Keywd [] = {
	{ "aftergraph",		SetAftGraph },		/* also -a */
	{ "ag",			SetAftGraph },		/* also -a */
	{ "bar",		SetBarChar },		/* also -b */
	{ "baudrate",		SetBaudRate },
	{ "br",			SetBaudRate },
	{ "background",		SetBkClr },
	{ "bk",			SetBkClr },
	{ "beforegraph",	SetBefGraph },		/* also -B */
	{ "b4g",		SetBefGraph },		/* also -B */
	{ "checksum",		SetCksum },		/* also -C */
	{ "cksum",		SetCksum },		/* also -C */
	{ "comport",		SetComPort },
	{ "DSRwait",		SetDSRwait },		/* also -D */
	{ "dump",		SetDumpFmt },		/* partially -q */
	{ "event",		SetEvent },
	{ "framesize",		SetFrameSz },
	{ "gmax",		SetGmaxLvl },		/* also -G */
	{ "gmin",		SetGminLvl },		/* also -g */
	{ "graphname",		SetGraphNm },
	{ "graphtitle",		SetGraphTtl },
	{ "gttl",		SetGraphTtl },
	{ "ignore",		SetIgnore },
	{ "intersection",	SetIntChar },		/* also -i */
	{ "ic",			SetIntChar },		/* also -i */
	{ "line",		SetLineChar },		/* also -l */
	{ "lpp",		SetLpp },		/* also -L */
	{ "pcd",		SetPcDisp },		/* also -P */
	{ "pgctdisp",		SetPcDisp },		/* also -P */
	{ "port",		SetComPort },
	{ "promptbk",		SetPromptBkClr },
	{ "pbk",		SetPromptBkClr },
	{ "prompttext",		SetPromptTxtClr },
	{ "ptxt",		SetPromptTxtClr },
	{ "qdumpname",		SetQDumpNm },
	{ "range",		SetTmRange },
	{ "rdumpname",		SetRDumpNm },
	{ "rng",		SetTmRange },
	{ "parity",		SetParity },
	{ "printer",		SetPrinter },		/* also -p */
	{ "startstop",		SetSSBits },
	{ "stst",		SetSSBits },
	{ "text",		SetTxtClr },
	{ "titlevar",		SetTtlVar },		/* also -v */
	{ "ttlIO",		SetTtlIO },
	{ "tv",			SetTtlVar },		/* also -v */
	{ "warn",		SetDispWarn, },
	{ NULL,			NULL },
};

char *AftGraph = NULL;			/* after graph printer control */
char *BefGraph = NULL;			/* before graph printer control */
char Gttl[GTTLLNL+1];			/* graph title line */
char GttlDumpf[GTTLLNL+1];		/* graph title line from dump file */
char *GttlOrig = NULL;			/* original graph title line */
short Lineval = 0;			/* line value definition status */
short PromptBkClrx = DEFPBCX;		/* prompt background color */
short PromptTxtClrx = DEFPTCX;		/* prompt text color */


/*
 * Asm1Ch() - assemble one character
 *
 * return = number of characters extracted from definition string
 *	    (-1 = error)
 */

int
Asm1Ch(s, r)
	char *s;			/* pointer to string that defines the
					 * character */
	char *r;			/* pointer to result */
{
	int b, i, n;
	register char *c;
	char tc;

	c = s;
	if (*c == '\\') {

	/*
	 * `\' is the prefix for special strings.
	 */
		c++;
		switch(*c) {

		case 'e':				/* \e = ESC */
		case 'E':
			*r = ESC;
			break;
		case 'f':				/* \f = FF */
		case 'F':
			*r =  '\f';
			break;
		case 'n':				/* \n = LF */
		case 'N':
			*r = '\n';
			break;
		case 'r':				/* \r = CR */
		case 'R':
			*r = '\r';
			break;
		case 't':				/* \t = TAB */
		case 'T':
			*r = '\t';
			break;
		case '^':				/* \^x = CTRL-x */
			tc = islower(*(++c)) ? toupper(*c) : *c;
			if (tc >= '@' && tc <= 'Z')
				*r = tc - '@';
			else
				return(0);
			break;
		case '\\':				/* \\ = \ */
			*r = '\\';
			break;
		default:
			if (*c == '0' && (*(c+1) == 't' || *(c+1) == 'o')) {

			/*
			 * \0tnnn = decimal character value
			 * \0onnn = octal character value
			 */
			    c++;
			    b = *c++ - 'o';
			    for (i = n = 0; i < 3; c++, i++) {
				if ( ! isascii(*c))
				    return(0);
				if (b) {
				    if (*c >= '0' && *c <= '9')
					n = (n * 10) + *c - '0';
				    else
					return(0);
				} else {
				    if (*c >= '0' && *c <= '7')
					n = (n << 3) + *c - '0';
				    else
					return(0);
				}
			    }
			    *r = n;
			    return(c - s);
			} else if (isxdigit(*c)) {

			/*
			 * \[0-9a-f][0-9a-f] = hexadecimal character value
			 */
			    if ((n = AsmHex(c)) == -1)
				return(0);
			    *r = n << 4;
			    if ((n = AsmHex(c+1)) == -1)
				return(0);
			    *r += n;
			    return(c + 2 - s);
			} else
			    return(0);
		}
		return(c + 1 - s);
	}
/*
 * Without a leading `\', characters represent themselves.
 */
	*r = *c;
	return(1);
}


/*
 * AsmHex() - assemble one hex character
 */

int
AsmHex(c)
	char *c;			/* character pointer */
{
	char tc;

	if (*c >= '0' && *c <= '9')
		return((int)*c - (int)'0');
	if (isascii(*c) && isalpha(*c)) {
		tc = isupper(*c) ? tolower(*c) : *c;
		if (tc >= 'a' && tc <= 'f')
			return(10 + tc - 'a');
	}
	return(-1);
}


/*
 * AsmPrStr() - assemble printer string
 */

static int
AsmPrStr(s, m, v)
	char **s;			/* string destination pointer */
	char *m;			/* error message value */
	char *v;			/* value to assemble */
{
	char bf[PRTRCTLLN+1];
	size_t l;

	if (CvtPrtrStr(v, bf, sizeof(bf)) == -1) {
		(void) fprintf(stderr, "%s: can't convert \"%s\" to %s string.",
			Pn, v, m);
		return(0);
	}
	if (*s) {
		(void) free(*s);
		*s = NULL;
	}
	l = strlen(v) + 1;
	if ((*s = (char *)malloc(l)) == NULL) {
		(void) fprintf(stderr, "%s: no space for %s.\n", Pn, v);
		return(0);
	}
	(void) strcpy(*s, v);
	return(1);
}


/*
 * Atof() - convert digits to floating point, recognizing either `.'
 *	    or `,' as the fractional separator
 */

int
Atof(s, d, m, t)
	char *s;			/* string to convert */
	double *d;			/* converted value */
	char *m;			/* conversion error message */
	char *t;			/* terminating character */
{
	char *cp;
	double dig, rv, sc;
	short f;

	for (cp = s, f = 0, rv = 0.0; *cp; cp++) {
		if (*cp == ' ')
			continue;
		if (*cp == '.' || *cp == ',') {
			if (f) {

Atof_error:
			    if (m)
				(void) fprintf(stderr,
				    "%s: can't convert %s: %s\n", Pn, m, s);
			    *d = rv;
			    if (t)
			        *t = *cp;
			    return(0);
			}
			f = 1;
			sc = 0.1;
			continue;
		}
		if (*cp < '0' || *cp > '9')
			goto Atof_error;
		dig = (double)(*cp - '0');
		if (f) {
			rv += dig * sc;
			sc *= 0.1;
		} else
			rv = (rv * 10.0) + dig;
	}
	*d = rv;
	if (t)
		*t = *cp;
	return(1);
}


/*
 * CvtPrtrStr() - convert spacing string
 */

int
CvtPrtrStr(s, r, n)
	char *s;			/* string to convert */
	char *r;			/* result buffer */	
	int n;				/* result buffer length */
{
	int i, l;

	for (i = 0; *s && i < (n - 1); i++) {
		if ((l = Asm1Ch(s, r + i)) < 1)
			return(-1);
		s += l;
	}
	if (*s)
		return(-1);
	r[i] = '\0';
	return(i);
}


/*
 * EntPrtStr() -- enter printer control strings
 */

int
EntPrtStr(a, b)
	char *a;			/* after graph control string */
	char *b;			/* before graph control string */
{
	size_t len;

	if (AftGraph) {	
		(void) free(AftGraph);
		AftGraph = NULL;
	}
	if (BefGraph) {
		(void) free(BefGraph);
		BefGraph = NULL;
	}
	len = strlen(a);
	if ((AftGraph = (char *)malloc(len + 1)) == NULL) {
		(void) fprintf(stderr,
			"%s: no space for after graph control string: %s\n",
			Pn, a);
		return(0);
	}
	(void) strcpy(AftGraph, a);
	len = strlen(b);
	if ((BefGraph = (char *)malloc(len + 1)) == NULL) {
		(void) fprintf(stderr,
			"%s: no space for before graph control string: %s\n",
			Pn, b);
		return(0);
	}
	(void) strcpy(BefGraph, b);
	return(1);
}


/*
 * FindClr() - find color
 */

static short
FindClr(v)
	char *v;			/* search value */
{
	short i;

	for (i = 0; Colors[i]; i++) {
		if (strcmpi(Colors[i], v) == 0)
			return(i);
		if (strnicmp(v, "bright", 6) == 0
		&&  v[6] != ' '
		&&  strnicmp(Colors[i], "bright ", 7) == 0
		&&  strcmpi(&Colors[i][7], &v[6]) == 0)
			return(i);
		if (strnicmp(v, "light", 5) == 0
		&&  v[5] != ' '
		&&  strnicmp(Colors[i], "light ", 6) == 0
		&&  strcmpi(&Colors[i][6], &v[5]) == 0)
			return(i);
	}
	return(-1);
}


/*
 * ReadConf() - read configuration file
 */

int
ReadConf()
{
	char buf[128], *cf, *tp, *vp;
	FILE *cfs;
	short e = 1;
	short f, len;
	struct kcmd *kp, *mp;
	short rv = 1;

	for (f = 0; f < 10; f++)
		TtlVar[f] = NULL;
	if ((cf = getenv("TOUCH2")) == NULL) {
		e = 0;
		cf = "touch2.cf";
	}
	if ((cfs = fopen(cf, "rt")) == NULL) {
		if ( ! e)
			return(1);
		(void) fprintf(stderr,
			"%s: can't open configuration file \"%s\"\n", Pn, cf);
		return(0);
	}
	while (fgets(buf, sizeof(buf), cfs)) {
		if (*buf == '#' || *buf == '\n')
			continue;
		if ((tp = strchr(buf, '\n')) != NULL)
			*tp = '\0';
		if ((vp = strchr(buf, '=')) == NULL) {
			(void) fprintf(stderr,
				"%s: configuration line lacks `=': %s\n",
				Pn, buf);
			rv = 0;
			continue;
		}
		for (tp = vp - 1; tp > buf; tp--) {
			if (*tp != ' ') 
				break;
		}
		len = tp + 1 - buf;
		for (vp++; *vp; vp++) {
			if (*vp != ' ');
				break;
		}
		for (f = 0, kp = Keywd; kp->nm; kp++) {
			if (strnicmp(buf, kp->nm, len) == 0) {
				f++;
				mp = kp;
			}
		}
		if (f == 1) {

#if	defined(_BCC)
#pragma warn -pro
#endif

			if ((*mp->fun)(buf, vp) == 0)

#if	defined(_BCC)
#pragma warn +pro
#endif

				rv = 0;
		} else {
			(void) fprintf(stderr,
				"%s: %s keyword in configuraton line: %s\n",
				Pn, (!f) ? "Unknown" : "Ambiguous", buf);
			rv = 0;
		}
	}
	(void) fclose(cfs);
	if ( ! TestGlev())
		rv = 0;
	return(rv);
}


/*
 * SetAftGraph() - set after graph printer control string
 */

#if	defined(_BCC)
#pragma argsused
#endif

int
SetAftGraph(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	return(AsmPrStr(&AftGraph, "after graph printer control string", v));
}


/*
 * SetBarChar() - set graph's bar character
 */

int
SetBarChar(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	int i;

	i = Asm1Ch(v, BarCh);
	if (i < 1 || v[i] != '\0') {
		(void) fprintf(stderr, "%s: bad bar character definition: %s\n",
			Pn, b);
		return(0);
	}
	return(1);
}


/*
 * SetBaudRate() - process "baudrate" line
 */

static int
SetBaudRate(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short br, bx;

	br = (short)atoi(v);
	for (bx = 0; Baud[bx]; bx++) {
		if (br == Baud[bx]) {
			Baudx = bx;
			return(1);
		}
	}
	(void) fprintf(stderr, "%s: unknown baud rate: %s\n", Pn, b);
	return(0);
}


/*
 * SetBefGraph() - set before graph printer control string
 */

#if	defined(_BCC)
#pragma argsused
#endif

int
SetBefGraph(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	return(AsmPrStr(&BefGraph, "before graph printer control string", v));
}


/*
 * SetBkClr() - set background color
 */

static int
SetBkClr(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short i;

	if ((i = FindClr(v)) < 0) {
		(void) fprintf(stderr, "%s: unknown background color: %s\n",
			Pn, b);
		return(0);
	}
	BkClrx = i;
	return(1);
}


/*
 * SetCksum() - set checksum status
 */

static int
SetCksum(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strcmpi(v, "on") == 0) {
		Cksum = 1;
		return(1);
	} else if (strcmpi(v, "off") == 0) {
		Cksum = 0;
		return(1);
	}
	(void) fprintf(stderr, "%s: unknown checksum status: %s\n", Pn, b);
	return(0);
}


/*
 * SetComPort() - set COM port number ([12])
 */

static int
SetComPort(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short hd, hv, nd;

	if (*v < '1' || *v > '4') {
		(void) fprintf(stderr, "%s: illegal port number: %s\n", Pn, b);
		return(0);
	}
	Port = (short)(*v - '1');
	v++;
	while (*v && *v == ' ')
		v++;
	if (*v == '\0')
		return(1);
	if (*v++ != ',') {

port_syntax_err:
		(void) fprintf(stderr, "%s: port command syntax error: %s\n",
			Pn, b);
		return(0);
	}
	while (*v && *v == ' ')
		v++;
	if (strnicmp(v, "irq", 3) == 0) {

	/*
	 * Assemble IRQ number; it must be 3, 4, 5, or 7.
	 */
		v += 3;
		if (*v != '3' && *v != '4' && *v != '5' && *v != '7') {
			(void) fprintf(stderr,
				"%s: IRQ not 3, 4, 5, or 7: %s\n", Pn, b);
			return(0);
		}
		Irq = (short)(*v++ - '0');
		while (*v && *v == ' ')
			v++;
		if (*v == '\0')
			return(1);
		if (*v++ != ',')
			goto port_syntax_err;
	}
/*
 * Assemble port address.
 */
	while (*v && *v == ' ')
		v++;
	for (hd = hv = nd = 0; *v; nd++, v++) {
		if (*v == ' ' || *v == '#' || *v == '\t')
			break;
		if ((hd = AsmHex(v)) < 0)
			break;
		hv = (hv << 4) + hd;
	}
	if (nd > 4 || hd < 0) {
		(void) fprintf(stderr, "%s: illegal port address: %s\n", Pn, b);
		return(0);
	}
	if (*v != '\0' && *v != ' ' && *v != '#' && *v != '\t')
		goto port_syntax_err;
	PortAddr = hv;
	return(1);
}


/*
 * SetDSRwait() - set wait for DSR on transmit status
 */

static int
SetDSRwait(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strcmpi(v, "on") == 0) {
		DSRwait = 1;
		return(1);
	} else if (strcmpi(v, "off") == 0) {
		DSRwait = 0;
		return(1);
	}
	(void) fprintf(stderr, "%s: unknown transmi DSR wait status: %s\n",
		Pn, b);
	return(0);
}


/*
 * SetDispWarn() - set display warning status
 */

static int
SetDispWarn(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strcmpi(v, "on") == 0) {
		Warn = 1;
		return(1);
	} else if (strcmpi(v, "off") == 0) {
		Warn = 0;
		return(1);
	}
	(void) fprintf(stderr, "%s: unknown display warning status: %s\n",
		Pn, b);
	return(0);
}


/*
 * SetDumpFmt() - set dump format, raw or QuattroPro
 */

int
SetDumpFmt(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	switch (*v) {

	case 'q':
	case 'Q':
		Qp = 1;
		break;
	case 'r':
	case 'R':
		Qp = 0;
		break;
	default:
		(void) fprintf(stderr, "%s: unknown dump type: %s\n", Pn, b);
		return(0);
	}
	return(1);
}


/*
 * SetEvent() - set event filter titles
 */

static int
SetEvent(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	char *cp;
	int ev;

	ev = *v++ - '1';
	if (ev < 0 || ev > 8) {
		(void) fprintf(stderr, "%s: illegal event number: %s\n",
			Pn, b);
		return(0);
	}
	if (*v && *v != ',') {
		(void) fprintf(stderr, "%s: format error: %s\n", Pn, b);
		return(0);
	}
	if ((cp = (char *)malloc(strlen(++v) + 1)) == NULL) {
		(void) fprintf(stderr, "%s: no space for %s.\n", Pn, v);
		return(0);
	}
	(void) strcpy(cp, v);
	EvtBuf[ev] = cp;
	return(1);
}


/*
 * SetPcDisp() - set page count display status
 */

static int
SetPcDisp(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strcmpi(v, "on") == 0) {
		PcDisp = 1;
		return(1);
	} else if (strcmpi(v, "off") == 0) {
		PcDisp = 0;
		return(1);
	}
	(void) fprintf(stderr, "%s: unknown page count status: %s\n", Pn, b);
	return(0);
}


/*
 * SetQDumpNm() - set QuattroPro dump file name
 */

static int
SetQDumpNm(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strlen(v) > DUMPFNL-1) {
		(void) fprintf(stderr,
			"%s: QuattroPro dump file name too long: %s\n",
			Pn, b);
		return(0);
	}
	(void) strcpy(DumpfnQ, v);
	return(1);
}


/*
 * SetRDumpNm() - set raw dump file name
 */

static int
SetRDumpNm(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strlen(v) > DUMPFNL-1) {
		(void) fprintf(stderr,
			"%s: raw dump file name too long: %s\n",
			Pn, b);
		return(0);
	}
	(void) strcpy(DumpfnR, v);
	return(1);
}


/*
 * SetFrameSz() - set frame size ([78] bits)
 */

static int
SetFrameSz(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short f;

	f = (short) atoi(v);
	if (f == 7 || f == 8) {
		Frame = f;
		return(1);
	}
	(void) fprintf(stderr, "%s: illegal frame size: %s\n", Pn, b);
	return(0);
}


/*
 * SetGmaxLvl() - set graph's maximum level
 */

#if	defined(_BCC)
#pragma argsused
#endif

int
SetGmaxLvl(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (Atof(v, &Gmaxl, "graph's maximum level", NULL)) {
		Lineval = 1;
		return(1);
	}
	return(0);
}

/*
 * SetGminLvl() - set graph's minimum level
 */

#if	defined(_BCC)
#pragma argsused
#endif

int
SetGminLvl(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (Atof(v, &Gminl, "graph's minimum level", NULL)) {
		Lineval = 1;
		return(1);
	}
	return(0);
}


/*
 * SetGraphNm() - set graph file name
 */

static int
SetGraphNm(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strlen(v) > DUMPFNL-1) {
		(void) fprintf(stderr, "%s: graph file name too long: %s\n",
			Pn, b);
		return(0);
	}
	(void) strcpy(Graphfn, v);
	return(1);
}


/*
 * SetGraphTtl() - set graph title line
 */

int
SetGraphTtl(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	int len;

	if ((len = strlen(v)) > GTTLLNL) {
		(void) fprintf(stderr, "%s: graph title line too long: %s\n",
			Pn, b);
		return(0);
	}
	(void) strcpy(Gttl, v);
	if (len > 0) {
		if ((GttlOrig = (char *)malloc(len + 1)) == NULL) {
			(void) fprintf(stderr, "%s: no space for title: %s\n",
				Pn, b);
			return(0);
		}
		(void) strcpy(GttlOrig, v);
	}
	return(1);
}


/*
 * SetIgnore() - set ignore event list
 */

static int
SetIgnore(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short n;

	for (n = 0; *v; v++) {
		if (*v == ' ' || *v == ',')
			continue;
		if (*v < '0' || *v > '9') {
			(void) fprintf(stderr,
				"%s: ignore-event number not [0 - 9]: %s\n",
				Pn, b);
			return(0);
		}
		IgnEvt[*v - '0'] = 1;
		n++;
	}
	if ( ! n) {
		(void) fprintf(stderr,
			"%s: no ignore-event numbers: %s\n", Pn, b);
		return(0);
	}
	for (n = 0; n < 10; n++)
		IgnEvtOrig[n] = IgnEvt[n];
	return(1);
}


/*
 * SetIntChar() - set graph's bar and max/min line intersection character
 */

int
SetIntChar(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	int i;

	i = Asm1Ch(v, IntCh);
	if (i < 1 || v[i] != '\0') {
		(void) fprintf(stderr,
			"%s: bad bar and max/min line intersection character definition: %s\n",
			Pn, b);
		return(0);
	}
	return(1);
}


/*
 * SetLineChar() - set graph's max/min line character
 */

int
SetLineChar(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	int i;

	i = Asm1Ch(v, LineCh);
	if (i < 1 || v[i] != '\0') {
		(void) fprintf(stderr,
			"%s: bad max/min line character definition: %s\n",
			Pn, b);
		return(0);
	}
	return(1);
}


/*
 * SetLpp() - set graph lines per page
 */

#if	defined(_BCC)
#pragma argsused
#endif

int
SetLpp(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	int t;

	if ((t = atoi(v)) <= HDRLPP) {
		(void) fprintf(stderr,
			"%s: lines per page too small (minimum = %d)\n",
			Pn, HDRLPP+1);
		return(0);
	}
	Lpp = t;
	return(1);
}


/*
 * SetPrinter() - set printer type
 */

int
SetPrinter(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strcmpi(v, "epson") == 0)
		return(EntPrtStr(AFTEPSON, BEFEPSON));
	else if (strcmpi(v, "pcl4") == 0)
		return(EntPrtStr(AFTPCL4, BEFPCL4));
	else if (strcmpi(v, "ppds") == 0)
		return(EntPrtStr(AFTPPDS, BEFPPDS));
	(void) fprintf(stderr, "%s: unidentified printer: %s\n", Pn, b);
	return(0);
}


/*
 * SetParity() - set parity ({none,even,odd})
 */

static int
SetParity(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short px;

	for (px = 0; px < 3; px++) {
		if (strcmpi(v, Par[px].nm) == 0) {
			Parx = px;
			return(1);
		}
	}
	(void) fprintf(stderr, "%s: unknown parity: %s\n", Pn, b);
	return(0);
}


/*
 * SetPromptBk() - set prompt background color
 */

static int
SetPromptBkClr(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short i;

	if ((i = FindClr(v)) < 0) {
		(void) fprintf(stderr,
			"%s: unknown prompt background color: %s\n", Pn, b);
		return(0);
	}
	if (i > 7) {
		(void) fprintf(stderr,
			"%s: illegal prompt background color: %s\n", Pn, b);
		return(0);
	}
	PromptBkClrx = i;
	return(1);
}


/*
 * SetPromptTxtClr() - set prompt text color
 */

static int
SetPromptTxtClr(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short i;

	if ((i = FindClr(v)) < 0) {
		(void) fprintf(stderr, "%s: unknown prompt text color: %s\n",
			Pn, b);
		return(0);
	}
	PromptTxtClrx = i;
	return(1);
}


/*
 * SetSSBits() - set start/stop bits ([12])
 */

static int
SetSSBits(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short s;

	s = (short) atoi(v);
	if (s == 1 || s == 2) {
		StSt = s;
		return(1);
	}
	(void) fprintf(stderr, "%s: illegal start/stop bit count: %s\n",
		Pn, b);
	return(0);
}


/*
 * SetTtlIO() - set title I/O state
 */

int
SetTtlIO(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	switch (*v) {
	case 'n':
	case 'N':
		SupTtlIO = 1;
		break;
	case 'y':
	case 'Y':
		SupTtlIO = 0;
		break;
	default:
		(void) fprintf(stderr, "%s: bad title I/O state: %s\n",
			Pn, b);
		return(0);
	}
	return(1);
}


/*
 * SetTtlVar() - set title variable value
 *
 * titleval=n,<value>
 */

int
SetTtlVar(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short i;

	if (*v < '0' || *v > '9' || *(v+1) != ',') {
		(void) fprintf(stderr, "%s: improper variable index: %s\n",
			Pn, b);
		return(0);
	}
	i = (short)(*v - '0');
	if (TtlVar[i])
		(void) free(TtlVar[i]);
	if ((TtlVar[i] = (char *)malloc(strlen(v+2) + 1)) == NULL) {
		(void) fprintf(stderr,
			"%s: no space for title variable %d: %s\n",
			Pn, i, v+2);
		return(0);
	}
	(void) strcpy(TtlVar[i], v+2);
	return(1);
}


/*
 * SetTmRange() - set time range
 */

#if	defined(_BCC)
#pragma argsused
#endif

static int
SetTmRange(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	char *cp, *cp1;
	short bt, et;
	char buf[128];
	char msg[128];

	if (NTmRange >= MAXTMRANGE) {
		(void) fprintf(stderr,
			"%s: maximum (%d) time ranges already defined: %s\n",
			Pn, b);
		return(0);
	}
	(void) strcpy(buf, v);
	if ((cp = strchr(buf, ',')) != NULL)
		*cp++ = '\0';
	if ( ! CvtTmRange(buf, msg, &bt, &et)) {
		(void) fprintf(stderr, "%s: %s: %s\n", Pn, msg, b);
		return(0);
	}
	TmRange[NTmRange].bt = bt;
	TmRange[NTmRange].et = et;
	if ((cp1 = (char *)malloc(strlen(buf) + 1)) == NULL) {
		(void) fprintf(stderr, "%s: no space for %s.\n", Pn, buf);
		return(0);
	}
	TmRange[NTmRange].tm = cp1;
	(void) strcpy(cp1, buf);
	if (cp == NULL) {
		TmRange[NTmRange++].ttl = NULL;
		return(1);
	}
	if ((cp1 = (char *)malloc(strlen(cp) + 1)) == NULL) {
		(void) fprintf(stderr, "%s: no space for %s.\n", Pn, cp);
		return(0);
	}
	TmRange[NTmRange++].ttl = cp1;
	(void) strcpy(cp1, cp);
	return(1);
}


/*
 * SetTxtClr() - set text color
 */

static int
SetTxtClr(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short i;

	if ((i = FindClr(v)) < 0) {
		(void) fprintf(stderr, "%s: unknown text color: %s\n", Pn, b);
		return(0);
	}
	TxtClrx = i;
	return(1);
}


/*
 * TestGlev() - test graph level line values
 */

int
TestGlev()

{
	if (Gminl && Gmaxl && Gminl >= Gmaxl) {
		(void) fprintf(stderr,
			"%s: graph minimum level (%d) >= maximum (%d)\n",
			Pn, Gminl, Gmaxl);
		return(0);
	}
	return(1);
}
