/*
 * dttm.c - manage date and time
 *
 * V. Abell
 */

/*
 * Copyright 1994 Victor A. Abell, Lafayette, Indiana  47906.  All rights
 * reserved.
 *
 * Written by Victor A. Abell.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Victor A. Abell is not responsible for any consequences of the use of
 * this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to Victor A. Abell must
 *    appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#if	!defined(lint)

# if	defined(_BCC)
#pragma warn -use
# endif

static char copyright[] =
"@(#) Copyright 1994 Victor A. Abell.\nAll rights reserved.\n";
#endif

#include "touch2.h"
#include <ctype.h>


struct menu MainDT[] = {
	{  2, 21, "Read and set the meter's date and time." },
	{  9, 21, "R - Read date and time" },
	{ 11, 21, "S - Set date and time" },
	{ 13, 21, "X - eXit" },
#define	PROMPTLN	15
#define	PROMPTCOL	11
	{  0,  0, NULL },
};

struct menu ReadWt[] = {
	{ 12, 26, "Waiting to read date and time" },
	{  0,  0, NULL },
};

struct menu DateErr[] = {
	{ 12, 33, "Date read/set error" },
	{  0,  0, NULL },
};

static char Instr[64];

struct menu SelEventMenu[] = {
	{  2,  6, "N - define and activate a new event filter" },
	{  4,  6, Instr },
	{  0,  0, NULL },
};

struct menu SelIgnMenu[] = {
#define	IGNROW	6
#define	IGNCOL	20
	{  8, 20, "E - Erase ignored event numbers" },
	{ 10, 20, "I - define events to be Ignored" },
	{ 12, 20, "X = eXit" },
#define	IGNIROW	14
	{  0,  0, NULL },
};

struct menu SelRange[] = {
	{  2,  6, "N - define or redefine time range 1" },
	{  4,  6, Instr },
	{  0,  0, NULL },
};

short month_days[] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

short NTmRange = 0;			/* number of ranges in TmRange[] */
short RangeInUse = 0;			/* index + 1 of TmRange[] that is
					 * in use (0 = none) */
short SkipAft = 0;			/* skip after in effect */
char SkipAftDt[SKIPDTLN];		/* skip after date */
char SkipAftTm[SKIPTMLN];		/* skip after time */
short SkipUnt = 0;			/* skip until in effect */
char SkipUntDt[SKIPDTLN];		/* skip until date */
char SkipUntTm[SKIPTMLN];		/* skip until time */
struct range TmRange[MAXTMRANGE];	/* time ranges */


#if	defined(UNIX)
static char *AsmNum(char *p, int *n);
#else
static char *AsmNum(char *p, short *n);
#endif

static void ReadDtTm(void);
static void SetDtTm(void);
static int SkipDtTm(char *ttl, char *dt, char *tm);


/*
 * AsmNum() - assemble number
 */

static char *
AsmNum(p, n)
	char *p;			/* pointer to number string */

#if	defined(UNIX)
	int *n;				/* pointer to result */
#else
	short *n;			/* pointer to result */
#endif

{
	for (*n = 0; *p; p++) {
		if ( ! isdigit(*p))
			break;
		*n = (*n * 10) + *p - '0';
	}
	return(p);
}


/*
 * CommDisp() - display communications input
 */

void
CommDisp(ch)
	char *ch;		/* character to display
				 * NULL = turn display on or off */
{
	static short state = 0;
	static short col = 1;
	static short row = 1;

	if (ch == NULL || ! state) {
		if (state)
			state = 0;
		else {
			col = row = state = 1;
			ClearRow(1,1);
		}
		if (ch == NULL)
			return;
	}
	if (*ch >= ' ' && *ch <= '~') {
		if (col > Vc.screenwidth) {
			col = 1;
			row++;
			if (row >= Vc.screenheight)
				row = 1;
			ClearRow(row, 1);
		}
		gotoxy(col++, row);
		putch(*ch);
		return;
	}
	if (*ch < ' ') {
		if ((col + 1) > Vc.screenwidth) {
			col = 1;
			row++;
			if (row >= Vc.screenheight)
				row = 1;
			ClearRow(row, 1);
		}
		gotoxy(col, row);
		cprintf("^%c", *ch + '@');
		col += 2;
		return;
	}
	if ((col + 3) > Vc.screenwidth) {
		col = 1;
		row++;
		if (row >= Vc.screenheight)
			row = 1;
		ClearRow(row, 1);
	}
	gotoxy(col, row);
	cprintf("0x%02x", (int) *ch);
	col += 4;
}


/*
 * CvtTm() - convert time
 */

int
CvtTm(char **tm, char *msg)
{
	register char *cp;
	int hr, min, nd;

	for (cp = *tm, hr = min = nd = 0; nd < 3 && *cp; cp++) {
		if (*cp == ' ')
			continue;
		if (*cp >= '0' && *cp <= '9') {
			hr = (hr * 10) + (int)(*cp - '0');
			nd++;
		} else
			break;
	}
	if (nd < 1 || nd > 2 || hr > 24) {

bad_hour:
		(void) strcpy(msg, "has a bad hour value.");
		return(-1);
	}
	if (*cp == ':') {
		for (++cp, nd = 0; nd < 3 && *cp; cp++) {
		    if (*cp == ' ')
			continue;
		    if (*cp >= '0' && *cp <= '9') {
			min = (min * 10) + (int)(*cp - '0');
			nd++;
		    } else
			break;
		}
		if (nd != 2 || min > 59) {
		    (void) strcpy(msg, "has a bad minute value.");
		    return(-1);
		}
		if (*cp == ':') {
		    cp++;
		/*
		 * Skip seconds and trailing space.
		 */
		    while (cp && (*cp == ' ' || (*cp >= '0' && *cp <= '9')))
			cp++;
		}
	} else {
		while (*cp && *cp == ' ')
		    cp++;
	}
/*
 * Handle AM suffix.
 */
	if (*cp == 'a' || *cp == 'A') {
		if (hr > 12)
		    goto bad_hour;
		if (hr == 12)
		    hr = 0;
		cp++;
		if (*cp == '.')
		    cp++;
		if (*cp == 'm' || *cp == 'M')
		    cp++;
		else {

bad_AM_PM:

		    (void) strcpy(msg, "has a bad AM/PM indicator.");
		    return(-1);
		}
		if (*cp == '.')
		    cp++;
/*
 * Handle PM suffix.
 */
	} else if (*cp == 'p' || *cp == 'P') {
		if (hr > 12)
		    goto bad_hour;
		if (hr < 12)
		    hr += 12;
		cp++;
		if (*cp == '.')
		    cp++;
		if (*cp == 'm' || *cp == 'M')
		    cp++;
		else
		    goto bad_AM_PM;
		if (*cp == '.')
		    cp++;
	}
	while (*cp && *cp == ' ')
		cp++;
	*tm = cp;
	return((hr * 60) + min);
}


/*
 * CvtTmRange() - convert time range
 */

int
CvtTmRange(tm, msg, bt, et)
	char *tm;			/* time range characters */
	char *msg;			/* error message buffer */
	short *bt;			/* computed start time */
	short *et;			/* computed end time */
{
	char *cp = tm;
	int t;

	(void) strcpy(msg, "The starting hour ");
	if ((t = CvtTm(&cp, &msg[strlen(msg)])) < 0)
		return(0);
	if (*cp != '-') {
		(void) strcpy(msg,
			"The times aren't separated by a minus sign.");
		return(0);
	}
	*bt = (short)t;
	cp++;
	(void) strcpy(msg, "The ending hour ");
	if ((t = CvtTm(&cp, &msg[strlen(msg)])) < 0)
		return(0);
	if (*cp) {
		(void) strcpy(msg,
			"The time range ends with unrecognized characters.");
		return(0);
	}
	*et = (short)t;
	if (*bt == *et) {
		(void) strcpy(msg,
			"The ending time can't equal the starting time.");
		return(0);
	}
	return(1);
}



/*
 * DateTime() - read and set date and time
 */

void
DateTime()
{
	int ch;

	DispMenu(MainDT, NULL);
	for (;;) {
		if ( !kbhit()) {
			AsynRstBf();
			continue;
		}
		if ((ch = getch()) == 0)
			ch = getch();
		switch (ch) {

		case ESC:
		case 'x':
		case 'X':
			return;

		case 'r':
		case 'R':
			ReadDtTm();
			break;

		case 's':
		case 'S':
			SetDtTm();
			break;

		default:
			putch(BELL);
		}
		DispMenu(MainDT, NULL);
	}
}


/*
 * ReadDtTm() - read the date and time
 */

static void
ReadDtTm()
{
	char b[64], *cp;
	int i;

	for (;;) {
		if (WaitRdy() == 0)
			return;
		DispMenu(ReadWt, NULL);
		if (WaitCmd("DMF", 'F'))
			break;
		DispMenu(DateErr,
			"Press ESC to exit; any other key to retry.");
		if ((char)WaitAnyKey() == ESC)
			return;
	}
	GetDataLn(DumpLine, DUMPLL);
	cp = DumpLine;
	if (*cp == 'F') {
		cp++;
		if (*cp == ' ')
			cp++;
	}
	for (i = 0; i < sizeof(b) - 1; cp++) {
		if (*cp == '"')
			continue;
		b[i++] = *cp;
	}
	b[i] = 0;
	ClearRow(12, 1);
	gotoxy(22, 12);
	(void) cputs(b);
	PromptMsg("Press any key to exit.");
	(void) WaitAnyKey();
}


/*
 * SelEvent() - select event filter (and its title)
 */

void
SelEvent()
{
	char buf[GTTLLNL+2+1], ch, ev[2], ttl[GTTLLNL+2+1];
	short ex, i, j, m;

	for (ex = 0; !ex;) {
		for (i = 0; i < 9; i++) {
			if (EvtBuf[i] != NULL)
				break;
		}
		if (i < 9) {
		    (void) strcpy(Instr,
			"Filter by one of these events by typing its number.");
		    j = 6;
		} else {
		    Instr[0] = '\0';
		    j = 4;
		}
		DispMenu(SelEventMenu, NULL);
		for (i = 0; i < 9; i++) {
			if (EvtBuf[i] == NULL)
				continue;
			ClearRow(j, 1);
			gotoxy(6, j++);
			cprintf("%c %d. %s",
			    ((Eventfilt - '1') == i) ? '>' : ' ',
			    i + 1, EvtBuf[i]);
		}
		if (j > 4)
			j++;
		if (Eventfilt) {
			ClearRow(j, 1);
			gotoxy(6, j);
			cprintf("S - stop using event %c", Eventfilt);
			j += 2;
		}
		gotoxy(6, j);
		cputs("X - eXit");
		switch((ch = (char)WaitAnyKey())) {
		case 'x':
		case 'X':
		case ESC:
			ex = 1;
			break;
		case 's':
		case 'S':
			if (Eventfilt) {
				Eventfilt = '\0';
				(void) strcpy(Gttl, GttlOrig);
			} else
				putch(BELL);
			break;
		case 'n':
		case 'N':
			for (m = 0; !m;) {
			    if (GetInp(20, 5, "Event number?", NULL, ev, 2)
			    == 0) {
				m = 1;
				break;
			    }
			    if ((i = ev[0] - '1') >= 0 && i < 9) {
				m = 2;
				break;
			    }
			    if ((char)WarnMsg(12,
				(short)((Vc.screenwidth - 36)/2 + 1),
				"An event number must be 1 through 9.",
				0, 0, NULL, 1)
			     == ESC) {
				m = 1;
				break;
			    }
			}
			if (m == 1)
			    break;
			if (GetInp(20, 5, "Title?", Gttl, ttl, sizeof(ttl))
			== 0)
			    ttl[0] = '\0';
			if (EvtBuf[i])
			    (void) free((void *)EvtBuf[i]);
			if ((EvtBuf[i] = (char *)malloc(strlen(ttl) + 1))
			== NULL) {
			    (void) fprintf(stderr,
				"%s: no space for title: %s\n",
				Pn, ttl);
			    exit(1);
			}
			(void) strcpy(EvtBuf[i], ttl);
			ch = i + '1';
			/* fall through */
		default:
			i = (short)(ch - '1');
			if (i < 0 || i > 8 || EvtBuf[i] == NULL) {
			    putch(BELL);
			    break;
			}
			(void) strcpy(Gttl, EvtBuf[i]);
			Eventfilt = ch;
			ex = 1;
		}
	}
}


/*
 * SelIgnEvt() - select ignored events
 */

void
SelIgnEvt()
{
	char buf[20], *cp;
	short i, j, m;

	for (m = 0; m == 0;) {
		DispMenu(SelIgnMenu, NULL);
		gotoxy(IGNCOL, IGNROW);
		for (i = j = 0; i < 10; i++) {
			if (IgnEvt[i] == 0)
				continue;
			if (j++ == 0)
				cprintf("Events being ignored: %d", i);
			else
				cprintf(", %d", i);
		}
		switch((char)WaitAnyKey()) {
		case ESC:
		case 'x':
		case 'X':
			m = 1;
			break;
		case 'e':
		case 'E':
			for (i = 0; i < 10; i++)
				IgnEvt[i] = 0;
			break;
		case 'i':
		case 'I':
			if (GetInp(IGNIROW, 1,
				   "Event numbers to ignore?",
				   "", buf, sizeof(buf))
			== 0)
				break;
			for (cp = buf; *cp; cp++) {
				if (*cp == ' ' || *cp == ',')
					continue;
				if (*cp < '0' || *cp > '9') {
					putch(BELL);
					continue;
				}
				IgnEvt[*cp - '0'] = 1;
			}
			break;
		default:
			putch(BELL);
		}
	}
}


/*
 * SelSkipAft() - select skip after
 */

void
SelSkipAft()
{
	char dt[SKIPDTLN], tm[SKIPTMLN];

	if (SkipDtTm("AFTER", dt, tm) == 1) {
		SkipAft = 1;
		(void) strcpy(SkipAftDt, dt);
		(void) strcpy(SkipAftTm, tm);
	}
}


/*
 * SelSkipUnt() - select skip until
 */

void
SelSkipUnt()
{
	char dt[SKIPDTLN], tm[SKIPTMLN];

	if (SkipDtTm("UNTIL", dt, tm) == 1) {
		SkipUnt = 1;
		(void) strcpy(SkipUntDt, dt);
		(void) strcpy(SkipUntTm, tm);
	}
}


/*
 * SelTmRange() - select time range
 */

void
SelTmRange()
{
	char ch;
	short bt, bufl, et, ex, i, j, m, msgl;
	char buf[GTTLLNL+2+1];
	char msg[128];
	char ttl[GTTLLNL+2+1];

	for (ex = 0; !ex;) {
		if (NTmRange > 0) {
			(void) strcpy(Instr,
			    "Use one of these ranges by typing its number:");
			j = 6;
		} else {
			Instr[0] = '\0';
			j = 4;
		}
		DispMenu(SelRange, NULL);
		for (i = 0; i < NTmRange; i++) {
			if (TmRange[i].tm == NULL)
				continue;
			ClearRow(j, 1);
			gotoxy(6, j++);
			cprintf("%c %d.  %s (%d:%02d-%d:%02d)",
				(RangeInUse == i+1) ? '>' : ' ',
				i+1, TmRange[i].tm,
				TmRange[i].bt / 60, TmRange[i].bt % 60,
				TmRange[i].et / 60, TmRange[i].et % 60);
			if (TmRange[i].ttl) {
				ClearRow(j, 1);
				gotoxy(12, j++);
				cprintf("\"%s\"", TmRange[i].ttl);
			}
		}
		if (i)
			j++;
		if (RangeInUse > 0) {
			ClearRow(j, 1);
			gotoxy(6, j);
			cprintf("S - Stop using range %d.",
				RangeInUse);
			j += 2;
		}
		ClearRow(j, 1);
		gotoxy(6, j);
		cputs("X - eXit");
		switch((ch = (char)WaitAnyKey())) {
		case 'x':
		case 'X':
		case ESC:
			ex = 1;
			break;
		case 's':
		case 'S':
			if (RangeInUse > 0) {
				RangeInUse = 0;
				(void) strcpy(Gttl, GttlOrig);
			} else
				putch(BELL);
			break;
		case 'n':
		case 'N':
			for (m = 0; m == 0;) {
			    if (GetInp(21, 5, "Range?", "", buf, sizeof(buf))
			    == 0)
				m = 1;
			    else {
				bufl = strlen(buf);
				if (CvtTmRange(buf, msg, &bt, &et))
				    m = 2;
				else {
				    msgl = strlen(msg);
				    (void) WarnMsg(11,
					(short)((Vc.screenwidth - bufl)/2 + 1),
					buf, 13,
					(short)((Vc.screenwidth - msgl)/2 + 1),
					msg, 0);
				    m = 1;
				}
			    }
			}
			if (m == 1)
			    break;
			ttl[0] = '\0';
			if (GetInp(21, 5, "Title?", Gttl, ttl, sizeof(ttl))
			== 0)
			    ttl[0] = '\0';
			TmRange[0].bt = bt;
			TmRange[0].et = et;
			if (TmRange[0].tm)
				(void) free((void *)TmRange[0].tm);
			if ((TmRange[0].tm = (char *)malloc(bufl + 1))
			== NULL) {
			    (void) fprintf(stderr,
				"%s: no space for time range: %s\n",
				Pn, buf);
			    exit(1);
			}
			(void) strcpy(TmRange[0].tm, buf);
			if (TmRange[0].ttl)
				(void) free((void *)TmRange[0].ttl);
			if ((TmRange[0].ttl = (char *)malloc(strlen(ttl) + 1))
			== NULL) {
			    (void) fprintf(stderr,
				"%s: no space for title: %s\n",
				Pn, ttl);
			    exit(1);
			}
			(void) strcpy(TmRange[0].ttl, ttl);
			(void) strcpy(Gttl, ttl);
			if (NTmRange == 0)
			    NTmRange = 1;
			RangeInUse = 1;
			ex = 1;
			break;
		default:
			i = (short)(ch - '0') - 1;
			if (i < 0 || i >= NTmRange || TmRange[i].tm == NULL) {
				putch(BELL);
				break;
			}
			(void) strcpy(Gttl, TmRange[i].ttl ? TmRange[i].ttl
							   : "");
			RangeInUse = i + 1;
			ex = 1;
		}
	}
}
 

/*
 * SetDtTm() - set the date and time
 */

static void
SetDtTm()
{
	char buf[19], cmd[32], *cp;
	int err;

#if	defined(UNIX)
	int day, hr, min, mo, sec, yr;
#else
	short day, hr, min, mo, sec, yr;
#endif
	int nc;

	for (buf[0] = '\0', err = 0;;) {
		ClearRow(PROMPTLN, PROMPTCOL);
		if (err) {
			putch(BELL);
			err = 0;
		}
		buf[17] = '\0';
		nc = GetInp(PROMPTLN, PROMPTCOL,
			"Enter date/time (mm/dd/yy hh:mm[:ss) ? ", buf, buf,
			sizeof(buf));
		if (nc == 0)
			return;
		err++;
		cp = AsmNum(buf, &mo);
		if (*cp++ != '/' || mo < 1 || mo > 12)
			continue;
		cp = AsmNum(cp, &day);
		if (*cp++ != '/' || day < 1 || day > 31)
			continue;
		cp = AsmNum(cp, &yr);
		if (*cp++ != ' ' || (yr > 14 && yr < 84))
			continue;
		if (mo == 2) {
			nc = (yr >= 84) ? yr + 1900 : yr + 2000;
			if ((((nc % 4) == 0) && ((nc % 100) != 0))
			|| ((nc % 400) == 0))
				month_days[1] = 29;
			else
				month_days[1] = 28;
		}
		if (day > month_days[mo - 1])
			continue;
		cp = AsmNum(cp, &hr);
		if (*cp++ != ':' || hr < 0 || hr > 23)
			continue;
		cp = AsmNum(cp, &min);
		if (min < 0 || min > 59)
			continue;
		if (*cp == ':') {
			cp = AsmNum(++cp, &sec);
			if (sec < 0 || sec > 59)
				continue;
		} else
			sec = 0;
		if (*cp != '\0')
			continue;
		(void) sprintf(cmd, "DMT%02d/%02d/%02d %02d:%02d:%02d",
			mo, day, yr, hr, min, sec);
		for (;;) {
			if (WaitRdy() == 0)
				return;
			DispMenu(ReadWt, NULL);
			if (WaitCmd(cmd, 'T')) {
				GetDataLn(DumpLine, DUMPLL);
				if (DumpLine[0] == ' ' && DumpLine[1] == '"')
					break;
			}
			DispMenu(DateErr,
				"Press ESC to exit; any other key to retry.");
			gotoxy(1, 14);
			cprintf("Command response: %s", DumpLine);
			if ((char)WaitAnyKey() == ESC)
				return;
			continue;
		}
		return;
	}
}


/*
 * SkipDtTm() - get skip date and time
 */

int
SkipDtTm(ttl, dt, tm)
	char *ttl;		/* title line */
	char *dt;		/* date destination */
	char *tm;		/* time destination */
{
	char buf[64];
	short dtl, i, j, k, tml;

	for (;;) {
	    for (i = 0; i < 2;) {
		clrscr();
		gotoxy(10, 10);
		cprintf("Enter some starting characters of the %s %s which",
			(i == 0) ? "date" : "time",
			ttl);
		gotoxy(10, 11);
		cputs("records are to be skipped; enter at least 1 character.");
		if (i == 0) {

		/*
		 * Get date.
		 */
		    if (GetInp(13, 10, "Date?", "", dt, SKIPDTLN) == 0)
			return(0);
		    dtl = strlen(dt);
		    if (dtl > 1 && dtl < (SKIPDTLN - 1) && *dt != ' '
		    &&  *(dt+1) == '/') {
			(void) strcpy(&buf[1], dt);
			buf[0] = ' ';
			(void) strcpy(dt, buf);
			dtl++;
	 	    }
		    i = 1;
		    continue;
		 }
	    /*
	     * Get time.
	     */
		if (GetInp(13, 10, "Time?", "", tm, SKIPTMLN) == 0)
		    return(0);
		tml = strlen(tm);
		if (tml > 1 && tml < (SKIPTMLN - 1) && *tm != ' '
		&&  *(tm+1) == ':') {
		    (void) strcpy(&buf[1], tm);
		    buf[0] = ' ';
		    (void) strcpy(tm, buf);
		    tml++;
		}
		break;
	    }
	/*
	 * Check for matching date and time in dump buffer.
	 */
	    for (j = DumpHs; j < DumpLc; j++) {
		k = ParseDumpLn(DumpLp[j], 0);
		if (k == 1 || k == 2) {
		    if (strncmp(Date, dt, dtl) == 0
		    &&  strncmp(Time, tm, tml) == 0)
			break;
		}
	    }
	    if (j < DumpLc) {

	    /*
	     * Display matching dump record.
	     */
		(void) sprintf(buf,
		    "This record has matching date and time (\"%s\",\"%s\"):",
		    dt, tm);
		if ((char)WarnMsg(11,
		    (short)(((Vc.screenwidth - strlen(buf))/2) + 1), buf,
		    13,
		    (short)(((Vc.screenwidth - strlen(DumpLp[j]))/2) + 1),
		    DumpLp[j], 1)
		== ESC)
		    continue;
	        break;
	    }
	    (void) sprintf(buf,
		"No record has a date and time of \"%s\",\"%s\".",
		dt, tm);
	    if ((char)WarnMsg(12, (short)(((Vc.screenwidth - strlen(buf))/2)+1),
		buf, 0, 0, NULL, 1)
	    == ESC)
		return(0);
	}
	return(1);
}


/*
 * WaitCmd() - issue command and wait for reply
 */

int
WaitCmd(c, e)
	char *c;                        /* command text */

#if	defined(UNIX)
	int e;				/* command echo character */
#else
	char e;				/* command echo character */
#endif

{
	char ch;
	int err;

	AsynRstBf();
	AsynSndStr(c, &err);
	if (err) {
	    switch (err) {
	    case 8:
		(void) sprintf(DumpLine, "COM%d send time out error",
			Port + 1);
		break;
	    case 10:
		(void) sprintf(DumpLine, "COM%d send port is not initialized.",
			Port + 1);
		break;
	    default:
		(void) sprintf(DumpLine, "Unknown COM%d send error: %d",
			Port + 1, err);
	    }
		(void) WarnMsg(12,
			(short)(((Vc.screenwidth - strlen(DumpLine))/2) + 1),
			DumpLine, 0, 0, NULL, 0);
		return(0);
	}
	do {
		if (CheckESC()) {
			err = -1;
			break;
		}
		AsynInp(&ch, &err);
		if (Debug &&  ! err && ch)
			CommDisp(&ch);
	} while (err == 6 || (err == 0 && ch != e));
	if ( ! err)
		return(1);
	if (err == -1)
		return(0);
	(void) sprintf(DumpLine, "Error reading command reply: %d", err);
	(void) WarnMsg(12, (short)(((Vc.screenwidth - strlen(DumpLine))/2)+1),
		DumpLine, 0, 0, NULL, 0);
	return(0);
}
