/*
 * dump.c - dump the meter's memory
 *
 * V. Abell
 */

/*
 * Copyright 1994 Victor A. Abell, Lafayette, Indiana  47906.  All rights
 * reserved.
 *
 * Written by Victor A. Abell.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Victor A. Abell is not responsible for any consequences of the use of
 * this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to Victor A. Abell must
 *    appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#if	!defined(lint)

# if	defined(_BCC)
#pragma warn -use
# endif
static char copyright[] =
"@(#) Copyright 1994 Victor A. Abell.\nAll rights reserved.\n";
#endif

#include "touch2.h"
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#define FIELDLN	16			/* dump field length */
#define	RDYCOL	31			/* ready message column */
#define	SMROW	12			/* status message row */

char CksumErrMsg[64];			/* checksum error message */
int DumpHs = 0;				/* dump header status */
int DumpLc = 0;				/* number of dump lines */
char DumpfnQ[DUMPFNL];			/* QuattroPro dump file name */
char DumpfnR[DUMPFNL];			/* raw dump file name */
FILE *Dumpfs = NULL;			/* write dump file stream */
char DumpLine[DUMPLL+1];		/* dump line */
char *DumpLp[DUMPNL];			/* dump line pointers */
char *EvtBuf[] = {
	NULL,				/* title for event filter '1' */
	NULL,				/* title for event filter '2' */
	NULL,				/* title for event filter '3' */
	NULL,				/* title for event filter '4' */
	NULL,				/* title for event filter '5' */
	NULL,				/* title for event filter '6' */
	NULL,				/* title for event filter '7' */
	NULL,				/* title for event filter '8' */
	NULL				/* title for event filter '9' */
};

char Graphfn[DUMPFNL];			/* graph file name */
FILE *Graphfs = NULL;			/* graph file stream */
unsigned char IgnEvt[] =		/* events to ignore */
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
unsigned char IgnEvtOrig[] =		/* original list of events to ignore */
	{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
FILE *Rdumpfs = NULL;			/* read dump file stream */
short Rtype;				/* dump line reading type */
short SupTtlIO = 0;			/* suppress title line I/O to and
					 * from the dump file */


struct menu DumpRdy[] = {
	{ 12, RDYCOL, "Waiting on meter" },
	{  0,  0, NULL },
};

struct menu DumpSt[] = {
	{ 12, 27, "Waiting for meter to start dump" },
	{  0,  0, NULL },
};

#define	TYPEBUFL	64
char CheckDump[TYPEBUFL];
char DumpFile[TYPEBUFL];
char DumpScreen[TYPEBUFL];
char EraseBuf[TYPEBUFL];
char FormatBuf[TYPEBUFL];
char GraphBuf[TYPEBUFL];
char GraphTtl[GTTLLNL+2+1];
char RangeTtl[TYPEBUFL];
char ReadBuf[TYPEBUFL];
char RevBuf[TYPEBUFL];
char SkipAftBuf[TYPEBUFL];
char SkipUntBuf[TYPEBUFL];
char SupTtlBuf[TYPEBUFL];
char EvtPromptBuf[TYPEBUFL];

struct menu DumpType[] = {
	{  6, 10, CheckDump },
	{  7, 10, EraseBuf },
	{  8, 10, DumpFile },
	{  9, 10, GraphBuf },
	{ 10, 10, SupTtlBuf },
	{ 11, 10, FormatBuf },
	{ 12, 10, ReadBuf },
	{ 13, 10, DumpScreen },
	{ 14, 10, "T - change graph Title from:" },
	{ 15, 12,  GraphTtl },
	{ 16, 10, RevBuf },
	{ 17, 10, "X - eXit" },
#define	DTIROW	19
	{  0,  0, NULL },
};

struct menu GraphType[] = {
	{  6, 10, SkipAftBuf },
	{  7, 10, "D - Disable event, range, and skip filtering" },
	{  8, 10, EvtPromptBuf },
	{  9, 10, "F - write graph to File" },
#define	IGNROW	10
#define	IGNCOL	10
	{ 11, 10, "R - select records to graph by time Range" },
	{ 12, 12, RangeTtl },
	{ 13, 10, "S - draw graph on Screen" },
	{ 14, 10, "T - change graph Title from:" },
	{ 15, 12, GraphTtl },
	{ 16, 10, SkipUntBuf },
	{ 17, 10, "X - eXit" },
	{  0,  0, NULL },
};

static int CheckCksum(int len, int x);
static void DispDumpLns(void);
static void EraseDump(void);

#if	defined(UNIX)
static int GetDispLn(char *s, int n);
#else
static int GetDispLn(char *s, short n);
#endif

static int CheckFileNm(char *fn);
static int GetDumpType(char *ty);
static void RcvErr(int err);
static void ReadDump(void);
static int StoreDumpLn(void);
static void TrimSpaces(char *s);
static void WarnCantOpen(char *f);

#if	defined(UNIX)
static int WarnCksum(int f, int x);
#else
static int WarnCksum(short f, int x);
#endif

static int WarnDumpHdr(void);
static void WarnDumpLen(void);
static int WarnExist(char *d, struct stat *s);
static void WrDumpF(void);


/*
 * CheckCksum() - check checksum
 */


static int
CheckCksum(len, x)
	int len;			/* length of line in DumpLine */
	int x;				/* index if line in DumpLp[] */
{
	char cksum[5], *lp;
	short i;
	unsigned int sum;

	if (!Cksum)
		return(0);
	if (len < 5) {
		(void) strcpy(CksumErrMsg, "Record is too short.");
		return(1);
	}
/*
 * A leading 'P' has been stripped from all dump header records,
 * and a leading "P " has been stripped from dump file records.
 * They need to be considered in computing the checksum.
 */
	lp = DumpLp[x];
	if (x == 0)
		sum = (unsigned int) 'P';
	else {
		if (lp[0] == 'P')
			sum = 0;
		else 
			sum = (unsigned int) 'P' + (unsigned int) ' ';
	}
	for (i = 0; i < (len - 5); i++)
		sum += (unsigned int)lp[i];
	(void) sprintf(cksum, "%04x", sum);
	if (strcmpi(lp + len - 4, cksum) == 0)
		return(0);
	for (i = 0; cksum[i]; i++) {
		if (isalpha(cksum[i]) && islower(cksum[i]))
			cksum[i] = toupper(cksum[i]);
	}
	(void) sprintf(CksumErrMsg, "Computed %s; read %s.",
		cksum, lp + len - 4);
	return(1);
}


/*
 * CheckESC() - check for ESC
 */

int
CheckESC()
{
	int ch;

	if (kbhit()) {
		if ((ch = getch()) == 0)
			ch = getch();
		if (ch == ESC)
			return(1);
		putch(BELL);
	}
	return(0);
}


/*
 * CheckFilNm() - check file name
 */

static int
CheckFileNm(fn)
	char *fn;			/* file name to check */
{
	char *cp = NULL;
	short err = 0;
	short fnl;
	short lc, p, rc;
	char *mp, msg[80];

/*
 * Skip leading alpha, followed by a colon -- i.e., a drive specification.
 * Skip leading "./", ".\", "../", and "..\".
 */
	fnl = strlen(fn);
	if (fnl > 1) {
		if ((isalpha(*fn) && fn[1] == ':')
		||   strncmp(fn, "./", 2) == 0
		||   strncmp(fn, ".\\", 2) == 0)
			cp = fn + 2;
	}
	if (cp == NULL && fnl > 2) {
		if (strncmp(fn, "../", 3) == 0
		||  strncmp(fn, "..\\", 3) == 0)
			cp = fn + 3;
	}
	if (cp == NULL)
		cp = fn;
/*
 * Check the file path components -- i.e., those between / and \.
 *
 * The left part (the name) may have no more than 8 characters.
 * The right part (the extension) may have no more than 3 characters.
 * There may be one period, separating the name and extension.
 * The characters must be alphanumeric or special characters from the set:
 *
 *	_ ^ $ ~ ! # % & - { } @ ` ' ( )
 */
	(void) strcpy(msg, "has ");
	mp = &msg[strlen(msg)];
	while (err == 0 && *cp) {
	    if (*cp == '/' || *cp == '\\') {
	    	cp++;
		continue;
	    }
	    lc = p = rc = 0;
	    while (*cp) {
		switch (*cp) {
	    /*
	     * These are the legal special characters.
	     */
		case '_':
		case '^':
		case '$':
		case '~':
		case '!':
		case '#':
		case '%':
		case '&':
		case '-':
		case '{':
		case '}':
		case '@':
		case '`':
		case '\'':
		case ')':
		case '(':
		    break;
	    /*
	     * The / and \ end a component.
	     */
		case '/':
		case '\\':
		    err = -1;
		    break;
	    /*
	     * The period separates the name and extension.
	     */
		case '.':
		    if (p) {
			(void) strcpy(mp, "more than one period.");
			err = 1;
		    } else if (lc == 0) {
			(void) strcpy(mp,
			    "nothing to the left of the period.");
			err = 1;
			break;
		    } else
			p = 1;
		    break;
	    /*
	     * Letters (either case) and numbers are legal.
	     */
		default:
		    if (isalnum(*cp))
			break;
		    (void) strcpy(mp, "an illegal character.");
		    err = 1;
		}
	    /*
	     * Stop on error or end of component.  Skip a period.
	     */
		if (err)
		    break;
		if (*cp++ == '.')
		    continue;
	    /*
	     * The name may have 8 characters; the extension, 3.
	     */
		if ( ! p) {
		    if (lc > 7) {
			(void) strcpy(mp,
			    "more than 8 characters in the name part.");
			err = 1;
		    } else
			lc++;
		} else {
		    if (rc > 2) {
			(void) strcpy(mp,
			    "more than 3 characters in the extension.");
			err = 1;
		    } else
			rc++;
		}
	    }
	/*
	 * Continue after the end of a component; stop on an error.
	 */
	    if (err == -1) {
		err = 0;
		continue;
	    }
	    if (err)
		break;
	}
/*
 * Return if no error; warn if there was one.
 */
	if (err == 0)
	    return(0);
	(void) WarnMsg(11, (short)((Vc.screenwidth - fnl)/2 + 1), fn,
	    13, (short)((Vc.screenwidth - strlen(msg))/2 + 1), msg, 0);
	return(1);
}


/*
 * DumpMtr() - dump the meter's memory
 */

void
DumpMtr()
{
	int i;
	char dt, msg[32];

	while (GetDumpType(&dt)) {
	    if ( ! DumpLc && dt != 'r') {
		for (;;) {
		    if (Debug)
			CommDisp(NULL);
		    if (WaitRdy() == 0) {
			CloseDump();
			break;
		    }
		    if ( ! Debug)
			DispMenu(DumpSt, NULL);
		    if (WaitCmd("DMP", 'P') == 0) {
			(void) strcpy(DumpLine, "Dump command failed.");
			if ((char)WarnMsg(12,
			    (short)(((Vc.screenwidth - strlen(DumpLine))/2)+1),
				DumpLine, 0, 0, NULL, 1)
			== ESC) {
				CloseDump();
				break;
			}
			continue;
		    }
		    if ( ! Debug)
			clrscr();
		    PromptMsg("Press ESC to exit.");
		    (void) GetDataLn(DumpLine, DUMPLL);
		    if (StoreDumpLn()) {
			CloseDump();
			(void) EraseDump();
			break;
		    }
		    if ( ! ParseHdr()) {
			if ((char)WarnDumpHdr() == ESC) {
				CloseDump();
				(void) EraseDump();
				break;
			}
		    }
		    gotoxy(RDYCOL, SMROW);
		    (void) cputs("Header record read.");
		    for (i = 0;;) {
			if (CheckESC()) {
				CloseDump();
				(void) EraseDump();
				break;
			}
			(void) GetDataLn(DumpLine, DUMPLL);
			if (DumpLine[0] != 'P')
				break;
			if (StoreDumpLn()) {
				CloseDump();
				(void) EraseDump();
				break;
			}
			if (i == 0) {
				ClearRow(SMROW, RDYCOL);
				(void) sprintf(msg,
					"Record %3d read.", DumpLc - 1);
				gotoxy(RDYCOL, SMROW);
				i = 1;
			} else {
				gotoxy(RDYCOL+7, SMROW);
				(void) sprintf(msg, "%3d", DumpLc - 1);
			}
			cputs(msg);
		    }
		    if (Ndump+1 != DumpLc)
			WarnDumpLen();
		    if (Cksum) {
			for (i = 0; i < DumpLc; i++) {
				if (CheckCksum(strlen(DumpLp[i]), i)) {
					if ((char)WarnCksum(1, i) == ESC) {
						CloseDump();
						(void) EraseDump();
						break;
					}
				}
			}
		    }
		    break;
	        }
		if ( ! DumpLc || (dt != 'f' && dt != 's'))
		    continue;
	    }
	    switch (dt) {
		case 'f':
			WrDumpF();
			break;
		case 'g':
		case 'G':
			DrawGraph((dt == 'g') ? 0 : 1);
			break;
		case 'r':
			ReadDump();
			break;
		case 's':
			DispDumpLns();
	    }
	}
}


/*
 * DispDumpLns() - display dump lines
 */

static void
DispDumpLns(void)
{
	char *bf, bot[80], *cp;
	int cpl, i, j, k, lps;

	if (DumpLc == 0)
		return;
	cpl = Vc.screenwidth;
	lps = Vc.screenheight;
	if ((bf = (char *)malloc((size_t)(cpl + 1))) == NULL) {
		clrscr();
		gotoxy(20, 12);
		(void) cputs("DispDump: no temporary line space");
		return;
	}

	i = 0;
	for (;;) {

	/*
	 * Display a screenload.
	 */
		clrscr();
		for (j = 1; j < lps && (i+j-1) < DumpLc;  j++) {
			for (cp = DumpLp[i+j-1], k = 0; k < cpl; cp++) {
				if (*cp == '"')
					continue;
				if (*cp == '\0')
					break;
				bf[k++] = *cp;
			}
			bf[k] = '\0';
			gotoxy(1, j);
			(void) cputs(bf);
		}
		(void) sprintf(bot,
		  "(%d of %d) Press ESC or X to exit; Page Up/Down; Arrow Up/Down.",
		  i+1, DumpLc);
		PromptMsg(bot);
	/*
	 * Wait for a command.
	 */
		for (k = 1; k;) {
			switch((char)WaitAnyKey()) {
			case ESC:
			case 'x':
			case 'X':
				return;
			case PGDN:
				if ((i + lps - 1) < DumpLc) {
					i += lps - 1;
					k = 0;
				} else
					putch(BELL);
				break;
			case PGUP:
				if (i > 0) {
					i -= lps - 1;
					if (i < 0)
						i = 0;
					k = 0;
				} else
					putch(BELL);
				break;
			case UARW:
				if (i > 0) {
					i--;
					k = 0;
				} else
					putch(BELL);
				break;
			case DARW:
				if (i < (DumpLc - 1)) {
					i++;
					k = 0;
					break;
				}
				/* fall through */
			default:
				putch(BELL);
			}
		}
	}
} 


/*
 * EraseDump() -- erase dump
 */

static void
EraseDump(void)
{
	int i;

	for (i = 0; i < DumpLc; i++) {
		(void) free(DumpLp[i]);
		DumpLp[i] = NULL;
	}
	DumpLc = Ndump = 0;
}


/*
 * GetDataLn(s, n) - get meter data line
 */

int
GetDataLn(s, n)
	char *s;
	int n;
{
	char c;
	int err, i;

	for (i = 0;;) {
		if (CheckESC()) {
			s[0] = '\0';
			return(0);
		}
		AsynInp(&c, &err);
		if ( ! err && c) {
			if (Debug)
				CommDisp(&c);
			if (c == CR)
				break;
			if (c == LF)
				continue;
			if (i+1 >= n)
				break;
			s[i++] = c;
		} else if (err == 6)
			continue;
		else if (err) {
			RcvErr(err);
			return(0);
		}
	}
	s[i] = '\0';
	return(i);
}


/*
 * GetDumpType() - get dump type
 */

static int
GetDumpType(ty)
	char *ty;			/* type response */
{
	char fn[DUMPFNL];
	short i, j, m, m1;
	struct stat sbuf;
	char *t;

	Dumpfs = Graphfs = NULL;
	for (m = 0;;) {
		if ( ! m) {
		    (void) sprintf(FormatBuf,
			"Q - change dump format to %s from %s",
			Qp ? "raw" : "QuattroPro",
			Qp ? "QuattroPro" : "raw");
		    if (Qp)
			(void) sprintf(CheckDump,
		    	    "C - %sdump Check, error, and high readings",
			    Ckdump ? "" : "don't ");
		    else
			CheckDump[0] = '\0';
		    if (DumpLc) {
			i = DumpLc - DumpHs;
			t = (i == 1) ? "record" : "records";
			(void) sprintf(EraseBuf,
				"E - Erase %d %s from dump buffer", i, t);
			(void) sprintf(DumpFile,
				"F - dump %d %s to File", i, t);
			(void) sprintf(GraphBuf, "G - Graph %d %s", i, t);
			ReadBuf[0] = '\0';
			(void) sprintf(DumpScreen,
				"S - dump %d %s to Screen", i, t);
			if (i > 1)
			    (void) sprintf(RevBuf,
				"V - reVerse order of %d %s", i, t);
			    else
				RevBuf[0] = '\0';
		    } else {
			EraseBuf[0] = RevBuf[0] = '\0';
			(void) strcpy(DumpFile, "F - dump to File");
			(void) strcpy(GraphBuf, "G - Graph");
			(void) strcpy(ReadBuf, "R - Read from file");
			(void) strcpy(DumpScreen, "S - dump to Screen");
		    }
		    if (Gttl[0] != '\0') {
			if (ExpandTtl())
				(void) sprintf(GraphTtl, "Error: %s",
					TtlErrMsg);
			else
				(void) sprintf(GraphTtl, "\"%s\"", Exttl);
		    } else
			GraphTtl[0] = '\0';
		    (void) sprintf(SupTtlBuf,
			"L - read/write graph titLe from/to dump file: %s",
			SupTtlIO ? "no" : "yes");
		    DispMenu(DumpType, NULL);
		    m = 1;
		}
		switch((char)WaitAnyKey()) {
		case 'x':
		case 'X':
		case ESC:
			return(0);
		case 'c':
		case 'C':
			Ckdump = Ckdump ? 0 : 1;
			m = 0;
			break;
		case 'e':
		case 'E':
			if ( ! DumpLc)
				putch(BELL);
			else {
				(void) EraseDump();
				if (GttlOrig)
					(void) strcpy(Gttl, GttlOrig);
				else
					Gttl[0] = '\0';
				m = 0;
			}
			break;
		case 'f':
		case 'F':
			(void) strcpy(fn, Qp ? DumpfnQ : DumpfnR);
			if (GetInp(DTIROW, 5, "Name?", fn, fn, sizeof(fn)) == 0
			||  CheckFileNm(fn)) {
				m = 0;
				break;
			}
			if (stat(fn, &sbuf) == 0) {
				if (WarnExist(fn, &sbuf) == 0) {
					m = 0;
					break;
				}
			}
			if ((Dumpfs = fopen(fn, "w+t")) != NULL) {
				(void) strcpy(Qp ? DumpfnQ : DumpfnR, fn);
				*ty = 'f';
				return(1);
			}
			WarnCantOpen(fn);
			m = 0;
			break;
		case 'g':
		case 'G':
			for (m1 = 0; m1 == 0;) {
			    if (Gttl[0] != '\0') {
				if (ExpandTtl())
				    (void) sprintf(GraphTtl, "Error: %s",
					TtlErrMsg);
				else
				    (void) sprintf(GraphTtl, "\"%s\"", Exttl);
			    } else
				GraphTtl[0] = '\0';
			    if (RangeInUse > 0)
			    	(void) sprintf(RangeTtl, "\"%s\"",
					TmRange[RangeInUse - 1].tm);
			    else
				RangeTtl[0] = '\0';
			    if (SkipAft)
				(void) sprintf(SkipAftBuf,
				    "A - skip After \"%s\",\"%s\"",
				    SkipAftDt, SkipAftTm);
			    else
				(void) strcpy(SkipAftBuf,
				    "A - skip After date and time");
			    if (SkipUnt)
				(void) sprintf(SkipUntBuf,
				    "U - skip Until \"%s\",\"%s\"",
				    SkipUntDt, SkipUntTm);
			    else
				(void) strcpy(SkipUntBuf,
				    "U - skip Until date and time");
			    if (Eventfilt)
				(void) sprintf(EvtPromptBuf,
				    "E - graph only records with Event %c",
				    Eventfilt);
			    else
				(void) strcpy(EvtPromptBuf,
				    "E - define Event to graph");
			    DispMenu(GraphType, NULL);
			    gotoxy(IGNCOL, IGNROW);
			    cprintf("I - Ignore events");
			    for (i = j = 0; i < 10; i++) {
			    	if (IgnEvt[i] == 0)
				    continue;
				cprintf("%c %d", (j++ == 0) ? ':' : ',', i);
			    }
			    switch ((char)WaitAnyKey()) {
			    case 'x':
			    case 'X':
			    case ESC:
			        m1 = 1;
				Eventfilt = '\0';
				RangeInUse = SkipAft = SkipUnt = 0;
				if (GttlDumpf[0] != '\0')
					(void) strcpy(Gttl, GttlDumpf);
				else if (GttlOrig)
					(void) strcpy(Gttl, GttlOrig);
				for (i = 0; i < 10; i++)
					IgnEvt[i] = IgnEvtOrig[i];
				break;
			    case 'a':
			    case 'A':
				SelSkipAft();
				break;
			    case 'd':
			    case 'D':
				Eventfilt = '\0';
				RangeInUse = SkipAft = SkipUnt = 0;
				if (GttlDumpf[0] != '\0')
					(void) strcpy(Gttl, GttlDumpf);
				else if (GttlOrig)
					(void) strcpy(Gttl, GttlOrig);
				break;
			    case 'e':
			    case 'E':
				SelEvent();
				break;
			    case 'f':
			    case 'F':
				(void) strcpy(fn, Graphfn);
				if (GetInp(19, 5, "Name?", fn, fn, sizeof(fn))
				== 0)
					break;
				if (CheckFileNm(fn))
					break;
				if (stat(fn, &sbuf) == 0) {
					if (WarnExist(fn, &sbuf) == 0)
						break;
				}
				if ((Graphfs = fopen(fn, "w+t")) != NULL) {
					(void) strcpy(Graphfn, fn);
					DrawGraph(1);
					break;
				}
				WarnCantOpen(fn);
				break;
			    case 'i':
			    case 'I':
				SelIgnEvt();
				break;
			    case 'r':
			    case 'R':
				SelTmRange();
				break;
			    case 's':
			    case 'S':
				DrawGraph(0);
				break;
			    case 't':
			    case 'T':
				if (GetInp(19, 7, "Title?", Gttl, GraphTtl,
				    GTTLLNL)
				!= 0)
				    (void) strcpy(Gttl, GraphTtl);
				else {
					if (GttlDumpf[0] != '\0')
						(void) strcpy(Gttl, GttlDumpf);
					else if (GttlOrig)
						(void) strcpy(Gttl, GttlOrig);
				}
				break;
			    case 'u':
			    case 'U':
				SelSkipUnt();
				break;
			    default:
				putch(BELL);
			    }
			}
			m = 0;
			break;
		case 'l':
		case 'L':
			SupTtlIO = SupTtlIO ? 0 : 1;
			m = 0;
			break;
		case 'q':
		case 'Q':
			Qp = (Qp == 1) ? 0 : 1;
			m = 0;
			break;
		case 'r':
		case 'R':
			(void) strcpy(fn, DumpfnR);
			if (GetInp(DTIROW, 5, "Name?", fn, fn, sizeof(fn)) == 0
			||  CheckFileNm(fn)) {
				m = 0;
				break;
			}
			if ((Rdumpfs = fopen(fn, "r+t")) != NULL) {
				(void) strcpy(DumpfnR, fn);
				*ty = 'r';
				return(1);
			}
			WarnCantOpen(fn);
			m = 0;
			break;
		case 's':
		case 'S':
			*ty = 's';
			return(1);
		case 't':
		case 'T':
			if (GetInp(DTIROW, 7, "Title?", Gttl, GraphTtl, GTTLLNL)
			!= 0)
				(void) strcpy(Gttl, GraphTtl);
			else {
				if (GttlDumpf[0] != '\0')
					(void) strcpy(Gttl, GttlDumpf);
				else if (GttlOrig)
					(void) strcpy(Gttl, GttlOrig);
			}
			m = 0;
			break;
		case 'v':
		case 'V':
			if ((i = DumpLc - DumpHs) < 2) {
			    if (i == 0)
			    	(void) strcpy(RevBuf,
				    "There are no dump records.");
			    else
				(void) strcpy(RevBuf,
				    "No need to reverse just 1 record.");
			    (void) WarnMsg(11,
				(short)((Vc.screenwidth-strlen(RevBuf))/2 + 1),
				RevBuf, 0, 0, NULL, 0);
			    m = 0;
			    break;
			}
			for (i = DumpHs, j = DumpLc - 1; i < j; i++, j--) {
			    t = DumpLp[j];
			    DumpLp[j] = DumpLp[i];
			    DumpLp[i] = t;
			}
			(void) sprintf(RevBuf,
			    "The order of %d dump records has been reversed.",
			    DumpLc - DumpHs);
			(void) WarnMsg(11,
			    (short)((Vc.screenwidth - strlen(RevBuf))/2 + 1),
			    RevBuf, 0, 0, NULL, 0);
			m = 0;
			break;
		default:
			putch(BELL);
		}
	}
}


/*
 * InitDump() - initialize dump
 */

void
InitDump()
{
	(void) strcpy(DumpfnQ, DEFQDMPF);
	(void) strcpy(DumpfnR, DEFRDMPF);
	(void) strcpy(Graphfn, DEFGRAPHF);
	Gttl[0] = GttlDumpf[0] = '\0';
}



/*
 * GetDispLn(s) - get meter display line
 */

static int
GetDispLn(s, n)
	char *s;			/* destination buffer */

#if	defined(UNIX)
	int n;				/* destination buffer length */
#else
	short n;			/* destination buffer length */
#endif

{
	char c;
	int err, i;

	for (i = 0;;) {
		if (CheckESC()) {
			s[0] = '\0';
			return(0);
		}
		AsynInp(&c, &err);
		if ( ! err && c) {
			if (Debug)
				CommDisp(&c);
			if (c == CR) {
				s[i] = '\0';
				break;
			}
			if (i < n - 1)
				s[i++] = c;
			else {
				s[i] = '\0';
				return(0);
			}
		} else if (err == 6)
			continue;
		else if (err) {
			RcvErr(err);
			return(0);
		}
	}
	return(1);
}


/*
 * Rcverr() - handle receive error
 */

static void
RcvErr(err)
	int err;
{
	switch (err) {
	case 7:
		(void) sprintf(DumpLine, "COM%d receive buffer overflow error",
			Port + 1);
		break;
	case 10:
		(void) sprintf(DumpLine,
			"COM%d receive port is not initialized.", Port + 1);
		break;
	default:
		(void) sprintf(DumpLine, "Unknown COM%d receive error: %d",
			Port + 1, err);
	}
	(void) WarnMsg(12,
		(short)(((Vc.screenwidth - strlen(DumpLine))/2) + 1),
		DumpLine,
		0, 0, NULL, 0);
}


/*
 * ReadDump() - read dump from file
 */

static void
ReadDump(void)
{
	char *cp, *rv;
	int i, len;
	short err = 0;

/*
 * Read, parse and store the dump title and header lines.
 */
	GttlDumpf[0] = '\0';
	while ((rv = fgets(DumpLine, DUMPLL, Rdumpfs)) != NULL) {
		if ((cp = strrchr(DumpLine, '\n')) != NULL) {
			*cp = '\0';
			len = cp - DumpLine;
		} else {
			DumpLine[DUMPLL-1] = '\0';
			len = strlen(DumpLine);
		}
		if (strncmp(DumpLine, "TTL:", 4) == 0) {
			if ( ! SupTtlIO) {
				(void) strncpy(Gttl, &DumpLine[4], GTTLLNL);
				Gttl[GTTLLNL] = '\0';
				(void) strcpy(GttlDumpf, Gttl);
			}
			continue;
		}
		if (StoreDumpLn()) {
			CloseDump();
			(void) EraseDump();
			return;
		}
		if (CheckCksum(len, 0)) {
			if ((char)WarnCksum(1, 0) == ESC) {
				CloseDump();
				(void) EraseDump();
				return;
			}
		}
		break;
	}
	if (rv == NULL)
		DumpLine[0] = '\0';
	if (rv == NULL || ! ParseHdr()) {

	/*
	 * Handle bad dump header line.
	 */
		i = WarnDumpHdr();
		if (DumpLc)
			(void) EraseDump();
		if ((char)i == ESC) {
			CloseDump();
			return;
		}
		err = 1;
	}
/*
 * Read and store the dump lines.
 */
	while (fgets(DumpLine, DUMPLL, Rdumpfs) != NULL) {
		if ((cp = strrchr(DumpLine, '\n')) != NULL) {
			*cp = '\0';
			len = cp - DumpLine;
		} else {
			DumpLine[DUMPLL-1] = '\0';
			len = strlen(DumpLine);
		}
		if (StoreDumpLn()) {
			err = 1;
			break;
		}
		if (CheckCksum(len, DumpLc - 1)) {
			if ((char)WarnCksum(1, DumpLc - 1) == ESC) {
				CloseDump();
				(void) EraseDump();
				return;
			}
		}
	}
	CloseDump();
	if (err == 0 && (Ndump+1 != DumpLc))
		WarnDumpLen();
}


/*
 * StoreDumpLn() - store dump line via DumpLp[]
 */

static int
StoreDumpLn(void)
{
	char *cp;
	int i;
	size_t l;
	char msg[128];

	if (DumpLc >= DUMPNL) {
		(void) sprintf(msg, "Dump buffer is full (limit = %d).",
			DUMPNL);
store_warn:
		i = WarnMsg(12, (short)(((Vc.screenwidth - strlen(msg))/2)+1),
		    msg, 14, (short)(((Vc.screenwidth - strlen(DumpLine))/2)+1),
		    DumpLine, 1);
		return(((char)i == ESC) ? 1 : 0);
	}
	cp = DumpLine;
	if (*cp == 'P') {
		cp++;
		if (*cp == ' ')
			cp++;
	}
	l = (size_t)(strlen(cp) + 1);
	if ((DumpLp[DumpLc] = (char *)malloc(l)) == NULL) {
		(void) sprintf(msg, "No memory for dump line number %d.",
			DumpLc);
		goto store_warn;
	}
	(void) strcpy(DumpLp[DumpLc], cp);
	DumpLc++;
	return(0);
}


/*
 * WaitRdy() - wait for the meter to become ready
 */

int
WaitRdy()
{
	char buf[64];

	DispMenu(DumpRdy, NULL);
	OpenCom();
	AsynRstBf();
	for (;;) {
		if (CheckESC())
			return(0);
		if (AsynBfLen == 0)
			continue;
		if (GetDispLn(buf, sizeof(buf)) == 0)
			return(0);
		if (strcmpi(buf, "INSERT")  == 0	/* ENGL */
		||  strcmpi(buf, "INSERT.") == 0	/* ESPAN */
		||  strcmpi(buf, "INSER. ") == 0	/* FRANC and ITALI */
		||  strcmpi(buf, "IN    ")  == 0	/* NEDER */
		||  strcmpi(buf, "INSIRA")  == 0	/* PORT */
		||  strcmpi(buf, "SATTIN")  == 0	/* SVENS */
		||  strcmpi(buf, "EINLEG")  == 0	/* DEUTS */
		||  strcmpi(buf, ")))   ")  == 0)	/* SYMB */
			break;
	}
	return(GetDispLn(buf, sizeof(buf)));
}


/*
 * TrimSpaces() - trim leading and trailing spaces from string
 */

static void
TrimSpaces(s)
	char *s;			/* string address */
{
	char *s1, *s2;
 
	for (s1 = s2 = s; *s1; s1++) {
		if (s2 > s || *s1 != ' ')
			*s2++ = *s1;
	}
	while (s2 > s) {
		if (*(s2 - 1) != ' ')
			break;
		s2--;
	}
	*s2 = '\0';
}


/*
 * WarnCantOpen() - issue can't open file warning
 */

static void
WarnCantOpen(f)
	char *f;			/* file name */
{
	char msg[80];

	(void) sprintf(msg, "Can't open %s", f);
	(void) WarnMsg(12, 1, msg, 0, 0, NULL, 0);
}


/*
 * WarnCksum()
 */

static int
WarnCksum(f, x)

#if	defined(UNIX)
	int f;			/* prompt flag for WarnMsg() */
	int x;			/* index of record in DumpLp[] */
#else
	short f;		/* prompt flag for WarnMsg() */
	int x;			/* index of record in DumpLp[] */
#endif

{
	int i;
	char m1[80], m2[80], m3[DUMPLL+1], *lp;

	(void) strcpy(m1, "Checksum doesn't match for dump ");
	if (x)
		(void) sprintf(m2, "record %d.  ", x);
	else
		(void) strcpy(m2, "header record.  ");
	(void) strcat(m1, m2);
	(void) strcat(m1, CksumErrMsg);
	for (i = 0, lp = DumpLp[x]; i < DUMPLL && *lp; lp++, i++) {
		if (isascii(*lp) && isprint(*lp))
			m3[i] = *lp;
		else
			m3[i] = '?';
	}
	m3[i] = '\0';
	return(WarnMsg(12, (short)(((Vc.screenwidth - strlen(m1))/2)+1), m1,
		14, (short)(((Vc.screenwidth - i)/2)+1), m3, f));
}


/*
 * WarnDumpHdr()
 */

static int
WarnDumpHdr(void)
{
	return(WarnMsg(12, 30, "Bad dump header line:", 14,
		(short)(((Vc.screenwidth - strlen(DumpLine))/2)+1),
		DumpLine, 1));
}


/*
 * WarnDumpLen() - warn about incorrect dump length
 */

static void
WarnDumpLen(void)
{
	char msg[80];

	(void) sprintf(msg,
		"Dump length incorrect: %d records expected, %d received.",
		Ndump, DumpLc - 1);
	(void) WarnMsg(12, 11, msg, 0, 0, NULL, 0);
}


/*
 * WarnExist() - warn that the dump file exists
 */

static int
WarnExist(d, s)
	char *d;			/* dump file name */
	struct stat *s;			/* stat buffer */
{
	char msg[64];

	clrscr();
	(void) sprintf(msg, "%s exists.", d);
	gotoxy((short)((Vc.screenwidth - strlen(msg))/2 + 1), 11);
	(void) cputs(msg);
	(void) sprintf(msg, "It is %ld bytes long.", s->st_size);
	gotoxy((short)((Vc.screenwidth - strlen(msg))/2 + 1), 13);
	(void) cputs(msg);
	PromptMsg("Press ESC to exit; any other key to overwrite.");
	if ((char)WaitAnyKey() == ESC)	
		return(0);
	return(1);
}


/*
 * WarnMsg() - issue warning message
 */

int
WarnMsg(r1, c1, m1, r2, c2, m2, f)

#if	defined(UNIX)
	int r1;				/* first message row */
	int c1;				/* first message column */
	char *m1;			/* first message */
	int r2;				/* second message row */
	int c2;				/* second message column */
	char *m2;			/* second message */
	int f;				/* prompt flag: 0 = any continue
					 *		1 = ESC abort
					 *		2 = any abort */
#else
	short r1;			/* first message row */
	short c1;			/* first message column */
	char *m1;			/* first message */
	short r2;			/* second message row */
	short c2;			/* second message column */
	char *m2;			/* second message */
	short f;			/* prompt flag: 0 = any continue
					 *		1 = ESC abort
					 *		2 = any abort */
#endif

{
	char *p;

	clrscr();
	gotoxy(c1, r1);
	(void) cputs(m1);
	if (m2) {
		gotoxy(c2, r2);
		(void) cputs(m2);
	}
	switch (f) {
	case 0:
		p = "Press any key to continue.";
		break;
	case 1:
		p = "Press ESC to exit; any other key to continue.";
		break;
	default:
		p = "Press any key to abort.";
	}
	PromptMsg(p);
	return(WaitAnyKey());
}


/*
 * WrDumpF() -- write dump lines to dump file
 */

static void
WrDumpF(void)
{

#if	defined(UNIX)
	int i;
#else
	short i;
#endif

	if (Qp) {
		for (i = 1; i < DumpLc; i++) {
			if (ParseDumpLn(DumpLp[i], 0) == 0)
				break;
			(void) TrimSpaces(Date);
			if (Ckdump == 0 && Rtype != RDREG)
				continue;
			(void) TrimSpaces(Time);
			(void) fprintf(Dumpfs, "%s,%s,%6.2f\n", Date, Time,
				Rval);
		}
	} else {
		if (Gttl[0] && !SupTtlIO)
			(void) fprintf(Dumpfs, "TTL:%s\n", Gttl);
		for (i = 0; i < DumpLc; i++)
			(void) fprintf(Dumpfs, "%s\n", DumpLp[i]);
	}
	(void) fclose(Dumpfs);
	Dumpfs = NULL;
}
