/*
 * grph.c - graph support
 */

/*
 * Copyright 1994 Victor A. Abell, Lafayette, Indiana  47906.  All rights
 * reserved.
 *
 * Written by Victor A. Abell.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Victor A. Abell is not responsible for any consequences of the use of
 * this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to Victor A. Abell must
 *    appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#if	!defined(lint)

# if	defined(_BCC)
#pragma warn -use
# endif

static char copyright[] =
"@(#) Copyright 1994 Victor A. Abell.\nAll rights reserved.\n";
#endif

#include "touch2.h"
#include <ctype.h>
#include <math.h>

#define	FIELDLN	16			/* dump line field length */
#define GPFX	31			/* graph prefix column count */
#define STARTPX	4			/* starting VGA y pixel */

short Bt;				/* beginning time of RangeInUse */
char Date[FIELDLN];			/* dump line date */
char Dow[FIELDLN];			/* dump line day of week */
short Et;				/* ending time of RangeInUse */
char Event;				/* dump line event value */
char Eventfilt = '\0';			/* event filter */
double Gmaxl;				/* graph maximum line */
double Gminl;				/* graph minimum line */
short Lpp = LPP;			/* lines per page */
int Maxx;				/* maximum VGA x coordinate */
int Maxy;				/* maximum VGA y coordinate */
double Rval;				/* dump line reading value */
char *TDumpLp[DUMPNL];			/* temporary dump line pointers */
char Time[FIELDLN];			/* dump line time */
short TimeB;				/* time converted to binary */
int TxtHt;				/* VGA text height */
int TxtWid;				/* VGA text width */

char *MeasName[] = { "Regular", "Check Solution (C)", "Check Strip (!)",
		     "Meter Error (*)" };

static int CheckRange(void);


/*
 * CheckRange() - check for time within range
 */

static int
CheckRange(void)
{
	if (Bt < Et) {
		if (TimeB >= Bt && TimeB < Et)
			return(1);
		return(0);
	}
	if (TimeB >= Bt || TimeB < Et)
		return(1);
	return(0);
}


/*
 * DrawGraph() - draw a graph of meter memory values
 */

void
DrawGraph(ty)
	int ty;				/* type: 0 = screen, 1 = file */
{

#if	defined(UNIX)
	int i;
#else
	short i;
#endif

	char ch, *cp1, *cp2, gttl[2][82], hdr[5][72], ln[82], pr[80];
	short dtla, dtlu, ska, sku, tmla, tmlu;
	short err, gmn, gmx, j, k, lpp, lps, mmol, n[4], nl, pc, pps, prow, row;
	double fn, dj, dk, iv, max[4], min[4], mn, mx, sumx[4], sumxx[4];
	char pca[PRTRCTLLN+1], pcb[PRTRCTLLN+1];
	int gdrv, gmode, hth, htw, pcal, pcbl;

#if	defined(UNIX)
	int g, gt, pcol;
#else
	short g, gt, pcol;
#endif

/*
 * If graphing to screen, enter high resolution VGA mode.
 */
	if ( !ty) {
		gdrv = VGA;
		gmode = VGAMED;
		registerbgidriver(EGAVGA_driver);
		initgraph(&gdrv, &gmode, "");
		if (graphresult() != 0) {
		    (void) WarnMsg(10, 20,
			"The graph function requires high resolution VGA.",
			12, 20,
			"This display does not support that mode.", 0);
		    return;
		}
		setbkcolor((int)(BkClrx & 15));
		setcolor((int)(TxtClrx & 15));
		Vmode = 1;
		Maxx = getmaxx();
		Maxy = getmaxy();
		TxtHt = textheight("H");
		TxtWid = textwidth("M");
	}
/*
 * Miscellaneous setup.
 */
	for (j = 0; j < 4; j++) {
		max[j] = 0.0;
		min[j] = 9999.0;
		n[j] = 0;
		sumx[j] = sumxx[j] = 0.0;
	}
	if (!ty) {
		hth = (TxtHt + 1) / 2;
		htw = TxtWid / 2;
		pps = Maxy - (2 * TxtHt) - 4;
		lps = pps / (TxtHt + 1);
	} else {
		pps = STARTPX+1;
		TxtHt = -1;
	}
	if (RangeInUse > 0) {
		Bt = TmRange[RangeInUse - 1].bt;
		Et = TmRange[RangeInUse - 1].et;
	}
	sku = SkipUnt;
	if (SkipAft) {
		dtla = strlen(SkipAftDt);
		tmla = strlen(SkipAftTm);
	}
	if (SkipUnt) {
		dtlu = strlen(SkipUntDt);
		tmlu = strlen(SkipUntTm);
	}
/*
 * Accumulate statistics values.
 */
	for (err = mmol = ska = 0, i = nl = DumpHs; i < DumpLc; i++) {
		if ((j = ParseDumpLn(DumpLp[i], 0)) == 0) {
		    err = 1;
		    if ((char)WarnMsg(10, 34, "Bad dump line:", 12,
			(short)(((Vc.screenwidth - strlen(DumpLp[i]))/2)+1),
			DumpLp[i], 1)
		    == ESC) {
graph_exit:
			if ( ! ty) {
				closegraph();
				Vmode = 0;
			}
		    	return;
		    }
		    continue;
		}
		if (j == 2)
			mmol = 1;
		if (j > 3)
			continue;
	/*
	 * Apply filters.
	 */
		if (ska)
			continue;
		if (sku) {
		    if (strncmp(Date, SkipUntDt, dtlu) != 0
		    ||  strncmp(Time, SkipUntTm, tmlu) != 0)
			continue;
		    sku = 0;
		}
		if (SkipAft) {
		    if (strncmp(Date, SkipAftDt, dtla) == 0
		    &&  strncmp(Time, SkipAftTm, tmla) == 0)
			ska = 1;
		}
		if (RangeInUse > 0 && CheckRange() == 0)
			continue;
		if (Eventfilt && (Event != Eventfilt))
			continue;
		if (Event >= '0' && Event <= '9' && IgnEvt[Event - '0'])
			continue;
	/*
	 * Graph this record.
	 */
		TDumpLp[nl++] = DumpLp[i];
		sumx[Rtype] += Rval;
		sumxx[Rtype] += Rval * Rval;
		n[Rtype]++;
		if (max[Rtype] < Rval)
			max[Rtype] = Rval;
		if (min[Rtype] > Rval)
			min[Rtype] = Rval;
	}
	if (err)
		goto graph_exit;
/*
 * Form statistics lines.
 */
	(void) sprintf(&hdr[0][0], "          %-18s %7s %6s %8s %6s %6s",
		"", "Number", "Mean", "Std Dev", "Min", "Max");
	for (j = 0; j < 4; j++) {
		if ( ! n[j]) {
			hdr[j+1][0] = '\0';
			continue;
		}
		fn = (float)n[j];
		(void) sprintf(&hdr[j+1][0],
			"          %-18s %7d %6.1f %8.2f %6.1f %6.1f",
			MeasName[j], n[j], sumx[j]/fn,
			sqrt(sumxx[j]/fn - (sumx[j]/fn * sumx[j]/fn)),
			min[j], max[j]);
	}
/*
 * Establish graph parameters.
 */
	if ( ! Lineval) {
		if (mmol) {
			Gmaxl = DEFMMMAX;
			Gminl = DEFMMMIN;
		} else {
			Gmaxl = DEFMGMAX;
			Gminl = DEFMGMIN;
		}
	}
	mn = (Gminl < min[0]) ? Gminl : min[0];
	mx = (Gmaxl > max[0]) ? Gmaxl : max[0];
	dk = (double)(((Vc.screenwidth > 80) ? 80 : Vc.screenwidth) - GPFX - 2);
	if ((iv = (mx - mn) / dk) < 0.1)
		iv = 0.1;
	gmn = (int)((Gminl - mn) / iv) + GPFX;
	gmx = (int)((Gmaxl - mn) / iv) + GPFX;
	(void) sprintf(&gttl[0][0], "%.3f %s per column", iv,
		mmol ? "mmol/l" : "mg/dl");
	for (j = strlen(&gttl[0][0]); j < GPFX; j++)
		gttl[0][j] = ' ';
 	for (j = 0; j < GPFX; j++)
		gttl[1][j] = ' ';
	cp1 = &gttl[0][GPFX-4];
	cp2 = &gttl[1][GPFX];
	for (dj = mn; dj < mx; ) {
		(void) sprintf(cp1, "%5.1f", dj);
		cp1 += 5;
		*cp2++ = '+';
		*cp2 = '\0';
		dj += iv;
		for (k = 0; k < 5 && dj <= mx; k++) {
			if (k < 1) {
				*cp1++ = ' ';
				*cp1 = '\0';
			}
			*cp2++ = '-';
			*cp2 = '\0';
			dj += iv;
		}
	}
/*
 * If drawing graph to file, create printer control strings.
 */
	if (ty) {
		if (AftGraph)
		    pcal = CvtPrtrStr(AftGraph, pca, sizeof(pca));
		if (BefGraph) {
		    if ((pcbl = CvtPrtrStr(BefGraph, pcb, sizeof(pcb))) > 0)
			(void) fwrite((void *)pcb, (size_t)pcbl, 1, Graphfs);
		}
		lpp = Lpp;
		pc = 0;
	}
/*
 * Refill screen.
 */
	for (j = pcol = 0;;) {
		if ( ! ty) {
			pcol = 0;
			cleardevice();
			(void) sprintf(pr,
				"(%d of %d) Press ESC or X to exit; Page Up/Down; Arrow Up/Down.",
				(j < 9) ? 1 : j - 9 + 1, nl - DumpHs);
			PromptMsg(pr);
		}
		for (prow = row = STARTPX, k = j;
		     row < pps && k < nl+9-DumpHs;
		     k++, row += (TxtHt + 1))
		{
			if ( ! ty)
				moveto(0, row);
			switch (k) {
			case 0:
				if ( ! ty)
					outtext(Exttl);
				break;

			case 1:
			case 2:
			case 3:
			case 4:
			case 5:
				if ( ! ty)
					outtext(&hdr[k-1][0]);
				break;
			case 6:
				break;
			case 7:
			case 8:
				if ( ! ty)
					outtext(&gttl[k-7][0]);
				break;
			default:
				(void) ParseDumpLn(TDumpLp[k-9+DumpHs], 1);
				if (Rtype == RDREG)
				    g = (int)((Rval - mn) / iv) + GPFX;
				if (Rtype == RDHIGH) {
				    (void) sprintf(ln,
					"%-3.3s %-8.8s %-8.8s  %c HIGH",
					Dow, Date, Time,
					Event == '0' ? ' ' : Event);
				} else {
				    switch (Rtype) {
					case RDSOL:
					    ch = 'C'; break;
					case RDSTRIP:
					    ch = '!'; break;
					case RDMTRERR:
					    ch = '*'; break;
					default:
					    ch = ' ';
				    }
				    (void) sprintf(ln,
					"%-3.3s %-8.8s %-8.8s %c%c%5.1f",
					Dow, Date, Time,
					Event == '0' ? ' ' : Event,
					ch, Rval);
				}
				if (ty) {
				    if (lpp >= Lpp) {
					pc++;
					if (PcDisp) {
					    (void) fprintf(Graphfs,
						"%s\nPage %2d:  %s\n",
						pc > 1 ? "\f" : "",
						pc, Exttl);
					} else {
					    (void) fprintf(Graphfs,
						"%s\n          %s\n",
						pc > 1 ? "\f" : "",
						Exttl);
					}
					(void) fprintf(Graphfs,
					    "\n%s\n%s\n%s\n%s\n%s\n\n%s\n%s\n",
					    &hdr[0][0],
					    &hdr[1][0],
					    &hdr[2][0],
					    &hdr[3][0],
					    &hdr[4][0],
					    &gttl[0][0],
					    &gttl[1][0]);
					lpp = HDRLPP;
				    }
				    if (Rtype != RDREG) {
					gt = gmx;
					g = GPFX-2;
				    } else
				        gt = (g > gmx) ? g : gmx;
				    ln[gt+1] = '\0';
				    for (; gt >= GPFX-2; gt--) {
					if (gt == gmn || gt == gmx) {
					    ln[gt] = (gt <= g) ? IntCh[0]
							       : LineCh[0];
					} else if (gt >= GPFX && gt <= g)
					    ln[gt] = BarCh[0];
					else
					    ln[gt] = ' ';
				    }
				    (void) fprintf(Graphfs, "%s\n", ln);
				    lpp++;
				} else {
					outtext(ln);
					if (gmn)
						outtextxy((gmn + 1) * 8, row,
							LineCh);
					if (gmx)
						outtextxy((gmx + 1) * 8, row,
							LineCh);
					if (Rtype != RDREG)
						break;
					if (pcol) {
						moveto((pcol * TxtWid) + htw,
							prow + hth);
						lineto((g * TxtWid) + htw,
							row + hth);
					}
					pcol = g;
					prow = row;
				}
			}
		}
	/*
	 * See if done graphing to file.
	 */
		if (ty) {
			if (k >= nl+9-DumpHs) {
				if (pcal > 0) {
					(void) fwrite((void *)pca,
						(size_t)pcal, 1, Graphfs);
				}
				(void) fclose(Graphfs);
				Graphfs = NULL;
				return;
			}
			j = k;
			continue;
		}
	/*
	 * Wait for keyboard input.
	 */
		for (k = 1; k;) {
			switch ((char)WaitAnyKey()) {
			case ESC:
			case 'x':
			case 'X':
				goto graph_exit;
			case PGDN:
				if ((j + lps) < (nl+9-DumpHs)) {
					j += lps;
					k = 0;
				} else
					putch(BELL);
				break;
			case PGUP:
				if (j < lps)
					j = 0;
				else
					j = j - lps;
				k = 0;
				break;
			case UARW:
				if (j) {
					j--;
					k = 0;
				} else
					putch(BELL);
				break;
			case DARW:
				if (j < (nl+9-DumpHs-1)) {
					j++;
					k = 0;
				} else
					putch(BELL);
				break;
			default:
				putch(BELL);
			}
		}
	}
}


/*
 * ParseDumpLn() - parse dump line
 */

int
ParseDumpLn(dl, s)
	char *dl;			/* dump line pointer */

#if	defined(UNIX)
	int s;				/* seconds flag */
#else
	short s;			/* seconds flag */
#endif

{
	char *cp, *cp1, msg[128], r[FIELDLN];
	int t;
	short rv;
/*
 * Parse day of week, date, time and reading.
 */
	if ((cp = ParseField(dl, Dow, sizeof(Dow))) == NULL)
		return(0);
	if ((cp = ParseField(cp, Date, sizeof(Date))) == NULL)
		return(0);
	if ((cp = ParseField(cp, Time, sizeof(Time))) == NULL)
		return(0);
	if ((cp = ParseField(cp, r, sizeof(r))) == NULL)
		return(0);
/*
 * Get event number.
 */
	if (*cp++ != ',')
		return(0);
	while (*cp && *cp == ' ')
		cp++;
	if ( ! *cp)
		return(0);
	Event = *cp;
/*
 * Convert last two characters of day of week to lower case.
 */
	if (Dow[1] && isascii(Dow[1]) && isupper(Dow[1]))
		Dow[1] = tolower(Dow[1]);
	if (Dow[2] && isascii(Dow[2]) && isupper(Dow[2]))
		Dow[2] = tolower(Dow[2]);
/*
 * Shorten the time by eliminating the :00 seconds.
 */
	if (s) {
		if (strncmp(&Time[5], ":00", 3) == 0) {
			for (cp = &Time[5];; cp++) {
				if ((*cp = *(cp+3)) == '\0')
					break;
			}
		}
	}
/*
 * If a time range specification is in use, convert the time
 * character string to binary.
 */
	if (RangeInUse > 0) {
		cp1 = Time;
		if ((t = CvtTm(&cp1, msg)) < 0)
			return(0);
		TimeB = (short)t;
	}
/*
 * Get reading type and convert the value.
 */
	cp = r;
	rv = 1;
	switch (*cp) {
		case ' ':
		case 'M':
			if (strcmpi(cp, " HIGH ") == 0) {
				Rtype = RDHIGH;
				return(4);
			}
			Rtype = RDREG;
			break;
		case '!':
			Rtype = RDSTRIP;
			break;
		case 'C':
		case 'K':			/* SVENS || DEUTS */
			Rtype = RDSOL;
			break;
		default:
			return(0);
	}
/*
 * Parse value.
 */
	cp++;
	if (*cp == 'M') {
		rv = 2;
		cp++;
	} else
		rv = 1;
	if ( ! Atof(cp, &Rval, NULL, cp)) {
		if (*cp == '?') {
			Rtype = RDMTRERR;
			return(3);
		}
		Rval = 0.0;
		return(0);
	}
	return(rv);
}
