/*
 * stat.c - manage meter status
 *
 * V. Abell
 */

/*
 * Copyright 1994 Victor A. Abell, Lafayette, Indiana  47906.  All rights
 * reserved.
 *
 * Written by Victor A. Abell.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Victor A. Abell is not responsible for any consequences of the use of
 * this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to Victor A. Abell must
 *    appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#if	!defined(lint)

# if	defined(_BCC)
#pragma warn -use
# endif

static char copyright[] =
"@(#) Copyright 1994 Victor A. Abell.\nAll rights reserved.\n";
#endif

#include "touch2.h"
#include <string.h>
#include <ctype.h>

struct menu BaudMenu[] = {
	{ 12, 19, "Changing the baud rate is not permitted." },
	{  0,  0,  NULL },
};

struct menu CmdFail[] = {
	{ 12, 32, "Command failed" },
	{  0,  0, NULL },
};

struct menu McommMenu[] = {
	{ 12, 14, "Changing the communications mode is not permitted." },
	{  0,  0,  NULL },
};

struct menu StatusType[] = {
	{  2, 25, "Inspect or change meter status." },
	{ 10, 25, "C - Change meter status" },
	{ 12, 25, "I - Inspect meter status" },
	{ 14, 25, "X - eXit" },
	{  0,  0, NULL},
};

#define BEEPLN		8
#define	COL		25
#define	CALLN		BEEPLN+1
#define	DTLN		CALLN+1
#define	LANLN		DTLN+1
#define COMMLN		LANLN+1
#define	PUNLN		COMMLN+1
#define	BAUDLN		PUNLN+1
#define STATLN		16
#define	TMLN		BAUDLN+1
#define	UNLN		TMLN+1
#define XLN		UNLN+1


char Mbaud[STATLN];			/* baud rate */
char Mbeep[STATLN];			/* beeper */
char Mcal[STATLN];			/* calibration */
char Mcomm[STATLN];			/* communications mode */
char Mdate[STATLN];                     /* date format */
char Mlang[STATLN];			/* language */
char Mmax[STATLN];			/* maximum reading */
char Mmin[STATLN];			/* minimum reading */
char Mpunc[STATLN];			/* MMOL/L punctuation */
char Mserial[STATLN];			/* meter serial number */
char Mtime[STATLN];			/* time format */
char Munits[STATLN];			/* MMOL/L units */
short Ndump = 0;			/* number of dump lines */

static int ChangeBeep(void);
static int ChangeCal(void);
static int ChangeDate(void);
static int ChangeLang(void);
static int ChangeMcomm(void);
static int ChangePunc(void);
static int ChangeBaud(void);
static int ChangeTime(void);
static int ChangeUnits(void);

#if	defined(UNIX)
static void CopyReply(char *s, char *d, int dl);
#else
static void CopyReply(char *s, char *d, short dl);
#endif

#if	defined(UNIX)
static int ReadVal(int c, char *b, int bl);
#else
static int ReadVal(char c, char *b, short bl);
#endif

static void StatusCh(void);
static void StatusIn(void);
static void StatusMenu(char *t, char *b);

#if	defined(UNIX)
static int WriteVal(char *c, char *r, int rl);
#else
static int WriteVal(char *c, char *r, short rl);
#endif


/*
 * ChangeBaud() - change baud rate (not permitted)
 */

static int
ChangeBaud(void)
{
	DispMenu(BaudMenu, "Press any key to continue.");
	(void) WaitAnyKey();
	return(1);
}


/*
 * ChangeBeep() - change beep status
 */

static int
ChangeBeep(void)
{
	char b[8], r[STATLN];

	for (;;) {
		clrscr();
		if (GetInp(12, 10, "Beep or nobeep?", "", b, sizeof(b)) == 0)
			return(0);
		if (strcmpi(b, "beep") == 0) {
			b[1] = '0';
			break;
		} else if (strcmpi(b, "nobeep") == 0) {
			b[1] = '1';
			break;
		}
		putch(BELL);
	}
	*b = 'B';
	b[2] = '\0';
	if (WriteVal(b, r, sizeof(r))) {
		(void) strcpy(Mbeep, r);
		return(1);
	}
	return(0);
}


/*
 * ChangeCal() - change the strip lot calibration code
 */

static int
ChangeCal(void)
{
	char b[4], r[STATLN];
	int n;

	for (;;) {
		clrscr();
		if (GetInp(12, 10, "1 through 16?", "", b, sizeof(b)) == 0)
			return(0);
		n = atoi(b);
		if (n >= 1 && n <= 16)
			break;
		putch(BELL);
	}
	*b = 'S';
	b[1] = (n < 11) ? (char)(n - 1 + '0') : (char)(n - 11 + 'A');
	b[2] = '\0';
	if (WriteVal(b, r, sizeof(r))) {
		(void) strcpy(Mcal, r);
		return(1);
	}
	return(0);
}


/*
 * ChangeDate() - change date format
 */

static int
ChangeDate(void)
{
	char b[8], r[STATLN];

	for (;;) {
		clrscr();
		if (GetInp(12, 10, "M.D.Y or D.M.Y?", "", b, sizeof(b)) == 0)
			return(0);
		if (strcmpi(b, "m.d.y") == 0) {
			b[1] = '0';
			break;
		} else if (strcmpi(b, "d.m.y") == 0) {
			b[1] = '1';
			break;
		}
		putch(BELL);
	}
	*b = 'D';
	b[2] = '\0';
	if (WriteVal(b, r, sizeof(r))) {
		(void) strcpy(Mdate, r);
		return(1);
	}
	return(0);
}


/*
 * ChangeLang() - change meter language
 */

static int
ChangeLang(void)
{
	char b[8], r[STATLN];

	for (;;) {
		clrscr();
		if (GetInp(12, 10,
			"ENGL|ESPAN|FRANC|ITALII|NEDER|PORT|SVENS|DEUTS|SYMB?",
			"", b, sizeof(b)) == 0)
			return(0);
		if (strcmpi(b, "engl") == 0) {
			b[1] = '0';
			break;
		} else if (strcmpi(b, "espan") == 0) {
			b[1] = '1';
			break;
		} else if (strcmpi(b, "franc") == 0) {
			b[1] = '2';
			break;
		} else if (strcmpi(b, "itali") == 0) {
			b[1] = '3';
			break;
		} else if (strcmpi(b, "neder") == 0) {
			b[1] = '4';
			break;
		} else if (strcmpi(b, "port") == 0) {
			b[1] = '5';
			break;
		} else if (strcmpi(b, "svens") == 0) {
			b[1] = '6';
			break;
		} else if (strcmpi(b, "deuts") == 0) {
			b[1] = '7';
			break;
		} else if (strcmpi(b, "symb") == 0) {
			b[1] = '8';
			break;
		}
		putch(BELL);
	}
	*b = 'L';
	b[2] = '\0';
	if (WriteVal(b, r, sizeof(r))) {
		(void) strcpy(Mlang, r);
		Mbeep[0] = Mcal[0] = Mpunc[0] ='\0';
		return(1);
	}
	return(0);
}


/*
 * ChangeMcomm() - change the communications mode (not permitted)
 */

static int
ChangeMcomm(void)
{
	DispMenu(McommMenu, "Press any key to continue.");
	(void) WaitAnyKey();
	return(1);
}


/*
 * ChangePunc() - change punctuation
 */

static int
ChangePunc(void)
{
	char b[8], r[STATLN];

	for (;;) {
		clrscr();
		if (GetInp(12, 10, "DEC PT or COMMA?", "", b, sizeof(b)) == 0)
			return(0);
		if (strcmpi(b, "dec pt") == 0) {
			b[1] = '0';
			break;
		} else if (strcmpi(b, "comma") == 0) {
			b[1] = '1';
			break;
		}
		putch(BELL);
	}
	*b = 'P';
	b[2] = '\0';
	if (WriteVal(b, r, sizeof(r))) {
		(void) strcpy(Mpunc, r);
		return(1);
	}
	return(0);
}


/*
 * ChangeTime() - change time format
 */

static int
ChangeTime(void)
{
	char b[8], r[STATLN];

	for (;;) {
		clrscr();
		if (GetInp(12, 10, "AM/PM or 24:00?", "", b, sizeof(b)) == 0)
			return(0);
		if (strcmpi(b, "am/pm") == 0) {
			b[1] = '0';
			break;
		} else if (strcmp(b, "24:00") == 0) {
			b[1] = '1';
			break;
		}
		putch(BELL);
	}
	*b = 'T';
	b[2] = '\0';
	if (WriteVal(b, r, sizeof(r))) {
		(void) strcpy(Mtime, r);
		return(1);
	}
	return(0);
}


/*
 * ChangeUnits() - change units
 */

static int
ChangeUnits(void)
{
	char b[8], r[STATLN];

	for (;;) {
		clrscr();
		if (GetInp(12, 10, "MG/DL or MMOL/L?", "", b, sizeof(b)) == 0)
			return(0);
		if (strcmpi(b, "mg/dl") == 0) {
			b[1] = '0';
			break;
		} else if (strcmpi(b, "mmol/l") == 0) {
			b[1] = '1';
			break;
		}
		putch(BELL);
	}
	*b = 'U';
	b[2] = '\0';
	if (WriteVal(b, r, sizeof(r))) {
		(void) strcpy(Munits, r);
		return(1);
	}
	return(0);
}


/*
 * CopyReply() - copy command reply
 */

static void
CopyReply(s, d, dl)
	char *s;			/* source */
	char *d;			/* destination */
	short dl;			/* destination limit */
{
	short i;

	for (i = 0; i < dl - 1; i++) {
		if (*s == '\0' || *s == '"')
			break;
		*d++ = *s++;
	}
	*d = '\0';
}


/*
 * InitStatus() - initialize status
 */

void
InitStatus(void)
{
	Mbaud[0] = Mbeep[0] = Mcal[0] = Mcomm[0] = Mdate[0] = '\0';
	Mlang[0] = Mpunc[0] = Mserial[0] = Mtime[0] = Munits[0] = '\0';
}


/*
 * ParseField() - parse '"' delimited field
 */

char *
ParseField(p, b, l)
	char *p;			/* field pointer */
	char *b;			/* destination buffer */
	int l;				/* destination length */
{
	int i;

	if ((p = strchr(p, '"')) == NULL)
		return(NULL);
	for (p++, i = 0; *p; p++, i++) {
		if (*p == '"')
			break;
		if (i > (l - 2))
			return(NULL);
		b[i] = *p;
	}
	b[i] = '\0';
	return(++p);
}


/*
 * ParseHdr() - parse dump header
 */

int
ParseHdr(void)
{
	char *cp;
	int i;

	DumpHs = 0;
/*
 * Get record count.
 */
	if ((cp = strchr(DumpLine, ' ')) == NULL)
		return(0);
	while(*cp == ' ') {
		cp++;
	}
	for (Ndump = 0; *cp; cp++) {
		if ( ! isdigit(*cp))
			break;
		Ndump = Ndump * 10 + *cp - '0';
	}
/*
 * Get serial number.
 */
	if ((cp = ParseField(cp, Mserial, sizeof(Mserial))) == NULL)
		return(0);
/*
 * Get language code.
 */
	if ((cp = ParseField(cp, Mlang, sizeof(Mlang))) == NULL)
		return(0);
/*
 * Get date format.
 */
	if ((cp = ParseField(cp, Mdate, sizeof(Mdate))) == NULL)
		return(0);
/*
 * Get time format.
 */
	if ((cp = ParseField(cp, Mtime, sizeof(Mtime))) == NULL)
		return(0);
/*
 * Get units format.
 */
	if ((cp = ParseField(cp, Munits, sizeof(Munits))) == NULL)
		return(0);
/*
 * Get minimum reading.
 */
	if ((cp = ParseField(cp, Mmin, sizeof(Mmin))) == NULL)
		return(0);
/*
 * Get maximum reading.
 */
	if ((cp = ParseField(cp, Mmax, sizeof(Mmax))) == NULL)
		return(0);
	DumpHs = 1;
	return(1);
}


/*
 * ReadVal() - read meter status value
 */

static int
ReadVal(c, b, bl)

#if	defined(UNIX)
	int c;				/* value code */
	char *b;			/* destination buffer */
	int bl;				/* buffer length */
#else
	char c;				/* value code */
	char *b;			/* destination buffer */
	short bl;			/* buffer length */
#endif

{
	char cmd[5], *cp;

	(void) sprintf(cmd, "DMS%c?", c);
	*b = '\0';
	for (;;) {
		if (WaitRdy() == 0)
			return(0);
		if (WaitCmd(cmd, c)) {
			(void) GetDataLn(DumpLine, DUMPLL);
			if ((cp = strchr(DumpLine, '"')) == NULL)
				return(0);
			break;
		}
		DispMenu(CmdFail,
			"Press ESC to exit; any other key to retry.");
		if ((char)WaitAnyKey() == ESC)
			return(0);
	}
	CopyReply(++cp, b, bl);
	return(1);
}


/*
 * StatusCh() - change meter status
 */
static void
StatusCh(void)
{
	int ch;

	StatusMenu("Change meter status", "Press ESC to exit.");
	for (;;) {
		if ( ! kbhit()) {
			AsynRstBf();
			continue;
		}
		ch = getch();
		switch (ch) {

		case 'b':
		case 'B':
			if ( ! ChangeBeep())
				return;
			break;
		case 'c':
		case 'C':
			if ( ! ChangeCal())
				return;
			break;
		case 'd':
		case 'D':
			if ( ! ChangeDate())
				return;
			break;
		case 'l':
		case 'L':
			if ( ! ChangeLang())
				return;
			break;
		case 'm':
		case 'M':
			if ( ! ChangeMcomm())
				return;
			break;
		case 'p':
		case 'P':
			if ( ! ChangePunc())
				return;
			break;
		case 'r':
		case 'R':
			if ( ! ChangeBaud())
				return;
			break;
		case 't':
		case 'T':
			if ( ! ChangeTime())
				return;
			break;
		case 'u':
		case 'U':
			if ( ! ChangeUnits())
				return;
			break;
		case 'x':
		case 'X':
		case ESC:
			return;
		default:
			if (ch == 0)
				ch = getch();
			putch(BELL);
		}
		StatusMenu("Change meter status", "Press ESC to exit.");
	}
}


/*
 * StatusIn() - inspect meter status
 */

static void
StatusIn(void)
{
	int ch;

	StatusMenu("Inspect meter status", "Press ESC to exit.");
	for (;;) {
		if ( ! kbhit()) {
			AsynRstBf();
			continue;
		}
		ch = getch();
		switch (ch) {

		case 'b':
		case 'B':
			if ( ! ReadVal('B', Mbeep, sizeof(Mbeep)))
				return;
			break;
		case 'c':
		case 'C':
			if ( ! ReadVal('S', Mcal, sizeof(Mcal)))
				return;
			break;
		case 'd':
		case 'D':
			if ( ! ReadVal('D', Mdate, sizeof(Mdate)))
				return;
			break;
		case 'l':
		case 'L':
			if ( ! ReadVal('L', Mlang, sizeof(Mlang)))
				return;
			break;
		case 'm':
		case 'M':
			if ( ! ReadVal('C', Mcomm, sizeof(Mcomm)))
				return;
			break;
		case 'p':
		case 'P':
			if ( ! ReadVal('P', Mpunc, sizeof(Mpunc)))
				return;
			break;
		case 'r':
		case 'R':
			if ( ! ReadVal('R', Mbaud, sizeof(Mbaud)))
				return;
			break;
		case 't':
		case 'T':
			if ( ! ReadVal('T', Mtime, sizeof(Mtime)))
				return;
			break;
		case 'u':
		case 'U':
			if ( ! ReadVal('U', Munits, sizeof(Munits)))
				return;
			break;
		case 'x':
		case 'X':
		case ESC:
			return;
		default:
			if (ch == 0)
				ch = getch();
			putch(BELL);
		}
		StatusMenu("Inspect meter status", "Press ESC to exit.");
	}
}


/*
 * StatusMenu() - display status menu
 */

static void
StatusMenu(ttl, bot)
	char *ttl;			/* menu title */
	char *bot;			/* text for bottom line */
{
	clrscr();
	gotoxy(COL, 2);
	(void) cputs(ttl);
	if (Mserial[0]) {
		gotoxy(COL+1, 3);
		cprintf("(serial: %s)", Mserial);
	}
	PromptMsg(bot);
	ClearRow(BEEPLN, COL);
	gotoxy(COL, BEEPLN);
	cprintf("B - Beeper status: %s", Mbeep);
	ClearRow(CALLN, COL);
	gotoxy(COL, CALLN);
	cprintf("C - strip lot Calibration code: %s", Mcal);
	ClearRow(DTLN, COL);
	gotoxy(COL, DTLN);
	cprintf("D - Date format: %s", Mdate);
	ClearRow(LANLN, COL);
	gotoxy(COL, LANLN);
	cprintf("L - message and prompt Language: %s", Mlang);
	ClearRow(COMMLN, COL);
	gotoxy(COL, COMMLN);
	cprintf("M - communications Mode: %s", Mcomm);
	ClearRow(PUNLN, COL);
	gotoxy(COL, PUNLN);
	cprintf("P - MMOL/L Punctuation: %s", Mpunc);
	ClearRow(BAUDLN, COL);
	gotoxy(COL, BAUDLN);
	cprintf("R - baud Rate: %s", Mbaud);
	ClearRow(TMLN, COL);
	gotoxy(COL, TMLN);
	cprintf("T - Time display format: %s", Mtime);
	ClearRow(UNLN, COL);
	gotoxy(COL, UNLN);
	cprintf("U - glucose Units: %s", Munits);
	gotoxy(COL, XLN);
	(void) cputs("X - eXit");
}


/*
 * StatusMtr() - inspect/change meter status
 */

void
StatusMtr(void)
{
	int ch;

	DispMenu(StatusType, NULL);
	for (;;) {
		if ( !kbhit()) {
			AsynRstBf();
			continue;
		}
		ch = getch();
		switch(ch) {

		case 'c':
		case 'C':
			StatusCh();
			break;
		case 'i':
		case 'I':
			StatusIn();
			break;
		case 'x':
		case 'X':
		case ESC:
			return;
		default:
			if (ch == 0)
				ch = getch();
			putch(BELL);
		}
		DispMenu(StatusType, NULL);
	}
}


/*
 * WriteVal() - write meter status value
 */

int
WriteVal(c, r, rl)
	char *c;			/* command */
	char *r;			/* result */

#if	defined(UNIX)
	int rl;				/* result buffer length */
#else
	short rl;			/* result buffer length */
#endif

{
	char cmd[6], *cp;

	(void) sprintf(cmd, "DMS%c%c", *c, *(c+1));
	*r = '\0';
	for (;;) {
		if (WaitRdy() == 0)
			return(0);
		if (WaitCmd(cmd, *c)) {
			(void) GetDataLn(DumpLine, DUMPLL);
			if ((cp = strchr(DumpLine, '"')) == NULL)
				return(0);
			break;
		}
		DispMenu(CmdFail,
			"Press ESC to exit; any other key to retry.");
		if ((char)WaitAnyKey() == ESC)
			return(0);
	}
	CopyReply(++cp, r, rl);
	return(1);
}
