/*
 * touch2.c - main program for LifeScan One Touch(R) II glucose meter
 * communication interface
 *
 * V. Abell
 */

/*
 * Copyright 1994 Victor A. Abell, Lafayette, Indiana  47906.  All rights
 * reserved.
 *
 * Written by Victor A. Abell.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Victor A. Abell is not responsible for any consequences of the use of
 * this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to Victor A. Abell must
 *    appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#if	!defined(lint)

# if	defined(_BCC)
#pragma warn -use
#endif

static char copyright[] =
"@(#) Copyright 1994 Victor A. Abell.\nAll rights reserved.\n";
#endif

#include "touch2.h"
#include "version.h"
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

struct menu ZeroFail[] = {
	{ 12, 14, "The command to zero the datalog may have failed." },
	{  0,  0, NULL },
};

struct menu ZeroMenu[] = {
	{  4, 29, "Zero the meter data log." },
	{  8, 26, "Are you sure you want to do this?" },
	{ 12, 23, "Please press Y (upper case) to confirm." },
	{ 13, 20, "Press any other key to exit without clearing." },
	{  0,  0, NULL },
};

struct menu MainMenu[] = {
	{  2, 13, "LifeScan One Touch(R) II Meter Communications Interface" },
	{  3, 29, NULL },
#define	VERSLN	1
	{  9, 18, "C - inspect/change Communications parameters" },
	{ 10, 18, "D - Dump memory" },
	{ 11, 18, "S - inspect/display meter Status" },
	{ 12, 18, "T - read/set date and Time" },
	{ 13, 18, "X - eXit" },
	{ 14, 18, "Z - Zero the meter datalog" },
	{ 20,  8, "Copyright (C) Victor A. Abell, 1994.  All rights reserved."},
	{ 21, 13, "1715 Summit Dr., W. Lafayette, IN (317) 463-6865" },
	{  0,  0, NULL },
};

int AttrNorm;				/* normal color attributes */
int AttrPrompt;				/* prompt color attributes */
char BarCh[] = { BARCH, '\0'};		/* bar graph character */
short BkClrx =  DEFBCX;			/* background color */
short Ckdump = 0;			/* check dump status */
short Cksum = 1;			/* checksum checking status */
short Debug = 0;			/* Debug mode */
struct text_info InitTextInfo;		/* initial text information */
char IntCh[] = { INTCH, '\0' };		/* bar and min/max intersection
					 * character */
char LineCh[] = { LINECH, '\0'};	/* min/max level line character */
short PcDisp = 1;			/* page count display state */
char *Pn;				/* program name */
short Qp = 0;				/* QuattroPro output mode */
short TxtClrx = DEFTCX;			/* text color */
short Warn = 1;				/* display warning status */

static void InitClr(void);
static void ParseArgs(int argc, char *argv[], int err);

#if	defined(UNIX)
static void SigHandle(int s);
#else
static void SigHandle(void);
#endif

static void Usage(void);
static void ZeroLog(void);


extern char *optarg;
extern int optind;


/*
 * main() - main program
 */

main(argc, argv )
	int argc;			/* argument count */
	char *argv[];			/* argument pointers */
{
	int err, i, vl;
	char *v;

/*
 * Save program name.
 */
	if ((Pn = strrchr(argv[0], '/')) != NULL)
		Pn++;
	else {
#if	!defined(UNIX)
		if ((Pn = strchr(argv[0], '\\')) != NULL)
			Pn++;
		else
#endif
		Pn = argv[0];
	}
/*
 * Construct version identification.
 */
	vl = strlen("Version ") + strlen(VN) + strlen(VNS) + strlen(", ")
	  + strlen(VDT);
	if ((v = (char *)malloc(vl + 1)) == NULL) {
		(void) fprintf(stderr,
			"%s: no space for version information\n", Pn);
		exit(1);
	}
	(void) sprintf(v, "Version %s%s, %s", VN, VNS, VDT);
	MainMenu[VERSLN].msg = v;
/*
 * Set up dump parameters.
 * Read configuration file.
 * Parse arguments (options) from command line.
 * Set up:
 *	menuing,
 *	display,
 *	meter status variables,
 *	signal handling.
 */
	InitDump();
	err = ! ReadConf();
	(void) ParseArgs(argc, argv, err);
	if (ExpandTtl()) {
		(void) fprintf(stderr, "%s: graph title error: %s\n",
			Pn, TtlErrMsg);
		Usage();
		exit(1);
	}
	gettextinfo(&InitTextInfo);
	InitMenu();
	InitClr();
	InitStatus();
	(void) signal(SIGINT, SigHandle);
/*
 * Display the main menu and wait for responses to it.
 */
	MainMenu[VERSLN].col = (short)(((Vc.screenwidth - vl)/2) + 1);
	DispMenu(MainMenu, NULL);
	for (;;) {
		if (kbhit()) {
			i = getch();
			switch(i) {
			case 'c':
			case 'C':
				DefCom();
				break;
			case 'd':
			case 'D':
				DumpMtr();
				break;
			case 's':
			case 'S':
				StatusMtr();
				break;
			case 't':
			case 'T':
				DateTime();
				break;
			case ESC:
			case 'x':
			case 'X':
				TouchExit(0);
			case 'z':
			case 'Z':
				ZeroLog();
				break;
			default:
				if (i == 0)
					i = getch();
				putch(BELL);
			}
			DispMenu(MainMenu, NULL);
		}
		AsynRstBf();
	}
}


/*
 * CloseDump() - close dump files (abnormally)
 */

void
CloseDump(void)
{
	struct stat sbuf;

	if (Dumpfs != NULL) {
		(void) fclose(Dumpfs);
		Dumpfs = NULL;
		if (stat(Qp ? DumpfnQ : DumpfnR, &sbuf) == 0
		&& sbuf.st_size == 0)
			(void) unlink(Qp ? DumpfnQ : DumpfnR);
	}
	if (Graphfs != NULL) {
		(void) fclose(Graphfs);
		Graphfs = NULL;
		if (stat(Graphfn, &sbuf) == 0 && sbuf.st_size == 0)
			(void) unlink(Graphfn);
	}
	if (Rdumpfs != NULL) {
		(void) fclose(Rdumpfs);
		Rdumpfs = NULL;
	}
}


/*
 * InitClr() - initialize colors
 */

static void
InitClr(void)
{
	char buf[80];

	clrscr();
	AttrNorm = (int)(((BkClrx & 7) << 4) | (TxtClrx & 15));
	AttrPrompt = (int)(((PromptBkClrx & 7)<<4) | (PromptTxtClrx & 15));
	textattr(AttrNorm);
	if (Vc.currmode != C80 && Warn) {
	    clrscr();
	    PromptMsg("Press ESC to exit; any other key to continue.");
	    gotoxy(15, 11);
	    (void) cputs("Warning: the display cannot be put in C80 mode.");
	    gotoxy(15, 13);
	    (void) cputs("This means that display operations, including color");
	    gotoxy(15, 14);
	    (void) cputs("selections, may be incorrect.");
	    if ((char)WaitAnyKey() == ESC)
	    	TouchExit(1);
	    clrscr();
	}
	if (Vc.currmode == C80 && BkClrx > 7 && Warn) {
	    clrscr();
	    PromptMsg("Press ESC to exit; any other key to continue.");
	    gotoxy(15, 11);
	    (void) cputs("Warning: for menus the display is in C80 mode.");
	    gotoxy(15, 13);
	    (void) cputs("Since C80 is limited to 8 background colors, the");
	    gotoxy(15, 14);
	    cprintf( "background color \"%s\" will be displayed as",
		Colors[BkClrx]);
	    gotoxy(15, 15);
	    cprintf( "\"%s\" on menus; \"%s\" will be used only on",
		Colors[BkClrx & 7], Colors[BkClrx]);
	    gotoxy(15, 16);
	    (void) cputs("graph output.");
	    if ((char)WaitAnyKey() == ESC)
	    	TouchExit(1);
	    clrscr();
	}
}


/*
 * ParseArgs() - parse command line arguments
 */

static void
ParseArgs(argc, argv, err)
	int argc;				/* argument count */
	char *argv[];				/* argument pointers */
	int err;				/* configuration file error
						 * status */
{
	int c;
	int hlp = 0;

/*
 * Process command line arguments.
 */
	while ((c = getopt(argc, argv, "a:b:B:cCdDg:G:hi:l:L:p:Pqrt:v:w"))
	!= EOF) {
		switch(c) {

		case 'a':	/* -a s -- set after graph printer control */
			if ( ! SetAftGraph(optarg, optarg))
				err = 1;
			break;
		case 'b':	/* -b char -- define bar graph character */
			if ( ! SetBarChar(optarg, optarg))
				err = 1;
			break;
		case 'c':	/* -c -- dump check, error and high readings */
			Ckdump = 1;
			break;
		case 'C':	/* -C -- don't check checksums */
			Cksum = 0;
			break;
		case 'd':	/* -d -- enable debug mode */
			Debug = 1;
			break;
		case 'D':	/* -D -- disable wait for DSR on transmit */
			DSRwait = 0;
			break;
		case 'B':	/* -B s -- set before graph printer control */
			if ( ! SetBefGraph(optarg, optarg))
				err = 1;
			break;
		case 'g':	/* -g min -- set graph minimum level value */
			if ( ! SetGmaxLvl(optarg, optarg))
				err = 1;
			break;
		case 'G':	/* -G max -- set graph maximum level value */
			if ( ! SetGminLvl(optarg, optarg))
				err = 1;
			break;
		case 'h':	/* -h -- display help */
			hlp = 1;
			break;
		case 'i':	/* -i char -- define bar and max/min
				 *	      intersection character */
			if ( ! SetIntChar(optarg, optarg))
				err = 1;
			break;
		case 'l':	/* -l char -- define max/min line character */
			if ( ! SetLineChar(optarg, optarg))
				err = 1;
			break;
		case 'L':	/* -L num -- define graph lines per page */
			if ( ! SetLpp(optarg, optarg))
				err = 1;
			break;
		case 'p':	/* -p name -- define printer */
			if ( ! SetPrinter(optarg, optarg))
				err = 1;
			break;
		case 'P':	/* -P -- turn off page count display */
			PcDisp = 0;
			break;
		case 'r':	/* -r -- turn off raw dump format */
			Qp = 1;
			break;
		case 't':	/* -t ttl -- set graph title line */
			if ( ! SetGraphTtl(optarg, optarg))
				err = 1;
			break;
		case 'v':	/* -v n,val -- define title variable n */
			if ( ! SetTtlVar(optarg, optarg))
				err = 1;
			break;
		case 'w':	/* disable display warnings */
			Warn = 0;
			break;
		case '?':
			err = 1;
		}
	}
	if (AftGraph == NULL) {
		if ( ! EntPrtStr(AFTEPSON, BEFEPSON))
			err = 1;
	}
	if ( ! TestGlev())
		err = 1;
	if (err || hlp) {
		Usage();
		if (err)
			exit(1);
		exit(0);
	}
}


/*
 * SigHandle() - handle signal
 */

static void

#if	defined(UNIX)
SigHandle(int s)
#else
SigHandle(void)
#endif

{
	TouchExit(1);
}


/*
 * TouchExit(xv) - exit
 */

void
TouchExit(xv)
	int xv;					/* exit value */
{
	CloseCom();
	CloseDump();
	if (Vmode)
		closegraph();
	textmode(InitTextInfo.currmode);
	textattr((int)InitTextInfo.attribute);
	_setcursortype(_NORMALCURSOR);
	gotoxy(InitTextInfo.screenwidth, InitTextInfo.screenheight);
	(void) putch('\n');
	exit(xv);
}


/*
 * Usage() -- display usage
 */

static void
Usage(void)
{
	(void) fprintf(stderr,
	    "%s, version %s%s, %s\n", Pn, VN, VNS, VDT);
	(void) fprintf(stderr,
	    "usage: [-a string] [-b char] [-B string] [-cdDChPrw] [-g min]");
	(void) fprintf(stderr, " [-G max]\n");
	(void) fprintf(stderr,
	 "       [-i char] [-l char] [-L num] [-p name] [-t ttl] [-v n,val]\n");
	(void) fprintf(stderr, "  %-37.37s",
	    "-a string  after graph = <string>");
	(void) fprintf(stderr,
	    "-b char    <char> = bar line\n");
	(void) fprintf(stderr, "  %-37.37s",
	    "-B string  before graph = <string>");
	(void) fprintf(stderr,
	    "-c         ck/err/hi in QuattoPro\n");
	(void) fprintf(stderr, "  %-37.37s",
	    "-C         no cksum verification");
	(void) fprintf(stderr,
	    "-d         enable Debug mode\n");
	(void) fprintf(stderr, "  %-37.37s",
	    "-D         disable DSR xmt wait");
	(void) fprintf(stderr,
	    "-h         display help (usage)\n");
	(void) fprintf(stderr, "  %-37.37s",
	    "-i char    intersection = <char>");
	(void) fprintf(stderr,
	    "-l char    max/min line = <char>\n");
	(void) fprintf(stderr, "  %-37.37s",
	    "-P         no page count display");
	(void) fprintf(stderr,
	    "-r         QuattroPro dump file\n");
	(void) fprintf(stderr, "  %-37.37s",
	    "-t ttl     graph title = <ttl>");
	(void) fprintf(stderr,
	    "-v n,val   ttl var <n> = <val>\n");
	(void) fprintf(stderr,
	    "  -w         no display mode warnings\n");
	(void) fprintf(stderr,
	    "  -g min     graph min = <min> ");
	fprintf(stderr, "(defaults: mg/dl = %.1f, mmol/l = %.1f)\n",
	    DEFMGMIN, DEFMMMIN);
	(void) fprintf(stderr,
	    "  -G max     graph max = <max> ");
	fprintf(stderr, "(defaults: mg/dl = %.1f, mmol/l = %.1f)\n",
	    DEFMGMAX, DEFMMMAX);
	(void) fprintf(stderr,
	    "  -L num     graph lines/page = <num> (default %d, minimum %d)\n",
	    LPP, HDRLPP+1);
	(void) fprintf(stderr,
	    "  -p name    define printer as <name>: EPSON, PCL4 or PPDS");
	(void) fprintf(stderr,
	    " (default = EPSON)\n");
}


/*
 * ZeroLog() - clear the meter datalog
 */

static void
ZeroLog(void)
{
	DispMenu(ZeroMenu, NULL);
	if ((char)WaitAnyKey() != 'Y')
		return;
	for (;;) {
		if (WaitRdy() == 0)
			return;
		if (WaitCmd("DMZ", 'Z')) {
			GetDataLn(DumpLine, DUMPLL);
			if (strcmpi(DumpLine, " 005A") == 0)
				return;
		}
		DispMenu(ZeroFail,
			"Press ESC to exit; any other key to retry.");
		if ((char)WaitAnyKey() == ESC)
			return;
	}
}
