/*
 * ttl.c - process graph title
 *
 * V. Abell
 */

/*
 * Copyright 1994 Victor A. Abell, Lafayette, Indiana  47906.  All rights
 * reserved.
 *
 * Written by Victor A. Abell.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Victor A. Abell is not responsible for any consequences of the use of
 * this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to Victor A. Abell must
 *    appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#if	!defined(lint)

# if	defined(_BCC)
#pragma warn -use
# endif

static char copyright[] =
"@(#) Copyright 1994 Victor A. Abell.\nAll rights reserved.\n";
#endif

#include "touch2.h"
#include <ctype.h>
#include <stdlib.h>
#include <time.h>

char Exttl[GTTLLNL+1];			/* expanded graph title */
char *Exp;				/* expansion pointer */
char TtlErrMsg[64];			/* title error message */
char *TtlVar[10];			/* special title variable */

static char *ExpEnv(char *ip);
static int PutExpCh(char *ip);
static int PutExpStr(char *ip);

static char *Day[] = { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };
static char *Month[] =
	{ "Jan", "Feb", "Mar", "Apr", "May", "Jun",
	  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };


/*
 * ExpandTtl() - expand title
 */

int
ExpandTtl()
{
	char buf[16], *cp, *ip;
	short i;
	struct tm *lt;
	time_t tm;

	tm = time(NULL);
	lt = localtime(&tm);
	Exp = Exttl;
	for (ip = Gttl; *ip; ip++) {
	    if (*ip != '$') {
	/*
	 * Put ordinary character.
	 */
		if (PutExpCh(ip))
		    return(1);
		continue;
	    }
	    ip++;
	    if (*ip == '$') {
	/*
	 * A double `$' is equivalent to one `$'.
	 */
		if (PutExpCh(ip))
		    return(1);
		continue;
	    }
	    if (*ip == '{') {
	/*
	 * ${NAME} means interpolate environment variable NAME.
	 */
		if ((ip = ExpEnv(++ip)) == NULL)
		    return(1);
	    } else if (strnicmp(ip, "dy", 2) == 0) {
	/*
	 * "Dy" means interpolate day of week name; "dy", day of month number
	 *  with leading zero; "dY", day of month number without leading zero.
	 */
		if (*ip == 'D')
			cp = Day[lt->tm_wday];
		else {
			(void) sprintf(buf,
				(*(ip+1) == 'Y') ? "%2d" : "%02d",
				lt->tm_mday);
			cp = buf;
		}
		if (PutExpStr(cp))
			return(1);
		ip++;
	    } else if (strnicmp(ip, "mo", 2) == 0) {
	/*
	 * "Mo" means interpolate month name; "mo", month number with leading
	 *  zero; * "mO", month number without leading zero.
	 */
		if (*ip == 'M')
			cp = Month[lt->tm_mon];
		else {
			(void) sprintf(buf,
				(*(ip+1) == 'O') ? "%2d" : "%02d",
				lt->tm_mon+1);
			cp = buf;
		}
		if (PutExpStr(cp))
			return(1);
		ip++;
	    } else if (strnicmp(ip, "yr", 2) == 0) {
	/*
	 * "Yr" means interpolate full year; "yr", last two year number digits.
	 */
		(void) sprintf(buf,
			(*ip == 'Y') ? "%04d" : "%02d",
			(*ip == 'Y') ? lt->tm_year+1900 : lt->tm_year);
		if (PutExpStr(buf))
			return(1);
		ip++;
	    } else if (strnicmp(ip, "hr", 2) == 0) {
	/*
	 * "Hr" means interpolate hour number with leading 0; "hr", without.
	 */
		(void) sprintf(buf, (*ip == 'H') ? "%02d" : "%2d", lt->tm_hour);
		if (PutExpStr(buf))
			return(1);
		ip++;
	    } else if (strnicmp(ip, "mn", 2) == 0) {
	/*
	 * "Mn" means interpolate minute number without leading 0; "mn", with.
	 */
		(void) sprintf(buf, (*ip == 'M') ? "%2d" : "%02d", lt->tm_min);
		if (PutExpStr(buf))
			return(1);
		ip++;
	    } else if (strnicmp(ip, "se", 2) == 0) {
	/*
	 * "Se" means interpolate second number without leading 0; "se", with.
	 */
		(void) sprintf(buf, (*ip == 'S') ? "%2d" : "%02d", lt->tm_sec);
		if (PutExpStr(buf))
			return(1);
		ip++;
	    } else if (*ip >= '0' && *ip <= '9') {
	/*
	 * [0-9] means interpolate variable [0-9].
	 */
		i = (short)(*ip - '0');
		if (TtlVar[i] == NULL) {
		     (void) sprintf(TtlErrMsg, "variable %d undefined", i);
		     return(1);
		}
		for (cp = TtlVar[i]; *cp; cp++) {
		    if (PutExpCh(cp))
			return(1);
		}
	    } else {
		(void) strncpy(buf, ip, sizeof(buf) - 1);
		buf[sizeof(buf) - 1] = '\0';
		(void) sprintf(TtlErrMsg,
			"unrecognized title macro, starting at \"$%s\"", buf);
		return(1);
	    }
	}
	*Exp = '\0';
	return(0);
}


/*
 * ExpEnv() - expand environment variable
 */

static char *
ExpEnv(ip)
	char *ip;			/* pointer to start of name */
{
	char *ep, nm[256];
	short i;

	for (i = 0; i < sizeof(nm) - 1; i++) {
		if (*ip == '\0' || *ip == '}')
			break;
		if (i > sizeof(nm) - 1) {
			nm[15] = '\0';
			(void) sprintf(TtlErrMsg,
			    "environment variable name \"%s...\" too long",
			    nm);
			return(NULL);
		}
		nm[i] = *ip++;
	}
	if (*ip == '\0') {
		nm[15] = '\0';
		(void) sprintf(TtlErrMsg,
			"unterminated environment variable, \"%s...\"", nm);
		return(NULL);
	}
	nm[i] = '\0';
	if ((ep = getenv(nm)) == NULL)
		ep = "(nil)";
	while (*ep) {
		if (PutExpCh(ep++))
			return(NULL);
	}
	return(ip);
}


/*
 * PutExpCh() - put character to expansion
 */

static int
PutExpCh(ip)
	char *ip;			/* character to put */
{
	if ((Exp - Exttl) >= (sizeof(Exttl) - 1)) {
		(void) strcpy(TtlErrMsg, "title expansion too large");
		return(1);
	}
	*Exp++ = *ip;
	return(0);
}


/*
 * PutExpStr() - put string to expansion
 */

static int
PutExpStr(ip)
	char *ip;			/* string to put */
{
	while (*ip) {
		if (PutExpCh(ip++))
			return(1);
	}
	return(0);
}
