// assumes large model and -DSTRICT!
#include <windows.h>
#include <windowsx.h>
#include <string.h>
#include <stdlib.h>

// custom message numbers for private window
const UINT WDM_REDRAW       = (WM_USER+1);
const UINT WDM_GROUPMOVE    = (WM_USER+2);
const UINT WDM_RESIZE       = (WM_USER+3);
const UINT WDM_SELECT       = (WM_USER+4);

// window class names
const char* WC_DLGEDIT      =   "WuiDialogEdit";
const char* WC_DLGEDITMASK  =   "WuiDialogEditMask";
const char* WC_DLGEDITCHILD =   "WuiDialogEditChild";

#define GETMETRIC(x)    GetSystemMetrics(x)
#define GETINSTANCE(w)  GetWindowInstance(w)
#define FRAMEINFLATE(r,i) InflateRect(r, i*\
    GETMETRIC(SM_CXFRAME), GETMETRIC(SM_CXFRAME))
#define W(r)    (r.right-r.left)
#define H(r)    (r.bottom-r.top)

// translate other window bound into rect in client coords
static RECT GetOtherRect(HWND Source, HWND Dest)
    {
    RECT ChildRect, New;
    POINT   P;
    GetWindowRect(Source, &ChildRect);
    P.x         = ChildRect.left;
    P.y         = ChildRect.top;
    ScreenToClient(Dest, &P);
    New.left    = P.x;    New.top     = P.y;
    New.right   = New.left + W(ChildRect);
    New.bottom  = New.top  + H(ChildRect);
    return New;
    }

// "global" variables attached to window to be reentrant.
typedef struct  TEditInfo
    {
    HWND    Child, DragSel, Dialog, OldParent, Mask;
    HWND    Selections[255];
    int     NSelected, Recurse, DragSelect;
    }   TEditInfo;

// make the selection window bound all selections
static void BoundSelections(TEditInfo *Edit, HWND Parent)
    {
    int i;
    RECT    SelectRect = GetOtherRect(Edit->Selections[0], Parent);
    for(i=1; i < Edit->NSelected; ++i)
        {
        RECT Other = GetOtherRect(Edit->Selections[i], Parent);
        RECT Temp  = SelectRect;
        UnionRect(&SelectRect, &Other, &Temp);
        }
    FRAMEINFLATE(&SelectRect, 1);
    SetWindowPos(Edit->Child, HWND_TOP,
        SelectRect.left, SelectRect.top,
        W(SelectRect), H(SelectRect), 0);
    ShowWindow(Edit->Child, SW_SHOWNA);
    }

// window proc for children of edit window
LRESULT CALLBACK __export ChildModifyProc(HWND Window,
        UINT Message, WPARAM Param1, LPARAM Param2)
    {
    // size changed -- resize if only one selected
    if(Message == WM_SYSCOMMAND && (Param1&0xFFF0) == SC_SIZE)
        PostMessage(GetParent(Window), WDM_RESIZE, 0, 0);    
    // mouse down -- initiate move
    else if(Message == WM_LBUTTONDOWN)
        {
        RECT    Start, Stop;
        GetWindowRect(Window, &Start);
        // undocumented, but reliable move mechanism
        DefWindowProc(Window, WM_SYSCOMMAND, SC_MOVE+2, 0);
        // window is now at new location, move selections
        GetWindowRect(Window, &Stop);
        if(memcmp(&Start, &Stop, sizeof(Start)))
            {
            PostMessage(GetParent(Window), WDM_GROUPMOVE,
                Start.left - Stop.left, Start.top - Stop.top);
            PostMessage(GetParent(Window), WDM_REDRAW, 0, 0);
            }
        }
    // default min size is not zero, so set it explicitly
    else if(Message == WM_GETMINMAXINFO)
        {
        MINMAXINFO FAR *Info = (MINMAXINFO FAR *)Param2;
        Info->ptMinTrackSize.x  = 0;
        Info->ptMinTrackSize.y  = 0;
        return 0;
        }
    else if(Message == WM_PAINT)
        PostMessage(GetParent(Window), WDM_REDRAW, 0, 0);
    return DefWindowProc(Window, Message, Param1, Param2);
    }

// create child window for moving or selecting
static HWND CreateChild(HWND Parent, int Move)
    {
    return CreateWindow(WC_DLGEDITCHILD, "",
        WS_CHILD|WS_BORDER|(Move?0:WS_THICKFRAME),
        0, 0, 1, 1, Parent, (HWND)NULL, GETINSTANCE(Parent), NULL);
    }

// make child window fill client area
static void FillChild(HWND Parent, HWND Child)
    {
    RECT    R;
    GetClientRect(Parent, &R);
    MoveWindow(Child, 0, 0, W(R), H(R), FALSE);
    }

// window procedure for transparent mask window
LRESULT CALLBACK __export MaskProc(HWND Window,
        UINT Message, WPARAM Param1, LPARAM Param2)
    {
    if(Message == WM_MOUSEMOVE && (Param1&MK_LBUTTON))
        {
        DWORD D     = (DWORD)GetWindowLong(Window, 0);
        POINT Start = MAKEPOINT(D);
        POINT Now   = MAKEPOINT(Param2);
        int Code;
        if(Now.x >= Start.x)
            Code    = (Now.y >= Start.y) ? 8 : 5;
        else
            Code    = (Now.y >= Start.y) ? 7 : 4;
        if(memcmp(&Start, &Now, sizeof Start))
            SendMessage(GetParent(Window), WDM_SELECT,
                Code, D);
        }
    // pass mouse messages to parent
    else if(Message==WM_LBUTTONDOWN || Message==WM_LBUTTONUP)
        {
        // remember possible start of drag-select
        if(Message == WM_LBUTTONDOWN)
            SetWindowLong(Window, 0, (LONG)Param2);
        return SendMessage(GetParent(Window), Message,
            Param1, Param2);
        }
    return DefWindowProc(Window, Message, Param1, Param2);
    }

// window procedure for dialog editor window
LRESULT CALLBACK __export ModifyProc(HWND Window,
        UINT Message, WPARAM Param1, LPARAM Param2)
    {
    TEditInfo  *Edit   = 0;
    HWND        Dialog = 0;
    if(Message == WM_CREATE)
        {
        // create and store "global" variables
        CREATESTRUCT *Info = (CREATESTRUCT *)Param2;
        Edit    = (TEditInfo *)malloc(sizeof(*Edit));
        memset(Edit, 0, sizeof(*Edit));
        Edit->Dialog    = (HWND)Info->lpCreateParams;
        Edit->OldParent = GetParent(Edit->Dialog);
        SetParent(Edit->Dialog, Window);
        FillChild(Window, Edit->Dialog);
        Edit->Mask      = CreateWindowEx(WS_EX_TRANSPARENT,
            WC_DLGEDITMASK, "", WS_CHILD | WS_VISIBLE,
            0, 0, 200, 200, Window, (HWND)NULL,
            GETINSTANCE(Window), NULL);
        FillChild(Window, Edit->Mask);
        Edit->Child     = CreateChild(Window, FALSE);
        Edit->DragSel    = CreateChild(Window, TRUE);

        SetWindowLong(Window, 0, (LONG)Edit);
        BringWindowToTop(Edit->Mask);
        return DefWindowProc(Window, Message, Param1, Param2);
        }
    Edit = (TEditInfo *)GetWindowLong(Window, 0);
    if(Edit != NULL)
        Dialog  = Edit->Dialog;
    else
        return DefWindowProc(Window, Message, Param1, Param2);

    // our keyboard interface
    if(Message == WM_KEYUP)
        {
        // delete key means delete current selection
        if(Param1 == VK_DELETE)
            {
            if(Edit->NSelected)
                {
                int i;
                for(i=0; i < Edit->NSelected; ++i)
                    DestroyWindow(Edit->Selections[i]);
                Edit->NSelected = 0;
                ShowWindow(Edit->Child, SW_HIDE);
                }
            }
        // tab key -> select next window in window list
        else if(Param1 == VK_TAB)
            {
            HWND Next = NULL;
            int Flag = GW_HWNDNEXT;
            if(GetAsyncKeyState(VK_SHIFT) & 0x8000)
                Flag = GW_HWNDPREV;
            if(Edit->NSelected > 0)
                Next    = GetWindow(Edit->Selections[Edit
                    ->NSelected-1], Flag);
            if(Next == NULL)
                {
                Next    = GetWindow(Dialog, GW_CHILD);
                // if backing up through tab order
                if(Next != NULL && Flag == GW_HWNDPREV)
                    Next    = GetWindow(Next, GW_HWNDLAST);
                }
            Edit->NSelected   = 0;
            if(Next != NULL)
                {
                Edit->Selections[Edit->NSelected++] = Next;
                BoundSelections(Edit, Window);
                PostMessage(Window, WDM_REDRAW, 0, 0);
                }
            }
        }
    // someone moved or resized us, alter dialog accordingly
    else if(Message == WM_WINDOWPOSCHANGED)
        {
        const WINDOWPOS *Info = (const WINDOWPOS FAR*)Param2;

        if((Info->flags & (SWP_NOMOVE|SWP_NOSIZE))
                       == (SWP_NOMOVE|SWP_NOSIZE))
            return TRUE;
        FillChild(Window, Edit->Dialog);
        FillChild(Window, Edit->Mask);
        return TRUE;
        }
    // someone wants us to move selected windows
    else if(Message == WDM_GROUPMOVE && Edit->NSelected)
        {
        int XOff = (int)Param1;
        int YOff = (int)Param2;
        int i;
        for(i = 0; i < Edit->NSelected; ++i)
            {
            RECT R;
            POINT P;
            GetWindowRect(Edit->Selections[i], &R);
            P.x = R.left - XOff;  P.y = R.top - YOff;
            ScreenToClient(Dialog, &P);
            MoveWindow(Edit->Selections[i], P.x, P.y,
                W(R), H(R), FALSE);
            }
        UpdateWindow(Edit->Dialog);
        ShowWindow(Edit->Child, SW_SHOW);
        }
    // selection has been resized
    else if(Message == WDM_RESIZE && Edit->NSelected == 1)
        {
        RECT    R;
        POINT   P;
        GetClientRect(Edit->Child, &R);
        P.x = R.left;   P.y = R.top;
        MapWindowPoints(Edit->Child, Dialog, &P, 1);
        MoveWindow(Edit->Selections[0], P.x, P.y,
            W(R), H(R), FALSE);
        }
    // someone requested we redraw our screen
    if(Message == WDM_REDRAW)
        {
        if(IsWindowVisible(Edit->Child))
            {
            if(Edit->Recurse)
                Edit->Recurse   = 0;
            else
                {
                ++Edit->Recurse;
                ShowWindow(Edit->Child, SW_HIDE);
                UpdateWindow(Edit->Dialog);
                ShowWindow(Edit->Child, SW_SHOW);
                }
            }
        return 0;
        }
    // mouse down, could be drag-select or just select
    else if(Message == WM_LBUTTONDOWN)
        {
        HWND    Found;
        POINT   P = MAKEPOINT(Param2);
        MapWindowPoints(Window, Dialog, &P, 1);
        Found = ChildWindowFromPoint(Dialog, P);
        if(Found == Dialog)
            Found   = NULL;
        // if landed outside any child window
        if(Found == NULL)
            {
            // then clear current selections and redraw
            if(Edit->NSelected)
                {
                Edit->NSelected   = 0;
                ShowWindow(Edit->Child, SW_HIDE);
                UpdateWindow(Edit->Dialog);
                }
            Edit->DragSelect    = TRUE;
            SetWindowPos(Edit->DragSel, HWND_TOP,
                LOWORD(Param2), HIWORD(Param2), 1, 1,
                SWP_NOACTIVATE|SWP_NOREDRAW);
            }
        else    // else mouse selected a child control
            {
            // shift key adds to current selections
            if(!(Param1 & MK_SHIFT))
                Edit->NSelected   = 0;
            Edit->Selections[Edit->NSelected++] = Found;
            BoundSelections(Edit, Window);
            InvalidateRect(Edit->Child, NULL, TRUE);
            }
        }
    else if(Message == WDM_SELECT && Edit->DragSelect)
        {
        int     j;
        HWND    Rover;
        RECT    Bound;
        // start drawing rubberband selection rectangle
        BringWindowToTop(Edit->DragSel);
        ShowWindow(Edit->DragSel, SW_SHOW);
        DefWindowProc(Edit->DragSel, WM_SYSCOMMAND,
            SC_SIZE+Param1, Param2);
        ShowWindow(Edit->DragSel, SW_HIDE);
        GetWindowRect(Edit->DragSel, &Bound);
        // now figure out which children were selected
        Edit->NSelected = 0;
        Rover = GetWindow(Dialog, GW_CHILD);
        for(j=0; j < 100 && Rover != NULL; ++j)
            {
            RECT    A, B;
            GetWindowRect(Rover, &A);
            if(IntersectRect(&B, &A, &Bound))
                Edit->Selections[Edit->NSelected++] = Rover;
            Rover   = GetWindow(Rover, GW_HWNDNEXT);
            if(Rover == GetWindow(Dialog, GW_CHILD))
                break;
            }
        if(Edit->NSelected)
            BoundSelections(Edit, Window);
        Edit->DragSelect    = FALSE;
        }
    else if(Message == WM_DESTROY)
        {
        // reparent dialog before we get destroyed
        RECT R;
        GetWindowRect(Dialog, &R);
        SetWindowPos(Edit->Dialog, (HWND)NULL, R.left, R.top,
            0, 0, SWP_NOSIZE|SWP_NOREDRAW);
        SetParent(Dialog, Edit->OldParent);
        EnableWindow(Dialog, TRUE);
        }
    // end of editing session
    else if(Message == WM_NCDESTROY)
        {
        // free memory
        free(Edit);
        // turn dialog back on and give it focus
        SetFocus(Dialog);
        }
    return DefWindowProc(Window, Message, Param1, Param2);
    }

static int RegisterWClass(HINSTANCE Instance, WNDPROC Proc,
        const char *ClassName, LPCSTR Icon)
    {
    WNDCLASS wc;
    memset(&wc, 0, sizeof(wc));
    wc.style         = CS_HREDRAW|CS_VREDRAW;
    wc.lpfnWndProc   = Proc;
    wc.hInstance     = Instance;
    wc.cbWndExtra    = 4;   // can store pointer or long
    wc.hCursor       = LoadCursor((HINSTANCE)NULL, Icon);
    wc.lpszClassName = ClassName;
    return RegisterClass(&wc);
    }

static void ModifyInit(HINSTANCE Instance)
    {
    static int Registered = FALSE;
    if(!Registered)
        {
        RegisterWClass(Instance, ModifyProc, WC_DLGEDIT,
            IDC_CROSS);
        RegisterWClass(Instance, MaskProc, WC_DLGEDITMASK,
            IDC_CROSS);
        RegisterWClass(Instance, ChildModifyProc,
            WC_DLGEDITCHILD, IDC_ARROW);
        Registered  = TRUE;
        }
    }

// main entry point for TINY Dynamic Dialog EDitor
void    TinyDded(HWND Dialog)
    {
    HINSTANCE   Instance    = GETINSTANCE(Dialog);
    RECT        DialogRect;
    ModifyInit(Instance);
    // don't want dialog getting any input
    EnableWindow(Dialog, FALSE);
    GetWindowRect(Dialog, &DialogRect);
    DialogRect.top -= GETMETRIC(SM_CYCAPTION);
    FRAMEINFLATE(&DialogRect, 1);
    CreateWindow(WC_DLGEDIT,
        "TINYDDED: Tiny Dynamic Dialog Editor",
        WS_CAPTION | WS_BORDER | WS_POPUP | WS_THICKFRAME
        | WS_SYSMENU | WS_VISIBLE | WS_CLIPSIBLINGS,
        DialogRect.left, DialogRect.top, W(DialogRect),
        H(DialogRect), 0, 0, Instance, (void FAR*)Dialog );
    }
