/*
 * The author of this software is William Dorsey.
 * Copyright (c) 1993, 1994, 1995 by William Dorsey.  All rights reserved.
 *
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY.  IN PARTICULAR, THE AUTHOR DOES NOT MAKE ANY CLAIM OR
 * WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR
 * ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */

/* comm.c
 *
 * REVISION HISTORY
 *
 * DATE      RESPONSIBLE PARTY  DESCRIPTION
 * -------------------------------------------------------------------------
 * 93/12/08  B. Dorsey      Module created by breakup of nautilus.c
 */

#include <stdio.h>
#include <stdlib.h>		/* plm */
#include <string.h>		/* plm */
#include <ctype.h>		/* plm */
#include <malloc.h>		/* plm */

#include "nautilus.h"

#define XCHG_TIMEOUT    15
#define RP_TIMEOUT      3

/* external variables */
extern struct param_t params;
extern struct coder_t coders[NCODERS];
extern struct negotiate_t negotiate;

/*
 * dial phone number specified by argument and wait for connection return 0
 * on success and -1 on failure
 */
int
Connect(char *phone, char *modem_init)
{
    char            buf[512];

    WritePort("AT\r", 3);
    if (WaitFor("OK", 2, 0) < 0) {
	WritePort("AT\r", 3);
	if (WaitFor("OK", 2, 0) < 0)
	    return -1;
    }
    PAUSE(100);			/* wait 100ms */
    if (strlen(modem_init)) {	/* config string defined? */
        strcpy(buf, "AT");
        strcat(buf, modem_init);
        strcat(buf, "\r");
        WritePort(buf, strlen(buf));
        if (WaitFor("OK", 2, 0) < 0) {
            return -1;
        }
    }
    PAUSE(100);			/* wait 100ms */
    strcpy(buf, "AT");
    if (strlen(phone))
        sprintf(buf + strlen(buf), "DT%s\r", phone);
    else
        strcat(buf, "D\r");
    WritePort(buf, strlen(buf));

    return WaitFor("CONNECT", 60, 1);
}

/*
 * if 'flag', put the modem in answer mode (ATS0=1) wait for incoming call.
 * else, take modem offhook immediately in answer mode. return 0 on success
 * and -1 on failure.
 */
int
AnswerMode(int flag)
{
    WritePort("AT\r", 3);
    if (WaitFor("OK", 2, 0) < 0) {
	WritePort("AT\r", 3);
	if (WaitFor("OK", 2, 0) < 0)
	    return -1;
    }
    PAUSE(100);			/* wait 100ms */
    if (flag) {
	WritePort("ATS0=1\r", 7);
	if (WaitFor("OK", 2, 0) < 0)
	    return -1;
	return WaitFor("CONNECT", 0, 1);
    }
    else {
	WritePort("ATA\r", 4);
	return WaitFor("CONNECT", 45, 1);
    }
}

/*
 * Monitor serial rx stream for 'string'.  Return 0 if found.  If not
 * found after 'wait' seconds, return -1 for failure.  If 'show' is
 * nonzero, seconds remaining and each string received are displayed on
 * console.  If 'wait' is zero, wait forever for desired string.
 */
int
WaitFor(char *string, int wait, int show)
{
    int		c, i=0, timeout, old=-1;
    unsigned	speed;
    char	ibuf[80], tbuf[16];
    UINT32	start = Clock();

    if (show) {
	sprintf(ibuf, "Waiting for \"%s\"", string);
	error(MSG_WARNING, ibuf);
    }

    for (;;) {
	/*
	 * Construct a line of input from the serial port.  Control
	 * characters are ignored with the exception of the linefeed
	 * which marks the end of the line.
	 */
	if (IncomingPort()) {
	    c = ReadPort(1);
	    if (i && (c == '\r')) {	/* end of line detected */
		ibuf[i] = '\0';
		if (show) {	/* print the buffer if show is true */
		    debug_puts(ibuf);
		    old = -1;
		}
		/*
		 * Check line against string to be matched and return
		 * a status or continue searching.
		 */
		if (strstr(ibuf, string)) {
		    /*
		     * User's string to be matched has been found.  If
		     * it's a "CONNECT" string, see if there is a connect
		     * speed (hopefully the DCE speed) present and store
		     * any such value in the params structure.
		     */
		    if (!strcmp(string, "CONNECT") && strlen(ibuf) > 7) {
			speed = atoi(&ibuf[8]);
			if (speed >= 300)
			    params.modem.speed = speed;
		    }
		    return 0;
		}
		/*
		 * If any of the following string(s) are detected,
		 * throw them out and continue waiting for the user's
		 * string.
		 */
		else if (strstr(ibuf, "AT") ||
		         strstr(ibuf, "RING"))
		    i = 0;
		/*
		 * We've reached the end of a line without finding the
		 * search string.  Return an error status.
		 */
		else
		    break;
	    }
	    else if (!iscntrl(c)) {
		ibuf[i++] = c;
		if (i == 80)	/* overflow detection */
		    break;
	    }
	}
	/*
	 * Check for timeout
	 */
	if (wait) {
	    timeout = Clock() - start;
	    if (timeout != old) {
		if (show) {
		    sprintf(tbuf, "\r%3d: ", wait-timeout);
		    debug_puts_nr(tbuf);
		}
		if (timeout >= wait)
		    break;
		old = timeout;
	    }
	}
	/*
	 * keypress will abort if waiting "forever".
	 */
	else
	    if (ptt())
	    	break;
    }

    if (show)
        debug_putc('\n');

    return -1;
}

int
XChange(int mode)
{
    int                i, done, speed;
    UINT32             start, start2;
    UINT8              *p1, *p2;
    struct negotiate_t *orig, *ans;
    struct keyexch_t   keyexch1, keyexch2;
    struct packet_t    packet;

    while (IncomingPort())
	ReadPort(1);
    sleep(1);

    /* Set modem speed in negotiation parameters (if known) */
    if (params.modem.speed > 0) {
        negotiate.modem_speed[0] = params.modem.speed % 256;	/* LSB */
        negotiate.modem_speed[1] = params.modem.speed / 256;	/* MSB */
    }

    /* Exchange negotiation parameters */
    switch (mode) {
    case ORIGINATE:
	done = 0;
	start = Clock();
	do {
	    SendPkt(DATA, (UINT8 *) &negotiate, sizeof(struct negotiate_t));
	    start2 = Clock();
	    do {
		if (IncomingPort()) {
		    if (ReadPkt(&packet) == 0) {
			done = 1;
			break;
		    }
		}
	    } while (Clock() - start2 < 2);
	    if (Clock() - start > XCHG_TIMEOUT)
		return -1;
	} while (done == 0);
	orig = &negotiate;
	ans = (struct negotiate_t *) packet.data;
	break;

    case ANSWER:
    case AUTO_ANSWER:
	start = Clock();
	for (;;) {
	    if (IncomingPort()) {
		if (ReadPkt(&packet) == 0)
		    break;
	    }
	    if (Clock() - start > XCHG_TIMEOUT)
		return -1;
	}
	SendPkt(DATA, (UINT8 *) &negotiate, sizeof(struct negotiate_t));
	orig = (struct negotiate_t *) packet.data;
	ans = &negotiate;
	break;
    }

    /* Verify that packet is the right type & size */
    if ((packet.type != DATA) || (packet.length != sizeof(struct negotiate_t)))
        return -1;

    /* Verify version compatibility */
    if ((orig->major != ans->major) || (orig->minor != ans->minor))
        return -1;

    /* Get modem speed or assume 14400 bps if unknown */
    speed = orig->modem_speed[0] + (orig->modem_speed[1] << 8);
    if (speed == 0)
        speed = ans->modem_speed[0] + (ans->modem_speed[1] << 8);
    if (speed == 0)
        speed = DEFAULT_DCE_SPEED;

    /* Find best coder to use */
    for (i=NCODERS-1; i>=0; i--) {
        if (coders[i].bandwidth <= speed) {
            if (i < 8) {
                if (orig->encode[0] & ans->encode[0] & (1<<i))
                    break;
            }
            else {
                if (orig->encode[1] & ans->encode[1] & (1<<(i-8)))
                    break;
            }
        }
    }
    if (i >= 0)
        params.coder.index = i;
    else
        return -1;

    /* Did answerer specify a coder? */
    if (ans->coder < NCODERS) {
        if (coders[ans->coder].bandwidth <= speed) {
            if (ans->coder < 8) {
                if (orig->encode[0] & ans->encode[0] & (1<<ans->coder)) {
                    params.coder.index = ans->coder;
		}
            }
            else {
                if (orig->encode[1] & ans->encode[1] & (1<<(ans->coder-8))) {
                    params.coder.index = ans->coder;
		}
            }
        }
    }

    /* Did originator specify a coder? */
    if (orig->coder < NCODERS) {
        if (coders[orig->coder].bandwidth <= speed) {
            if (orig->coder < 8) {
                if (orig->encode[0] & ans->encode[0] & (1<<orig->coder)) {
                    params.coder.index = orig->coder;
		}
            }
            else {
                if (orig->encode[1] & ans->encode[1] & (1<<(orig->coder-8))) {
                    params.coder.index = orig->coder;
		}
            }
        }
    }

    /*
     * For now, make sure that both ends have specified the same
     * type of encryption (or none).  If they haven't, treat it as
     * an error.
     */
    if (orig->encrypt_type != ans->encrypt_type) {
	error(MSG_ERROR, "selected encryption must match");
	return -1;
    }

    if (params.crypto.key.type != NONE) {
        /*
         * Generate keyexch structure.
         */
        params.crypto.skey_len = MAX_SKEY_LEN;
        random_bytes((UINT8 *) &keyexch1.sess_iv, 8);
        random_bytes((UINT8 *) &keyexch1.xmit_iv, 8);
        random_bytes((UINT8 *) &keyexch1.recv_iv, 8);
        random_bytes((UINT8 *) &keyexch1.skey, params.crypto.skey_len);

	/*
	 * Copy the keyexch structure just generated and encrypt it
	 * in CFB mode using the sess_iv element (first 8 bytes) as
	 * the IV.  Obviously, the first 8 bytes are not encrypted.
	 */
	keyexch2 = keyexch1;
	cfb_init(&params.crypto.sess_ctx, &params.crypto.key, &keyexch2.sess_iv);
	cfb_encrypt(&params.crypto.sess_ctx, (UINT8 *) &keyexch2.xmit_iv,
	            sizeof(struct keyexch_t) - sizeof(BLOCK));

    	/*
    	 * Exchange keyexch structures.  Originator goes first.
    	 */
	if (mode == ORIGINATE) {
	    SendPkt(DATA, (UINT8 *) &keyexch2, sizeof(struct keyexch_t));
	    if (ReadPkt(&packet) < 0) {
		error(MSG_ERROR, "timeout during crypto parameter negotiation");
		return -1;
	    }
	    if ((packet.type != DATA) || (packet.length != sizeof(struct keyexch_t))) {
		error(MSG_ERROR, "negotiation of crypto parameters failed");
		return -1;
	    }
	    /*
	     * Make sure the other end isn't attempted a replay attack
	     * against us by playing back the keyexch packet we just
	     * sent it.  If this is allowed to happen, the session key
	     * as well as transmit and receive IVs will be all zeros --
	     * a bad thing.
	     */
	    if (!memcmp((char *) &keyexch2, (char *) packet.data, sizeof(struct keyexch_t))) {
		error(MSG_ERROR, "key exchange failed -- possible replay attack detected");
		return -1;
	    }
	}
	else {
	    if (ReadPkt(&packet) < 0) {
		error(MSG_ERROR, "timeout during crypto parameter negotiation");
		return -1;
	    }
	    if ((packet.type != DATA) || (packet.length != sizeof(struct keyexch_t))) {
	        error(MSG_ERROR, "negotiation of crypto parameters failed");
	        return -1;
	    }
	    SendPkt(DATA, (UINT8 *) &keyexch2, sizeof(struct keyexch_t));
	}

	/*
	 * Copy the received packet data into the second keyexch structure.
	 * Decrypt the second keyexch structure using the sess_iv element
	 * (first 8 bytes) as the iv.  Since we're done using the key
	 * schedule generated from the pass phrase, burn it.
	 */
	memcpy((char *) &keyexch2, (char *) packet.data, sizeof(struct keyexch_t));
	cfb_init(&params.crypto.sess_ctx, &params.crypto.key, &keyexch2.sess_iv);
	cfb_decrypt(&params.crypto.sess_ctx, (UINT8 *) &keyexch2.xmit_iv,
		    sizeof(struct keyexch_t) - sizeof(BLOCK));
	keydestroy(&params.crypto.key);

	/*
	 * Check the CRC on the received keyexch structure to make sure
	 * the other user typed the same passphrase we did (optional)
	 */
	/* XXX */

	/*
	 * XOR the two decrypted keyexch structures together and copy
	 * the result to the params structure.
	 */
	p1 = (UINT8 *) &keyexch1.xmit_iv;
	p2 = (UINT8 *) &keyexch2.xmit_iv;
	for (i=0; i<sizeof(struct keyexch_t) - sizeof(BLOCK); i++) {
	    *p1++ ^= *p2++;
	}
	memcpy((char *) params.crypto.skey, (char *) keyexch1.skey, params.crypto.skey_len);

	/*
	 * Call keyinit with the session key.
	 */
	if (keyinit(&params.crypto.key, params.crypto.skey, params.crypto.skey_len) < 0) {
	    error(MSG_ERROR, "key initialization failed");
	    return -1;
	}
	memset(params.crypto.skey, '\0', MAX_SKEY_LEN);

	/*
	 * Initialize crypto contexts for transmitting/receiving.
	 */
	if (mode == ORIGINATE) {
	    cfb_init(&params.crypto.xmit_ctx, &params.crypto.key, &keyexch1.xmit_iv);
	    cfb_init(&params.crypto.recv_ctx, &params.crypto.key, &keyexch1.recv_iv);
	}
	else {
	    cfb_init(&params.crypto.xmit_ctx, &params.crypto.key, &keyexch1.recv_iv);
	    cfb_init(&params.crypto.recv_ctx, &params.crypto.key, &keyexch1.xmit_iv);
	}

	/*
	 * Burn sensitive data
	 */
	memset((char *) &keyexch1, '\0', sizeof(struct keyexch_t));
	memset((char *) &keyexch2, '\0', sizeof(struct keyexch_t));
    }

    return 0;
}

int
ReadPkt(struct packet_t *pkt)
{
    int             i, c;
    UINT16          recv_crc, crc;
    UINT8           buf[2 * MAX_PKT_DATA];

    /* Initialize packet structure */
    pkt->type = (UINT8) - 1;
    pkt->length = (UINT16) - 1;
    pkt->crc = 0;

    /*
     * Expect next incoming character to be a FRAME character. If it isn't,
     * there's a framing problem so advance to the beginning of the next
     * frame and return an error (-1).
     */
    if ((c = ReadPort(RP_TIMEOUT)) != FRAME) {
	SyncPkt();
	return -1;
    }

    /* Allow up to two FRAME characters to separate frames. */
    if ((c = ReadPort(RP_TIMEOUT)) == FRAME)
	c = ReadPort(RP_TIMEOUT);

    /* Check packet type for validity */
    if ((c < 0) || (c > MAX_PACKET_TYPE)) {
	SyncPkt();
	return -1;
    }

    pkt->type = c;

#if DEBUG > 0
    switch (pkt->type) {
    case DATA:
	break;
    case EOT:
	debug_puts("Reading packet type EOT");
	break;
    case XMIT:
	debug_puts("Reading packet type TRANSMIT");
	break;
    case RECV:
	debug_puts("Reading packet type RECEIVE");
	break;
    case ACK:
	debug_puts("Reading packet type ACK");
	break;
    default:
	debug_puts("Reading packet with undefined type");
	break;
    }
#endif

    /* Get characters until we encounter another FRAME character */
    for (i = 0; ((c = GetByte()) >= 0) && i < 2 * MAX_PKT_DATA; i++) {
	buf[i] = c;
    }
    if (c == -1) {
    	/*
    	 * A timeout occurred while in the middle of reading a
    	 * packet.  Return an error.
    	 */
#if DEBUG > 0
	debug_puts("timeout while reading packet");
#endif
        return -1;
    }
    else if (c == -2) {
	/*
	 * We've found the trailing FRAME character.  The last two bytes are
	 * the crc.  Compute the length, copy the buffer into pkt->data, and
	 * finally check the crcs.
	 */
	recv_crc = buf[--i];
	recv_crc += buf[--i] << 8;
	pkt->length = i;
	memcpy(pkt->data, buf, pkt->length);
	crc = ComputeCRC(pkt->type, pkt->data, pkt->length);
	if (crc != recv_crc) {
#if DEBUG > 1
	    debug_putc('%');
#endif
	    error(MSG_WARNING, "crc error in received packet");
	    return -1;
	}
	else {
#if DEBUG > 1
	    debug_putc('.');
#endif
	}
    }
    else {
	/*
	 * Something screwy happened -- perhaps a FRAME character was
	 * garbled.  Anyway, we've overflowed our buffer so we discard all
	 * data up to the next FRAME character we can find and return an
	 * error.
	 */
#if DEBUG > 0
	debug_puts("Packet overflow -- data discarded");
#endif
	SyncPkt();
	return -1;
    }

    return 0;
}

void
SendPkt(int type, UINT8 * data, int len)
{
    int             i, j;
    UINT16          crc;
    UINT8           buf[2 * MAX_PKT_DATA];

#if DEBUG > 0
    switch (type) {
    case DATA:
#if DEBUG > 1
	debug_putc('*');
#endif
	break;
    case EOT:
	debug_puts("Sending packet type EOT");
	break;
    case XMIT:
	debug_puts("Sending packet type TRANSMIT");
	break;
    case RECV:
	debug_puts("Sending packet type RECEIVE");
	break;
    case ACK:
	debug_puts("Sending packet type ACK");
	break;
    default:
	debug_puts("Sending packet with undefined type");
	break;
    }
#endif

    i = 0;
    buf[i++] = FRAME;
    buf[i++] = type;
    for (j = 0; j < len; j++)
	AddByte(buf, i, data[j]);
    crc = ComputeCRC((UINT8) type, data, len);
    AddByte(buf, i, (crc >> 8) & 0xff);	/* msb first */
    AddByte(buf, i, crc & 0xff);
    buf[i++] = FRAME;
    WritePort(buf, i);
}

void
SyncPkt(void)
{
    int             c;

    /*
     * Advance to beginning of next frame by discarding incoming characters
     * until a FRAME character is received.
     */
    while (IncomingPort()) {
	c = ReadPort(1);
	if (c == FRAME)
	    break;
    }
}

int
GetByte()
{
    int             c;

    switch (c = ReadPort(RP_TIMEOUT)) {
    case FRAME:
	c = -2;
	break;
    case ESCAPE:
	c = ReadPort(RP_TIMEOUT) ^ 0x20;
	break;
    default:
	break;
    }

    return c;
}
