/*
 * The author of this software is William Dorsey.
 * Copyright (c) 1993, 1994, 1995 by William Dorsey.  All rights reserved.
 *
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY.  IN PARTICULAR, THE AUTHOR DOES NOT MAKE ANY CLAIM OR
 * WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR
 * ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */

/*
 *  iapsd.c -- fixed point apsd coder
 */

/* REVISION HISTORY
 *
 * DATE      RESPONSIBLE PARTY  DESCRIPTION
 * -------------------------------------------------------------------------
 * 93/05/13  B. Dorsey          Wrote original version
 * 93/09/12  B. Dorsey		Changes for nautilus (no functional changes)
 * 93/09/16  R. Berry		Optimized encode()
 * 93/12/11  B. Dorsey		Modified to use fixed point arithmetic
 */

#include <stdio.h>

#include "machine.h"

#define	ZNORM		(1L << FR_BITS)
#define FR_BITS		20

INT32           zmul(INT32, INT32);
INT32           zdiv(INT32, INT32);
UINT32          zsqrt(UINT32 v);

/* Operating Parameters */
#define FRAME_SIZE      120
#define YPSIZE          6
#define	YPMIN		((INT32)(0.001 * ZNORM))
#define YPMAX		((INT32)(0.150 * ZNORM))
#define FSROOT		((INT32)(10.9087 * ZNORM))	/* sqrt(FRAME_SIZE-1) */

static struct iapsd_t {
    INT32           y[2];
    INT32           ypdelta;
}               handle;

static struct ifilter_t {
    INT32           h1;
    INT32           h2;
}               filters[] = {	/* filter coefficients */
    ((INT32) (1.53 * ZNORM)), ((INT32) (-0.72 * ZNORM)),
    ((INT32) (0.95 * ZNORM)), ((INT32) (-0.34 * ZNORM)),
    ((INT32) (0.48 * ZNORM)), ((INT32) (-0.21 * ZNORM)),
    ((INT32) (-0.63 * ZNORM)), ((INT32) (-0.36 * ZNORM))
};

void
apsd_init(void)
{
    /* initializations */
    handle.y[0] = 0;
    handle.y[1] = 0;
    handle.ypdelta = zdiv(YPMAX - YPMIN, ((1L << YPSIZE) - 1) * ZNORM);
}

void
apsd_encode(INT16 * x, UINT8 * bits)
{
    int             i, state;
    INT32           phi0, phi1, step_size, step, y, yy;
    UINT8           mask = 0x80, data = 0;

    /*
     * Compute normalized autocorrelation at lag 0 & 1
     * Compute step size based on RMS value
     */
    yy = ((INT32) x[0]) << (FR_BITS - 15);
    phi0 = zmul(yy, yy) + 1;		/* + 1 to prevent divide by 0 */
    step_size = phi1 = 0;

    for (i = 1; i < FRAME_SIZE; i++) {
	y = ((INT32) x[i]) << (FR_BITS - 15);	/* convert to fixed pt */
	phi0 += zmul(y, y);			/* autocorr at lag 0 */
	phi1 += zmul(yy, y);			/* autocorr at lag 1 */
	step_size += zmul(y - yy, y - yy);	/* rms calc */
	yy = y;
    }
    phi1 = zdiv(phi1, phi0);

    /* select predictor state */
    if (phi1 > ((INT32) (0.7 * ZNORM)))
	state = 0;
    else if (phi1 > ((INT32) (0.4 * ZNORM)))
	state = 1;
    else if (phi1 > 0)
	state = 2;
    else
	state = 3;

    /* compute step sized based on RMS value of input */
    step_size = zsqrt(step_size);
    step_size = zdiv(step_size, FSROOT);

    /* check step size for bounds */
    if (step_size < YPMIN)
	step_size = YPMIN;
    else if (step_size > YPMAX)
	step_size = YPMAX;

    /* quantize step size to YPSIZE bits */
    step = zdiv(step_size - YPMIN, handle.ypdelta);
    step >>= FR_BITS;

    step_size = zmul(step << FR_BITS, handle.ypdelta);
    step_size += YPMIN;

    /* save predictor state and quantized step size in output */
    *bits++ = state + (step << 2);

    /* compute output bits */
    for (i = 0; i < FRAME_SIZE; i++) {
	y = zmul(filters[state].h1, handle.y[0]);
	y += zmul(filters[state].h2, handle.y[1]);

	handle.y[1] = handle.y[0];

	if ((((INT32) x[i]) << (FR_BITS - 15)) > y) {
	    y += step_size;
	    if (y >= ZNORM)
		y = 0x7fff << (FR_BITS - 15);
	    data |= mask;
	}
	else {
	    y -= step_size;
	    if (y < -ZNORM)
		y = -ZNORM;
	}

	if (!(mask >>= 1)) {
	    *bits++ = data;
	    data = mask;
	    mask = 0x80;
	}
	handle.y[0] = y;
    }
}

void
apsd_decode(UINT8 * bits, INT16 * x)
{
    int             i, state;
    INT32           y, step_size, step;

    /* get predictor state and step size from input */
    state = *bits & 0x3;	/* ls 2 bits */
    step = *bits++ >> 2;	/* ms 6 bits */

    /* decode quantized step size */
    step_size = zmul(step << FR_BITS, handle.ypdelta);
    step_size += YPMIN;

    /* compute output from input bits */
    for (i = 0; i < FRAME_SIZE; i++) {
	y = zmul(filters[state].h1, handle.y[0]);
	y += zmul(filters[state].h2, handle.y[1]);

	handle.y[1] = handle.y[0];

	if (bits[i>>3] & 0x80>>(i&7)) {		/* Mycal's suggestion */
	    y += step_size;
#ifdef CLIP
	    if (y >= ZNORM)
		y = 0x7fff << (FR_BITS - 15);
#endif
	}
	else {
	    y -= step_size;
#ifdef CLIP
	    if (y < -ZNORM)
		y = -ZNORM;
#endif
	}
	handle.y[0] = y;
	x[i] = (INT16) (y >> (FR_BITS - 15));
    }
}
