// ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO 
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A 
// PARTICULAR PURPOSE.
//
// Copyright  1993, 1994  Microsoft Corporation.  All Rights Reserved.
//
//   PROGRAM:   Generic.c
//
//   PURPOSE:   Generic template for Windows applications
//   
//   
//   PLATFORMS: Chicago,NT
//   
//   FUNCTIONS:
//      WinMain() - calls initialization function, processes message loop
//      InitApplication() - initializes window data and registers window
//      InitInstance() - saves instance handle and creates main window
//      WndProc() - processes messages
//      CenterWindow() - used to center the "About" box over application window
//      About() - processes messages for "About" dialog box
//   
//   COMMENTS:
//   
//      The Windows SDK Generic Application Example is a sample application
//      that you can use to get an idea of how to perform some of the simple
//      functionality that all Applications written for Microsoft Windows
//      should implement. You can use this application as either a starting
//      point from which to build your own applications, or for quickly
//      testing out functionality of an interesting Windows API.
//      
//      This application is source compatible for with Windows 3.1 and
//      Windows NT.
//      
//   SPECIAL INSTRUCTIONS: N/A
//   

#include <windows.h>                // required for all Windows applications
#if !defined(_WIN32)
#include <ver.h>
#endif
#include "generic.h"                // specific to this program
#include "resource.h"
#include "commctrl.h"               // Common controls
#include <stdio.h>                  // for atoi

// Windows NT defines APIENTRY, but 3.x doesn't
#if !defined (APIENTRY)
#define APIENTRY far pascal
#endif

// Windows 3.x uses a FARPROC for dialogs
#if !defined(_WIN32)
#define DLGPROC FARPROC
#endif

HINSTANCE   hInst;                     // current instance
HWND        ghWnd; 					   // Main window handle.

char szAppName[] = "Generic";        // The name of this application
char szTitle[]   = "Generic Sample Application"; // The title bar text

//************** NEW CODE START *********

COLORREF 	crColorWindow = RGB(255,0,0); // Color of client area
int		   iPattern = IDD_NONE;          // Pattern of client area

VOID PrePlannedPropertySheet(HWND);  // Test functions for
VOID DynamicPropertySheet   (HWND);  // each type of property
VOID SingleProcPropertySheet(HWND);  // sheet

// This is an app-defined message to make this sample
// small enough to fit in a magazine article.

#define WM_MYCHANGESETTINGS (WM_USER+1000)  // Send to ghWnd to indicate
                                            // the color/pattern has changed

//************** NEW CODE END *********

//
//   FUNCTION: WinMain(HINSTANCE, HINSTANCE, LPSTR, int)
//
//   PURPOSE: calls initialization function, processes message loop
//   
//   COMMENTS:
//   
//      Windows recognizes this function by name as the initial entry point
//      for the program.  This function calls the application initialization
//      routine, if no other instance of the program is running, and always
//      calls the instance initialization routine.  It then executes a 
//      message retrieval and dispatch loop that is the top-level control 
//      structure for the remainder of execution.  The loop is terminated 
//      when a WM_QUIT	message is received, at which time this function 
//      exits the application instance by returning the value passed by 
//      PostQuitMessage().
//      
//      If this function must abort before entering the message loop, it
//      returns the conventional value NULL.
//

int APIENTRY WinMain(
               HINSTANCE hInstance,
               HINSTANCE hPrevInstance,
               LPSTR lpCmdLine,
               int nCmdShow
               )
{
   MSG msg;
   HANDLE hAccelTable;

   // Other instances of app running?
   if (!hPrevInstance) {
      // Initialize shared things 
      if (!InitApplication(hInstance)) {
         return (FALSE);               // Exits if unable to initialize
      }
   }

   // Perform initializations that apply to a specific instance 
   if (!InitInstance(hInstance, nCmdShow)) {
      return (FALSE);
   }

   hAccelTable = LoadAccelerators (hInstance, szAppName);

   // Acquire and dispatch messages until a WM_QUIT message is received.
   while (GetMessage(&msg,   // message structure
                     NULL,   // handle of window receiving the message
                     0,      // lowest message to examine
                     0)){    // highest message to examine
       if (!TranslateAccelerator (msg.hwnd, hAccelTable, &msg)) {
         TranslateMessage(&msg);// Translates virtual key codes
         DispatchMessage(&msg); // Dispatches message to window
       }
   }

   // Returns the value from PostQuitMessage
   return (msg.wParam);

   // This will prevent 'unused formal parameter' warnings
   lpCmdLine;
}


//
//   FUNCTION: InitApplication(HINSTANCE)
//
//   PURPOSE: Initializes window data and registers window class
//   
//   COMMENTS:
//   
//      This function is called at initialization time only if no other
//      instances of the application are running.  This function performs
//      initialization tasks that can be done once for any number of running
//      instances.
//      
//      In this case, we initialize a window class by filling out a data
//      structure of type WNDCLASS and calling the Windows RegisterClass()
//      function.  Since all instances of this application use the same 
//      window class, we only need to do this when the first instance is 
//      initialized.
//

BOOL InitApplication(HINSTANCE hInstance)
{
   WNDCLASS  wc;

   // Fill in window class structure with parameters that describe the
   // main window.
   wc.style         = CS_HREDRAW | CS_VREDRAW; // Class style(s).           
   wc.lpfnWndProc   = (WNDPROC)WndProc;        // Window Procedure          
   wc.cbClsExtra    = 0;                       // No per-class extra data.  
   wc.cbWndExtra    = 0;                       // No per-window extra data. 
   wc.hInstance     = hInstance;               // Owner of this class       
   wc.hIcon         = LoadIcon (hInstance, szAppName);// Icon name from .RC 
   wc.hCursor       = LoadCursor(NULL, IDC_ARROW); // Cursor                
   wc.hbrBackground = (HBRUSH)(COLOR_WINDOW+1);// Default color             
   wc.lpszMenuName  = szAppName;               // Menu name from .RC        
   wc.lpszClassName = szAppName;               // Name to register as       

   // Register the window class and return success/failure code.
   return (RegisterClass(&wc));
}


//
//   FUNCTION:  InitInstance(HINSTANCE, int)
//
//   PURPOSE:  Saves instance handle and creates main window
//   
//   COMMENTS:
//   
//      This function is called at initialization time for every instance of
//      this application.  This function performs initialization tasks that
//      cannot be shared by multiple instances.
//      
//      In this case, we save the instance handle in a static variable and
//      create and display the main program window.
//

BOOL InitInstance(
         HINSTANCE     hInstance,
         int           nCmdShow
         )
{


   // Save the instance handle in static variable, which will be used in
   // many subsequence calls from this application to Windows.

   hInst = hInstance; // Store instance handle in our global variable

   // Create a main window for this application instance.
   ghWnd = CreateWindow(
      szAppName,	        // See RegisterClass() call.             
      szTitle,	           // Text for window title bar.            
      WS_OVERLAPPEDWINDOW,// Window style.                         
      CW_USEDEFAULT, 0, CW_USEDEFAULT, 0,// Use default positioning
      NULL,		           // Overlapped windows have no parent.    
      NULL,		           // Use the window class menu.            
      hInstance,	        // This instance owns this window.       
      NULL		           // We don't use any data in our WM_CREATE
      );

   // If window could not be created, return "failure"
   if (!ghWnd) {
      return (FALSE);
   }

//************** NEW CODE START *********

   InitCommonControls();  // This MUST be called once per instance
                          // to register the common controls

//************** NEW CODE END *********

   // Make the window visible; update its client area; and return "success"
   ShowWindow(ghWnd, nCmdShow); // Show the window        
   UpdateWindow(ghWnd);         // Sends WM_PAINT message 

   return (TRUE);              // We succeeded...        
}

//
//   FUNCTION: WndProc(HWND, UINT, WPARAM, LPARAM)
//
//   PURPOSE:  Processes messages
//   
//   MESSAGES:
//   
//      WM_COMMAND    - application menu (About dialog box)
//      WM_DESTROY    - destroy window
//   
//   COMMENTS:
//   
//      To process the IDM_ABOUT message, call MakeProcInstance() to get the
//      current instance address of the About() function.  Then call Dialog
//      box which will create the box according to the information in your
//      generic.rc file and turn control over to the About() function.  When
//      it returns, free the intance address.
//

LRESULT CALLBACK WndProc(
      HWND hWnd,         // window handle
      UINT message,      // type of message
      WPARAM uParam,     // additional information
      LPARAM lParam      // additional information
      )
{
   FARPROC lpProcAbout;  // pointer to the "About" function
   int wmId, wmEvent;
   HDC hDC;
   PAINTSTRUCT ps;
   RECT rc;
   HBRUSH hBrush;
   int  x, y;

   switch (message) {

//************** NEW CODE START *********

      case WM_PAINT:

         // Get the client area size
         GetClientRect ( hWnd, &rc );

         // Get a DC to paint to
         hDC = BeginPaint ( hWnd, &ps );

         // Use ExtTextOut as a fast way to fill the rect with color
         SetBkColor ( hDC, crColorWindow);
         ExtTextOut ( hDC, 0, 0, ETO_OPAQUE, &rc, NULL, 0, NULL );

         // Draw the pattern
         switch (iPattern)
           {
           case IDD_CIRCLES:
            
             // Hollow circles
             hBrush = SelectObject ( hDC, GetStockObject ( NULL_BRUSH ));

             // Basic loop to draw circles
             for ( x = 0; x < ((rc.right+31)/32); x++ )
               for ( y = 0; y < ((rc.bottom+31)/32); y++ )
                 {
                 Ellipse ( hDC, x*32, y*32, (x+1)*32, (y+1)*32 );
                 }

             // Miss Manners sez clean up
             SelectObject ( hDC, hBrush );
             break;

           case IDD_SQUARES:

             // Hollow squares
             hBrush = SelectObject ( hDC, GetStockObject ( NULL_BRUSH ));

             // Basic loop to draw squares
             for ( x = 0; x < ((rc.right+31)/32); x++ )
               for ( y = 0; y < ((rc.bottom+31)/32); y++ )
                 {
                 Rectangle ( hDC, x*32, y*32, (x+1)*32, (y+1)*32 );
                 }

             // Get rid of that evidence
             SelectObject ( hDC, hBrush );
             break;

           }

         // Relase the DC
         EndPaint ( hWnd, &ps );
         break;

      case WM_MYCHANGESETTINGS:

         // This message is sent by the property sheet page
         // for color and pattern. If the user clicks the
         // Apply button, the associated dialog procedure for
         // that page sends the WM_MYCHANGESETTINGS message here
         // to instantly update the client area.
         
         switch ( uParam )
           {
           case IDD_RED:   crColorWindow = RGB ( 255,   0,   0 ); break;
           case IDD_GREEN: crColorWindow = RGB (   0, 255,   0 ); break;
           case IDD_BLUE:  crColorWindow = RGB (   0,   0, 255 ); break;
           case IDD_CIRCLES:
           case IDD_SQUARES:
           case IDD_NONE:  iPattern = uParam; break;
           }

         // Force the repaint
         InvalidateRect ( hWnd, NULL, FALSE );
         break;

//************** NEW CODE END *********

      case WM_COMMAND:  // message: command from application menu

         // Message packing of uParam and lParam have changed for Win32,
         // let us handle the differences in a conditional compilation:
         #if defined (_WIN32)
             wmId    = LOWORD(uParam);
			    wmEvent = HIWORD(uParam);
         #else
            wmId    = uParam;
            wmEvent = HIWORD(lParam);
         #endif

         switch (wmId) {
            case IDM_ABOUT:
               lpProcAbout = MakeProcInstance((FARPROC)About, hInst);

               DialogBox(hInst,           // current instance         
                  "AboutBox",             // dlg resource to use      
                  hWnd,                   // parent handle            
                  (DLGPROC)lpProcAbout);  // About() instance address 

               FreeProcInstance(lpProcAbout);
               break;

            case IDM_EXIT:
               DestroyWindow (hWnd);
               break;

            case IDM_HELPCONTENTS:
               if (!WinHelp (hWnd, "GENERIC.HLP", HELP_KEY,(DWORD)(LPSTR)"CONTENTS")) {
                  MessageBox (GetFocus(),
                     "Unable to activate help",
                     szAppName, MB_SYSTEMMODAL|MB_OK|MB_ICONHAND);
				  
               }
               break;

            case IDM_HELPSEARCH:
               if (!WinHelp(hWnd, "GENERIC.HLP", HELP_PARTIALKEY, (DWORD)(LPSTR)"")) {
                  MessageBox (GetFocus(),
                     "Unable to activate help",
                     szAppName, MB_SYSTEMMODAL|MB_OK|MB_ICONHAND);
               }
               break;

            case IDM_HELPHELP:
               if(!WinHelp(hWnd, (LPSTR)NULL, HELP_HELPONHELP, 0)) {
                  MessageBox (GetFocus(),
                     "Unable to activate help",
                     szAppName, MB_SYSTEMMODAL|MB_OK|MB_ICONHAND);
               }
               break;


//************** NEW CODE START *********

            // Call the three test functions

            case IDM_PREPLANNED:

               PrePlannedPropertySheet(hWnd);
               break;

            case IDM_DYNAMIC:

               DynamicPropertySheet(hWnd);
               break;

            case IDM_SINGLEPROC:

               SingleProcPropertySheet(hWnd);
               break;

//************** NEW CODE END *********

            // Here are all the other possible menu options,
            // all of these are currently disabled:
			case IDM_NEW:
            case IDM_OPEN:
            case IDM_SAVE:
            case IDM_SAVEAS:
            case IDM_UNDO:
            case IDM_CUT:
            case IDM_COPY:
            case IDM_PASTE:
            case IDM_LINK:
            case IDM_LINKS:

            default:
               return (DefWindowProc(hWnd, message, uParam, lParam));
         }
         break;

      case WM_DESTROY:  // message: window being destroyed

         PostQuitMessage(0);
         break;

      default:          // Passes it on if unproccessed
         return (DefWindowProc(hWnd, message, uParam, lParam));
   }
   return (0);
}

//
//   FUNCTION: CenterWindow (HWND, HWND)
//
//   PURPOSE:  Center one window over another
//   
//   COMMENTS:
//   
//      Dialog boxes take on the screen position that they were designed 
//      at, which is not always appropriate. Centering the dialog over a 
//      particular window usually results in a better position.
//

BOOL CenterWindow (HWND hwndChild, HWND hwndParent)
{
   RECT    rChild, rParent;
   int     wChild, hChild, wParent, hParent;
   int     wScreen, hScreen, xNew, yNew;
   HDC     hdc;

   // Get the Height and Width of the child window
   GetWindowRect (hwndChild, &rChild);
   wChild = rChild.right - rChild.left;
   hChild = rChild.bottom - rChild.top;

   // Get the Height and Width of the parent window
   GetWindowRect (hwndParent, &rParent);
   wParent = rParent.right - rParent.left;
   hParent = rParent.bottom - rParent.top;

   // Get the display limits
   hdc = GetDC (hwndChild);
   wScreen = GetDeviceCaps (hdc, HORZRES);
   hScreen = GetDeviceCaps (hdc, VERTRES);
   ReleaseDC (hwndChild, hdc);

   // Calculate new X position, then adjust for screen
   xNew = rParent.left + ((wParent - wChild) /2);
   if (xNew < 0) {
      xNew = 0;
   }
   else if ((xNew+wChild) > wScreen) {
      xNew = wScreen - wChild;
   }

   // Calculate new Y position, then adjust for screen
   yNew = rParent.top  + ((hParent - hChild) /2);
   if (yNew < 0) {
      yNew = 0;
   }
   else if ((yNew+hChild) > hScreen) {
      yNew = hScreen - hChild;
   }

   // Set it, and return
   return SetWindowPos (hwndChild, NULL,
      xNew, yNew, 0, 0, SWP_NOSIZE | SWP_NOZORDER);
}


//
//   FUNCTION: About(HWND, UINT, WPARAM, LPARAM)
//
//   PURPOSE:  Processes messages for "About" dialog box
//   
//   MESSAGES:
//   
//      WM_INITDIALOG - initialize dialog box
//      WM_COMMAND    - Input received
//   
//   COMMENTS:
//   
//      Display version information from the version section of the
//      application resource.
//      
//      Wait for user to click on "Ok" button, then close the dialog box.
//

LRESULT CALLBACK About(
      HWND hDlg,           // window handle of the dialog box
      UINT message,        // type of message
      WPARAM uParam,       // message-specific information
      LPARAM lParam
      )
{
   static  HFONT hfontDlg;

   switch (message) {
      case WM_INITDIALOG:  // message: initialize dialog box
         // Create a font to use
         hfontDlg = CreateFont(14, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0,
            VARIABLE_PITCH | FF_SWISS, "");

         // Center the dialog over the application window
         CenterWindow (hDlg, GetWindow (hDlg, GW_OWNER));

         return (TRUE);

      case WM_COMMAND:                      // message: received a command
         if (LOWORD(uParam) == IDOK         // "OK" box selected?
            || LOWORD(uParam) == IDCANCEL) {// System menu close command? 
            EndDialog(hDlg, TRUE);          // Exit the dialog
            DeleteObject (hfontDlg);
            return (TRUE);
         }
         break;
   }
   return (FALSE); // Didn't process the message

   lParam; // This will prevent 'unused formal parameter' warnings
}

//************** NEW CODE START *********

// MyPropSheetPageEnumProc is called by each property sheet
// page receiving the PSM_QUERYSIBLINGS message. In this
// ultra cheezy example, param1 is a pointer to an integer,
// and we simply increment the value pointed to by the
// pointer.

BOOL MyPropSheetPageEnumProc( HWND hDlgPage, WPARAM param1, LPARAM param2 )
{
  LPINT lpCount = (LPINT) param2;
  (*lpCount)++;
  return TRUE;
}

// The SingleProcDialogProc is the dialog procedure for
// both the "Gender" and "Age" pages in the SingleProcPropertySheet
// example function.

LRESULT CALLBACK SingleProcDialogProc(HWND hDlg,
                             UINT uMessage,
                             WPARAM uParam,
                             LPARAM lParam)
{
  static char szName[32]; // Static buffer to hold name between
                          // page transitions. Must be static to
                          // hold contents between PSN_KILLACTIVE
                          // and PSN_SETACTIVE messages.

  switch ( uMessage )
    {
    case WM_INITDIALOG:

      // If this is the first WM_INITDIALOG, page #0 (stored in lParam of
      // the PROPSHEETPAGE structure, then clear the static text

      if (!((LPCPROPSHEETPAGE)lParam)->lParam) *szName = 0;

      // Cache away the pointer to this pages PROPSHEETPAGE struct so
      // we can tell in the future what page we are in if we desire.

      SetWindowLong ( hDlg, DWL_USER, (LONG)lParam ); 
      break;

    case WM_NOTIFY:

      switch (((LPNMHDR)lParam)->code)
        {
        // When a page loses focus, store away the contents of the
        // edit control into the static variable.

        case PSN_KILLACTIVE:

          SetWindowLong ( hDlg, DWL_MSGRESULT, 0 );
          GetDlgItemText ( hDlg, IDD_NAME, szName, sizeof(szName));
          return TRUE;

        // When a page gains focus, set the contents of the
        // edit control from the static variable.

        case PSN_SETACTIVE:

          SetWindowLong ( hDlg, DWL_MSGRESULT, 0 );
          SetDlgItemText ( hDlg, IDD_NAME, szName);
          return TRUE;

        default:

          return FALSE;
        }
      break;

    default:

      break;
    }
    return FALSE;
}

// The PatternPageDialogProc is the dialog procedure for
// the "Pattern" pages in the PrePlannedPropertySheet and
// DynamicPropertySheet sample functions.

LRESULT CALLBACK PatternPageDialogProc(HWND hDlg,
                             UINT uMessage,
                             WPARAM uParam,
                             LPARAM lParam)
{
   int wmId, wmEvent;

   switch ( uMessage )
    {
    // This is our Poor Man's enum child pages procedure.
    // If we recieve this message, call the MyPropSheetPageEnumProc
    // function.

    case PSM_QUERYSIBLINGS:

      return (!MyPropSheetPageEnumProc( hDlg, uParam, lParam ));

    case WM_NOTIFY:

      switch (((LPNMHDR)lParam)->code)
        {
        case PSN_APPLY: // User pressed the Apply button

          // Send the appropriate WM_MYCHANGESETTINGS message
          // back to the top level main window.

          if (IsDlgButtonChecked(hDlg, IDD_NONE)) 
            SendMessage (ghWnd, WM_MYCHANGESETTINGS, IDD_NONE, 0L );
          if (IsDlgButtonChecked(hDlg, IDD_CIRCLES)) 
            SendMessage (ghWnd, WM_MYCHANGESETTINGS, IDD_CIRCLES, 0L );
          if (IsDlgButtonChecked(hDlg, IDD_SQUARES)) 
            SendMessage (ghWnd, WM_MYCHANGESETTINGS, IDD_SQUARES, 0L );
          return TRUE;

        default:

          return FALSE;
        }
      break;

    case WM_COMMAND:

       // Message packing of uParam and lParam have changed for Win32,
       // let us handle the differences in a conditional compilation:
       #if defined (_WIN32)
          wmId    = LOWORD(uParam);
          wmEvent = HIWORD(uParam);
       #else
          wmId    = uParam;
          wmEvent = HIWORD(lParam);
       #endif
       switch (wmEvent)
         {
         case BN_CLICKED:

           // If the user clicks on a radio button, enable
           // the Apply button.

           PropSheet_Changed(GetParent(hDlg), hDlg);
           break;
         }
       break;

    default:

      break;
    }
  return FALSE;
}

// The ColorPageDialogProc is the dialog procedure for
// the "Pattern" pages in the PrePlannedPropertySheet and
// DynamicPropertySheet sample functions.

LRESULT CALLBACK ColorPageDialogProc(HWND hDlg,
                            UINT uMessage,
                            WPARAM uParam,
                            LPARAM lParam)
{
   int wmId, wmEvent;
   int iCount;

   switch ( uMessage )
    {
    // This is our Poor Man's enum child pages procedure.
    // If we recieve this message, call the MyPropSheetPageEnumProc
    // function.

    case PSM_QUERYSIBLINGS:

      return (!MyPropSheetPageEnumProc( hDlg, uParam, lParam ));

    case WM_NOTIFY:

      switch (((LPNMHDR)lParam)->code)
        {
        case PSN_APPLY: // User pressed the Apply button

          // Send the appropriate WM_MYCHANGESETTINGS message
          // back to the top level main window.

          if (IsDlgButtonChecked(hDlg, IDD_RED)) 
            SendMessage (ghWnd, WM_MYCHANGESETTINGS, IDD_RED, 0L );
          if (IsDlgButtonChecked(hDlg, IDD_GREEN)) 
            SendMessage (ghWnd, WM_MYCHANGESETTINGS, IDD_GREEN, 0L );
          if (IsDlgButtonChecked(hDlg, IDD_BLUE)) 
            SendMessage (ghWnd, WM_MYCHANGESETTINGS, IDD_BLUE, 0L );
          return TRUE;

        default:

          return FALSE;
        }
      break;

    case WM_COMMAND:

       // Message packing of uParam and lParam have changed for Win32,
       // let us handle the differences in a conditional compilation:
       #if defined (_WIN32)
          wmId    = LOWORD(uParam);
          wmEvent = HIWORD(uParam);
       #else
          wmId    = uParam;
          wmEvent = HIWORD(lParam);
       #endif
       switch (wmEvent)
         {
         case BN_CLICKED:

           switch ( wmId )
             {
             case IDD_SOMEBUTTON: // Add a new page if allowed

               // First, let's count the number of buttons

               // Reset counter
               iCount = 0;

               // Invoke the poor man's version of enumerating pages
               PropSheet_QuerySiblings ( GetParent(hDlg), (WPARAM)(&iCount), (LPARAM) (LPINT) &iCount );

               // Only add a page if we have less than two pages
               if ( iCount < 2 )
                 {
                 HPROPSHEETPAGE hPage;
                 PROPSHEETPAGE psp;

                 // Fill out the "Pattern" page structure
                 psp.dwSize      = sizeof(PROPSHEETPAGE);
                 psp.dwFlags     = PSP_USETITLE;
                 psp.hInstance   = hInst;
                 psp.pszTemplate = "PATTERNPAGE";
                 psp.pfnDlgProc  = PatternPageDialogProc;
                 psp.pszTitle    = "&Pattern";
                 psp.lParam      = 0;

                 // Create a page, get back a handle
                 hPage = CreatePropertySheetPage ( &psp );

                 // Add the page to the property sheet
                 PropSheet_AddPage ( GetParent (hDlg), hPage);

                 // Select that new page (this insures the dialog
                 // window for that page is created, and the
                 // Poor Man's enum pages idea from above works.
                 // Since this is the second page, it has the index
                 // of 1.
                 PropSheet_SetCurSel ( GetParent(hDlg), NULL, 1 );
                 }
               else
                 {
                 // Kibosh!
                 MessageBox ( hDlg, "Is this trip really necessary?", "MSJ", MB_OK );
                 }
               break;

             default: // Must be one of the radio buttons

               // If the user clicks on a radio button, enable
               // the Apply button.

               PropSheet_Changed(GetParent(hDlg), hDlg);
               break;
             } /* end switch wmID */
           break;
         } /* end switch wmEvent */
       break;

    default:

      break;
    } /* end switch uMessage */
  return FALSE;
}

// The PrePlannedPropertySheet sample function
// creates a property sheet with two pages, the
// "Color" page and the "Pattern" page. Each page
// has it's own dialog procedure and dialog template.

VOID PrePlannedPropertySheet(HWND hWndParent)
{
    PROPSHEETPAGE psp[2];
    PROPSHEETHEADER psh;

    // First, initalize the array of pages to
    // indicate each page.

    psp[0].dwSize      = sizeof(PROPSHEETPAGE);
    psp[0].dwFlags     = PSP_USETITLE;
    psp[0].hInstance   = hInst;
    psp[0].pszTemplate = "COLORPAGE";
    psp[0].pfnDlgProc  = ColorPageDialogProc;
    psp[0].pszTitle    = "&Color";
    psp[0].lParam      = 0;

    psp[1].dwSize      = sizeof(PROPSHEETPAGE);
    psp[1].dwFlags     = PSP_USETITLE;
    psp[1].hInstance   = hInst;
    psp[1].pszTemplate = "PATTERNPAGE";
    psp[1].pfnDlgProc  = PatternPageDialogProc;
    psp[1].pszTitle    = "&Pattern";
    psp[1].lParam      = 0;

    // Now, initialize the property sheet strucuture
    // to use these two pages and parent it to the
    // main window.

    psh.dwSize         = sizeof(PROPSHEETHEADER);
    psh.dwFlags        = PSH_PROPSHEETPAGE | PSH_PROPTITLE;
    psh.hwndParent     = hWndParent;
    psh.hInstance      = hInst;
    psh.pszIcon        = NULL;
    psh.pszCaption     = (LPSTR)"Client Area";
    psh.nPages         = sizeof(psp) / sizeof(PROPSHEETPAGE);
    psh.ppsp           = (LPCPROPSHEETPAGE) &psp;

    // And finally, display the property sheet.

    PropertySheet(&psh);

    return;
}

// The DynamicPropertySheet is exactly like the
// PrePlannedPropertySheet function, except that it
// only adds the first page, and uses a slightly different
// template for that first page (this template has an
// "Add Page" button on it.

VOID DynamicPropertySheet(HWND hWndParent)
{
    // Still use an array because this is cut & paste code
    PROPSHEETPAGE   psp[1];
    PROPSHEETHEADER psh;

    // First, initalize the (small) array of pages to
    // indicate each page.

    psp[0].dwSize      = sizeof(PROPSHEETPAGE);
    psp[0].dwFlags     = PSP_USETITLE;
    psp[0].hInstance   = hInst;
    psp[0].pszTemplate = "COLORPAGEDYNO";
    psp[0].pfnDlgProc  = ColorPageDialogProc;
    psp[0].pszTitle    = "&Color";
    psp[0].lParam      = 0;

    // Now, initialize the property sheet strucuture
    // to use the page and parent it to the
    // main window.

    psh.dwSize         = sizeof(PROPSHEETHEADER);
    psh.dwFlags        = PSH_PROPSHEETPAGE | PSH_PROPTITLE;
    psh.hwndParent     = hWndParent;
    psh.hInstance      = hInst;
    psh.pszIcon        = NULL;
    psh.pszCaption     = (LPSTR)"Client Area";
    psh.nPages         = sizeof(psp) / sizeof(PROPSHEETPAGE);
    psh.ppsp           = (LPCPROPSHEETPAGE) &psp;

    // And finally, display the property sheet.

    PropertySheet(&psh);

    return;
}

// The SingleProcPropertySheet sample function
// creates a property sheet with two pages, the
// "Gender" page and the "Age" page. Each page
// uses the same dialog procedure, but each page
// uses a different dialog template. Controls that
// are shred between pages (such as the edit control
// in this sample) are placed at the same coorindates, and
// use the same ID. In addition, the lParam field of
// the PROPSHEETPAGE structure is used to indicate the page.
// The dialog procedure can access the instance-specific
// PROPSHEETPAGE structure for each page, and the lParam
// can then be referenced.

VOID SingleProcPropertySheet(HWND hWndParent)
{
    PROPSHEETPAGE psp[2];
    PROPSHEETHEADER psh;

    // First, initalize the array of pages to
    // indicate each page.

    psp[0].dwSize      = sizeof(PROPSHEETPAGE);
    psp[0].dwFlags     = PSP_USETITLE;
    psp[0].hInstance   = hInst;
    psp[0].pszTemplate = "PAGE0SINGLEPROC";
    psp[0].pfnDlgProc  = SingleProcDialogProc;
    psp[0].pszTitle    = "&Gender";
    psp[0].lParam      = 0; // Page number

    psp[1].dwSize      = sizeof(PROPSHEETPAGE);
    psp[1].dwFlags     = PSP_USETITLE;
    psp[1].hInstance   = hInst;
    psp[1].pszTemplate = "PAGE1SINGLEPROC";
    psp[1].pfnDlgProc  = SingleProcDialogProc;
    psp[1].pszTitle    = "&Age";
    psp[1].lParam      = 1; // Page number

    // Now, initialize the property sheet strucuture
    // to use these two pages and parent it to the
    // main window.

    psh.dwSize         = sizeof(PROPSHEETHEADER);
    psh.dwFlags        = PSH_PROPSHEETPAGE | PSH_PROPTITLE;
    psh.hwndParent     = hWndParent;
    psh.hInstance      = hInst;
    psh.pszIcon        = NULL;
    psh.pszCaption     = (LPSTR)"Carbon Based Unit";
    psh.nPages         = sizeof(psp) / sizeof(PROPSHEETPAGE);
    psh.ppsp           = (LPCPROPSHEETPAGE) &psp;

    // And finally, display the property sheet.

    PropertySheet(&psh);

    return;
}

//************** NEW CODE END *********


