/* diskio.c - disk benchmark
 *
 * Author:  Kai Uwe Rommel <rommel@ars.muc.de>
 * Created: Fri Jul 08 1994
 */
 
static char *rcsid =
"$Id: diskio.c,v 1.9 1995/12/31 20:24:09 rommel Exp rommel $";
static char *rcsrev = "$Revision: 1.9 $";

/*
 * $Log: diskio.c,v $
 * Revision 1.9  1995/12/31 20:24:09  rommel
 * Changed CPU load calculation
 * General cleanup
 *
 * Revision 1.8  1995/12/28 11:28:07  rommel
 * Fixed async timer problem.
 *
 * Revision 1.7  1995/12/28 10:04:15  rommel
 * Added CPU benchmark (concurrently to disk I/O)
 *
 * Revision 1.6  1995/11/24 16:02:10  rommel
 * Added bus/drive cache speed test by 
 * repeatedly reading a small amount of data
 *
 * Revision 1.5  1995/08/09 13:07:02  rommel
 * Changes for new diskacc2 library, minor corrections, arguments.
 *
 * Revision 1.4  1994/07/11 14:23:00  rommel
 * Changed latency timing
 *
 * Revision 1.3  1994/07/09 13:07:20  rommel
 * Changed transfer speed test
 *
 * Revision 1.2  1994/07/08 21:53:05  rommel
 * Cleanup
 *
 * Revision 1.1  1994/07/08 21:29:41  rommel
 * Initial revision
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define INCL_DOS
#define INCL_DOSDEVICES
#define INCL_DOSDEVIOCTL
#define INCL_DOSERRORS
#define INCL_NOPM
#include <os2.h>

#include "diskacc2.h"

#define INTERVAL 10

int nHandle;
unsigned nSides, nTracks, nSectors;
char *pBuffer;

int time_over;
long dhry_time, dhry_result;

extern unsigned long Number_Of_Runs;
extern long dhry_stone(void);

VOID APIENTRY timer_thread(ULONG nArg)
{
#if 1
  HEV hSem;
  HTIMER hTimer;

  DosCreateEventSem(0, &hSem, DC_SEM_SHARED, 0);
  DosAsyncTimer(nArg * 1000, (HSEM) hSem, &hTimer);
  DosWaitEventSem(hSem, SEM_INDEFINITE_WAIT);
  DosStopTimer(hTimer);
  DosCloseEventSem(hSem);
#else
  DosSleep(nArg * 1000);
#endif

  time_over = 1;
  Number_Of_Runs = 0;

  DosExit(EXIT_THREAD, 0);
}

int start_alarm(ULONG nSeconds)
{ 
  TID ttid;

  time_over = 0;
  Number_Of_Runs = -1;

  if (DosCreateThread(&ttid, timer_thread, nSeconds, 0, 8192))
    return printf("Cannot create timer thread.\n"), -1;

  return 0;
}

int start_timer(QWORD *nStart)
{
  if (DosTmrQueryTime(nStart))
    return printf("Timer error.\n"), -1;

  return 0;
}

int stop_timer(QWORD *nStart, int accuracy)
{
  QWORD nStop;
  ULONG nFreq;

  if (DosTmrQueryTime(&nStop))
    return printf("Timer error.\n"), -1;
  if (DosTmrQueryFreq(&nFreq))
    return printf("Timer error.\n"), -1;

  nFreq = (nFreq + accuracy / 2) / accuracy;

  return (nStop.ulLo - nStart->ulLo) / nFreq;
}

int bench_transfer(int nTrack, int nDirection)
{
  int nCnt, nData = 0, nTime;
  QWORD nLocal;

  printf("Data transfer rate on track %-4d: ", nTrack);
  fflush(stdout);

  if (start_alarm(INTERVAL))
    return -1;

  if (start_timer(&nLocal))
    return -1;

  for (nCnt = 0; !time_over; nCnt++)
  {
    if (DskRead(nHandle, nCnt % nSides, 
		nTrack + (nCnt / nSides) * nDirection, 1, nSectors, pBuffer))
      return printf("Disk read error.\n"), -1;

    nData += nSectors * 512;
  }

  if ((nTime = stop_timer(&nLocal, 1024)) == -1)
    return -1;

  printf("%d k/sec\n", nData / nTime);

  return 0;
}

int bench_bus(void)
{
  int nCnt, nData = 0, nTime;
  QWORD nLocal;

  printf("Drive cache/bus transfer rate: ");
  fflush(stdout);

  if (start_alarm(INTERVAL))
    return -1;

  if (start_timer(&nLocal))
    return -1;

  for (nCnt = 0; !time_over; nCnt++)
  {
    if (DskRead(nHandle, 0, 0, 1, nSectors, pBuffer))
      return printf("Disk read error.\n"), -1;

    nData += nSectors * 512;
  }

  if ((nTime = stop_timer(&nLocal, 1024)) == -1)
    return -1;

  printf("%d k/sec\n", nData / nTime);

  return 0;
}

VOID APIENTRY dhry_thread(ULONG nArg)
{
  DosSetPriority(PRTYS_THREAD, PRTYC_IDLETIME, PRTYD_MAXIMUM, 0);

  dhry_time = dhry_stone();

  DosExit(EXIT_THREAD, 0);
}

int bench_concurrent(void)
{
  int nCnt, nData = 0, nTime;
  long percent;
  QWORD nLocal;
  TID dtid;
  APIRET rc;

  printf("Concurrent CPU activity and disk transfers: ");
  fflush(stdout);

  if (start_alarm(INTERVAL))
    return -1;

  if (DosCreateThread(&dtid, dhry_thread, 0, 0, 8192))
    return -1;

  if (start_timer(&nLocal))
    return -1;

  for (nCnt = 0; !time_over; nCnt++)
  {
    if (DskRead(nHandle, 0, 0, 1, nSectors, pBuffer))
      return printf("Disk read error.\n"), -1;

    nData += nSectors * 512;
  }

  if ((nTime = stop_timer(&nLocal, 1024)) == -1)
    return -1;

  if ((rc = DosWaitThread(&dtid, DCWW_WAIT)) && rc != ERROR_INVALID_THREADID)
    return -1;                             /* it may have already terminated */

  dhry_time = (dhry_time + 500) / 1000;
  percent = (dhry_result - Number_Of_Runs / dhry_time) * 100 / dhry_result;

  printf("%d%% CPU usage for disk I/O\n", percent);

  return 0;
}

int bench_latency(void)
{
  int nCnt, nSector, nTime;
  QWORD nLocal;

  printf("Average latency time: ");
  fflush(stdout);

  srand(1);

  if (start_alarm(INTERVAL))
    return -1;

  if (start_timer(&nLocal))
    return -1;

  for (nCnt = 0; !time_over; nCnt++)
  {
    nSector = rand() * nSectors / RAND_MAX + 1;

    if (DskRead(nHandle, 0, 0, nSector, 1, pBuffer))
      return printf("Disk read error.\n"), -1;
  }

  if ((nTime = stop_timer(&nLocal, 1000)) == -1)
    return -1;

  nTime = nTime * 10 / nCnt;

  printf("%d.%d ms\n", nTime / 10, nTime % 10);

  return nTime;
}

int bench_seek(void)
{
  int nCnt, nTrack, nTime;
  QWORD nLocal;

  printf("Average data access time: ");
  fflush(stdout);

  srand(1);

  if (start_alarm(INTERVAL))
    return -1;

  if (start_timer(&nLocal))
    return -1;

  for (nCnt = 0; !time_over; nCnt++)
  {
    nTrack = rand() * nTracks / RAND_MAX;

    if (DskRead(nHandle, 0, nTrack, 1, 1, pBuffer))
      return printf("Disk read error.\n"), -1;
  }

  if ((nTime = stop_timer(&nLocal, 1000)) == -1)
    return -1;

  nTime = nTime * 10 / nCnt;

  printf("%d.%d ms\n", nTime / 10, nTime % 10);

  return 0;
}

int bench_disk(int nDisk)
{
  char szName[8];

  sprintf(szName, "$%d:", nDisk);

  if ((nHandle = DskOpen(szName, 0, 0, &nSides, &nTracks, &nSectors)) < 0)
    return printf("\nCannot access disk %d.\n", nDisk), -1;

  printf("\nDisk %d: %d sides, %d cylinders, %d sectors per track = %d MB\n", 
	 nDisk, nSides, nTracks, nSectors,
	 nSides * nTracks * nSectors / 2048);

  if ((pBuffer = malloc(nSectors * 512)) == NULL)
    return printf("\nNot enough memory.\n"), -1;

  bench_bus();
  bench_transfer(0, 1);
  bench_transfer(nTracks - 1, -1);
  bench_concurrent();
  bench_latency();
  bench_seek();

  free(pBuffer);
  DskClose(nHandle);

  return 0;
}

int bench_dhry(void)
{
  printf("Dhrystone benchmark for this CPU: ");
  fflush(stdout);

  if (start_alarm(INTERVAL / 2))
    return -1;

  dhry_time = dhry_stone();

  dhry_time = (dhry_time + 500) / 1000;
  dhry_result = Number_Of_Runs / dhry_time;

  printf("%d runs/sec\n", dhry_result);

  return 0;
}

int main(int argc, char **argv)
{
  char szVersion[32];
  USHORT nDisks;
  int nCount;

  strcpy(szVersion, rcsrev + sizeof("$Revision: ") - 1);
  *strchr(szVersion, ' ') = 0;

  printf("\nDISKIO - Fixed Disk Benchmark, Version %s"
	 "\n(C) 1994-1995 Kai Uwe Rommel\n"
	 "\nDhrystone 2.1 C benchmark routines (C) 1988 Reinhold P. Weicker\n",
	 szVersion);

  bench_dhry();

  if (DosPhysicalDisk(INFO_COUNT_PARTITIONABLE_DISKS, &nDisks, sizeof(nDisks), 0, 0))
    return printf("\nCannot determine number of disks.\n"), 1;

  printf("\nNumber of fixed disks: %d\n", nDisks);

  if (argc > 1)
    for (nCount = 1; nCount < argc; nCount++)
      bench_disk(atoi(argv[nCount]));
  else
    for (nCount = 1; nCount <= nDisks; nCount++)
      bench_disk(nCount);

  return 0;
}

/* end of diskio.c */
