/**********************************************************************/
/*                                                                    */
/*  REXX Drive Waste Scanner v.02                                     */
/*                                                                    */
/*  Written by Mike Ruskai <mruskai@microfone.net>                    */
/*                                                                    */
/*  Distribute and modify freely, but it'd be nice if you give me     */
/*  credit if you distribute a modified form.                         */
/*                                                                    */
/*  Usage:  DRIVECHK.CMD [<drive letter>: [/S] [/O:<filename>]]       */
/*                                                                    */
/*  Parameters are optional.  <drive letter>: is self explanatory.    */
/*  The /S switch tells the program skip extended attributes, and     */
/*  can only be used when the drive letter is passed.  The /O switch  */
/*  tells the program to write the report to <filename>.  If no       */
/*  options are chosen, the program will prompt for the input of the  */
/*  drive to scan, whether or not to check extended attributes, and   */
/*  the method of report output.                                      */
/*                                                                    */
/*  This REXX script will scan an entire drive to determine how much  */
/*  space would be wasted for your specific files with the HPFS and   */
/*  FAT file systems.  The display of this information depends on     */
/*  which file system is actually being used.                         */
/*                                                                    */
/*  I've tried to include as much information as I know how.  In      */
/*  addition to the normal file system usage for a file and a         */
/*  directory, the extended attributes are also added up and used     */
/*  in space and slack space determinations.  For files that do not   */
/*  have any extended attributes, the size of the Fnode (512 bytes)   */
/*  for the HPFS scenario is added to the slack space for the file,   */
/*  since it would not exist in the FAT scenario.  That also goes     */
/*  for zero-byte files, which take up 512 bytes on HPFS drives, but  */
/*  zero bytes on FAT drives.                                         */
/*                                                                    */
/*  What's *not* included is the space consumed by the system areas   */
/*  specific to the file system.  The reason for this exclusion is    */
/*  that I don't know how to measure this size for HPFS volumes       */
/*  without capturing the output of CHKDSK (time-consuming), since I  */
/*  see no correlation between system area size and total disk size.  */
/*  My own HPFS drives have system areas ranging from 2MB to 6.5MB,   */
/*  with smaller drives sometimes having bigger system areas than     */
/*  some larger drives.                                               */
/*                                                                    */
/*  As stated, extended attributes are included, but the process by   */
/*  which they are is quite slow.  The only way via REXX to find the  */
/*  size of a file's extended attributes was to use the output of     */
/*  CMD.EXE or 4OS2.EXE's DIR command (using the /N parameter on FAT  */
/*  volumes).  Piping the output to RXQUEUE was very slow, and        */
/*  redirecting the output to a file, then reading from the file was  */
/*  much faster, but still slow.  4OS2.EXE tends to be about 60%      */
/*  slower than CMD.EXE in loading itself into memory (more features  */
/*  to load).  This would normally prompt you to type CMD before      */
/*  running this, given the frequency with which the interpreter is   */
/*  called (if you knew this about CMD and 4OS2, that is), but here,  */
/*  you needn't bother.  Because of some extra work necessitated by   */
/*  the output of CMD, the speed balances out - both CMD and 4OS2     */
/*  are equally snail-like in this process.  Because of the horrible  */
/*  speed, you might want to skip EA checking altogether.  The        */
/*  significance of EA's in space determinations vary from drive to   */
/*  drive, though.  If you know that your files have quite a lot of   */
/*  extended attributes, you should just suffer to get results which  */
/*  more accurately represent reality.                                */
/*                                                                    */
/**********************************************************************/

call RxFuncAdd 'SysLoadFuncs','RexxUtil','SysLoadFuncs'
call SysLoadFuncs

/* Checks to see if a drive was passed on the command line, and       */
/* whether or not to skip extended attributes.                        */

CurrDir=directory()
arg TestDrive SkipSwitch OutFile

if TestDrive='/?' | TestDrive='?' then signal Usage

if TestDrive\='' then do
    if directory(TestDrive||'\')\=TestDrive||'\' then do
        say 'Invalid drive, '||'"'TestDrive'"'||'.'
        exit
    end
    else call directory CurrDir
    if translate(SkipSwitch)='/S' then SkipEA=1
        else SkipEA=0
    if OutFile\='' then do
        if substr(OutFile,1,2)\='/O' then do
            say 'Invalid outfile option.'
            exit
        end
        else do
            OutFile=substr(OutFile,4)
            if substr(OutFile,2,1)\=':' then do
                if pos('\',OutFile)=0 then 
                    OutFile=CurrDir||'\'||OutFile
                else if pos('\',OutFile)=1 then
                    OutFile=left(CurrDir,2)||OutFile
                    else OutFile=CurrDir||'\'||OutFile
            end
            check=lineout(OutFile,'1')
            if check\=0 then do
                say 'Invalid outfile option, '||OutFile
                exit
            end
            else do
                call stream OutFile,'c','close'
                call SysFileDelete OutFile
            end
        end
    end
    else OutFile='CON'
end

/* If no drive was passed to the program, it prompts the user to      */
/* input one, repeating if an invalid drive is given.                 */

else do
    do until ans=1
        call SysCls
        if ans=0 then do
            call SysCurPos 8,0
            say '"'TestDrive'"'||', is an invalid drive.'
        end
        call SysCurPos 10,0
        say 'Which drive do you wish to check (e.g. C)?'
        call charout ,': '
        TDrive=translate(SysGetKey('echo'))
        TestDrive=TDrive||':'        
        if directory(TestDrive||'\')\=TestDrive||'\' then ans=0
            else ans=1
    end
    ans=2
    do until ans=1
        call SysCls
        if ans=0 then do
            call SysCurPos 8,0
            say "Please answer only with 'Y' or 'N'."
        end
        call SysCurPos 10,0
        say 'Do you wish to skip EA checking? (Y/N)'
        call charout ,': '
        res=translate(SysGetKey('echo'))
        if res\='Y' & res\='N' then ans=0
            else do
                ans=1
                if res='Y' then SkipEA=1
                else SkipEA=0
            end
    end
    ans=2
    do until ans=1
        call SysCls
        if ans=0 then do
            call SysCurPos 8,0
            say "Please answer only with 'Y' or 'N'."
        end
        call SysCurPos 10,0
        say 'Do you wish to direct output to a file? (Y/N)'
        call charout ,': '
        res=translate(SysGetKey('echo'))
        if res\='Y' & res\='N' then ans=0
            else do
                ans=1
                if res='Y' then GetOF=1
                else OutFile='CON'
            end
    end
    if GetOF=1 then do
        ans=2
        do until ans=1
            call SysCls
            if ans=0 then do
                call SysCurPos 8,0
                say '"'OFName'"'||' is not a valid filename.'
            end
            call SysCurPos 10,0
            say 'Enter the name for the output file.'
            call charout ,': '
            parse pull OFName
            if substr(OFName,2,1)\=':' then do
                if pos('\',OutFile)=0 then 
                    OFName=CurrDir||'\'||OFName
                else if pos('\',OFName)=1 then
                    OFName=left(CurrDir,2)||OFName
                    else OFName=CurrDir||'\'||OFName
            end
            check=lineout(OFName,'1')
            if check\=0 then ans=0
                else do
                    OutFile=OFName
                    call stream OFName,'c','close'
                    call SysFileDelete OFName
                    do until ans2=1
                        call SysCls
                        if ans2=0 then do
                            call SysCurPos 8,0
                            say "Please answer only with 'Y' or 'N'."
                        end
                        call SysCurPos 10,0
                        say 'Use '||'"'OutFile'"'||' for output? (Y/N)'
                        call charout ,': '
                        res=translate(SysGetKey('echo'))
                        if res\='Y' & res\='N' then ans2=0
                            else do
                                ans2=1
                                if res='Y' then ans=1
                                else ans=2
                            end
                    end
                end
        end
    end
end

call directory TestDrive||'\'

/* Uses the total drive space as reported by SysDriveInfo to deter-   */
/* mine the cluster size that FAT would use on the drive.  It then    */
/* attempts to write a file with a long name to determine if the      */
/* drive being scanned is HPFS or FAT, and orders the allocation unit */
/* variables accordingly (affects report at program conclusion).      */

DriveSize=word(SysDriveInfo(TestDrive),3)
if DriveSize<16777216 then ClustSize=4096
else do
    if DriveSize>=16777216 & DriveSize<134217728 then ClustSize=2048
    else do
        RawClusterSize=format((DriveSize/65536),,0)
        BinClust=x2b(d2x(RawClusterSize))
        ValidPartStart=pos('1',BinClust)
        DigitValue=length(substr(BinClust,ValidPartStart))
        ClustSize=2**DigitValue
    end
end
tfile='Testing For HPFS File System'
rc=lineout(tfile,'test')
if rc=1 then do
    FileSystem.1='FAT'
    FileSystem.2='HPFS'
    AUnit.1=ClustSize
    AUnit.2=512
    Fnode.1=0
    Fnode.2=512
end
else do
    FileSystem.1='HPFS'
    FileSystem.2='FAT'
    call stream tfile,'c','close'
    call SysFileDelete tfile
    AUnit.1=512
    AUnit.2=ClustSize
    Fnode.1=512
    Fnode.2=0
end

/* Quick check to see if CMD.EXE or 4OS2.EXE is being used.  It has   */
/* an effect on how EA size checking is done, since the display for   */
/* each command processor is slightly different.                      */

p1=right(d2x(random(65535)),4,'0')
p2=right(d2x(random(65535)),4,'0')
TShChk='\'||p1||p2||'.TMP'
'@ver>'||TShChk
call SysFileSearch '4OS2',TShChk,'ShCheck.'
call stream TShChk,'c','close'
call SysFileDelete TShChk
if ShCheck.0=0 then UShell='CMD'
    else UShell='4OS2'

/* Initializes the variables used for the main procedure.             */

TotFiles=0
TotSize=0
TotDirs=0
TotWaste.1=0
TotWaste.2=0
TotDirSize.1=0
TotDirSize.2=0

/* Uses CMD.EXE or 4OS2.EXE to get a quick list of all subdirectories */
/* on the drive for the computation of their space consumption, and   */
/* that of the files they contain.                                    */

call SysCls
ClrStr="                                  "
ClrStr=ClrStr||ClrStr||' '
call SysCurPos 12,0
call charout ,'Scanning : '||TestDrive||'\'
Data=ProcDir(TestDrive||'\')
TotSize=TotSize+word(Data,3)
do i=1 to 2
    TotWaste.i=TotWaste.i+word(Data,i)
    TotDirSize.i=TotDirSize.i+AUnit.i
end
TotFiles=TotFiles+word(Data,4)
p1=right(d2x(random(65535)),4,'0')
p2=right(d2x(random(65535)),4,'0')
TDList='\'||p1||p2||'.TMP'
call SysCurPos 10,0
call SysCurState 'off'
call charout ,"Getting list of directories... "
if UShell='4OS2' then 
    '@dir/b/ad/s > '||TDList
else '@dir/f/ad/s >'||TDList
call charout ,"Done"
check=lines(TDList)
call SysCurPos 12,0
call charout ,'Scanning : '
do while check=1
    WDir=linein(TDList)
    if length(WDir)>68 then 
        DDir=left(WDir,68)||'>'
    else
        DDir=WDir
    call SysCurPos 12,11
    call charout ,ClrStr
    call SysCurPos 12,11
    call charout ,DDir
    TotDirs=TotDirs+1
    Data=ProcDir(WDir)
    TotSize=TotSize+word(Data,3)
    do i=1 to 2
        TotWaste.i=TotWaste.i+word(Data,i)
        TotDirSize.i=TotDirSize.i+AUnit.i
    end
    TotFiles=TotFiles+word(Data,4)
    DEAData=TallyDirEA(WDir)
    do i=1 to 2
        TotDirSize.i=TotDirSize.i+word(DEAData,i)
    end
    check=lines(TDList)
end
call stream TDList,'c','close'
call SysFileDelete TDList

AllWaste=TotWaste.1-TotWaste.2
AllDir=TotDirSize.1-TotDirSize.2
AllTot=AllWaste+AllDir
if AllTot>=0 then 
    if AllTot=0 then
        TotalDiff=0
else do
    Modifier='*LOST*'
    TotalDiff=abs(AllTot)
end
else do
    Modifier='*SAVED*'
    TotalDiff=abs(AllTot)
end

if OutFile='CON' then do
    call SysCurPos 16,0
    call lineout ,'Press any key for statistics...'
    scrap=SysGetKey('noecho')
end

/* Reports the data in a formatted manner.  Each of the numerical     */
/* variables is deliminated with via the CoDel procedure.  Then the   */
/* screen is cleared, data presented, and conclusion stated.          */

FormDriveSize=CoDel(DriveSize)
FormTotalDiff=CoDel(TotalDiff)
do i=1 to 2
    FormDirSpace.i=CoDel(TotDirSize.i)
    FormWasteSpace.i=CoDel(TotWaste.i)
    FormUnitSize.i=CoDel(AUnit.i)
end
FormTotSize=CoDel(TotSize)
FormTotFiles=CoDel(TotFiles)
FormTotDirs=CoDel(TotDirs)

call SysCls
call SysCurPos 5,0
call lineout Outfile,"Drive "||TestDrive||" "||FormDriveSize||" bytes"
call lineout OutFile, ""
call lineout OutFile, "Total files                           : "||FormTotFiles
if SkipEA=1 then 
call lineout OutFile, "Total size of files                   : "||FormTotSize||" bytes"
else
call lineout OutFile, "Total size of files and EA's          : "||FormTotSize||" bytes"
call lineout OutFile, ""
call lineout OutFile, "Current statistics using "||FileSystem.1||" :"
call lineout OutFile, ""
call lineout OutFile, "Allocation unit size                  : "||FormUnitSize.1||" bytes"
call lineout OutFile, "Total space consumed by directories   : "||FormDirSpace.1||" bytes"
if SkipEA=1 then
call lineout OutFile, "Total wasted space for files          : "||FormWasteSpace.1||" bytes"
else                                         
call lineout OutFile, "Total wasted space for files and EA's : "||FormWasteSpace.1||" bytes"
call lineout OutFile, ""
call lineout OutFile, "If it was being used with "||FileSystem.2||" :"
call lineout OutFile, ""
call lineout OutFile, "Allocation unit size                  : "||FormUnitSize.2||" bytes"
call lineout OutFile, "Total space consumed by directories   : "||FormDirSpace.2||" bytes"
if SkipEA=1 then
call lineout OutFile, "Total wasted space for files          : "||FormWasteSpace.2||" bytes"
else
call lineout OutFile, "Total wasted space for files and EA's : "||FormWasteSpace.2||" bytes"
call lineout OutFile, ""
if TotalDiff=0 then do
call lineout OutFile, "Oddly enough, by using "||FileSystem.1||" instead of "||FileSystem.2||","
call lineout OutFile, "you've neither saved nor lost any disk space on drive "||TestDrive||"."
end
else do
call lineout OutFile, "By using "||FileSystem.1||" instead of "||FileSystem.2||", you've"
call lineout OutFile, Modifier||" "||FormTotalDiff||" bytes on drive "||TestDrive||"."
end
call directory CurrDir
exit
Usage:
call SysCls
call SysCurPos 5,0
say "DriveCheck v.02  REXX disk space waste scanner, written by:"
say ""
say "Mike Ruskai <mruskai@microfone.net>"
say ""
say "Usage: DRIVECHK.CMD [<drive letter>: [/S] [/O:<filename>]]"
say ""
say "<drive letter>:  - Drive to scan"
say "/S               - Skip extended attributes checking"
say "/O:<filename>    - Write output to <filename>"
say ""
say "Example:"
say ""
say "DRIVECHK C: /S /O:scan.log"
say ""
say " - Scans drive C:, skips EA's, writes output to 'scan.log'"
say ""
say "Run DRIVECHK.CMD with no parameters to be prompted for input."
say ""
exit
/******************** Begin TallyFiles Procedure **********************/
/* Procedure to determine total size and waste of all files in a      */
/* given directory, for both FAT and HPFS file systems.  The list of  */
/* files is obtained using SysFileTree.  For each file, it's total    */
/* size and the size of its EA's (unless skipped) are tallied for the */
/* file size total, and the waste of the file and it's EA's is added  */
/* to the waste total.  Calculations are performed by the SlackSpace  */
/* procedure, for both FAT and HPFS file systems.                     */

TallyFiles: Procedure expose Fnode.1 Fnode.2 SkipEA UShell AUnit.1 AUnit.2 

ClrStr='                               '
ClrStr=ClrStr||ClrStr||' '
TotSize=0
TotWaste.1=0
TotWaste.2=0
arg Dir
call directory Dir
call SysFileTree '*','files.','F'
call SysCurPos 14,0
call charout ,'Working on file: '
do i=1 to files.0
    if SkipEA=1 then do
        CurFile=filespec('name',(substr(files.i,38)))
        if length(CurFile)>63 then
            CurFile=substr(CurFile,1,62)||'>'
        call SysCurPos 14,17
        say ClrStr
        call SysCurPos 14,17
        call charout ,CurFile        
        FSize=word(files.i,3)
        TotSize=TotSize+FSize
        Waste=SlackSpace(Fsize)
        TotWaste.1=TotWaste.1+word(Waste,1)+Fnode.1
        TotWaste.2=TotWaste.2+word(Waste,2)+Fnode.2
    end
    else do
        file=substr(files.i,38)
        p1=right(d2x(random(65535)),4,'0')
        p2=right(d2x(random(65535)),4,'0')
        tfile='\'||p1||p2||'.TMP'
        if UShell='4OS2' then do
            '@dir/a/knm '||'"'file'"'||'>'||tfile
            CurFile=filespec('name',file)
                if length(CurFile)>63 then
            CurFile=substr(CurFile,1,62)||'>'
            call SysCurPos 14,17
            say ClrStr
            call SysCurPos 14,17
            call charout ,CurFile        
            filestring=linein(tfile)
            UFEASize=word(filestring,4)
            UFFSize=word(filestring,3)
            EACommaPos=pos(',',UFEASize)
            if EACommaPos\=0 then
                EASize=delstr(UFEASize,EACommaPos,1)
            else EASize=UFEASize
            FCommaPos=pos(',',UFFSize)
            if FCommaPos\=0 then do until FCommaPos=0
                UFFSize=delstr(UFFSize,FCommaPos,1)
                FCommaPos=pos(',',UFFSize)
            end 
            FSize=UFFSize
        end
        else do
            '@dir/a/n '||'"'file'"'||'>'||tfile
            CurFile=filespec('name',file)
                if length(CurFile)>63 then
            CurFile=substr(CurFile,1,62)||'>'
            call SysCurPos 14,17
            say ClrStr
            call SysCurPos 14,17
            call charout ,CurFile        
            do 5
                scrap=linein(tfile)
            end
            filestring=linein(tfile)
        EASize=word(filestring,4)
        FSize=word(filestring,3)
        end
        TotSize=TotSize+FSize+EASize
        FWaste=SlackSpace(FSize)
        FWaste.1=word(FWaste,1)
        FWaste.2=word(FWaste,2)
        if EASize=0 then do
            TotWaste.1=TotWaste.1+FWaste.1+Fnode.1
            TotWaste.2=TotWaste.2+FWaste.2+Fnode.2
        end
        else do
            EAWaste=SlackSpace(EASize)
            EAWaste.1=word(EAWaste,1)
            EAWaste.2=word(EAWaste,2)
            TotWaste.1=TotWaste.1+FWaste.1+EAWaste.1
            TotWaste.2=TotWaste.2+FWaste.2+EAWaste.2
        end
        call stream tfile,'c','close'
        call SysFileDelete tfile
    end
end
return TotSize TotWaste.1 TotWaste.2 files.0
/******************** End TallyFiles Procedure ************************/

/******************** Begin SlackSpace Procddure **********************/
/* Procedure to determine the wasted space of the given file size.    */
/* This procedure receives the file size and allocation unit size     */
/* from the main program.  It divides the file size by the size of    */
/* the allocation unit, then multiplies the integer portion the       */
/* result to obtain the space occupied only by completely used        */
/* allocation units.  The difference between this and the size of the */
/* file is then subtracted from the size of the allocation unit which */
/* results in the amount of wasted space, in bytes.  Of course, if    */
/* there is no partial allocation unit usage, there is no wasted      */
/* space, and 0 is returned in such a case.                           */

SlackSpace: Procedure expose AUnit.1 AUnit.2

arg FileSize
do i=1 to 2
    IntUnit.i=FileSize%AUnit.i
    FullUnits.i=IntUnit.i*AUnit.i
    Diff.i=FileSize-FullUnits.i
    if Diff.i=0 then Waste.i=0
        else Waste.i=AUnit.i-Diff.i
end
return Waste.1 Waste.2
/******************** End SlackSpace Procedure ************************/

/******************** Begin ProcDir Procedure *************************/
/* Procedure to get all information about a specific directory, which */
/* includes total file and EA size and wasted space (for both FAT and */
/* HPFS).                                                             */

ProcDir: Procedure expose FNode.1 FNode.2 AUnit.1 AUnit.2 UShell SkipEA

arg DirName
TSize=TallyFiles(DirName)
FSize=word(TSize,1)
Waste.1=word(TSize,2)
Waste.2=word(TSize,3)
NumFiles=word(TSize,4)
return Waste.1 Waste.2 FSize NumFiles
/******************** End ProcDir Procedure ***************************/

/******************** Begin TallyDirEA Procedure **********************/
/* Procedure to tally the extended attributes of a given directory,   */
/* which is done regardless of the SkipEA status, since it doesn't    */
/* really take that long.  The parent of the given directory is       */
/* changed to, so we can do a DIR listing of all directories, and     */
/* parse the information about the specific one we want from the list */
/* returned.  There seems to be no way to list only a specific        */
/* directory in either CMD.EXE or 4OS2.EXE.  Once the EA size for the */
/* directory is found, the total actual space taken up for each       */
/* allocation unit is calculated, and returned.                       */

TallyDirEA: Procedure expose AUnit.1 AUnit.2 UShell

arg DirName
tDirName=reverse(DirName)
BSPos=pos('\',tDirName)
BDName=translate(reverse(substr(tDirName,1,(BSPos-1))))
call directory DirName||'\..'
p1=right(d2x(random(65535)),4,'0')
p2=right(d2x(random(65535)),4,'0')
tfile='\'||p1||p2||'.TMP'
if UShell='4OS2' then do
    '@dir/n/m/k/h/ad > '||tfile
    do until found=1
        dirstring=linein(tfile)
        if translate(substr(dirstring,46))=BDName then do
            UFEASize=word(dirstring,4)           
            EACommaPos=pos(',',UFEASize)
            if EACommaPos\=0 then
                EASize=delstr(UFEASize,EACommaPos,1)
            else EASize=UFEASize
            found=1
        end
        else found=0
    end
end
else do
    '@dir/n/ad > '||tfile
    do 5
        scrap=linein(tfile)
    end
    do until found=1
        dirstring=linein(tfile)
        if translate(substr(dirstring,41))=BDName then do
            EASize=word(dirstring,4)
            found=1
        end
        else found=0
    end
end
call stream tfile,'c','close'
call SysFileDelete tfile
DSData=SlackSpace(EASize)
DirEATot.1=EASize+word(DSData,1)
DirEATot.2=EASize+word(DSData,2)
return DirEATot.1 DirEATot.2
/******************** End TallyDirEA Procedure ************************/

/******************** Begin CoDel Procedure ***************************/
/* Procedure to take a long decimal number, and deliminate it with a  */
/* comma for each group of three digits, starting from the end of the */
/* number.  First you reverse the string.  Then you take the first    */
/* three characters of the reversed string (i.e. last three           */
/* characters of the original string), writing the remaining          */
/* characters back to the variable.  The three characters are         */
/* appended to the final string with a comma at the end, and the      */
/* process is repeated until there are three or less characters left  */
/* in the reversed string variable.  The final write of the formatted */
/* string appends the remaining digits.  The string is then reversed  */
/* again (putting it back to normal), which results in a readable     */
/* number:                                                            */
/*                                                                    */
/*  2484693813       - Original string                                */
/*  3183964842       - Reversed                                       */
/*  318,             - Three chars and comma appended to final string */
/*  318,396,         - Next three and a comma                         */
/*  318,396,484,     - Next three and a comma                         */
/*  318,396,484,2    - Last char(s), three or less, no comma          */
/*  2,484,693,813    - Reversed back to normal                        */

CoDel: Procedure

arg RNum
rRNum=reverse(RNum)
FNum=''
do while length(rRNum)>3
    parse var rRNum TriDig +3 rRNum
    FNum=FNum||TriDig||','
end
FNum=reverse(FNum||rRNum)
return FNum
/******************** End CoDel Procedure *****************************/
