;;; pm-win.el --- code specific to the OS/2 Presentation Manager

;; Copyright (C) 1993-1995 Eberhard Mattes
;; Copyright (C) 1995 Patrick Nadeau (scroll bar code)

;; Author: Eberhard Mattes <mattes@azu.informatik.uni-stuttgart.de>
;; Keywords: emx

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

(if (not (eq window-system 'pm))
    (error "Loading pm-win.el but not compiled for PM"))

(require 'frame)
(require 'mouse)
(require 'faces)
(require 'lucid)

(defvar pm-invocation-args)

(emacs-priority 'foreground-server)
(setq min-skip-run 10)

(setq pm-color-alist
      '(
        ;;
        ;; OS/2 color names
        ;;
        ("os2black"             . [  0   0   0])
        ("os2blue"              . [  0   0 255])
        ("os2green"             . [  0 255   0])
        ("os2cyan"              . [  0 255 255])
        ("os2red"               . [255   0   0])
        ("os2pink"              . [255   0 255])
        ("os2yellow"            . [255 255   0])
        ("os2white"             . [255 255 255])
        ("os2darkblue"          . [  0   0 128])
        ("os2darkgreen"         . [  0 128   0])
        ("os2darkcyan"          . [  0 128 128])
        ("os2darkred"           . [128   0   0])
        ("os2darkpink"          . [128   0 128])
        ("os2brown"             . [128 128   0])
        ("os2darkgray"          . [128 128 128])
        ("os2palegray"          . [204 204 204])
        ;;
        ;; X color names
        ;;
        ("green"                . [  0 255   0])
        ("green1"               . [  0 255   0])
        ("green2"               . [  0 238   0])
        ("green3"               . [  0 205   0])
        ("green4"               . [  0 139   0])
        ("darkgreen"            . [  0 100   0])
        ("darkolivegreen"       . [ 85 107  47])
        ("darkolivegreen1"      . [202 255 112])
        ("darkolivegreen2"      . [188 238 104])
        ("darkolivegreen3"      . [162 205  90])
        ("darkolivegreen4"      . [110 139  61])
        ("palegreen"            . [152 251 152])
        ("palegreen1"           . [154 255 154])
        ("palegreen2"           . [144 238 144])
        ("palegreen3"           . [124 205 124])
        ("palegreen4"           . [ 84 139  84])
        ("springgreen"          . [  0 255 127])
        ("springgreen1"         . [  0 255 127])
        ("springgreen2"         . [  0 238 118])
        ("springgreen3"         . [  0 205 102])
        ("springgreen4"         . [  0 139  69])
        ("mediumspringgreen"    . [  0 250 154])
        ("lawngreen"            . [124 252   0])
        ("seagreen"             . [ 46 139  87])
        ("seagreen1"            . [ 84 255 159])
        ("seagreen2"            . [ 78 238 148])
        ("seagreen3"            . [ 67 205 128])
        ("seagreen4"            . [ 46 139  87])
        ("darkseagreen"         . [143 188 143])
        ("darkseagreen1"        . [193 255 193])
        ("darkseagreen2"        . [180 238 180])
        ("darkseagreen3"        . [155 205 155])
        ("darkseagreen4"        . [105 139 105])
        ("mediumseagreen"       . [ 60 179 113])
        ("lightseagreen"        . [ 32 178 170])
        ("forestgreen"          . [ 34 139  34])
        ("aquamarine"           . [127 255 212])
        ("aquamarine1"          . [127 255 212])
        ("aquamarine2"          . [118 238 198])
        ("aquamarine3"          . [102 205 170])
        ("aquamarine4"          . [ 69 139 116])
        ("mediumaquamarine"     . [102 205 170])
        ("limegreen"            . [ 50 205  50])
        ("chartreuse"           . [127 255   0])
        ("chartreuse1"          . [127 255   0])
        ("chartreuse2"          . [118 238   0])
        ("chartreuse3"          . [102 205   0])
        ("chartreuse4"          . [ 69 139   0])
        ("greenyellow"          . [173 255  47])
        ("yellowgreen"          . [154 205  50])
        ("yellow"               . [255 255   0])
        ("yellow1"              . [255 255   0])
        ("yellow2"              . [238 238   0])
        ("yellow3"              . [205 205   0])
        ("yellow4"              . [139 139   0])
        ("lightyellow"          . [255 255 224])
        ("lightyellow1"         . [255 255 224])
        ("lightyellow2"         . [238 238 209])
        ("lightyellow3"         . [205 205 180])
        ("lightyellow4"         . [139 139 122])
        ("blue"                 . [  0   0 255])
        ("blue1"                . [  0   0 255])
        ("blue2"                . [  0   0 238])
        ("blue3"                . [  0   0 205])
        ("blue4"                . [  0   0 139])
        ("royalblue"            . [ 65 105 225])
        ("royalblue1"           . [ 72 118 255])
        ("royalblue2"           . [ 67 110 238])
        ("royalblue3"           . [ 58  95 205])
        ("royalblue4"           . [ 39  64 139])
        ("skyblue"              . [135 206 235])
        ("skyblue1"             . [135 206 255])
        ("skyblue2"             . [126 192 238])
        ("skyblue3"             . [108 166 205])
        ("skyblue4"             . [ 74 112 139])
        ("deepskyblue"          . [  0 191 255])
        ("deepskyblue1"         . [  0 191 255])
        ("deepskyblue2"         . [  0 178 238])
        ("deepskyblue3"         . [  0 154 205])
        ("deepskyblue4"         . [  0 104 139])
        ("lightskyblue"         . [135 206 250])
        ("lightskyblue1"        . [176 226 255])
        ("lightskyblue2"        . [164 211 238])
        ("lightskyblue3"        . [141 182 205])
        ("lightskyblue4"        . [ 96 123 139])
        ("lightblue"            . [173 216 230])
        ("lightblue1"           . [191 239 255])
        ("lightblue2"           . [178 223 238])
        ("lightblue3"           . [154 192 205])
        ("lightblue4"           . [104 131 139])
        ("midnightblue"         . [ 25  25 112])
        ("navyblue"             . [  0   0 128])
        ("aliceblue"            . [240 248 255])
        ("cornflowerblue"       . [100 149 237])
        ("darkslateblue"        . [ 72  61 139])
        ("slateblue"            . [106  90 205])
        ("slateblue1"           . [131 111 255])
        ("slateblue2"           . [122 103 238])
        ("slateblue3"           . [105  89 205])
        ("slateblue4"           . [ 71  60 139])
        ("mediumslateblue"      . [123 104 238])
        ("lightslateblue"       . [132 112 255])
        ("mediumblue"           . [  0   0 205])
        ("dodgerblue"           . [ 30 144 255])
        ("dodgerblue1"          . [ 30 144 255])
        ("dodgerblue2"          . [ 28 134 238])
        ("dodgerblue3"          . [ 24 116 205])
        ("dodgerblue4"          . [ 16  78 139])
        ("steelblue"            . [ 70 130 180])
        ("steelblue1"           . [ 99 184 255])
        ("steelblue2"           . [ 92 172 238])
        ("steelblue3"           . [ 79 148 205])
        ("steelblue4"           . [ 54 100 139])
        ("lightsteelblue"       . [176 196 222])
        ("lightsteelblue1"      . [202 225 255])
        ("lightsteelblue2"      . [188 210 238])
        ("lightsteelblue3"      . [162 181 205])
        ("lightsteelblue4"      . [110 123 139])
        ("powderblue"           . [176 224 230])
        ("cadetblue"            . [ 95 158 160])
        ("cadetblue1"           . [152 245 255])
        ("cadetblue2"           . [142 229 238])
        ("cadetblue3"           . [122 197 205])
        ("cadetblue4"           . [ 83 134 139])
        ("azure"                . [240 255 255])
        ("azure1"               . [240 255 255])
        ("azure2"               . [224 238 238])
        ("azure3"               . [193 205 205])
        ("azure4"               . [131 139 139])
        ("cyan"                 . [  0 255 255])
        ("cyan1"                . [  0 255 255])
        ("cyan2"                . [  0 238 238])
        ("cyan3"                . [  0 205 205])
        ("cyan4"                . [  0 139 139])
        ("lightcyan"            . [224 255 255])
        ("lightcyan1"           . [224 255 255])
        ("lightcyan2"           . [209 238 238])
        ("lightcyan3"           . [180 205 205])
        ("lightcyan4"           . [122 139 139])
        ("lavender"             . [230 230 250])
        ("lavenderblush"        . [255 240 245])
        ("lavenderblush1"       . [255 240 245])
        ("lavenderblush2"       . [238 224 229])
        ("lavenderblush3"       . [205 193 197])
        ("lavenderblush4"       . [139 131 134])
        ("turquoise"            . [ 64 224 208])
        ("turquoise1"           . [  0 245 255])
        ("turquoise2"           . [  0 229 238])
        ("turquoise3"           . [  0 197 205])
        ("turquoise4"           . [  0 134 139])
        ("paleturquoise"        . [175 238 238])
        ("paleturquoise1"       . [187 255 255])
        ("paleturquoise2"       . [174 238 238])
        ("paleturquoise3"       . [150 205 205])
        ("paleturquoise4"       . [102 139 139])
        ("darkturquoise"        . [  0 206 209])
        ("mediumturquoise"      . [ 72 209 204])
        ("purple"               . [160  32 240])
        ("purple1"              . [155  48 255])
        ("purple2"              . [145  44 238])
        ("purple3"              . [125  38 205])
        ("purple4"              . [ 85  26 139])
        ("mediumpurple"         . [147 112 219])
        ("mediumpurple1"        . [171 130 255])
        ("mediumpurple2"        . [159 121 238])
        ("mediumpurple3"        . [137 104 205])
        ("mediumpurple4"        . [ 93  71 139])
        ("firebrick"            . [178  34  34])
        ("firebrick1"           . [255  48  48])
        ("firebrick2"           . [238  44  44])
        ("firebrick3"           . [205  38  38])
        ("firebrick4"           . [139  26  26])
        ("magenta"              . [255   0 255])
        ("magenta1"             . [255   0 255])
        ("magenta2"             . [238   0 238])
        ("magenta3"             . [205   0 205])
        ("magenta4"             . [139   0 139])
        ("pink"                 . [255 192 203])
        ("pink1"                . [255 181 197])
        ("pink2"                . [238 169 184])
        ("pink3"                . [205 145 158])
        ("pink4"                . [139  99 108])
        ("deeppink"             . [255  20 147])
        ("deeppink1"            . [255  20 147])
        ("deeppink2"            . [238  18 137])
        ("deeppink3"            . [205  16 118])
        ("deeppink4"            . [139  10  80])
        ("hotpink"              . [255 105 180])
        ("hotpink1"             . [255 110 180])
        ("hotpink2"             . [238 106 167])
        ("hotpink3"             . [205  96 144])
        ("hotpink4"             . [139  58  98])
        ("lightpink"            . [255 182 193])
        ("lightpink1"           . [255 174 185])
        ("lightpink2"           . [238 162 173])
        ("lightpink3"           . [205 140 149])
        ("lightpink4"           . [139  95 101])
        ("mistyrose"            . [255 228 225])
        ("mistyrose1"           . [255 228 225])
        ("mistyrose2"           . [238 213 210])
        ("mistyrose3"           . [205 183 181])
        ("mistyrose4"           . [139 125 123])
        ("orchid"               . [218 112 214])
        ("orchid1"              . [255 131 250])
        ("orchid2"              . [238 122 233])
        ("orchid3"              . [205 105 201])
        ("orchid4"              . [139  71 137])
        ("mediumorchid"         . [186  85 211])
        ("mediumorchid1"        . [224 102 255])
        ("mediumorchid2"        . [209  95 238])
        ("mediumorchid3"        . [180  82 205])
        ("mediumorchid4"        . [122  55 139])
        ("darkorchid"           . [153  50 204])
        ("darkorchid1"          . [191  62 255])
        ("darkorchid2"          . [178  58 238])
        ("darkorchid3"          . [154  50 205])
        ("darkorchid4"          . [104  34 139])
        ("thistle"              . [216 191 216])
        ("thistle1"             . [255 225 255])
        ("thistle2"             . [238 210 238])
        ("thistle3"             . [205 181 205])
        ("thistle4"             . [139 123 139])
        ("maroon"               . [176  48  96])
        ("maroon1"              . [255  52 179])
        ("maroon2"              . [238  48 167])
        ("maroon3"              . [205  41 144])
        ("maroon4"              . [139  28  98])
        ("gold"                 . [255 215   0])
        ("gold1"                . [255 215   0])
        ("gold2"                . [238 201   0])
        ("gold3"                . [205 173   0])
        ("gold4"                . [139 117   0])
        ("goldenrod"            . [218 165  32])
        ("goldenrod1"           . [255 193  37])
        ("goldenrod2"           . [238 180  34])
        ("goldenrod3"           . [205 155  29])
        ("goldenrod4"           . [139 105  20])
        ("darkgoldenrod"        . [184 134  11])
        ("darkgoldenrod1"       . [255 185  15])
        ("darkgoldenrod2"       . [238 173  14])
        ("darkgoldenrod3"       . [205 149  12])
        ("darkgoldenrod4"       . [139 101   8])
        ("palegoldenrod"        . [238 232 170])
        ("lightgoldenrod"       . [238 221 130])
        ("lightgoldenrod1"      . [255 236 139])
        ("lightgoldenrod2"      . [238 220 130])
        ("lightgoldenrod3"      . [205 190 112])
        ("lightgoldenrod4"      . [139 129  76])
        ("lightgoldenrodyellow" . [250 250 210])
        ("olivedrab"            . [107 142  35])
        ("olivedrab1"           . [192 255  62])
        ("olivedrab2"           . [179 238  58])
        ("olivedrab3"           . [154 205  50])
        ("olivedrab4"           . [105 139  34])
        ("khaki"                . [240 230 140])
        ("khaki1"               . [255 246 143])
        ("khaki2"               . [238 230 133])
        ("khaki3"               . [205 198 115])
        ("khaki4"               . [139 134  78])
        ("darkkhaki"            . [189 183 107])
        ("moccasin"             . [255 228 181])
        ("plum"                 . [221 160 221])
        ("plum1"                . [255 187 255])
        ("plum2"                . [238 174 238])
        ("plum3"                . [205 150 205])
        ("plum4"                . [139 102 139])
        ("sienna"               . [160  82  45])
        ("sienna1"              . [255 130  71])
        ("sienna2"              . [238 121  66])
        ("sienna3"              . [205 104  57])
        ("sienna4"              . [139  71  38])
        ("peru"                 . [205 133  63])
        ("beige"                . [245 245 220])
        ("wheat"                . [245 222 179])
        ("wheat1"               . [255 231 186])
        ("wheat2"               . [238 216 174])
        ("wheat3"               . [205 186 150])
        ("wheat4"               . [139 126 102])
        ("brown"                . [165  42  42])
        ("brown1"               . [255  64  64])
        ("brown2"               . [238  59  59])
        ("brown3"               . [205  51  51])
        ("brown4"               . [139  35  35])
        ("rosybrown"            . [188 143 143])
        ("rosybrown1"           . [255 193 193])
        ("rosybrown2"           . [238 180 180])
        ("rosybrown3"           . [205 155 155])
        ("rosybrown4"           . [139 105 105])
        ("saddlebrown"          . [139  69  19])
        ("burlywood"            . [222 184 135])
        ("burlywood1"           . [255 211 155])
        ("burlywood2"           . [238 197 145])
        ("burlywood3"           . [205 170 125])
        ("burlywood4"           . [139 115  85])
        ("sandybrown"           . [244 164  96])
        ("salmon"               . [250 128 114])
        ("salmon1"              . [255 140 105])
        ("salmon2"              . [238 130  98])
        ("salmon3"              . [205 112  84])
        ("salmon4"              . [139  76  57])
        ("darksalmon"           . [233 150 122])
        ("lightsalmon"          . [255 160 122])
        ("lightsalmon1"         . [255 160 122])
        ("lightsalmon2"         . [238 149 114])
        ("lightsalmon3"         . [205 129  98])
        ("lightsalmon4"         . [139  87  66])
        ("chocolate"            . [210 105  30])
        ("chocolate1"           . [255 127  36])
        ("chocolate2"           . [238 118  33])
        ("chocolate3"           . [205 102  29])
        ("chocolate4"           . [139  69  19])
        ("orange"               . [255 165   0])
        ("orange1"              . [255 165   0])
        ("orange2"              . [238 154   0])
        ("orange3"              . [205 133   0])
        ("orange4"              . [139  90   0])
        ("darkorange"           . [255 140   0])
        ("darkorange1"          . [255 127   0])
        ("darkorange2"          . [238 118   0])
        ("darkorange3"          . [205 102   0])
        ("darkorange4"          . [139  69   0])
        ("orangered"            . [255  69   0])
        ("orangered1"           . [255  69   0])
        ("orangered2"           . [238  64   0])
        ("orangered3"           . [205  55   0])
        ("orangered4"           . [139  37   0])
        ("red"                  . [255   0   0])
        ("red1"                 . [255   0   0])
        ("red2"                 . [238   0   0])
        ("red3"                 . [205   0   0])
        ("red4"                 . [139   0   0])
        ("violetred"            . [208  32 144])
        ("violetred1"           . [255  62 150])
        ("violetred2"           . [238  58 140])
        ("violetred3"           . [205  50 120])
        ("violetred4"           . [139  34  82])
        ("palevioletred"        . [219 112 147])
        ("palevioletred1"       . [255 130 171])
        ("palevioletred2"       . [238 121 159])
        ("palevioletred3"       . [205 104 137])
        ("palevioletred4"       . [139  71  93])
        ("mediumvioletred"      . [199  21 133])
        ("indianred"            . [205  92  92])
        ("indianred1"           . [255 106 106])
        ("indianred2"           . [238  99  99])
        ("indianred3"           . [205  85  85])
        ("indianred4"           . [139  58  58])
        ("violet"               . [238 130 238])
        ("darkviolet"           . [148   0 211])
        ("blueviolet"           . [138  43 226])
        ("tomato"               . [255  99  71])
        ("tomato1"              . [255  99  71])
        ("tomato2"              . [238  92  66])
        ("tomato3"              . [205  79  57])
        ("tomato4"              . [139  54  38])
        ("coral"                . [255 127  80])
        ("coral1"               . [255 114  86])
        ("coral2"               . [238 106  80])
        ("coral3"               . [205  91  69])
        ("coral4"               . [139  62  47])
        ("lightcoral"           . [240 128 128])
        ("tan"                  . [210 180 140])
        ("tan1"                 . [255 165  79])
        ("tan2"                 . [238 154  73])
        ("tan3"                 . [205 133  63])
        ("tan4"                 . [139  90  43])
        ("white"                . [255 255 255])
        ("antiquewhite"         . [250 235 215])
        ("antiquewhite1"        . [255 239 219])
        ("antiquewhite2"        . [238 223 204])
        ("antiquewhite3"        . [205 192 176])
        ("antiquewhite4"        . [139 131 120])
        ("navajowhite"          . [255 222 173])
        ("navajowhite1"         . [255 222 173])
        ("navajowhite2"         . [238 207 161])
        ("navajowhite3"         . [205 179 139])
        ("navajowhite4"         . [139 121  94])
        ("ivory"                . [255 255 240])
        ("ivory1"               . [255 255 240])
        ("ivory2"               . [238 238 224])
        ("ivory3"               . [205 205 193])
        ("ivory4"               . [139 139 131])
        ("snow"                 . [255 250 250])
        ("snow1"                . [255 250 250])
        ("snow2"                . [238 233 233])
        ("snow3"                . [205 201 201])
        ("snow4"                . [139 137 137])
        ("seashell"             . [255 245 238])
        ("seashell1"            . [255 245 238])
        ("seashell2"            . [238 229 222])
        ("seashell3"            . [205 197 191])
        ("seashell4"            . [139 134 130])
        ("oldlace"              . [253 245 230])
        ("floralwhite"          . [255 250 240])
        ("ghostwhite"           . [248 248 255])
        ("whitesmoke"           . [245 245 245])
        ("linen"                . [250 240 230])
        ("gainsboro"            . [220 220 220])
        ("papayawhip"           . [255 239 213])
        ("blanchedalmond"       . [255 235 205])
        ("bisque"               . [255 228 196])
        ("bisque1"              . [255 228 196])
        ("bisque2"              . [238 213 183])
        ("bisque3"              . [205 183 158])
        ("bisque4"              . [139 125 107])
        ("peachpuff"            . [255 218 185])
        ("peachpuff1"           . [255 218 185])
        ("peachpuff2"           . [238 203 173])
        ("peachpuff3"           . [205 175 149])
        ("peachpuff4"           . [139 119 101])
        ("cornsilk"             . [255 248 220])
        ("cornsilk1"            . [255 248 220])
        ("cornsilk2"            . [238 232 205])
        ("cornsilk3"            . [205 200 177])
        ("cornsilk4"            . [139 136 120])
        ("lemonchiffon"         . [255 250 205])
        ("lemonchiffon1"        . [255 250 205])
        ("lemonchiffon2"        . [238 233 191])
        ("lemonchiffon3"        . [205 201 165])
        ("lemonchiffon4"        . [139 137 112])
        ("honeydew"             . [240 255 240])
        ("honeydew1"            . [240 255 240])
        ("honeydew2"            . [224 238 224])
        ("honeydew3"            . [193 205 193])
        ("honeydew4"            . [131 139 131])
        ("mintcream"            . [245 255 250])
        ("lightgray"            . [211 211 211])
        ("lightgrey"            . [211 211 211])
        ("dimgray"              . [105 105 105])
        ("dimgrey"              . [105 105 105])
        ("slategray"            . [112 128 144])
        ("slategrey"            . [112 128 144])
        ("slategray1"           . [198 226 255])
        ("slategray2"           . [185 211 238])
        ("slategray3"           . [159 182 205])
        ("slategray4"           . [108 123 139])
        ("darkslategray"        . [ 47  79  79])
        ("darkslategrey"        . [ 47  79  79])
        ("darkslategray1"       . [151 255 255])
        ("darkslategray2"       . [141 238 238])
        ("darkslategray3"       . [121 205 205])
        ("darkslategray4"       . [ 82 139 139])
        ("lightslategray"       . [119 136 153])
        ("lightslategrey"       . [119 136 153])
        ("black"                . [  0   0   0])
        ("gray"                 . [192 192 192])
        ("grey"                 . [192 192 192])
        ("gray0"                . [  0   0   0])
        ("grey0"                . [  0   0   0])
        ("gray1"                . [  3   3   3])
        ("grey1"                . [  3   3   3])
        ("gray2"                . [  5   5   5])
        ("grey2"                . [  5   5   5])
        ("gray3"                . [  8   8   8])
        ("grey3"                . [  8   8   8])
        ("gray4"                . [ 10  10  10])
        ("grey4"                . [ 10  10  10])
        ("gray5"                . [ 13  13  13])
        ("grey5"                . [ 13  13  13])
        ("gray6"                . [ 15  15  15])
        ("grey6"                . [ 15  15  15])
        ("gray7"                . [ 18  18  18])
        ("grey7"                . [ 18  18  18])
        ("gray8"                . [ 20  20  20])
        ("grey8"                . [ 20  20  20])
        ("gray9"                . [ 23  23  23])
        ("grey9"                . [ 23  23  23])
        ("gray10"               . [ 26  26  26])
        ("grey10"               . [ 26  26  26])
        ("gray11"               . [ 28  28  28])
        ("grey11"               . [ 28  28  28])
        ("gray12"               . [ 31  31  31])
        ("grey12"               . [ 31  31  31])
        ("gray13"               . [ 33  33  33])
        ("grey13"               . [ 33  33  33])
        ("gray14"               . [ 36  36  36])
        ("grey14"               . [ 36  36  36])
        ("gray15"               . [ 38  38  38])
        ("grey15"               . [ 38  38  38])
        ("gray16"               . [ 41  41  41])
        ("grey16"               . [ 41  41  41])
        ("gray17"               . [ 43  43  43])
        ("grey17"               . [ 43  43  43])
        ("gray18"               . [ 46  46  46])
        ("grey18"               . [ 46  46  46])
        ("gray19"               . [ 48  48  48])
        ("grey19"               . [ 48  48  48])
        ("gray20"               . [ 51  51  51])
        ("grey20"               . [ 51  51  51])
        ("gray21"               . [ 54  54  54])
        ("grey21"               . [ 54  54  54])
        ("gray22"               . [ 56  56  56])
        ("grey22"               . [ 56  56  56])
        ("gray23"               . [ 59  59  59])
        ("grey23"               . [ 59  59  59])
        ("gray24"               . [ 61  61  61])
        ("grey24"               . [ 61  61  61])
        ("gray25"               . [ 64  64  64])
        ("grey25"               . [ 64  64  64])
        ("gray26"               . [ 66  66  66])
        ("grey26"               . [ 66  66  66])
        ("gray27"               . [ 69  69  69])
        ("grey27"               . [ 69  69  69])
        ("gray28"               . [ 71  71  71])
        ("grey28"               . [ 71  71  71])
        ("gray29"               . [ 74  74  74])
        ("grey29"               . [ 74  74  74])
        ("gray30"               . [ 77  77  77])
        ("grey30"               . [ 77  77  77])
        ("gray31"               . [ 79  79  79])
        ("grey31"               . [ 79  79  79])
        ("gray32"               . [ 82  82  82])
        ("grey32"               . [ 82  82  82])
        ("gray33"               . [ 84  84  84])
        ("grey33"               . [ 84  84  84])
        ("gray34"               . [ 87  87  87])
        ("grey34"               . [ 87  87  87])
        ("gray35"               . [ 89  89  89])
        ("grey35"               . [ 89  89  89])
        ("gray36"               . [ 92  92  92])
        ("grey36"               . [ 92  92  92])
        ("gray37"               . [ 94  94  94])
        ("grey37"               . [ 94  94  94])
        ("gray38"               . [ 97  97  97])
        ("grey38"               . [ 97  97  97])
        ("gray39"               . [ 99  99  99])
        ("grey39"               . [ 99  99  99])
        ("gray40"               . [102 102 102])
        ("grey40"               . [102 102 102])
        ("gray41"               . [105 105 105])
        ("grey41"               . [105 105 105])
        ("gray42"               . [107 107 107])
        ("grey42"               . [107 107 107])
        ("gray43"               . [110 110 110])
        ("grey43"               . [110 110 110])
        ("gray44"               . [112 112 112])
        ("grey44"               . [112 112 112])
        ("gray45"               . [115 115 115])
        ("grey45"               . [115 115 115])
        ("gray46"               . [117 117 117])
        ("grey46"               . [117 117 117])
        ("gray47"               . [120 120 120])
        ("grey47"               . [120 120 120])
        ("gray48"               . [122 122 122])
        ("grey48"               . [122 122 122])
        ("gray49"               . [125 125 125])
        ("grey49"               . [125 125 125])
        ("gray50"               . [127 127 127])
        ("grey50"               . [127 127 127])
        ("gray51"               . [130 130 130])
        ("grey51"               . [130 130 130])
        ("gray52"               . [133 133 133])
        ("grey52"               . [133 133 133])
        ("gray53"               . [135 135 135])
        ("grey53"               . [135 135 135])
        ("gray54"               . [138 138 138])
        ("grey54"               . [138 138 138])
        ("gray55"               . [140 140 140])
        ("grey55"               . [140 140 140])
        ("gray56"               . [143 143 143])
        ("grey56"               . [143 143 143])
        ("gray57"               . [145 145 145])
        ("grey57"               . [145 145 145])
        ("gray58"               . [148 148 148])
        ("grey58"               . [148 148 148])
        ("gray59"               . [150 150 150])
        ("grey59"               . [150 150 150])
        ("gray60"               . [153 153 153])
        ("grey60"               . [153 153 153])
        ("gray61"               . [156 156 156])
        ("grey61"               . [156 156 156])
        ("gray62"               . [158 158 158])
        ("grey62"               . [158 158 158])
        ("gray63"               . [161 161 161])
        ("grey63"               . [161 161 161])
        ("gray64"               . [163 163 163])
        ("grey64"               . [163 163 163])
        ("gray65"               . [166 166 166])
        ("grey65"               . [166 166 166])
        ("gray66"               . [168 168 168])
        ("grey66"               . [168 168 168])
        ("gray67"               . [171 171 171])
        ("grey67"               . [171 171 171])
        ("gray68"               . [173 173 173])
        ("grey68"               . [173 173 173])
        ("gray69"               . [176 176 176])
        ("grey69"               . [176 176 176])
        ("gray70"               . [179 179 179])
        ("grey70"               . [179 179 179])
        ("gray71"               . [181 181 181])
        ("grey71"               . [181 181 181])
        ("gray72"               . [184 184 184])
        ("grey72"               . [184 184 184])
        ("gray73"               . [186 186 186])
        ("grey73"               . [186 186 186])
        ("gray74"               . [189 189 189])
        ("grey74"               . [189 189 189])
        ("gray75"               . [191 191 191])
        ("grey75"               . [191 191 191])
        ("gray76"               . [194 194 194])
        ("grey76"               . [194 194 194])
        ("gray77"               . [196 196 196])
        ("grey77"               . [196 196 196])
        ("gray78"               . [199 199 199])
        ("grey78"               . [199 199 199])
        ("gray79"               . [201 201 201])
        ("grey79"               . [201 201 201])
        ("gray80"               . [204 204 204])
        ("grey80"               . [204 204 204])
        ("gray81"               . [207 207 207])
        ("grey81"               . [207 207 207])
        ("gray82"               . [209 209 209])
        ("grey82"               . [209 209 209])
        ("gray83"               . [212 212 212])
        ("grey83"               . [212 212 212])
        ("gray84"               . [214 214 214])
        ("grey84"               . [214 214 214])
        ("gray85"               . [217 217 217])
        ("grey85"               . [217 217 217])
        ("gray86"               . [219 219 219])
        ("grey86"               . [219 219 219])
        ("gray87"               . [222 222 222])
        ("grey87"               . [222 222 222])
        ("gray88"               . [224 224 224])
        ("grey88"               . [224 224 224])
        ("gray89"               . [227 227 227])
        ("grey89"               . [227 227 227])
        ("gray90"               . [229 229 229])
        ("grey90"               . [229 229 229])
        ("gray91"               . [232 232 232])
        ("grey91"               . [232 232 232])
        ("gray92"               . [235 235 235])
        ("grey92"               . [235 235 235])
        ("gray93"               . [237 237 237])
        ("grey93"               . [237 237 237])
        ("gray94"               . [240 240 240])
        ("grey94"               . [240 240 240])
        ("gray95"               . [242 242 242])
        ("grey95"               . [242 242 242])
        ("gray96"               . [245 245 245])
        ("grey96"               . [245 245 245])
        ("gray97"               . [247 247 247])
        ("grey97"               . [247 247 247])
        ("gray98"               . [250 250 250])
        ("grey98"               . [250 250 250])
        ("gray99"               . [252 252 252])
        ("grey99"               . [252 252 252])
        ("gray100"              . [255 255 255])
        ("grey100"              . [255 255 255])
        ))

(setq system-key-alist
      '(
        ( 1 . button1)
        ( 2 . button2)
        ( 3 . button3)
        ( 4 . break)
        ( 5 . backspace)
        ( 6 . tab)
        ( 7 . backtab)
        ( 8 . newline)
        ( 9 . shift)
        (10 . ctrl)
        (11 . alt)
        (12 . altgraf)
        (13 . pause)
        (14 . capslock)
        (15 . esc)
        (16 . space)
        (17 . pageup)
        (18 . pagedown)
        (19 . end)
        (20 . home)
        (21 . left)
        (22 . up)
        (23 . right)
        (24 . down)
        (25 . printscrn)
        (26 . insert)
        (27 . delete)
        (28 . scrllock)
        (29 . numlock)
        (30 . kp-enter)
        (31 . sysrq)
        (32 . f1)
        (33 . f2)
        (34 . f3)
        (35 . f4)
        (36 . f5)
        (37 . f6)
        (38 . f7)
        (39 . f8)
        (40 . f9)
        (41 . f10)
        (42 . f11)
        (43 . f12)
        (44 . f13)
        (45 . f14)
        (46 . f15)
        (47 . f16)
        (48 . f17)
        (49 . f18)
        (50 . f19)
        (51 . f20)
        (52 . f21)
        (53 . f22)
        (54 . f23)
        (55 . f24)
        (56 . enddrag)
        (57 . close-frame)
        (58 . kp-center)
        (59 . kp-add)
        (60 . kp-subtract)
        (61 . kp-multiply)
        (62 . kp-divide)
        (63 . kp-0)
        (64 . kp-1)
        (65 . kp-2)
        (66 . kp-3)
        (67 . kp-4)
        (68 . kp-5)
        (69 . kp-6)
        (70 . kp-7)
        (71 . kp-8)
        (72 . kp-9)
        (73 . kp-separator)
        (74 . kp-decimal)
        ))

(defconst pm-option-alist
  '(("-name"       . pm-handle-name-switch)
    ("-T"          . pm-handle-switch)
    ("-r"          . pm-handle-switch)
    ("-fn"         . pm-handle-switch)
    ("-font"       . pm-handle-switch)
    ("-fg"         . pm-handle-switch)
    ("-foreground" . pm-handle-switch)
    ("-bg"         . pm-handle-switch)
    ("-background" . pm-handle-switch)
    ("-iconic"     . pm-handle-iconic)))

(defconst pm-long-option-alist
  '(("--name"             . "-name")
    ("--title"            . "-T")
    ("--font"             . "-font")
    ("--foreground-color" . "-fg")
    ("--background-color" . "-bg")
    ("--iconic"           . "-iconic")))

(defconst pm-switch-definitions
  '(("-name" name)
    ("-T" name)
    ("-fn" font)
    ("-font" font)
    ("-fg" foreground-color)
    ("-foreground" foreground-color)
    ("-bg" background-color)
    ("-background" background-color)))

(pm-open-connection)

;; Redefine colors for 16-color displays
(if (= (pm-display-planes) 4)
    (progn
      (setcdr (assoc "paleturquoise" pm-color-alist) '[0 255 255])
      (setcdr (assoc "darkseagreen2" pm-color-alist) '[0 255   0])))

(setq suspend-hook
      '(lambda ()
	 (error "Suspending an Emacs running under PM isn't a good idea")))

;; Note: does *not* take an argument
(defun pm-handle-display (switch))

;; Handler for switches of the form "-switch value" or "-switch".
(defun pm-handle-switch (switch)
  (let ((aelt (assoc switch pm-switch-definitions)))
    (if aelt
	(if (nth 2 aelt)
	    (setq default-frame-alist
		  (cons (cons (nth 1 aelt) (nth 2 aelt))
			default-frame-alist))
	  (setq default-frame-alist
		(cons (cons (nth 1 aelt)
			    (car pm-invocation-args))
		      default-frame-alist)
		pm-invocation-args (cdr pm-invocation-args))))))

;; Make -iconic apply only to the initial frame!
(defun pm-handle-iconic (switch)
  (setq initial-frame-alist
	(cons '(visibility . icon) initial-frame-alist)))

;; Handle the -name option.
(defun pm-handle-name-switch (switch)
  (or (consp pm-invocation-args)
      (error "%s: missing argument to `%s' option" (invocation-name) switch))
  (setq initial-frame-alist (cons (cons 'name (car pm-invocation-args))
                                  initial-frame-alist))
  (setq pm-invocation-args (cdr pm-invocation-args)))

(defun pm-handle-args (args)
  "Process the PM-related command line options in ARGS.
This is done before the user's startup file is loaded.  They are copied to
pm-invocation args from which the PM-related things are extracted, first
the switch (e.g., \"-fg\") in the following code, and possible values
\(e.g., \"black\") in the option handler code (e.g., pm-handle-switch).
This returns ARGS with the arguments that have been processed removed."
  (message "%s" args)
  (setq pm-invocation-args args
	args nil)
  (while pm-invocation-args
    (let* ((this-switch (car pm-invocation-args))
	   (orig-this-switch this-switch)
	   completion argval aelt)
      (setq pm-invocation-args (cdr pm-invocation-args))
      ;; Check for long options with attached arguments
      ;; and separate out the attached option argument into argval.
      (if (string-match "^--[^=]*=" this-switch)
	  (setq argval (substring this-switch (match-end 0))
		this-switch (substring this-switch 0 (1- (match-end 0)))))
      (setq completion (try-completion this-switch pm-long-option-alist))
      (if (eq completion t)
	  ;; Exact match for long option.
	  (setq this-switch (cdr (assoc this-switch pm-long-option-alist)))
	(if (stringp completion)
	    (let ((elt (assoc completion pm-long-option-alist)))
	      ;; Check for abbreviated long option.
	      (or elt
		  (error "Option `%s' is ambiguous" this-switch))
	      (setq this-switch (cdr elt)))
	  ;; Check for a short option.
	  (setq argval nil this-switch orig-this-switch)))
      (setq aelt (assoc this-switch pm-option-alist))
      (if aelt
	  (if argval
	      (let ((pm-invocation-args
		     (cons argval pm-invocation-args)))
		(funcall (cdr aelt) this-switch))
	    (funcall (cdr aelt) this-switch))
	(setq args (cons this-switch args)))))
  (setq args (nreverse args)))

(setq command-line-args (pm-handle-args command-line-args))

(defun pm-paste ()
  "Paste text from clipboard into buffer at point."
  (interactive "*")
  (insert (pm-get-clipboard)))

(defun pm-copy (beg end)
  "Copy the region to the clipboard.\n\
Note that several programs cannot handle more than 64KB of text in the\n\
clipboard.  Therefore you should avoid copying more than 64KB of text."
  (interactive "r")
  (pm-put-clipboard (buffer-substring beg end)))

(defun pm-cut (beg end)
  "Kill between point and mark and place the killed text in the clipboard.\n\
Note that several programs cannot handle more than 64KB of text in the\n\
clipboard.  Therefore you should avoid killing more than 64KB of text."
  (interactive "*r")
  (pm-copy beg end)
  (delete-region beg end))

(defun pm-close-frame ()
  "Close the current frame or kill Emacs, if this is the last frame."
  (interactive)
  ;; Fake a menu-bar event to get a dialog
  (setq last-nonmenu-event
        (list 'close-frame
              (list (selected-frame) '(menu-bar) '(0 . 0) 0)))
  (if (= (length (frame-list)) 1)
      (save-buffers-kill-emacs)
    (delete-frame)))

(defun pm-find-file ()
  "Like find-file, but use a file dialog box."
  (interactive)
  (let ((file (pm-file-dialog nil "Open File"
                              default-directory
                              (and buffer-file-name
                                   (file-name-nondirectory buffer-file-name))
                              nil nil "Open")))
    (if file (find-file file))))

(defun pm-write-file ()
  "Like write-file, but use a file dialog box."
  (interactive)
  (let ((file (pm-file-dialog nil "Save Buffer As"
                              default-directory
                              (and buffer-file-name
                                   (file-name-nondirectory buffer-file-name))
                              nil t "Write")))
    (if file (write-file file))))

(setq split-window-keep-point t)
(setq frame-creation-function 'x-create-frame-with-faces)

(defun iconify-or-deiconify-frame ()
  "Iconify the selected frame, or deiconify if it's currently an icon."
  (interactive)
  (if (eq (cdr (assq 'visibility (frame-parameters))) t)
      (iconify-frame)
    (make-frame-visible)))

(substitute-key-definition 'suspend-emacs 'iconify-or-deiconify-frame
			   global-map)

(define-key global-map [S-left]      'scroll-right)
(define-key global-map [S-right]     'scroll-left)
(define-key global-map [S-insert]    'pm-paste)
(define-key global-map [S-delete]    'pm-cut)
(define-key global-map [C-insert]    'pm-copy)
(define-key global-map [close-frame] 'pm-close-frame)

(define-key function-key-map [backspace]   [127])
(define-key function-key-map [M-backspace] [?\M-\177])
(define-key function-key-map [M-delete]    [?\M-\d])
(define-key function-key-map [backtab]     [S-tab])

; This is easier to type than C-_ (doesn't need the SHIFT key)
(define-key global-map [?\C--]       'undo)

; Put this on a more convenient key
(define-key global-map [?\C-.]       'repeat-complex-command)

; Disable extended scan codes, re-enable C-@
(define-key global-map "\C-@" 'set-mark-command)

(defun pm-defined-colors (&optional frame)
  "Return a list of colors supported for a particular frame.
The argument FRAME specifies which frame to try."
  (or frame (setq frame (selected-frame)))
  (let ((all-colors pm-color-alist)
	(this-color nil)
        (defined-colors nil))
    (while all-colors
      (setq this-color (car (car all-colors))
	    all-colors (cdr all-colors))
      (and (face-color-supported-p frame this-color t)
           (setq defined-colors (cons this-color defined-colors))))
    defined-colors))

;;
;; Font menu
;;
(setq x-fixed-font-alist
  '("Font menu"
    ("Misc"
     ("Default" "-*-Courier-medium-r-normal--*-100-*-*-m-*-cp850")
     ("Small"   "-*-Small-medium-r-normal--*-60-*-*-m-*-cp850"))
    ("System Monospaced"
     ("8pt"  "-os2-System Monospaced-medium-r-normal--*-80-*-*-m-*-cp850")
     ("10pt" "-os2-System Monospaced-medium-r-normal--*-100-*-*-m-*-cp850"))
    ("System VIO"
     (" 8x8 (8)"   "-os2-System VIO-medium-r-normal--*-80-*-*-m-*-cp850")
     ("10x6 (4)"   "-os2-System VIO-medium-r-normal--*-40-*-*-m-*-cp850")
     ("10x8 (9)"   "-os2-System VIO-medium-r-normal--*-90-*-*-m-*-cp850")
     ("12x5 (2)"   "-os2-System VIO-medium-r-normal--*-20-*-*-m-*-cp850")
     ("12x8 (10)"  "-os2-System VIO-medium-r-normal--*-100-*-*-m-*-cp850")
     ("14x6 (5)"   "-os2-System VIO-medium-r-normal--*-50-*-*-m-*-cp850")
     ("14x8 (11)"  "-os2-System VIO-medium-r-normal--*-110-*-*-m-*-cp850")
     ("15x7 (6)"   "-os2-System VIO-medium-r-normal--*-60-*-*-m-*-cp850")
     ("16x5 (3)"   "-os2-System VIO-medium-r-normal--*-30-*-*-m-*-cp850")
     ("16x8 (12)"  "-os2-System VIO-medium-r-normal--*-120-*-*-m-*-cp850")
     ("16x12 (15)" "-os2-System VIO-medium-r-normal--*-150-*-*-m-*-cp850")
     ("18x8 (13)"  "-os2-System VIO-medium-r-normal--*-130-*-*-m-*-cp850")
     ("18x10 (14)" "-os2-System VIO-medium-r-normal--*-140-*-*-m-*-cp850")
     ("20x12 (16)" "-os2-System VIO-medium-r-normal--*-160-*-*-m-*-cp850")
     ("22x12 (17)" "-os2-System VIO-medium-r-normal--*-170-*-*-m-*-cp850")
     ("25x7 (7)"   "-os2-System VIO-medium-r-normal--*-70-*-*-m-*-cp850")
     ("30x12 (18)" "-os2-System VIO-medium-r-normal--*-180-*-*-m-*-cp850"))
    ("Courier"
     ("8pt"  "-*-Courier-medium-r-normal--*-80-*-*-m-*-cp850")
     ("10pt" "-*-Courier-medium-r-normal--*-100-*-*-m-*-cp850")
     ("12pt" "-*-Courier-medium-r-normal--*-120-*-*-m-*-cp850"))
    ))

;;
;; This can be run from the initialization file
;;
(defun pm-edit-menu ()
  "Modify the Edit menu of the menu bar to use the clipboard."
  (define-key menu-bar-edit-menu [paste] '("Paste" . pm-paste))
  (define-key menu-bar-edit-menu [copy] '("Copy" . pm-copy))
  (define-key menu-bar-edit-menu [cut] '("Cut" . pm-cut))
  (define-key menu-bar-edit-menu [choose-next-paste] 'undefined)
  (put 'pm-copy 'menu-enable 'mark-active)
  (put 'pm-cut 'menu-enable 'mark-active)
  (put 'pm-paste 'menu-enable '(pm-clipboard-ready-p))
  nil)

;;
;; This can be run from the initialization file
;;
(defun pm-files-menu ()
  "Modify the Files menu of the menu bar to use the file dialog box."
  (define-key menu-bar-files-menu [open-file] '("Open File..." . pm-find-file))
  (define-key menu-bar-files-menu [write-file]
    '("Save Buffer As..." . pm-write-file)))

;;
;; Drag & Drop
;;
;; This command must be bound to the `drop' event.
;;
;; A magic cookie, to be used for pm-get-drop, is passed instead
;; of a timestamp.
;;
(defun pm-drop (event)
  (interactive "e")
  (let* ((pos (nth 1 event))
	 (obj (pm-get-drop (posn-timestamp pos))))
    (if obj
	(progn
	  (select-window (nth 0 pos))
	  (find-file obj)))))

(global-set-key [drop] 'pm-drop)

;; Don't show the frame name as it's displayed in the title bar.
(setq-default mode-line-buffer-identification '("Emacs: %12b"))

;;
;; Dummy function
;;
(defun x-get-resource (attribute class &optional component subclass)
  nil)

(defalias 'x-create-frame 'pm-create-frame)
(defalias 'x-color-defined-p 'pm-color-defined-p)
(defalias 'x-color-values 'pm-color-values)
(defalias 'x-display-color-p 'pm-display-color-p)
(defalias 'x-display-grayscale-p 'pm-display-grayscale-p)
(defalias 'x-display-planes 'pm-display-planes)
(defalias 'x-display-color-cells 'pm-display-color-cells)
(defalias 'x-defined-colors 'pm-defined-colors)
(defalias 'x-list-fonts 'pm-list-fonts)

;;
;; Handle scroll bar events
;;
;; Written by Patrick Nadeau, modified by Eberhard Mattes
;;
(defun pm-scroll-bar-scroll (event)
  (interactive "e")
  (or point-before-scroll
      (setq point-before-scroll (point)))
  (let* ((end-position (event-end event))
 	 (window (nth 0 end-position))
 	 (part   (nth 4 end-position))
 	 (saved-window (selected-window))
         point-before-scroll)
    (select-window window)
    (cond ((eq part 'above-handle) (scroll-down))
 	  ((eq part 'below-handle)
           (condition-case nil
               (scroll-up)
             (end-of-buffer (goto-char (point-max)))))
 	  ((eq part 'up-arrow)     (scroll-down 1))
 	  ((eq part 'down-arrow)   (scroll-up 1))
 	  ((eq part 'handle)
           (save-excursion
             (goto-char (+ (point-min)
                           (scroll-bar-scale (nth 2 end-position)
                                             (- (point-max) (point-min)))))
             (beginning-of-line)
             (set-window-start window (point))))
          (t nil))
    (select-window saved-window)))

(global-set-key [vertical-scroll-bar mouse-1] 'pm-scroll-bar-scroll)
