/* mony_api.c   */
/* 24-Jan-96 - original (chh) */

// ------------------------------------------------------------
//
// A simplified API for using the Mony MDA BIOS emulator driver
// for HLL access follows after this IDC-access description for
// DD programmers.  The protos/structures are in mony_api.h.
//
// ------------------------------------------------------------
// For IDC/device driver access, use:
//
// AttachDD to the Mony driver at init or task time (MONY.SYS must already be loaded).
//
// IDCstruc STRUC
//  userRez        WORD 3 DUP (0)
//  entryPtr       WORD 0
//  entrySel       WORD 0
//  entryDS        WORD 0
// IDCstruc ENDS
//
// IDCname      BYTE "MONY$   "         ;that's MONY$ followed by 3 spaces
// IDC          IDCstruc <>             ;as defined above
//
// ;attach to MONY$ driver if available
//
//      mov     bx,OFFSET IDCname
//      mov     di,OFFSET IDC
//      mov     dl,42           ;ATTACH_DD in devhlp.inc
//      call    [Device_Help]
//      jc      @F
//      ...
//
// ;You cannot call Mony during your device driver init since you are at ring3.
// ;You have to wait for task (kernel/strategy/IOCtl) or interrupt time.
// ;For example, in a generic IOCtl call you can print a string by:
//
//      sub     al,al                  ;al=0: bl=attr ,cursorPos not updated
//      mov     bx,0007h               ;bh=page, (bl=attr if al=0/1)
//      mov     cx,SIZEOF stringMsg    ;cx=string length
//      mov     dx,0100h               ;dh=row, dl=col
//      mov     di,OFFSET stringMsg    ;es:di->string
//      push    ds
//      pop     es
//      mov     ah,13h
//      ;awkward loading ds, so I do it for you (i.e., don't load ds=IDC.entryDS)
//      call    DWORD PTR IDC.entryPtr
//      sub     ax,ax
//
// ;The virtual address must be global (i.e., in the GDT).  Shouldn't be a problem,
// ;but if it is (an LDT entry), just use Device_Help to make a suitable pointer.
//
// ;As you see, pointers down here are now virtual (sel:off), and can be easily used.
// ;Unlike BIOS, I don't use BP for passing data.  Instead, di is used whenever
// ;a pointer offset is required.  Functions 13h, 14h, 15h, and 1Bh uses es:di
// ;(the pointer use should be obvious).  Register arguments are like the standard
// ;INT10 interface.  Consult the mony_api.c file for any exceptions with regard
// ;to registers ax,bx,cx,dx (there may not be any exceptions, except that BP is
// ;not used by Mony as an argument, while INT10 BIOS calls do use it -- I just
// ;use di, instead).  The IDC entry is semaphore guarded, and returns 0FE14h if
// ;busy (i.e., at interrupt time).  Retry until you succeed, if you want.  The
// ;HLL interface returns busy as 0xFF14h, and this indicates that a DD is using
// ;Mony at interrupt time.  Retry, again, as required.
//
// ;If you are doing device drivers in something other than assembly, you need
// ;to find out how to go about doing the above tasks before you can use Mony.
// ;For applicaton level stuff, you can forget about all this above, and just use
// ;the stuff below.
//
// ------------------------------------------------------------------------------
// Now, back to our regular programming

#define INCL_BASE
#define INCL_DOSDEVIOCTL

#include <os2.h>

#include <stdlib.h>
#include <string.h>     // only strlen() is used

#include "mony_api.h"  // protos to functions in this module are in mony_api.h


// 00 -----------------------------------
// Zeroes all page memory (unless bit7=1)
// Displays page 0
// Inits all cursor positions to 0,0
// Disables blink/enables intensity

ULONG MonySetVideoMode (HFILE monyID, UCHAR videoMode) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   REGPACK iREG;

   iREG.ah = 0x00;
   iREG.al = videoMode;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    NULL,0,NULL);

   return rc;
}


// 01 ------------
// Set cursor size
// topLine=0x20 disables the cursor

ULONG MonySetCursorSize (HFILE monyID, UCHAR topLine, UCHAR bottomLine) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   REGPACK iREG;

   iREG.ah = 0x01;
   iREG.ch = topLine;
   iREG.cl = bottomLine;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    NULL,0,NULL);
   return rc;
}


// 02 --------------------------
// Set Cursor Position for Page

ULONG MonySetCursorPos (HFILE monyID, UCHAR page, UCHAR row, UCHAR column) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   REGPACK iREG;

   iREG.ah = 0x02;
   iREG.bh = page;
   iREG.dh = row;
   iREG.dl = column;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    NULL,0,NULL);
   return rc;
}


// 03 --------------
// Get Cursor Status

ULONG MonyGetCursorStatus (HFILE monyID, UCHAR page, PUCHAR row, PUCHAR column, PUCHAR topLine, PUCHAR bottomLine) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   ULONG dataLen = sizeof(REGPACK);
   REGPACK iREG, oREG;

   iREG.ah = 0x03;
   iREG.bh = page;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    &oREG,sizeof(REGPACK),&dataLen);
   *row = oREG.dh;
   *column = oREG.dl;
   *topLine = oREG.ch;
   *bottomLine = oREG.cl;
   return rc;
}


// 05 -----------
// Set Video Page

ULONG MonySetVideoPage (HFILE monyID, UCHAR page) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   REGPACK iREG;

   iREG.ah = 0x05;
   iREG.al = page;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    NULL,0,NULL);
   return rc;
}


// 06 ------
// Scroll Up

ULONG MonyScrollUp (HFILE monyID, UCHAR lines, UCHAR attr, UCHAR r0, UCHAR c0, UCHAR r1, UCHAR c1) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   REGPACK iREG;

   iREG.ah = 0x06;
   iREG.al = lines;
   iREG.bh = attr;
   iREG.ch = r0;
   iREG.cl = c0;
   iREG.dh = r1;
   iREG.dl = c1;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    NULL,0,NULL);
   return rc;
}


// 07 --------
// Scroll Down

ULONG MonyScrollDown (HFILE monyID, UCHAR lines, UCHAR attr, UCHAR r0, UCHAR c0, UCHAR r1, UCHAR c1) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   REGPACK iREG;

   iREG.ah = 0x07;
   iREG.al = lines;
   iREG.bh = attr;
   iREG.ch = r0;
   iREG.cl = c0;
   iREG.dh = r1;
   iREG.dl = c1;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    NULL,0,NULL);
   return rc;
}


// 08 ----------------------------------
// Get Character and Attribute at Cursor
//
// An easy enchancement to this routine is to include a cursor position and
// get-save the current position, set to the new, get char/attr, then restore
// the original cursor position.

ULONG MonyGetCharAttrAtCursor (HFILE monyID, UCHAR page, PUCHAR character, PUCHAR attr) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   ULONG dataLen = sizeof(REGPACK);
   REGPACK iREG, oREG;

   iREG.ah = 0x08;
   iREG.bh = page;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    &oREG,sizeof(REGPACK),&dataLen);
   *character = oREG.al;
   *attr = oREG.ah;
   return rc;
}


// 09 ---------------------------------------
// Write Character(s) and Attribute at Cursor
//
// Does not change cursor position.
//
// An easy enchancement to this routine is to include a cursor position and
// get-save the current position, set to the new, write char/attr, then restore
// the original cursor position.

ULONG MonyWriteCharAttrAtCursor (HFILE monyID, UCHAR page, UCHAR character, UCHAR attr, USHORT repCount) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   REGPACK iREG;

   iREG.ah = 0x09;
   iREG.al = character;
   iREG.bh = page;
   iREG.bl = attr;
   iREG.cx = repCount;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    NULL,0,NULL);
   return rc;
}


// 0A -------------------------
// Write Character(s) at Cursor
//
// Does not change cursor position.
//
// Uses existing attribute, otherwise similar to 09.

ULONG MonyWriteCharAtCursor (HFILE monyID, UCHAR page, UCHAR character, USHORT repCount) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   REGPACK iREG;

   iREG.ah = 0x0A;
   iREG.al = character;
   iREG.bh = page;
   iREG.cx = repCount;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    NULL,0,NULL);
   return rc;
}


// 0E ------------------------
// Write Character in TTY mode
//
// Processing control characters:  7=bell, 8=BS, 10=LF, 13=CR.
// Cursor position is updated as appropriate.

ULONG MonyWriteCharTTY (HFILE monyID, UCHAR page, UCHAR character) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   REGPACK iREG;

   iREG.ah = 0x0E;
   iREG.al = character;
   iREG.bh = page;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    NULL,0,NULL);
   return rc;
}


// 0F -------------
// Get Video Status
//
// Gets the number of columns (80), video mode (7), and active page (0-7).

ULONG MonyGetVideoStatus (HFILE monyID, PUCHAR page, PUCHAR columns, PUCHAR videoMode) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   ULONG dataLen = sizeof(REGPACK);
   REGPACK iREG, oREG;

   iREG.ah = 0x0F;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    &oREG,sizeof(REGPACK),&dataLen);
   *columns = oREG.ah;
   *videoMode = oREG.al;
   *page = oREG.bh;
   return rc;
}


// 10/3 --------------
// Set Blink/Intensity
//
// Select either blink mode or intensity mode when bit7=1 of attr byte.
// blink=1 sets blink mode.

ULONG MonySetBlinkMode (HFILE monyID, UCHAR blink) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   REGPACK iREG;

   iREG.ah = 0x10;
   iREG.al = 0x03;
   iREG.bl = blink;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    NULL,0,NULL);
   return rc;
}

// 10/4 -----------------
// Set Bell Freq/Duration
//
// Set frequency (Hz) and duration (ms) of bell sound on Ctrl-G processing
// during Write TTY or String, where 0 for BOTH turns off the bell.  The
// Ctrl-G is not displayed in any case using WriteTTY or WriteString.  Use
// one of the WriteChar functions to display code 7.

ULONG MonySetBellMode (HFILE monyID, USHORT freq, USHORT duration) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   REGPACK iREG;

   iREG.ah = 0x10;
   iREG.al = 0x04;
   iREG.bx = freq;
   iREG.cx = duration;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    NULL,0,NULL);
   return rc;
}


// 13 ----------
// Writes String
// without cursor position update

ULONG MonyWriteString (HFILE monyID, PSZ strgPtr, UCHAR page, UCHAR attr, UCHAR row, UCHAR col) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   ULONG dataLen;
   REGPACK iREG;

   dataLen = strlen(strgPtr);
   if (dataLen) {

      iREG.ah = 0x13;
      iREG.al = 0;         // use attribute in .bl, do not update cursor position
      iREG.bh = page;
      iREG.bl = attr;
      iREG.cx = dataLen;
      iREG.dh = row;
      iREG.dl = col;

      rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                       &iREG,sizeof(REGPACK),&parmLen,
                       strgPtr,strlen(strgPtr),&dataLen);
   }
   return rc;
}


// 14 -------
// Read Block
//
// Read page memory block to buffer.  Buffer must be large enough to contain area
// (rows * columns * 2 (*2 for char/attr pairs)).  Page memory is read, and need
// not be the active display.  This is not a BIOS INT10 routine.

ULONG MonyReadBlock (HFILE monyID, PVOID bufferPtr, UCHAR page, UCHAR r0, UCHAR c0, UCHAR rows, UCHAR columns) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   ULONG dataLen;
   REGPACK iREG;

   dataLen = rows * columns * 2;

   iREG.ah = 0x14;
   iREG.bh = page;
   iREG.ch = r0;
   iREG.cl = c0;
   iREG.dh = rows;
   iREG.dl = columns;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    bufferPtr,dataLen,&dataLen);
   return rc;
}


// 15 --------
// Write Block
//
// Write buffer to page memory and screen if page is active.
// Buffer must be same size as (rows * columns * 2 (*2 for char/attr pairs)).
// This is not a BIOS INT10 routine.

ULONG MonyWriteBlock (HFILE monyID, PVOID bufferPtr, UCHAR page, UCHAR r0, UCHAR c0, UCHAR rows, UCHAR columns) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   ULONG dataLen;
   REGPACK iREG;

   dataLen = rows * columns *2;

   iREG.ah = 0x15;
   iREG.bh = page;
   iREG.ch = r0;
   iREG.cl = c0;
   iREG.dh = rows;
   iREG.dl = columns;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    bufferPtr,dataLen,&dataLen);
   return rc;
}


// 1B -----------------------------------
// Fill 64-byte Table with Mony BIOS Info
// The returned info is slightly modified from standard INT10 to better suit Mony.

ULONG MonyGetBiosData (HFILE monyID, PBIOSDATA VBDptr) {

   ULONG rc=0;
   ULONG parmLen = sizeof(REGPACK);
   ULONG dataLen = sizeof(BIOSDATA);
   REGPACK iREG;

   iREG.ah = 0x1B;
   iREG.bx = 0;

   rc = DosDevIOCtl(monyID,IOCTL_MONY,IOCTL_40TH,
                    &iREG,sizeof(REGPACK),&parmLen,
                    VBDptr,sizeof(BIOSDATA),&dataLen);
   return rc;
}

// o-la-la
// -------

