//***************************************************************************
//
// ZipDrop.cpp
//
//***************************************************************************
 
#include <windows.h>
#include <initguid.h>
#include <shlobj.h>
#include "Resource.h"
#include "ZipDrop.h"

/////////////////////////////////////////////////////////////////////////////
// Nonmember function prototypes

STDAPI DllGetClassObject (REFCLSID, REFIID, LPVOID*);
STDAPI DllCanUnloadNow ();

BOOL CALLBACK OptionsProc (HWND, UINT, WPARAM, LPARAM);

/////////////////////////////////////////////////////////////////////////////
// Global variables

UINT        g_cRefThisDll = 0;          // Reference count for this DLL
HINSTANCE   g_hInstance;                // Instance handle for this DLL

BOOL    g_bCopy = TRUE;
BOOL    g_bRecurse = FALSE;
BOOL    g_bPreservePath = FALSE;
BOOL    g_bFolder = FALSE;

/////////////////////////////////////////////////////////////////////////////
// DLL entry point

extern "C" BOOL WINAPI DllMain (HINSTANCE hInstance, DWORD dwReason,
    LPVOID lpReserved)
{
    if (dwReason == DLL_PROCESS_ATTACH)
        g_hInstance = hInstance;
    return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// Exported functions

STDAPI DllGetClassObject (REFCLSID rclsid, REFIID riid, LPVOID* ppvObj)
{
    *ppvObj = NULL; 
    if (rclsid != CLSID_ShellExtension)
        return CLASS_E_CLASSNOTAVAILABLE;

    CClassFactory* pClassFactory = new CClassFactory;

    if (pClassFactory == NULL)
        return E_OUTOFMEMORY;

    HRESULT hr = pClassFactory->QueryInterface (riid, ppvObj);
    pClassFactory->Release ();
    return hr;
}

STDAPI DllCanUnloadNow ()
{
    return (g_cRefThisDll == 0) ? S_OK : S_FALSE;
}

/////////////////////////////////////////////////////////////////////////////
// CClassFactory member functions

CClassFactory::CClassFactory ()
{
    m_cRef = 1;
    g_cRefThisDll++;
}

CClassFactory::~CClassFactory ()
{
    g_cRefThisDll--;
}

STDMETHODIMP CClassFactory::QueryInterface (REFIID riid, LPVOID* ppvObj)
{
    *ppvObj = NULL;
    HRESULT hr = E_NOINTERFACE;

    if ((riid == IID_IUnknown) || (riid == IID_IClassFactory)) {
        *ppvObj = this;
        hr = NOERROR;
        m_cRef++;
    }
    return hr;
}

STDMETHODIMP_(ULONG) CClassFactory::AddRef ()
{
    return ++m_cRef;
}

STDMETHODIMP_(ULONG) CClassFactory::Release ()
{
    if (--m_cRef == 0)
        delete this;
    return m_cRef;
}

STDMETHODIMP CClassFactory::CreateInstance (LPUNKNOWN pUnkOuter, REFIID riid,
    LPVOID* ppvObj)
{
    *ppvObj = NULL;
    if (pUnkOuter != NULL)
        return CLASS_E_NOAGGREGATION;

    CShellExtension* pShellExtension = new CShellExtension;

    if (pShellExtension == NULL)
        return E_OUTOFMEMORY;

    HRESULT hr = pShellExtension->QueryInterface (riid, ppvObj);
    pShellExtension->Release ();
    return hr;
}

STDMETHODIMP CClassFactory::LockServer (BOOL fLock)
{
    return E_NOTIMPL;
}

/////////////////////////////////////////////////////////////////////////////
// CShellExtension member functions

CShellExtension::CShellExtension ()
{
    m_cRef = 1;
    g_cRefThisDll++;
}

CShellExtension::~CShellExtension ()
{
    g_cRefThisDll--;
}

STDMETHODIMP CShellExtension::QueryInterface (REFIID riid, LPVOID* ppvObj)
{
    *ppvObj = NULL;
    HRESULT hr = E_NOINTERFACE;

    if (riid == IID_IUnknown) {
        *ppvObj = (LPUNKNOWN) (LPDROPTARGET) this;
        hr = NOERROR;
        m_cRef++;
    }
    else if (riid == IID_IDropTarget) {
        *ppvObj = (LPDROPTARGET) this;
        hr = NOERROR;
        m_cRef++;
    }
    else if (riid == IID_IPersistFile) {
        *ppvObj = (LPPERSISTFILE) this;
        hr = NOERROR;
        m_cRef++;
    }
    return hr;
}

STDMETHODIMP_(ULONG) CShellExtension::AddRef ()
{
    return ++m_cRef;
}
 
STDMETHODIMP_(ULONG) CShellExtension::Release ()
{
    if (--m_cRef == 0)
        delete this;
    return m_cRef;
}

STDMETHODIMP CShellExtension::DragEnter (IDataObject* pDataObj,
    DWORD grfKeyState, POINTL pt, DWORD* pdwEffect)
{
    static FORMATETC fe = { CF_HDROP, NULL, DVASPECT_CONTENT,
        -1, TYMED_HGLOBAL };

    HRESULT hr = pDataObj->QueryGetData (&fe);

    if (FAILED (hr)) {
        m_bFormatSupported = FALSE;
        *pdwEffect = DROPEFFECT_NONE;
    }
    else {
        m_bFormatSupported = TRUE;
        *pdwEffect = (grfKeyState & MK_CONTROL) ? DROPEFFECT_COPY :
            DROPEFFECT_MOVE;
    }
    m_bRightDrag = grfKeyState & MK_RBUTTON;
    return NOERROR;
}

STDMETHODIMP CShellExtension::DragOver (DWORD grfKeyState, POINTL pt,
    DWORD* pdwEffect)
{
    if (m_bFormatSupported)
        *pdwEffect = (grfKeyState & MK_CONTROL) ? DROPEFFECT_COPY :
            DROPEFFECT_MOVE;
    else
        *pdwEffect = DROPEFFECT_NONE;

    m_bRightDrag = grfKeyState & MK_RBUTTON;
    return NOERROR;
}

STDMETHODIMP CShellExtension::DragLeave ()
{
    return NOERROR;
}

STDMETHODIMP CShellExtension::Drop (IDataObject* pDataObj,
    DWORD grfKeyState, POINTL pt, DWORD* pdwEffect)
{
    static FORMATETC fe = { CF_HDROP, NULL, DVASPECT_CONTENT, -1,
        TYMED_HGLOBAL };

    //
    // Ask the drop source to render the data in CF_HDROP format.
    //
    STGMEDIUM medium;
    HRESULT hr = pDataObj->GetData (&fe, &medium);

    if (FAILED (hr)) {
        *pdwEffect = DROPEFFECT_NONE;
        return E_FAIL;
    }

    //
    // Add the files to the archive.
    //
    *pdwEffect = (grfKeyState & MK_CONTROL) ? DROPEFFECT_COPY :
        DROPEFFECT_MOVE;
    hr = NOERROR;

    if (!ZipFiles (&medium, grfKeyState & MK_CONTROL, m_bRightDrag)) {
        *pdwEffect = DROPEFFECT_NONE;
        hr = E_FAIL;
    }

    //
    // Release the storage medium and we're done.
    //
    ReleaseStgMedium (&medium);
    return hr;
}

STDMETHODIMP CShellExtension::Load (LPCOLESTR pszFile, DWORD dwMode)
{
    if (!WideCharToMultiByte (CP_ACP, WC_SEPCHARS, pszFile, -1, m_szFile,
        sizeof (m_szFile), NULL, NULL)) {
        LPSTR psz = m_szFile;
        while (*psz++ = (char) *pszFile++);
    }
    GetShortPathName (m_szFile, m_szFile, sizeof (m_szFile));
    return NOERROR;
}

STDMETHODIMP CShellExtension::GetClassID (LPCLSID lpClsID)
{
    return E_NOTIMPL;
}

STDMETHODIMP CShellExtension::IsDirty ()
{
    return E_NOTIMPL;
}

STDMETHODIMP CShellExtension::Save (LPCOLESTR pszFile, BOOL fRemember)
{
    return E_NOTIMPL;
}

STDMETHODIMP CShellExtension::SaveCompleted (LPCOLESTR pszFile)
{
    return E_NOTIMPL;
}

STDMETHODIMP CShellExtension::GetCurFile (LPOLESTR FAR *ppszFile)
{
    return E_NOTIMPL;
}

BOOL CShellExtension::ZipFiles (STGMEDIUM* pMedium, BOOL bCopy, BOOL bPrompt)
{
    char szCmdLine[(MAX_PATH * 2) + 32];
    char szTempFile[MAX_PATH], szTempPath[MAX_PATH];
    static char crlf[2] = { 0x0D, 0x0A };

    //
    // Get a count of files to be added to the archive.
    //
    UINT nFiles = DragQueryFile ((HDROP) pMedium->hGlobal, (UINT) -1,
        NULL, 0);

    if (nFiles == 0)
        return FALSE;

    //
    // Create a list file containing the file names.
    //
    GetTempPath (sizeof (szTempPath), szTempPath);
    GetTempFileName (szTempPath, "LST", 0, szTempFile);
    GetShortPathName (szTempFile, szTempFile, sizeof (szTempFile));

    HANDLE hFile = CreateFile (szTempFile, GENERIC_WRITE, 0, NULL,
        CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);

    if (hFile == INVALID_HANDLE_VALUE)
        return FALSE;

    DWORD dwBytesWritten;
    char szFile[MAX_PATH];
    g_bFolder = FALSE;

    for (UINT i=0; i<nFiles; i++) {
        DragQueryFile ((HDROP) pMedium->hGlobal, i, szFile, sizeof (szFile));
        GetShortPathName (szFile, szFile, sizeof (szFile));
        if (GetFileAttributes (szFile) & FILE_ATTRIBUTE_DIRECTORY) {
            lstrcat (szFile, "\\*.*");
            g_bFolder = TRUE;
        }
        WriteFile (hFile, szFile, lstrlen (szFile), &dwBytesWritten, NULL);
        WriteFile (hFile, crlf, 2, &dwBytesWritten, NULL);
    }

    CloseHandle (hFile);

    //
    // Formulate a "PKZIP -a -whs [-m] [-r] [-p] zipfile @listfile" command.
    //
    lstrcpy (szCmdLine, "pkzip -a -whs ");

    if (!bPrompt) {
        if (!bCopy)
            lstrcat (szCmdLine, "-m ");
    }
    else {
        g_bCopy = bCopy;
        if (!DialogBox (g_hInstance, "Options", NULL,
            (DLGPROC) OptionsProc)) {
            DeleteFile (szTempFile);
            return FALSE;
        }
        if (!g_bCopy)
            lstrcat (szCmdLine, "-m ");
        if (g_bFolder) {
            if (g_bRecurse) {
                lstrcat (szCmdLine, "-r ");
                if (g_bPreservePath)
                    lstrcat (szCmdLine, "-p ");
            }
        }
    }
    lstrcat (szCmdLine, m_szFile);
    lstrcat (szCmdLine, " @");
    lstrcat (szCmdLine, szTempFile);

    //
    // Execute the PKZIP command.
    //
    STARTUPINFO si;
    ZeroMemory (&si, sizeof (STARTUPINFO));
    si.cb = sizeof (STARTUPINFO);
    PROCESS_INFORMATION pi;

    BOOL bReturn = TRUE;

    if (CreateProcess (NULL, szCmdLine, NULL, NULL, FALSE, 0,
        NULL, NULL, &si, &pi)) {
        WaitForSingleObject (pi.hProcess, INFINITE);
        CloseHandle (pi.hProcess);
        CloseHandle (pi.hThread);
    }
    else {
        MessageBox (NULL, "Unable to launch PKZIP. Make sure PKZIP.EXE " \
            "is in a PATHed directory and try again.", "Error",
            MB_ICONINFORMATION | MB_OK);
        bReturn = FALSE;
    }

    //
    // Clean up and exit.
    //
    DeleteFile (szTempFile);
    return bReturn;
}

/////////////////////////////////////////////////////////////////////////////
// Dialog procedures

BOOL CALLBACK OptionsProc (HWND hwnd, UINT uMessage, WPARAM wParam,
    LPARAM lParam)
{
    int nWidth, nHeight;
    int nScreenWidth, nScreenHeight;

    switch (uMessage) {

    case WM_INITDIALOG:
        //
        // Make sure the dialog isn't off the screen.
        //
        POINT point;
        GetCursorPos (&point);

        RECT rect;
        GetWindowRect (hwnd, &rect);
        nWidth = rect.right - rect.left;
        nHeight = rect.bottom - rect.top;

        nScreenWidth = GetSystemMetrics (SM_CXSCREEN);
        nScreenHeight = GetSystemMetrics (SM_CYSCREEN);

        if ((point.x + nWidth) > nScreenWidth)
            point.x  = nScreenWidth - nWidth - 8;

        if ((point.y + nHeight) > nScreenHeight)
            point.y = nScreenHeight - nHeight - 8;

        SetWindowPos (hwnd, NULL, point.x, point.y, 0, 0,
            SWP_NOSIZE | SWP_NOZORDER);

        //
        // Initialize the dialog's controls.
        //
        CheckRadioButton (hwnd, IDD_MOVE, IDD_COPY,
            g_bCopy ? IDD_COPY : IDD_MOVE);

        SendDlgItemMessage (hwnd, IDD_RECURSE, BM_SETCHECK,
            g_bRecurse ? 1 : 0, 0);
        SendDlgItemMessage (hwnd, IDD_PRESERVEPATH, BM_SETCHECK,
            g_bPreservePath ? 1 : 0, 0);

        EnableWindow (GetDlgItem (hwnd, IDD_RECURSE), g_bFolder);
        EnableWindow (GetDlgItem (hwnd, IDD_PRESERVEPATH), g_bFolder &&
            g_bRecurse);
        return TRUE;

    case WM_ACTIVATE:
        //
        // Kill the dialog if another window is activated.
        //
        if (LOWORD (wParam) == WA_INACTIVE) {
            EndDialog (hwnd, 0);
            return TRUE;
        }
        break;

    case WM_COMMAND:
        //
        // Respond to control notifications.
        //
        UINT nID = LOWORD (wParam);

        switch (nID) {

        case IDD_RECURSE:
            EnableWindow (GetDlgItem (hwnd, IDD_PRESERVEPATH),
                IsDlgButtonChecked (hwnd, IDD_RECURSE));
            return TRUE;

        case IDOK:
            g_bCopy = IsDlgButtonChecked (hwnd, IDD_COPY);
            g_bRecurse = IsDlgButtonChecked (hwnd, IDD_RECURSE);
            g_bPreservePath = IsDlgButtonChecked (hwnd, IDD_PRESERVEPATH);
            EndDialog (hwnd, 1);
            return TRUE;

        case IDCANCEL:
            EndDialog (hwnd, 0);
            return TRUE;
        }
        break;
    }
    return FALSE;
}
