//***************************************************************************
//
// ZipView.cpp
//
//***************************************************************************
 
#include <windows.h>
#include <initguid.h>
#include <shlobj.h>
#include "Resource.h"
#include "ZipView.h"

/////////////////////////////////////////////////////////////////////////////
// Nonmember function prototypes

STDAPI DllGetClassObject (REFCLSID, REFIID, LPVOID*);
STDAPI DllCanUnloadNow ();

BOOL CALLBACK PageProc (HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK OptionsProc (HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK GetPathProc (HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK AboutProc (HWND, UINT, WPARAM, LPARAM);
int CALLBACK CompareFunc (LPARAM, LPARAM, LPARAM);

BOOL IsAnyItemSelected (HWND);
int GetZipFileInfo (LPSTR, PZIPFILEINFO*);
void InitListView (HWND, int, PZIPFILEINFO);
void GetDispInfo (LV_DISPINFO*);
void ToAnsi (DWORD, char*);
void ToDate (WORD, char*);
void ToTime (WORD, char*);
void SaveColumnWidths (HWND);
void GetColumnWidths (int*);
void GetPathToPkunzip (LPSTR, UINT, BOOL*, UINT);
void WritePathToPkunzip (LPSTR, BOOL);
void ExtractFiles (HWND, HWND, LPSTR);
int CreateListFile (HWND, LPSTR);
void WriteListFileEntry (HANDLE, LPSTR);
int DoPathSetup (HWND);
BOOL IsPathValid (LPSTR);
void GetExtractPath ();
void WriteExtractPath ();

/////////////////////////////////////////////////////////////////////////////
// Global variables

UINT        g_cRefThisDll = 0;          // Reference count for this DLL
HINSTANCE   g_hInstance;                // Instance handle for this DLL

char* g_pszMethods[] = {
    "Stored",
    "Shrunk",
    "Reduced (CF=1)",
    "Reduced (CF=2)",
    "Reduced (CF=3)",
    "Reduced (CF=4)",
    "Imploded",
    "Tokenized",
    "Deflated"
};

char g_szSubkey[] = "Software\\PC Magazine Utilities\\ZipView\\V1.0";
char g_szDir[MAX_PATH];

/////////////////////////////////////////////////////////////////////////////
// DLL entry point

extern "C" BOOL WINAPI DllMain (HINSTANCE hInstance, DWORD dwReason,
    LPVOID lpReserved)
{
    if (dwReason == DLL_PROCESS_ATTACH)
        g_hInstance = hInstance;
    return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// Exported functions

STDAPI DllGetClassObject (REFCLSID rclsid, REFIID riid, LPVOID* ppvObj)
{
    *ppvObj = NULL; 
    if (rclsid != CLSID_ShellExtension)
        return CLASS_E_CLASSNOTAVAILABLE;

    CClassFactory* pClassFactory = new CClassFactory;

    if (pClassFactory == NULL)
        return E_OUTOFMEMORY;

    HRESULT hr = pClassFactory->QueryInterface (riid, ppvObj);
    pClassFactory->Release ();
    return hr;
}

STDAPI DllCanUnloadNow ()
{
    return (g_cRefThisDll == 0) ? S_OK : S_FALSE;
}

/////////////////////////////////////////////////////////////////////////////
// CClassFactory member functions

CClassFactory::CClassFactory ()
{
    m_cRef = 1;
    g_cRefThisDll++;
}

CClassFactory::~CClassFactory ()
{
    g_cRefThisDll--;
}

STDMETHODIMP CClassFactory::QueryInterface (REFIID riid, LPVOID* ppvObj)
{
    *ppvObj = NULL;
    HRESULT hr = E_NOINTERFACE;

    if ((riid == IID_IUnknown) || (riid == IID_IClassFactory)) {
        *ppvObj = this;
        hr = NOERROR;
        m_cRef++;
    }
    return hr;
}

STDMETHODIMP_(ULONG) CClassFactory::AddRef ()
{
    return ++m_cRef;
}

STDMETHODIMP_(ULONG) CClassFactory::Release ()
{
    if (--m_cRef == 0)
        delete this;
    return m_cRef;
}

STDMETHODIMP CClassFactory::CreateInstance (LPUNKNOWN pUnkOuter, REFIID riid,
    LPVOID* ppvObj)
{
    *ppvObj = NULL;
    if (pUnkOuter != NULL)
        return CLASS_E_NOAGGREGATION;

    CShellExtension* pShellExtension = new CShellExtension;

    if (pShellExtension == NULL)
        return E_OUTOFMEMORY;

    HRESULT hr = pShellExtension->QueryInterface (riid, ppvObj);
    pShellExtension->Release ();
    return hr;
}

STDMETHODIMP CClassFactory::LockServer (BOOL fLock)
{
    return E_NOTIMPL;
}

/////////////////////////////////////////////////////////////////////////////
// CShellExtension member functions

CShellExtension::CShellExtension ()
{
    m_cRef = 1;
    g_cRefThisDll++;
}

CShellExtension::~CShellExtension ()
{
    g_cRefThisDll--;
}

STDMETHODIMP CShellExtension::QueryInterface (REFIID riid, LPVOID* ppvObj)
{
    *ppvObj = NULL;
    HRESULT hr = E_NOINTERFACE;

    if (riid == IID_IUnknown) {
        *ppvObj = (LPUNKNOWN) (LPSHELLPROPSHEETEXT) this;
        hr = NOERROR;
        m_cRef++;
    }
    else if (riid == IID_IShellPropSheetExt) {
        *ppvObj = (LPSHELLPROPSHEETEXT) this;
        hr = NOERROR;
        m_cRef++;
    }
    else if (riid == IID_IShellExtInit) {
        *ppvObj = (LPSHELLEXTINIT) this;
        hr = NOERROR;
        m_cRef++;
    }
    return hr;
}

STDMETHODIMP_(ULONG) CShellExtension::AddRef ()
{
    return ++m_cRef;
}
 
STDMETHODIMP_(ULONG) CShellExtension::Release ()
{
    if (--m_cRef == 0)
        delete this;
    return m_cRef;
}

STDMETHODIMP CShellExtension::AddPages (LPFNADDPROPSHEETPAGE lpfnAddPage,
    LPARAM lParam)
{
    PROPSHEETPAGE psp;
    HPROPSHEETPAGE hPage;

    psp.dwSize = sizeof (psp);
    psp.dwFlags = PSP_USEREFPARENT | PSP_USETITLE;
    psp.hInstance = g_hInstance;
    psp.pszTemplate = "ZipContents";
    psp.pszTitle = "Contents";
    psp.pfnDlgProc = (DLGPROC) PageProc;
    psp.lParam = (LPARAM) m_szFile;
    psp.pcRefParent = &g_cRefThisDll;

    hPage = CreatePropertySheetPage (&psp);

    if (hPage != NULL)
        if (!lpfnAddPage (hPage, lParam))
            DestroyPropertySheetPage (hPage);

    return NOERROR;
}
    
STDMETHODIMP CShellExtension::ReplacePage (UINT uPageID,
    LPFNADDPROPSHEETPAGE lpfnReplaceWith, LPARAM lParam)
{
    return E_FAIL;
}

STDMETHODIMP CShellExtension::Initialize (LPCITEMIDLIST pidlFolder,
    LPDATAOBJECT lpdobj, HKEY hKeyProgID)
{
    STGMEDIUM medium;
    FORMATETC fe = { CF_HDROP, NULL, DVASPECT_CONTENT, -1, TYMED_HGLOBAL };

    if (lpdobj == NULL)
        return E_FAIL;

    //
    // Render the data referenced by the IDataObject pointer to an HGLOBAL
    // storage medium in CF_HDROP format.
    //
    HRESULT hr = lpdobj->GetData (&fe, &medium);
    if (FAILED (hr))
        return E_FAIL;

    //
    // If only one file is selected, retrieve the file name and store it in
    // m_szFile. Otherwise fail the call.
    //
    hr = E_FAIL;
    if (DragQueryFile ((HDROP) medium.hGlobal, (UINT) -1, NULL, 0) == 1) {
        DragQueryFile ((HDROP) medium.hGlobal, 0, m_szFile, sizeof (m_szFile));
        // Convert the file name to a *short* file name so it's compatible
        // with the 16-bit PKUNZIP utility...
        GetShortPathName (m_szFile, m_szFile, sizeof (m_szFile));
        hr = NOERROR;
    }
   
    //
    // Release the storage medium and return.
    //
    ReleaseStgMedium (&medium);
    return hr;
}

/////////////////////////////////////////////////////////////////////////////
// Dialog procedures and other callback functions

BOOL CALLBACK PageProc (HWND hwnd, UINT uMessage, WPARAM wParam, LPARAM lParam)
{
    int nCount, i;
    NM_LISTVIEW* pnmlv;
    HWND hwndListView;
    PZIPFILEINFO pzfi;
    UINT nID;

    switch (uMessage) {
    
    case WM_INITDIALOG:
        //
        // Initialize the ListView control.
        //
        hwndListView = GetDlgItem (hwnd, IDD_LISTVIEW);
        SetWindowLong (hwnd, DWL_USER, ((LPPROPSHEETPAGE) lParam)->lParam);
        nCount = GetZipFileInfo ((LPSTR) ((LPPROPSHEETPAGE) lParam)->lParam,
            &pzfi);
        SetWindowLong (hwnd, GWL_USERDATA, (LONG) pzfi);
        InitListView (hwndListView, nCount, pzfi);

        if (nCount > 0) {
            EnableWindow (GetDlgItem (hwnd, IDD_SELECTALL), TRUE);
            EnableWindow (GetDlgItem (hwnd, IDD_SELECTNONE), TRUE);
        }
        return TRUE;    

    case WM_COMMAND:
        //
        // Process clicks of the Extract and Setup buttons.
        //
        nID = LOWORD (wParam);
        hwndListView = GetDlgItem (hwnd, IDD_LISTVIEW);

        switch (nID) {

        case IDD_EXTRACT:
            ExtractFiles (hwnd, hwndListView,
                (LPSTR) GetWindowLong (hwnd, DWL_USER));
            return TRUE;

        case IDD_SELECTALL:
            nCount = ListView_GetItemCount (hwndListView);
            for (i=0; i<nCount; i++)
                ListView_SetItemState (hwndListView, i, LVIS_SELECTED,
                    LVIS_SELECTED);
            EnableWindow (GetDlgItem (hwnd, IDD_EXTRACT), TRUE);
            return TRUE;

        case IDD_SELECTNONE:
            nCount = ListView_GetItemCount (hwndListView);
            for (i=0; i<nCount; i++)
                ListView_SetItemState (hwndListView, i, 0, LVIS_SELECTED);
            EnableWindow (GetDlgItem (hwnd, IDD_EXTRACT), FALSE);
            return TRUE;

        case IDD_OPTIONS:
            DoPathSetup (hwnd);
            return TRUE;

        case IDD_ABOUT:
            DialogBox ((HINSTANCE) GetWindowLong (hwnd, GWL_HINSTANCE),
                "About", hwnd, (DLGPROC) AboutProc);
            return TRUE;
        }
        break;

    case WM_NOTIFY:
        //
        // Respond to notifications.
        //    
        hwndListView = GetDlgItem (hwnd, IDD_LISTVIEW);

        switch (((NMHDR*) lParam)->code) {

        case PSN_APPLY:
        case PSN_RESET:
            SaveColumnWidths (hwndListView);
            break;

        case LVN_COLUMNCLICK:
            pnmlv = (NM_LISTVIEW*) lParam;
            ListView_SortItems (hwndListView, CompareFunc, pnmlv->iSubItem);
            break;

        case LVN_GETDISPINFO:
            GetDispInfo ((LV_DISPINFO*) lParam);
            break;

        case LVN_ITEMCHANGED:
            pnmlv = (NM_LISTVIEW*) lParam;
            EnableWindow (GetDlgItem (hwnd, IDD_EXTRACT),
                IsAnyItemSelected (hwndListView));
            break;
        }
        SetWindowLong (hwnd, DWL_MSGRESULT, FALSE);
        return TRUE;        

    case WM_DESTROY:
        //
        // Delete the ZIPFILEINFO array before closing.
        //
        pzfi = (PZIPFILEINFO) GetWindowLong (hwnd, GWL_USERDATA);
        if (pzfi != NULL)
            delete[] pzfi;
        return TRUE;
    }
    return FALSE;
}

BOOL CALLBACK OptionsProc (HWND hwnd, UINT uMessage, WPARAM wParam,
    LPARAM lParam)
{
    OPENFILENAME ofn;
    char szPathName[MAX_PATH];
    char szTemp[MAX_PATH];
    static char szFilters[] = "Executable files (*.exe)\0*.exe\0" \
        "All files (*.*)\0*.*\0\0";
    BOOL bPrompt;

    switch (uMessage) {
    
    case WM_INITDIALOG:
        //
        // Initialize the dialog box controls.
        //
        GetPathToPkunzip (szPathName, sizeof (szPathName), &bPrompt,
            sizeof (bPrompt));
        SetDlgItemText (hwnd, IDD_PATHNAME, szPathName);
        if (szPathName[0] != 0)
            EnableWindow (GetDlgItem (hwnd, IDOK), TRUE);
        CheckRadioButton (hwnd, IDD_DONTPROMPT, IDD_DOPROMPT,
            bPrompt ? IDD_DOPROMPT : IDD_DONTPROMPT);
        return TRUE;    

    case WM_COMMAND:
        //
        // Process messages from the controls.
        //
        UINT nID = LOWORD (wParam);

        switch (nID) {

        case IDOK:
            bPrompt = IsDlgButtonChecked (hwnd, IDD_DOPROMPT) ? TRUE : FALSE;
            GetDlgItemText (hwnd, IDD_PATHNAME, szPathName,
                sizeof (szPathName));
            WritePathToPkunzip (szPathName, bPrompt);
            EndDialog (hwnd, 1);
            return TRUE;

        case IDCANCEL:
            EndDialog (hwnd, 0);
            return TRUE;

        case IDD_BROWSE:
            szTemp[0] = 0;
            ZeroMemory (&ofn, sizeof (OPENFILENAME));

            ofn.lStructSize = sizeof (OPENFILENAME);
            ofn.hwndOwner = hwnd;
            ofn.lpstrFilter = szFilters;
            ofn.nFilterIndex = 1;
            ofn.lpstrFile = szTemp;
            ofn.nMaxFile = sizeof (szTemp);
            ofn.lpstrTitle = "Browse";
            ofn.Flags = OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;

            if (GetOpenFileName (&ofn)) {
                lstrcpy (szPathName, szTemp);
                SetDlgItemText (hwnd, IDD_PATHNAME, szPathName);
            }
            return TRUE;

        case IDD_PATHNAME:
            if (HIWORD (wParam) == EN_CHANGE) {
                int nChars = GetDlgItemText (hwnd, IDD_PATHNAME, szTemp, 8);
                EnableWindow (GetDlgItem (hwnd, IDOK), nChars);
                return TRUE;
            }
            break;
        }
        break;
    }
    return FALSE;
}

BOOL CALLBACK GetPathProc (HWND hwnd, UINT uMessage, WPARAM wParam,
    LPARAM lParam)
{
    char szDir[MAX_PATH + 32];

    switch (uMessage) {

    case WM_INITDIALOG:
        SetDlgItemText (hwnd, IDD_PATHNAME, g_szDir);
        if (g_szDir[0] != 0)
            EnableWindow (GetDlgItem (hwnd, IDOK), TRUE);
        return TRUE;

    case WM_COMMAND:
        UINT nID = LOWORD (wParam);

        switch (nID) {

        case IDOK:
            GetDlgItemText (hwnd, IDD_PATHNAME, szDir, sizeof (szDir));
            if (IsPathValid (szDir)) {
                lstrcpy (g_szDir, szDir);
                WriteExtractPath ();
                EndDialog (hwnd, 1);
            }
            else { // Invalid path name
                lstrcat (szDir, " is not a valid path name");
                MessageBox (hwnd, szDir, "Error", MB_OK);
                SetFocus (GetDlgItem (hwnd, IDD_PATHNAME));
                SendDlgItemMessage (hwnd, IDD_PATHNAME, EM_SETSEL, 0, -1);
            }
            return TRUE;

        case IDCANCEL:
            EndDialog (hwnd, 0);
            return TRUE;

        case IDD_PATHNAME:
            if (HIWORD (wParam) == EN_CHANGE) {
                int nChars = GetDlgItemText (hwnd, IDD_PATHNAME, szDir, 8);
                EnableWindow (GetDlgItem (hwnd, IDOK), nChars);
                return TRUE;
            }
            break;
        }
        break;
    }
    return FALSE;
}

BOOL CALLBACK AboutProc (HWND hwnd, UINT uMessage, WPARAM wParam,
    LPARAM lParam)
{
    switch (uMessage) {

    case WM_COMMAND:
        UINT nID = LOWORD (wParam);

        switch (nID) {

        case IDOK:
            EndDialog (hwnd, 1);
            return TRUE;

        case IDCANCEL:
            EndDialog (hwnd, 0);
            return TRUE;
        }
        break;
    }
    return FALSE;
}

/////////////////////////////////////////////////////////////////////////////
// Helper functions

BOOL IsPathValid (LPSTR pszPath)
{
    char szTemp[MAX_PATH];
    UINT nReturn = GetTempFileName (pszPath, "TMP", 0, szTemp);
    if (nReturn)
        DeleteFile (szTemp);
    return nReturn;
}

BOOL IsAnyItemSelected (HWND hwndListView)
{
    int nItems = ListView_GetItemCount (hwndListView);
    if (nItems > 0) {
        for (int i=0; i<nItems; i++) {
            if (ListView_GetItemState (hwndListView, i, LVIS_SELECTED) &
                LVIS_SELECTED)
                return TRUE;
        }
    }
    return FALSE;
}

void ExtractFiles (HWND hwnd, HWND hwndListView, LPSTR pszZipFile)
{
    //
    // Make sure we have a path to PKUNZIP and a prompt setting.
    //
    BOOL bPrompt;
    char szCmdLine[(MAX_PATH * 3) + 16];

    GetPathToPkunzip (szCmdLine, sizeof (szCmdLine), &bPrompt,
        sizeof (bPrompt));

    if (szCmdLine[0] == 0) {
        if (!DoPathSetup (hwnd))
            return;
        GetPathToPkunzip (szCmdLine, sizeof (szCmdLine), &bPrompt,
            sizeof (bPrompt));
    }

    //
    // If bPrompt is TRUE, get a path name from the user.
    //
    lstrcpy (g_szDir, pszZipFile);
    int i = lstrlen (g_szDir);
    while (g_szDir[i--] != '\\');
    g_szDir[++i] = 0;

    int nLen = i;
    BOOL bAddTrailingSlash = TRUE;

    for (i=0; i<nLen; i++) {
        if ((g_szDir[i] == '\\') && (g_szDir[i + 1] != '\\')) {
            bAddTrailingSlash = FALSE;
            break;
        }
    }

    if (bAddTrailingSlash)
        lstrcat (g_szDir, "\\");

    if (bPrompt) {
        GetExtractPath ();
        if (!DialogBox ((HINSTANCE) GetWindowLong (hwnd, GWL_HINSTANCE),
            "GetPath", hwnd, (DLGPROC) GetPathProc))
                return;
    }

    //
    // Create a temporary file containing the names of the files to extract.
    //
    char szTempFile[MAX_PATH];
    if (GetTempFileName (g_szDir, "LST", 0, szTempFile))
        DeleteFile (szTempFile);
    else {
        MessageBox (hwnd, "Extraction failed, possibly because you're " \
        "trying to extract files to a read-only directory on a CD-ROM " \
        "or a similar device", "Error", MB_ICONSTOP | MB_OK);
        return;
    }

    if (!CreateListFile (hwndListView, szTempFile))
        return;

    //
    // Execute PKUNZIP with a -e switch and a list file.
    //
    lstrcat (szCmdLine, " -e ");
    lstrcat (szCmdLine, pszZipFile);
    lstrcat (szCmdLine, " @");
    lstrcat (szCmdLine, szTempFile);

    STARTUPINFO si;
    ZeroMemory (&si, sizeof (STARTUPINFO));
    si.cb = sizeof (STARTUPINFO);
    PROCESS_INFORMATION pi;

    if (CreateProcess (NULL, szCmdLine, NULL, NULL, FALSE, 0,
        NULL, g_szDir, &si, &pi))
        WaitForSingleObject (pi.hProcess, INFINITE);
    else
        MessageBox (hwnd, "The attempt to launch PKUNZIP failed, possibly " \
            "because the path name provided in the Options dialog box is " \
            "no longer valid. After closing this message box, please click " \
            "the Options button and enter an updated path to PKUNZIP.",
            "Error", MB_ICONSTOP | MB_OK);

    //
    // Clean up and exit.
    //
    DeleteFile (szTempFile);
    return;
}

int CreateListFile (HWND hwndListView, LPSTR pszTempFile)
{
    char szFile[MAX_PATH];

    HANDLE hFile = CreateFile (pszTempFile, GENERIC_WRITE, 0, NULL,
        CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);

    if (hFile == INVALID_HANDLE_VALUE)
        return 0;

    int nSel = 0;
    int nItems = ListView_GetItemCount (hwndListView);

    for (int i=0; i<nItems; i++) {
        if (ListView_GetItemState (hwndListView, i, LVIS_SELECTED) &
            LVIS_SELECTED) {
            ListView_GetItemText (hwndListView, i, 0, szFile, sizeof (szFile));
            WriteListFileEntry (hFile, szFile);
            nSel++;
        }
    }
    CloseHandle (hFile);
    return nSel;
}

void WriteListFileEntry (HANDLE hFile, LPSTR pszFile)
{
    static char crlf[2] = { 0x0D, 0x0A };

    DWORD dwBytesWritten;
    WriteFile (hFile, pszFile, lstrlen (pszFile), &dwBytesWritten, NULL);
    WriteFile (hFile, crlf, 2, &dwBytesWritten, NULL);
}

int CALLBACK CompareFunc (LPARAM lParam1, LPARAM lParam2, LPARAM lParamSort)
{
    static char szUnknown[] = "Unknown";

    int iSubItem = (int) lParamSort;
    PZIPFILEINFO pzfi1 = (PZIPFILEINFO) lParam1;
    PZIPFILEINFO pzfi2 = (PZIPFILEINFO) lParam2;
    int nReturn = 0;

    switch (iSubItem) {

    case 0: // File name
        nReturn = lstrcmpi (pzfi1->zfiFileName, pzfi2->zfiFileName);
        break;

    case 1: // Date
        nReturn = (int) pzfi1->zfiDate - (int) pzfi2->zfiDate;
        if (nReturn != 0)
            break; // Fall through if dates are equal

    case 2: // Time
        nReturn = (int) pzfi1->zfiTime - (int) pzfi2->zfiTime;
        break;

    case 3: // Compressed Size
        if (pzfi1->zfiSizeCompressed < pzfi2->zfiSizeCompressed)
            nReturn = -1;
        else if (pzfi1->zfiSizeCompressed > pzfi2->zfiSizeCompressed)
            nReturn = 1;
        break;

    case 4: // Uncompressed size
        if (pzfi1->zfiSizeUncompressed < pzfi2->zfiSizeUncompressed)
            nReturn = -1;
        else if (pzfi1->zfiSizeUncompressed > pzfi2->zfiSizeUncompressed)
            nReturn = 1;
        break;

    case 5: // Method
        LPSTR pszMethod1, pszMethod2;
        pszMethod1 = (pzfi1->zfiMethod <= 8) ?
            g_pszMethods[pzfi1->zfiMethod] : szUnknown;
        pszMethod2 = (pzfi2->zfiMethod <= 8) ?
            g_pszMethods[pzfi2->zfiMethod] : szUnknown;
        nReturn = lstrcmpi (pszMethod1, pszMethod2);
        break;

    case 6: // CRC-32
        if (pzfi1->zfiCrc < pzfi2->zfiCrc)
            nReturn = -1;
        else if (pzfi1->zfiCrc > pzfi2->zfiCrc)
            nReturn = 1;
        break;
    }
    return nReturn;
}

int GetZipFileInfo (LPSTR pszFile, PZIPFILEINFO* ppzfi)
{
    //
    // Open the zip file and map a view of it.
    //
    HANDLE hFile;
    if ((hFile = CreateFile (pszFile, GENERIC_READ, FILE_SHARE_READ, NULL,
        OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0)) == INVALID_HANDLE_VALUE)
        return 0;

    HANDLE hFileMapping;
    if ((hFileMapping = CreateFileMapping (hFile,
        NULL, PAGE_READONLY, 0, 0, NULL)) == NULL) {
        CloseHandle (hFile);
        return 0;
    }

    LPVOID pBase;
    if ((pBase = MapViewOfFile (hFileMapping,
        FILE_MAP_READ, 0, 0, 0)) == NULL) {
        CloseHandle (hFileMapping);
        CloseHandle (hFile);
        return 0;
    }

    //
    // Find the zip file's central directory and count the number of
    // files along the way.
    //
    int nCount = 0;
    PLOCALFILEHEADER plfh = (PLOCALFILEHEADER) pBase;

    int nOffset;
    while (plfh->lfhSignature == 0x04034B50) {
        nOffset = sizeof (LOCALFILEHEADER) + plfh->lfhFileNameLength +
            plfh->lfhExtraLength + plfh->lfhSizeCompressed +
            ((plfh->lfhFlag & 0x08) ? 12 : 0);
        plfh = (PLOCALFILEHEADER) ((PBYTE) plfh + nOffset);
        nCount++;
    }

    //
    // Fill an array of ZIPFILEINFO structures with information about
    // the files stored in the archive.
    //
    *ppzfi = NULL;
    PFILEHEADER pfh = (PFILEHEADER) plfh;

    if (nCount > 0) {
        PZIPFILEINFO pzfi = new ZIPFILEINFO[nCount];
        if (pzfi != NULL) {
            *ppzfi = pzfi;
            int i = 0;
            while ((pfh->fhSignature == 0x02014B50) && (i < nCount)) {
                lstrcpyn (pzfi[i].zfiFileName,
                (LPCTSTR) ((PBYTE) pfh + sizeof (FILEHEADER)),
                pfh->fhFileNameLength + 1);

                pzfi[i].zfiDate = pfh->fhDate;
                pzfi[i].zfiTime = pfh->fhTime;
                pzfi[i].zfiSizeCompressed = pfh->fhSizeCompressed;
                pzfi[i].zfiSizeUncompressed = pfh->fhSizeUncompressed;
                pzfi[i].zfiMethod = pfh->fhMethod;
                pzfi[i].zfiCrc = pfh->fhCrc;

                nOffset = sizeof (FILEHEADER) + pfh->fhFileNameLength +
                    pfh->fhExtraLength + pfh->fhCommentLength;
                pfh = (PFILEHEADER) ((PBYTE) pfh + nOffset);
                i++;
            }
        }
    }

    //
    // Clean up and exit.
    //
    UnmapViewOfFile (pBase);
    CloseHandle (hFileMapping);
    CloseHandle (hFile);
    return nCount;
}

void InitListView (HWND hwndListView, int nFiles, PZIPFILEINFO pzfi)
{
    static char* pszLabels[] = {
        "File Name",
        "Date",
        "Time",
        "Size",
        "Length",
        "Method",
        "CRC-32"        
    };

    //
    // Read the column widths from the registry and create the columns.
    //
    int nWidths[7];
    GetColumnWidths (nWidths);

    LV_COLUMN lvc;
    for (int i=0; i<7; i++) {
        lvc.mask = LVCF_FMT | LVCF_WIDTH | LVCF_TEXT;
        lvc.fmt = LVCFMT_LEFT;
        lvc.cx = nWidths[i];
        lvc.pszText = pszLabels[i];
        ListView_InsertColumn (hwndListView, i, &lvc);
    }

    //
    // Add nFiles items to the ListView control.
    //
    if (nFiles > 0) {
        LV_ITEM lvi;
        for (i=0; i<nFiles; i++) {
            lvi.mask = LVIF_TEXT | LVIF_PARAM;
            lvi.iItem = i;
            lvi.iSubItem = 0;
            lvi.pszText = LPSTR_TEXTCALLBACK;
            lvi.lParam = (LPARAM) &pzfi[i];
            ListView_InsertItem (hwndListView, &lvi);
        }
    }
}

void GetDispInfo (LV_DISPINFO* plvdi)
{
    if (plvdi->item.mask & LVIF_TEXT) {
        PZIPFILEINFO pzfi = (PZIPFILEINFO) plvdi->item.lParam;

        switch (plvdi->item.iSubItem) {

        case 0: // File name
            lstrcpy (plvdi->item.pszText, pzfi->zfiFileName);
            break;

        case 1: // Date
            ToDate (pzfi->zfiDate, plvdi->item.pszText);
            break;

        case 2: // Time
            ToTime (pzfi->zfiTime, plvdi->item.pszText);
            break;

        case 3: // Compressed size
            ToAnsi (pzfi->zfiSizeCompressed, plvdi->item.pszText);
            break;

        case 4: // Uncompressed size
            ToAnsi (pzfi->zfiSizeUncompressed, plvdi->item.pszText);
            break;

        case 5: // Method
            lstrcpy (plvdi->item.pszText, (pzfi->zfiMethod <= 8) ?
                g_pszMethods[pzfi->zfiMethod] : "Unknown");
            break;

        case 6: // CRC-32
            wsprintf (plvdi->item.pszText, "%0.8lx", pzfi->zfiCrc);
            break;
        }
    }
}

void ToAnsi (DWORD dwVal, char* pDest)
{
    if (dwVal == 0) {   // Special case (dwVal == 0)
        pDest[0] = 0x30;
        pDest[1] = 0;
        return;
    }

    int nCount = 0;
    char szTemp[32];
    szTemp[31] = 0;

    while (dwVal > 0) {
        if (((nCount + 1) % 4) == 0)
            szTemp[30 - nCount++] = 0x2C;
        szTemp[30 - nCount++] = (char) (dwVal % 10) + 0x30;
        dwVal /= 10;
    }

    for (int i=0; i<=nCount; i++)
        pDest[i] = szTemp[31 - nCount + i];
}

void ToDate (WORD wDate, char* pDest)
{
    int nDay = (int) (wDate & 0x001F);
    int nMonth = (int) (wDate & 0x01E0) >> 5;
    int nYear = ((int) (wDate & 0xFE00) >> 9) + 80;
    wsprintf (pDest, "%0.2d-%0.2d-%0.2d", nMonth, nDay, nYear);
}

void ToTime (WORD wTime, char* pDest)
{
    int nMinutes = (int) (wTime & 0x07E0) >> 5;
    int nHours = (int) (wTime & 0xF800) >> 11;

    BOOL pm = FALSE;
    if (nHours == 0)
        nHours = 12;
    else if (nHours == 12)
        pm = TRUE;
    else if (nHours > 12) {
        nHours -= 12;
        pm = TRUE;
    }

    wsprintf (pDest, "%d:%0.2d", nHours, nMinutes);
    lstrcat (pDest, pm ? "p" : "a");
}

void SaveColumnWidths (HWND hwndListView)
{
    HKEY hKey;
    DWORD dwWidths[7], i;
    char szValueName[16];

    for (i=0; i<7; i++)
        dwWidths[i] = (DWORD) ListView_GetColumnWidth (hwndListView, i);

    if (RegCreateKeyEx (HKEY_CURRENT_USER, g_szSubkey, 0, NULL,
        REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL, &hKey,
        NULL) != ERROR_SUCCESS)
        return;

    for (i=0; i<7; i++) {
        wsprintf (szValueName, "Col%dWidth", i + 1);    
        RegSetValueEx (hKey, szValueName, 0, REG_DWORD,
            (PBYTE) &dwWidths[i], 4);
    }
    RegCloseKey (hKey);
}

void GetColumnWidths (int* nWidths)
{
    HKEY hKey;
    char szValueName[16];
    DWORD dwSize;

    nWidths[0] = 128;
    for (int i=1; i<7; i++)
        nWidths[i] = 64;

    if (RegOpenKeyEx (HKEY_CURRENT_USER, g_szSubkey, 0, KEY_QUERY_VALUE,
        &hKey) != ERROR_SUCCESS)
        return;

    for (i=0; i<7; i++) {
        dwSize = 4;
        wsprintf (szValueName, "Col%dWidth", i + 1);    
        RegQueryValueEx (hKey, szValueName, NULL, NULL,
            (PBYTE) &nWidths[i], &dwSize);
    }
    RegCloseKey (hKey);
}

int DoPathSetup (HWND hwnd)
{
    return DialogBox ((HINSTANCE) GetWindowLong (hwnd, GWL_HINSTANCE),
        "Options", hwnd, (DLGPROC) OptionsProc);
}

void GetPathToPkunzip (LPSTR pszPathName, UINT nMaxStr, BOOL* pbPrompt,
    UINT nMaxBool)
{
    *pbPrompt = FALSE;
    pszPathName[0] = 0;

    HKEY hKey;
    if (RegOpenKeyEx (HKEY_CURRENT_USER, g_szSubkey, 0, KEY_QUERY_VALUE,
        &hKey) == ERROR_SUCCESS) {
        DWORD dwSize = (DWORD) nMaxStr;
        RegQueryValueEx (hKey, "PathToPkunzip", NULL, NULL,
            (PBYTE) pszPathName, &dwSize);
        dwSize = (DWORD) nMaxBool;
        RegQueryValueEx (hKey, "PromptSetting", NULL, NULL,
            (PBYTE) pbPrompt, &dwSize);
        RegCloseKey (hKey);
    }
}

void WritePathToPkunzip (LPSTR pszPathName, BOOL bPrompt)
{
    HKEY hKey;
    if (RegCreateKeyEx (HKEY_CURRENT_USER, g_szSubkey, 0, NULL,
        REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL, &hKey,
        NULL) == ERROR_SUCCESS) {
        RegSetValueEx (hKey, "PathToPkunzip", 0, REG_SZ, (PBYTE) pszPathName,
            lstrlen (pszPathName));
        RegSetValueEx (hKey, "PromptSetting", 0, REG_BINARY, (PBYTE) &bPrompt,
            sizeof (bPrompt));
        RegCloseKey (hKey);
    }
}

void GetExtractPath ()
{
    HKEY hKey;
    if (RegOpenKeyEx (HKEY_CURRENT_USER, g_szSubkey, 0, KEY_QUERY_VALUE,
        &hKey) == ERROR_SUCCESS) {
        DWORD dwSize = sizeof (g_szDir);
        RegQueryValueEx (hKey, "ExtractPath", NULL, NULL,
            (PBYTE) g_szDir, &dwSize);
        RegCloseKey (hKey);
    }
}

void WriteExtractPath ()
{
    HKEY hKey;
    if (RegCreateKeyEx (HKEY_CURRENT_USER, g_szSubkey, 0, NULL,
        REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL, &hKey,
        NULL) == ERROR_SUCCESS) {
        RegSetValueEx (hKey, "ExtractPath", 0, REG_SZ, (PBYTE) g_szDir,
            lstrlen (g_szDir));
        RegCloseKey (hKey);
    }
}
