/****************************************************************************

    Module  : DockUtil.c

    Version : v1.0alpha

    Date    : 20/8/93

    Changes : 20/8/93 - Added use of RePaintSlot function to selectively
                        repaint a single slot instead of using invalidate
                        clientrect to paint whole window when only a single
                        slot changed.
              24/6/94 - rewrote code to position dock & abstracted code into
                        function SetDockPos()

*****************************************************************************/


#include <windows.h>
#include <commdlg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <direct.h>
#include <ctype.h>
#include <shellapi.h>
#include "freedock.h"

/*********************************************************
	Static (local) slot structure used in window enum
	function.
*********************************************************/
static SLOT_ENTRY TmpSlot;

/**********************************************************
	Global Data for the various system metrics we need
**********************************************************/
extern GLOBAL_METRICS gMetrics;
extern GLOBAL_OPTIONS gOptions;

/**********************************************************
	Name of the .INI file, local to this module
	Different name depending on OS
**********************************************************/
#ifdef WIN32
	static char           *IniFile = "FreeDk32.ini";
#else
	static char           *IniFile = "FreeDk16.ini";
#endif


/****************************************************************
	ReadGlobalOptions()
	This function reads in all the options which have a 
	global effect on FreeDock and all the Slots/sub-docks.
	These global options are read into the Global Options
	data structure gOptions. Any read errors are ignored
	and sensible defaults assumed.
****************************************************************/
void ReadGlobalOptions( void ){

    char     TmpBuffer[MAX_FPATH_LEN];

	/*********************************
		Initialise Mail box Flag
	*********************************/                           
    gOptions.bMailInBox = FALSE;          
    
	 // Read in and setup the slot scale.
     gOptions.SlotScale   = GetPrivateProfileInt("Main Options", "SlotScale", DEF_SLOT_SCALE, IniFile);
 	 gOptions.SlotHeight  = gOptions.SlotWidth = gOptions.SlotScale;
     gOptions.MaxDockSize = GetPrivateProfileInt("Main Options", "MaxDockSize", DEF_MAX_SLOTCOUNT, IniFile);
     gOptions.SleepTime   = GetPrivateProfileInt("Main Options", "SleepTime", 1, IniFile);

	// Handle Win Exit Options
    GetPrivateProfileString("Main Options", "WinExitActive", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y') {
        gOptions.WinExitActive = TRUE;
        GetPrivateProfileString("Main Options", "WinExitConfirm", "Yes", TmpBuffer, MAX_FPATH_LEN, IniFile);
        if (toupper(TmpBuffer[0]) == 'N')
            gOptions.WinExitConfirm = FALSE;
        else
            gOptions.WinExitConfirm = TRUE;
    } 
    else
        gOptions.WinExitActive = FALSE;

	// Handle Clock Options
    GetPrivateProfileString("Main Options", "ClockActive", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y') {
        gOptions.ClockActive = TRUE;
    } 
    else
        gOptions.ClockActive = FALSE;

	// Handle Mail Options
    GetPrivateProfileString("Main Options", "MailActive", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y') {
        gOptions.MailActive = TRUE;
        GetPrivateProfileString("Main Options", "MailSound", "Yes", TmpBuffer, MAX_FPATH_LEN, IniFile);
        if (toupper(TmpBuffer[0]) == 'N')
            gOptions.MailSound = FALSE;
        else
            gOptions.MailSound = TRUE;

        GetPrivateProfileString("Main Options", "MailPath", "c:\\mail.box", gOptions.MailPath, MAX_FPATH_LEN, IniFile);
        /* GetPrivateProfileString Cannot handle long, so do it manually */
        GetPrivateProfileString("Main Options", "MailBoxSize", "0", TmpBuffer, MAX_FPATH_LEN, IniFile);
        sscanf(TmpBuffer, "%lu", &gOptions.MailBoxSize);

        GetPrivateProfileString("Main Options", "MailBoxDateTime", "0", TmpBuffer, MAX_FPATH_LEN, IniFile);
        sscanf(TmpBuffer, "%lu", &gOptions.MailBoxDateTime);

        gOptions.MailFreq = GetPrivateProfileInt("Main Options", "MailFreq", 10, IniFile);
    } 
    else
        gOptions.MailActive = FALSE;
	
	// Handle Hide Dialog On Grab Option
    GetPrivateProfileString("Main Options", "HideDlgOnGrab", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y')
        gOptions.HideDlgOnGrab = TRUE;
    else
        gOptions.HideDlgOnGrab = FALSE;

	// Handle Always on top option
    GetPrivateProfileString("Main Options", "AlwaysOnTop", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y')
        gOptions.AlwaysOnTop = TRUE;
    else
        gOptions.AlwaysOnTop = FALSE;

	// Handle Single Click Start Option
    GetPrivateProfileString("Main Options", "SingleClickStart", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y')
        gOptions.SingleClickStart = TRUE;
    else
        gOptions.SingleClickStart = FALSE;

	// Handle Max View Option
    GetPrivateProfileString("Main Options", "MaxView", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y')
        gOptions.MaxView = TRUE;
    else
        gOptions.MaxView = FALSE;

	// Handle DockLocked Option
    GetPrivateProfileString("Main Options", "DockLocked", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y')
        gOptions.DockLocked = TRUE;
    else
        gOptions.DockLocked = FALSE;
}


void ReadDockOptions( DOCK_STRUCT *Dock ){

    char     TmpBuffer[MAX_FPATH_LEN];

    /**********************************************************************
        Setup the dock options, only the root dock has a position &
		orientation, sub-docks derive this from their parent.
    ***********************************************************************/
	if(Dock->DockType == ROOT_DOCK){
	    GetPrivateProfileString("Main Options", "Position", "Horizontal", TmpBuffer, MAX_FPATH_LEN, IniFile);
	    if (toupper(TmpBuffer[0]) == 'H') {
    	    Dock->Orientation = DOCK_HORZ;
	    } 
    	else
	        Dock->Orientation = DOCK_VERT;

	    Dock->DockLeft = GetPrivateProfileInt("Main Options", "DockLeft", 0, IniFile);
    	Dock->DockTop = GetPrivateProfileInt("Main Options", "DockTop", 0, IniFile);
	}

    Dock->SlotCount = GetPrivateProfileInt("Main Options", "SlotCount", DEFAULT_DOCK_SIZE, IniFile);
}

/****************************************************************
   Read in and process all slot data, also carry out any setup
   which is required.
****************************************************************/
 
void ReadAllSlotOptions(DOCK_STRUCT *Dock, SLOT_ENTRY * Slot)
{
    char            SlotSection[MAX_FPATH_LEN], TmpBuffer[MAX_FPATH_LEN];
    int             StartSlot, SlotNum;
    int             i;
                          

    /**********************************************************************
       Set the index to the first free slot to the slot after the title
       slot	and any actice special slots.
    ***********************************************************************/
    StartSlot = 0;

	// Only the root Dock can have special built-in slots.
	if(Dock->DockType == ROOT_DOCK){
	    /**********************************************************************
    	       Start by initialising the first slot to be the title icon
	    ***********************************************************************/
    	Slot[StartSlot].SlotType = SLOT_SPECIAL_TITLE;
        Slot[StartSlot].ID = StartSlot;
	    UtilLoadIcon( Dock, &Slot[StartSlot] );
		StartSlot ++;

		if(gOptions.WinExitActive){
    	    Slot[StartSlot].SlotType = SLOT_SPECIAL_EXIT;
	        Slot[StartSlot].ID = StartSlot;
        	UtilLoadIcon(Dock, &Slot[StartSlot] );
			StartSlot ++;
		}
		if(gOptions.MailActive){
    	    Slot[StartSlot].SlotType = SLOT_SPECIAL_MAIL;
	        Slot[StartSlot].ID = StartSlot;
	       	UtilLoadIcon(Dock, &Slot[StartSlot] );
			StartSlot ++;
		}
		if(gOptions.ClockActive){
        	Slot[StartSlot].SlotType = SLOT_SPECIAL_CLOCK;
	        Slot[StartSlot].ID = StartSlot;
	        UtilLoadIcon(Dock, &Slot[StartSlot] );
			StartSlot ++;
		}
	}

    /*******************************************************************************
       SlotNum is used to index the sections in the .INI file, each slot is
       uniquely numbered in a section of the type [Slot xx] where xx is the slot
       number.
       The for loop attempts to read the AppName field from each slot, if this
       fails the slot is empty so the slot type is set to SLOT_FREE and the next
       slot is read. If the reading of AppName succeeds, the slot is used and
       the rest of the details are read in.
    *******************************************************************************/
    SlotNum = 1;

    for (i = StartSlot; i < gOptions.MaxDockSize; i++) {

       	UtilEmptySlot( &Slot[i] );	// Ensure Default Empty struct is in Slot 

        Slot[i].ID = i;

        sprintf(SlotSection, "Slot %d", SlotNum++);

        GetPrivateProfileString(SlotSection, "AppName", "", Slot[i].AppName, MAX_FPATH_LEN, IniFile);
        if (Slot[i].AppName[0] != '\0') {
            GetPrivateProfileString(SlotSection, "CmdLine", "", Slot[i].CmdLine, MAX_CMDLINE_LEN, IniFile);
            GetPrivateProfileString(SlotSection, "RunTimeDir", "", Slot[i].RunTimeDir, MAX_FPATH_LEN, IniFile);

            Slot[i].StartState = GetPrivateProfileInt(SlotSection, "StartState", START_NORMAL, IniFile);

            GetPrivateProfileString(SlotSection, "StartOnTop", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
            if (toupper(TmpBuffer[0]) == 'N')
                Slot[i].StartOnTop = FALSE;
            else
                Slot[i].StartOnTop = TRUE;

            GetPrivateProfileString(SlotSection, "StartSeparate", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
            if (toupper(TmpBuffer[0]) == 'N')
                Slot[i].StartSeparate = FALSE;
            else
                Slot[i].StartSeparate = TRUE;

            Slot[i].WinX = GetPrivateProfileInt(SlotSection, "WinX", 0, IniFile);
            Slot[i].WinY = GetPrivateProfileInt(SlotSection, "WinY", 0, IniFile);
            Slot[i].WinWidth = GetPrivateProfileInt(SlotSection, "WinWidth", DEF_STORE_W, IniFile);
            Slot[i].WinHeight = GetPrivateProfileInt(SlotSection, "WinHeight", DEF_STORE_H, IniFile);

            GetPrivateProfileString(SlotSection, "IconFile", "", Slot[i].IconFile, MAX_FPATH_LEN, IniFile);

            Slot[i].IconPos = GetPrivateProfileInt(SlotSection, "IconPos", 0, IniFile);

            // Load the Icon into the slot's own bitmap & update the IconTotal Field
            UtilLoadIcon( Dock, &Slot[i] );
      
            Slot[i].SlotType = SLOT_USED;
        } 
        else
            Slot[i].SlotType = SLOT_FREE;
    }
}

void WriteGlobalOptions( void ){

    char     TmpBuffer[MAX_FPATH_LEN];

    /* Flush any cached .INI file to disk */
//    WritePrivateProfileString(NULL, NULL, NULL, IniFile);

    WritePrivateProfileString("Main Options", "MaxView", (gOptions.MaxView) ? "Yes" : "No", IniFile);
    WritePrivateProfileString("Main Options", "AlwaysOnTop", (gOptions.AlwaysOnTop) ? "Yes" : "No", IniFile);
    WritePrivateProfileString("Main Options", "SingleClickStart", (gOptions.SingleClickStart) ? "Yes" : "No", IniFile);
    WritePrivateProfileString("Main Options", "HideDlgOnGrab", (gOptions.HideDlgOnGrab) ? "Yes" : "No", IniFile);

    WritePrivateProfileString("Main Options", "ClockActive", (gOptions.ClockActive) ? "Yes" : "No", IniFile);

    WritePrivateProfileString("Main Options", "MailActive", (gOptions.MailActive) ? "Yes" : "No", IniFile);
    WritePrivateProfileString("Main Options", "MailPath", gOptions.MailPath, IniFile);
    WritePrivateProfileString("Main Options", "MailSound", (gOptions.MailSound) ? "Yes" : "No", IniFile);
    sprintf(TmpBuffer, "%d", gOptions.MailFreq);
    WritePrivateProfileString("Main Options", "MailFreq", TmpBuffer, IniFile);

    WritePrivateProfileString("Main Options", "WinExitActive", (gOptions.WinExitActive) ? "Yes" : "No", IniFile);
    WritePrivateProfileString("Main Options", "WinExitConfirm", (gOptions.WinExitConfirm) ? "Yes" : "No", IniFile);

    WritePrivateProfileString("Main Options", "DockLocked", (gOptions.DockLocked) ? "Yes" : "No", IniFile);

    sprintf(TmpBuffer, "%lu", gOptions.MailBoxDateTime);
    WritePrivateProfileString("Main Options", "MailBoxDateTime", TmpBuffer, IniFile);

    sprintf(TmpBuffer, "%lu", gOptions.MailBoxSize);
    WritePrivateProfileString("Main Options", "MailBoxSize", TmpBuffer, IniFile);

    sprintf(TmpBuffer, "%d", gOptions.SlotScale);
    WritePrivateProfileString("Main Options", "SlotScale", TmpBuffer, IniFile);

}

void WriteDockOptions ( DOCK_STRUCT *Dock ){

    char            TmpBuffer[MAX_FPATH_LEN];
    char            PosStr[2][11] =  { {"Horizontal"}, {"Vertical"}};

    /* Flush any cached .INI file to disk */
//    WritePrivateProfileString(NULL, NULL, NULL, IniFile);

    WritePrivateProfileString("Main Options", "Position", PosStr[Dock->Orientation], IniFile);
    sprintf(TmpBuffer, "%d", Dock->SlotCount);
    WritePrivateProfileString("Main Options", "SlotCount", TmpBuffer, IniFile);
    sprintf(TmpBuffer, "%d", Dock->DockLeft);
    WritePrivateProfileString("Main Options", "DockLeft", TmpBuffer, IniFile);
    sprintf(TmpBuffer, "%d", Dock->DockTop);
    WritePrivateProfileString("Main Options", "DockTop", TmpBuffer, IniFile);
}


/* Updates a slot entry within an INI file */
void WriteSlotOptions(DOCK_STRUCT *Dock, SLOT_ENTRY * Slot)
{
    char    TmpBuffer[MAX_FPATH_LEN], SlotName[MAX_FPATH_LEN];
	int 	SlotOff;

	// Calculate offset for correct slot number

	SlotOff = ((Dock->Slot[1].SlotType!=SLOT_USED) && (Dock->Slot[1].SlotType!=SLOT_FREE))?1:0 + 
			  ((Dock->Slot[2].SlotType!=SLOT_USED) && (Dock->Slot[2].SlotType!=SLOT_FREE))?1:0 + 
			  ((Dock->Slot[3].SlotType!=SLOT_USED) && (Dock->Slot[3].SlotType!=SLOT_FREE))?1:0 ;

    /* Flush any cached .INI file to disk */
//    WritePrivateProfileString(NULL, NULL, NULL, IniFile);
    sprintf(SlotName, "Slot %d", (Slot->ID) - SlotOff);

        if (Slot->SlotType == SLOT_FREE){
            WritePrivateProfileString(SlotName, NULL, NULL, IniFile);
        }
        if (Slot->SlotType == SLOT_USED) {
//            WritePrivateProfileString(SlotName, NULL, NULL, IniFile);
            WritePrivateProfileString(SlotName, "AppName", Slot->AppName, IniFile);
            WritePrivateProfileString(SlotName, "CmdLine", Slot->CmdLine, IniFile);
            WritePrivateProfileString(SlotName, "RunTimeDir", Slot->RunTimeDir, IniFile);

            sprintf(TmpBuffer, "%d", Slot->StartState);
            WritePrivateProfileString(SlotName, "StartState", TmpBuffer, IniFile);
            WritePrivateProfileString(SlotName, "StartOnTop", (Slot->StartOnTop) ? "Yes" : "No", IniFile);
            WritePrivateProfileString(SlotName, "StartSeparate", (Slot->StartSeparate) ? "Yes" : "No", IniFile);
            sprintf(TmpBuffer, "%d", Slot->WinX);
            WritePrivateProfileString(SlotName, "WinX", TmpBuffer, IniFile);
            sprintf(TmpBuffer, "%d", Slot->WinY);
            WritePrivateProfileString(SlotName, "WinY", TmpBuffer, IniFile);
            sprintf(TmpBuffer, "%d", Slot->WinWidth);
            WritePrivateProfileString(SlotName, "WinWidth", TmpBuffer, IniFile);
            sprintf(TmpBuffer, "%d", Slot->WinHeight);
            WritePrivateProfileString(SlotName, "WinHeight", TmpBuffer, IniFile);

            WritePrivateProfileString(SlotName, "IconFile", Slot->IconFile, IniFile);

            sprintf(TmpBuffer, "%d", Slot->IconPos);
            WritePrivateProfileString(SlotName, "IconPos", TmpBuffer, IniFile);
        }
}

int FindSlotHit(DOCK_POS Orientation, int XHit, int YHit)
{
    if (Orientation == DOCK_HORZ) {
        return (XHit / gOptions.SlotWidth);
    } 
    else {
        return (YHit / gOptions.SlotHeight);
    }


}

BOOL IsCursorOutOfDock(DOCK_STRUCT *Dock, int X, int Y)
{
    if (Dock->Orientation == DOCK_HORZ) {
        if( (X < (0-(gOptions.SlotWidth/2))) ||
            (Y < (0-(gOptions.SlotHeight/2))) ||
            (Y > (gOptions.SlotHeight+(gOptions.SlotHeight/2))) ||
            (X > ((Dock->SlotCount * gOptions.SlotWidth)+(gOptions.SlotWidth/2))) ){
            return TRUE;
        }
    } 
    else {
        if( (X < (0-(gOptions.SlotHeight/2))) ||
            (Y < (0-(gOptions.SlotWidth/2))) ||
            (X > (gOptions.SlotWidth+(gOptions.SlotWidth/2)) ) ||
            (Y > ((Dock->SlotCount * gOptions.SlotHeight)+(gOptions.SlotHeight/2))) ){
            return TRUE;
        }
    }

    return( FALSE );
}


BOOL FAR PASCAL	AppOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
	static SLOT_ENTRY TmpSlot;		// Temporary work space for slot structure
	static SLOT_ENTRY *Slot;		// Pointer to actual Slot
	static DOCK_STRUCT  *Dock;		// Pointer to dock containing above slot.

    static BOOL     Grabbing, bWasEmpty;
    static int      NumIcons;
    RECT            rect;
    HWND            hwndGrab, hwndTmp;
    POINT           pt;
    char            TmpBuffer[MAX_FPATH_LEN], *TmpCharPtr;
    int             Status;
    short int       TmpX, TmpY;
    static HCURSOR  hGrabCur;
    static FARPROC  lpfnChooseIconDlgProc, lpfnPreviewerDlgProc;
    DRAWITEMSTRUCT *ItemStruct;
           RECT     rcItem;
           HWND     hwndItem;
           HDC      hdcGrab, hdcTmp;
    static HBITMAP  hbmGrab, hbmGrabbing;

    switch (iMessage) {
        case WM_INITDIALOG:

			Slot = (SLOT_ENTRY *)lParam; // Retrieve pointer to slot from parameter
			Dock = Slot->Dock;			 // Retrieve pointer to parent dock

            SendDlgItemMessage(hDlg, QX_FILE, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
            SendDlgItemMessage(hDlg, QX_CMDLINE, EM_LIMITTEXT, MAX_CMDLINE_LEN, 0L);
            SendDlgItemMessage(hDlg, QX_RUNTIMEDIR, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
            SendDlgItemMessage(hDlg, QX_ICONFILE, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);

            lpfnChooseIconDlgProc = MakeProcInstance((FARPROC) ChooseIconDlgProc, gOptions.hAppInst);
            lpfnPreviewerDlgProc  = MakeProcInstance((FARPROC) PreviewerDlgProc,  gOptions.hAppInst);

			// Copy Selected Slot into Workspace
            memcpy(&TmpSlot, Slot, sizeof(SLOT_ENTRY));

            // Initialise TmpSlots Scratch Space
            TmpSlot.IconIndex = (gMetrics.IconWidth * (gOptions.MaxDockSize));
            // Load the current Iocn into the Scratch Space
            UtilLoadIcon( Dock, &TmpSlot );

            CheckDlgButton(hDlg, QX_HIDE_ON_GRAB, gOptions.HideDlgOnGrab);

			// Set the Start in Separate Memory check (NT only)
#ifdef WIN32
                    CheckDlgButton(hDlg, QX_START_SEPARATE, TmpSlot.StartSeparate);
#else
                    CheckDlgButton(hDlg, QX_START_SEPARATE, FALSE);
#endif

            switch (TmpSlot.StartState) {

                case START_MINIMUM:
                    CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_MINIMUM);
                    break;

                case START_MAXIMUM:
                    CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_MAXIMUM);
                    break;

                case START_STORE:
                    CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_STORE);
                    break;

                default:
                    CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_NORMAL);
                    break;
            }

            CheckDlgButton(hDlg, QX_START_TOP, TmpSlot.StartOnTop);

            if ( TmpSlot.SlotType == SLOT_USED ){
                NumIcons = TmpSlot.IconTotal;
            }
            else{
                NumIcons = 0;
            }

            SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
            SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);

            /* Set starting values. */
            SetDlgItemText(hDlg, QX_FILE, TmpSlot.AppName);
            SetDlgItemText(hDlg, QX_CMDLINE, TmpSlot.CmdLine);
            SetDlgItemText(hDlg, QX_RUNTIMEDIR, TmpSlot.RunTimeDir);
            SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);

            Grabbing = FALSE;

			/********************************************************
				There is a special condition when an empty slot is
				left clicked and the App name specified
			********************************************************/
			bWasEmpty = (TmpSlot.AppName[0] == '\0');

            return (TRUE);

        case WM_PAINT:
            /** Paint the Grabber bitmap **/
            hwndTmp = GetDlgItem( hDlg, QX_GRABWIN );
            hdcGrab = GetDC( hwndTmp );
            hbmGrab = LoadBitmap(gOptions.hAppInst, "GRABBER");
            hbmGrabbing = LoadBitmap(gOptions.hAppInst, "GRABBER2");
			if(Grabbing){
            	DrawBitmap(hdcGrab, hbmGrabbing, 0, 0);
			}
			else{
            	DrawBitmap(hdcGrab, hbmGrab, 0, 0);
			}
            ReleaseDC( hwndTmp, hdcGrab );
            DeleteObject( hbmGrab );
            DeleteObject( hbmGrabbing );
            /** return false to force the default message handler to paint **/
            /** the rest of the diaog box **/
            return(FALSE);

        case WM_DRAWITEM :  // Draw the Current Icon
            ItemStruct = (DRAWITEMSTRUCT *)lParam;
            UtilDrawIcon( ItemStruct->hDC, Dock, &TmpSlot, 0, 0, FALSE );
            return (TRUE);

//		case WM_MOUSEMOVE :
//			if (Grabbing){
//			    pt.x = LOWORD(lParam);
//				pt.y = HIWORD(lParam);
//
//            /* Climb back up to the top of a parent-child list */
//            hwndTmp = GetParent(hwndGrab);
//            while (hwndTmp != NULL) {
//                hwndGrab = hwndTmp;
//                hwndTmp = GetParent(hwndGrab);
//            }
//
//  	        GetWindowRect(WindowFromPoint(pt), &rect);
//				hdcTmp = GetWindowDC( WindowFromPoint(pt) );
//				DrawFocusRect( hdcTmp, &rect );
//			}
//			return (TRUE);

		case WM_MOVE :
			GetWindowRect(hDlg, &rect);
			InvalidateRect(hDlg, &rect, FALSE);
			UpdateWindow(hDlg);
			return (TRUE);

        case WM_HSCROLL:
            if (wParam == SB_LINEDOWN)
                if (NumIcons > 0)
                    TmpSlot.IconPos = min(NumIcons - 1, TmpSlot.IconPos + 1);

            if (wParam == SB_LINEUP)
                if (NumIcons > 0)
                    if (TmpSlot.IconPos > 0)
                        TmpSlot.IconPos = TmpSlot.IconPos - 1;

            // Update icon number
            SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
            // Load the Icon into the slot's own bitmap  & update the IconTotal Field
            UtilLoadIcon( Dock, &TmpSlot );
            // ReDraw Icon by invalidating the button area
            hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
            GetClientRect( hwndItem, &rcItem );
            InvalidateRect( hwndItem, &rcItem, TRUE );

            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {

			    case QX_START_SEPARATE:
#ifndef WIN32		// This option is not valid in 16 bit windows
                    CheckDlgButton(hDlg, QX_START_SEPARATE, FALSE);
#endif
					return(TRUE);

				case QX_HIDE_ON_GRAB:
                    gOptions.HideDlgOnGrab = IsDlgButtonChecked(hDlg, QX_HIDE_ON_GRAB);
					return(TRUE);

                case QX_OK:
                    if (SendDlgItemMessage(hDlg, QX_ICONFILE, EM_GETMODIFY, 0, 0L)) {
                        SendDlgItemMessage(hDlg, QX_ICONFILE, EM_SETMODIFY, FALSE, 0L);
                        GetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile, MAX_FPATH_LEN);
                        // Load the Icon into the slot's own bitmap & Initialise IconTotal Field
                        TmpSlot.IconPos = 0;
                        UtilLoadIcon( Dock, &TmpSlot );
                        if (TmpSlot.IconTotal == 0) {
                            MessageBox(hDlg, "This file contains no icons",
                                "FreeDock", MB_OK | MB_ICONSTOP);
                            TmpSlot.IconPos = Slot->IconPos;
                            strcpy(TmpSlot.IconFile, Slot->IconFile);
                            SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
	                        UtilLoadIcon( Dock, &TmpSlot );
                        } 

                        NumIcons = TmpSlot.IconTotal;

                        SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
                        SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
                        // ReDraw Icon by invalidating the button area
                        hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
                        GetClientRect( hwndItem, &rcItem );
                        InvalidateRect( hwndItem, &rcItem, TRUE );
                        return (TRUE);
                    }

                    /*****************************************
                        Try to fully qualify the pathname 
                    *****************************************/
                    GetDlgItemText(hDlg, QX_FILE, TmpBuffer, MAX_FPATH_LEN);

                    Status = (UINT)FindExecutable(TmpBuffer,
                                                  TmpSlot.RunTimeDir,
                                                  TmpSlot.AppName);
                    if( Status <= 31 ){
                            MessageBox(hDlg, "Error, File or Path Not Found",
                                       "FreeDock Error", MB_OK | MB_ICONSTOP);
                            return(TRUE);
                    }
                    else{
                        SetDlgItemText(hDlg, QX_FILE, TmpSlot.AppName);
                    }
                    
                    GetDlgItemText(hDlg, QX_FILE, TmpSlot.AppName, MAX_FPATH_LEN);
                    GetDlgItemText(hDlg, QX_CMDLINE, TmpSlot.CmdLine, MAX_CMDLINE_LEN);
                    GetDlgItemText(hDlg, QX_RUNTIMEDIR, TmpSlot.RunTimeDir, MAX_FPATH_LEN);
                    GetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile, MAX_FPATH_LEN);

					/*****************************************************************
						If this is an empty slot which is being filled out by hand
					*****************************************************************/
					if(bWasEmpty){
                        TmpSlot.SlotType = SLOT_USED;
                        TmpSlot.StartState = START_NORMAL;
                        TmpSlot.WinX = DEF_STORE_X;
                        TmpSlot.WinY = DEF_STORE_Y;
                        TmpSlot.WinWidth = DEF_STORE_W;
                        TmpSlot.WinHeight = DEF_STORE_H;
						if( TmpSlot.RunTimeDir[0] == '\0' ){
	                        strcpy(TmpSlot.RunTimeDir, TmpSlot.AppName);
						}
						if( TmpSlot.IconFile[0] == '\0' ){
	                        strcpy(TmpSlot.IconFile, TmpSlot.AppName);
						}
                        TmpSlot.IconPos = 0;

                        /************************************************
                           Set Run Time Dir to be same path as filename
                        ************************************************/
                        TmpCharPtr = strrchr(TmpSlot.RunTimeDir, '\\');
						if (TmpCharPtr != NULL ){
	                        if (*(TmpCharPtr-1) == ':')
    	                        *(TmpCharPtr+1) = '\0';
        	                else{
            	                *TmpCharPtr = '\0';
							}
						}

                        /*****************************************************************
                           Check What type of file has been dropped, is it a .PIF, .COM
                           or a .BAT ?
                        *****************************************************************/
                        if (!stricmp(&TmpSlot.AppName[strlen(TmpSlot.AppName) - 4], ".PIF") ||
                            !stricmp(&TmpSlot.AppName[strlen(TmpSlot.AppName) - 4], ".COM") ||
                            !stricmp(&TmpSlot.AppName[strlen(TmpSlot.AppName) - 4], ".BAT")) {


		                    Status = (UINT)FindExecutable("progman.exe",
    		                                              NULL,
        		                                          TmpSlot.IconFile);
            		        if( Status <= 31 ){
                		            MessageBox(hDlg, "Program Manager not found, cannot select default Icon.",
                    		                   "FreeDock Error", MB_OK | MB_ICONSTOP);
		                    }
                            TmpSlot.IconPos = 1;
                            // Load the Icon into the slot's own bitmap
                            UtilLoadIcon( Dock, &TmpSlot );
                        }
                        /******************************************************************
                            Is is a .EXE, if so is it a Windows EXE or a DOS EXE ?
                        ******************************************************************/
                        else if (!stricmp(&TmpSlot.AppName[strlen(TmpSlot.AppName) - 4], ".EXE")) {
                            // Load the Icon into the slot's own bitmap
                            UtilLoadIcon( Dock, &TmpSlot );
                            if (TmpSlot.IconTotal == 0) {

			                    Status = (UINT)FindExecutable("progman.exe",
    			                                              NULL,
        			                                          TmpSlot.IconFile);
            			        if( Status <= 31 ){
                			            MessageBox(hDlg, "Program Manager not found, cannot select default Icon.",
                    			                   "FreeDock Error", MB_OK | MB_ICONSTOP);
		                    	}

                                TmpSlot.IconPos = 1;
                                // Load the Icon into the slot's own bitmap
                                UtilLoadIcon( Dock, &TmpSlot );

                            }
                        }
                        /******************************************************************
                            It must be a document file, try to find an association for it
                        ******************************************************************/
                        else {
                            strcpy(TmpSlot.CmdLine, TmpSlot.AppName);
                            // Read the executable name in,
                            // then call SearchPath to get entire path for application.
                            Status = (UINT)FindExecutable(TmpSlot.CmdLine,
                                						  TmpSlot.RunTimeDir,
                                						  TmpSlot.AppName);
                            
                            // Now setup the parameter which is still in TmpBuffer
                            strcpy(TmpSlot.CmdLine, TmpBuffer);
                                
                            // Now call find executable to ensure we have a complete
                            // path to the application
                            if( FindExecutable(TmpSlot.AppName,
                                TmpSlot.RunTimeDir,
                                TmpBuffer) > (HINSTANCE)32 ){
	                          	strcpy( TmpSlot.AppName, TmpBuffer);
	                        }
                            
                            if (Status > 32) {
                                strcpy(TmpSlot.IconFile, TmpSlot.AppName);
                                TmpSlot.IconPos = 0;
                                // Load the Icon into the slot's own bitmap
                                UtilLoadIcon( Dock, &TmpSlot );
                            } 
                            else {
                                sprintf(TmpBuffer, "%s\nis not an executable or an associated file", TmpSlot.CmdLine);
                                MessageBox(hDlg, TmpBuffer, "FreeDock", MB_OK | MB_ICONSTOP);
                                TmpSlot.SlotType = SLOT_FREE;
                            }
                        }
					}


                    /*************************************************************************
                        If the user has just entered an application name into an empty slot
                        then mark this slot as being in use.
                    *************************************************************************/
                    if (TmpSlot.AppName[0] != '\0'){
                        TmpSlot.SlotType = SLOT_USED;
                    }

                    if (IsDlgButtonChecked(hDlg, QX_MINIMUM))
                        TmpSlot.StartState = START_MINIMUM;
                    else if (IsDlgButtonChecked(hDlg, QX_MAXIMUM))
                        TmpSlot.StartState = START_MAXIMUM;
                    else if (IsDlgButtonChecked(hDlg, QX_STORE))
                        TmpSlot.StartState = START_STORE;
                    else
                        TmpSlot.StartState = START_NORMAL;

                    if (IsDlgButtonChecked(hDlg, QX_START_TOP))
                        TmpSlot.StartOnTop = TRUE;
                    else
                        TmpSlot.StartOnTop = FALSE;

#ifdef WIN32		// is the Start in Separate memory space option set ?
                    TmpSlot.StartSeparate = IsDlgButtonChecked(hDlg, QX_START_SEPARATE);
#endif
					// Restore Icon Index to it's original value
		            TmpSlot.IconIndex = Slot->IconIndex;

					// Copy modified WorkSpace into Selected Slot
                    memcpy(Slot, &TmpSlot, sizeof(SLOT_ENTRY));

					// Reload the Icon incase it was changed
                    UtilLoadIcon( Dock, Slot );

                    RePaintSlot(Dock, Slot, FALSE);
                    FreeProcInstance(lpfnChooseIconDlgProc);
                    EndDialog(hDlg, TRUE);
                    return(TRUE);

                case QX_REMOVE:
                	UtilEmptySlot( Slot );
					WriteSlotOptions( Dock, Slot );	// Delete entry in ini file
                    RePaintSlot(Dock, Slot, TRUE);
                    EndDialog(hDlg, TRUE);
                    return (TRUE);

                case QX_BROWSEICON :
                    Status = BrowseIconFile( hDlg, TmpSlot.IconFile );
                    if(Status){
                        TmpSlot.IconPos = 0;
                        // Load the Icon into the slot's own bitmap  & update the IconTotal Field
                        UtilLoadIcon( Dock, &TmpSlot );
                        if (TmpSlot.IconTotal == 0) {
                            MessageBox(hDlg, "This file contains no icons",
                                "FreeDock", MB_OK | MB_ICONSTOP);
                            TmpSlot.IconPos = Slot->IconPos;
                            strcpy(TmpSlot.IconFile, Slot->IconFile);
                            SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
	                        UtilLoadIcon( Dock, &TmpSlot );
                        } 
                        NumIcons = TmpSlot.IconTotal;
                        SetDlgItemInt(hDlg, QX_ICONNUM, (TmpSlot.IconPos+1), TRUE);
                        SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
                        SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
                        // ReDraw Icon by invalidating the button area
                        hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
                        GetClientRect( hwndItem, &rcItem );
                        InvalidateRect( hwndItem, &rcItem, TRUE );
                    }
                    return (TRUE);

                case QX_BROWSEFILE :
                    Status = BrowseFileName( hDlg, TmpSlot.AppName );
                    if(Status){
                        SetDlgItemText(hDlg, QX_FILE, TmpSlot.AppName);
                        strcpy( TmpSlot.IconFile, TmpSlot.AppName );
                        TmpSlot.IconPos = 0;
                        // Load the Icon into the slot's own bitmap & update the IconTotal Field
                        UtilLoadIcon( Dock, &TmpSlot );
                        if( TmpSlot.IconTotal == 0 ){
                            // File has no Icons so select PROGMAN.EXE : Icon (2)

		                    Status = (UINT)FindExecutable("progman.exe",
    		                                              NULL,
        		                                          TmpSlot.IconFile);
            		        if( Status <= 31 ){
                		            MessageBox(hDlg, "Program Manager not found, cannot select default Icon.",
                    		                   "FreeDock Error", MB_OK | MB_ICONSTOP);
		                    }
	                        TmpSlot.IconPos = 1;
	                        // Load the Icon into the slot's own bitmap & update the IconTotal Field
    	                    UtilLoadIcon( Dock, &TmpSlot );
                        }

                        NumIcons = TmpSlot.IconTotal;
                        SetDlgItemInt(hDlg, QX_ICONNUM, (TmpSlot.IconPos+1), TRUE);
                        SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
                        SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
                        // ReDraw Icon by invalidating the button area
                        hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
                        GetClientRect( hwndItem, &rcItem );
                        InvalidateRect( hwndItem, &rcItem, TRUE );
                    }
                    return (TRUE);

                case QX_CANCEL:
                    FreeProcInstance(lpfnChooseIconDlgProc);
                    EndDialog(hDlg, FALSE);
                    return (TRUE);

				case QX_PREVIEW :
                    Status = DialogBoxParam(gOptions.hAppInst, "PREVIEWERDLG", hDlg, lpfnPreviewerDlgProc, (LPARAM)&TmpSlot);
					if(Status){
						TmpSlot.StartState = START_STORE;
    	    	        CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_STORE);
					}
					return (TRUE);

                case QX_CHOOSEICON :
                    /*************************************************************
                    	If we are showing a dialog on an empty slot, then don't
                    	allow an icon to be chosen
                    *************************************************************/
                    if ( NumIcons == 0 ) return(TRUE);

                    Status = DialogBoxParam(gOptions.hAppInst, "CHOOSEICON", hDlg, lpfnChooseIconDlgProc, (LPARAM)&TmpSlot);
                    if(Status){
                        UtilLoadIcon( Dock, &TmpSlot );
                        hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
                        GetClientRect( hwndItem, &rcItem );
                        InvalidateRect( hwndItem, &rcItem, TRUE );
                        SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
                    }
                    return (TRUE);
            }
            return (FALSE);
 
  		case WM_LBUTTONDOWN:
                GetWindowRect( hDlg, &rect );
                /** Correct for position of dialogbox       **/
				/** and correct for dialog border & frame **/
                TmpX = (short int)LOWORD(lParam) + (short int)rect.left + gMetrics.DlgFrameWidth;
                TmpY = (short int)HIWORD(lParam) + (short int)rect.top + gMetrics.DlgFrameHeight + gMetrics.DlgTitleHeight;
                /** Get the location of the Grabber Item **/
                hwndTmp = GetDlgItem( hDlg, QX_GRABWIN );
                GetWindowRect( hwndTmp, &rect );
                /** Next see if the button was pressed here **/
                if( (TmpX <= rect.right && TmpX >= rect.left ) &&
                    (TmpY <= rect.bottom && TmpY >= rect.top ) ){
                    /** Then click hit Grabber Item **/
                        SetCapture(hDlg);
                        hGrabCur = LoadCursor(gOptions.hAppInst, "GrabberCur");
                        SetCursor(hGrabCur);
                        Grabbing = TRUE;
						GetWindowRect(hDlg, &rect);
						InvalidateRect(hDlg, &rect, FALSE);
						UpdateWindow(hDlg);
						/** Hide the Application Options Dialog window from the grabber **/
						/** If this option is selected **/
						if(gOptions.HideDlgOnGrab){
    	                    SetWindowPos( hDlg, 0, 0,0,0,0,
	                                      SWP_HIDEWINDOW|SWP_NOSIZE|SWP_NOMOVE|SWP_NOZORDER );
						}
  			    }
                return (TRUE);
  
        case WM_LBUTTONUP:
            if (!Grabbing)
                return TRUE;
            /********************************************************
               Button click positions are reported relative to the
               Dialog window, so correct for this by adding the x,y
               coordinates of the top left corner of the dialog to
               the click position. Also correct for the dialog
			   frame and title bar sizes.
            ********************************************************/
            GetWindowRect(hDlg, &rect);
            pt.x = rect.left + (short int)LOWORD(lParam) + gMetrics.DlgFrameWidth;
            pt.y = rect.top + (short int)HIWORD(lParam) + gMetrics.DlgFrameHeight + gMetrics.DlgTitleHeight;
            hwndGrab = WindowFromPoint(pt);

			/** Show the the dialog window again if it was hidden **/
			if(gOptions.HideDlgOnGrab){
	            SetWindowPos( hDlg, 0, 0,0,0,0,
    	                      SWP_SHOWWINDOW|SWP_NOSIZE|SWP_NOMOVE|SWP_NOZORDER );
			}

            /* Climb back up to the top of a parent-child list */
            hwndTmp = GetParent(hwndGrab);
            while (hwndTmp != NULL) {
                hwndGrab = hwndTmp;
                hwndTmp = GetParent(hwndGrab);
            }
            GetWindowRect(hwndGrab, &rect);
            TmpSlot.WinX = rect.left;
            TmpSlot.WinY = rect.top;
            TmpSlot.WinWidth = rect.right - rect.left;
            TmpSlot.WinHeight = rect.bottom - rect.top;

            if( hwndGrab == GetDesktopWindow() ){
                MessageBox(hDlg, "Oops, that's the Desktop window, please try again.",
                    "FreeDock", MB_OK | MB_ICONINFORMATION);
            }
            else if( hwndGrab == Dock->hwndDock ){
                MessageBox(hDlg, "Oops, that's a FreeDock window, please try again.",
                    "FreeDock", MB_OK | MB_ICONINFORMATION);
            }
            else{
                MessageBox(hDlg, "Window Grabbed Successfully.",
                    "FreeDock", MB_OK | MB_ICONINFORMATION);
				TmpSlot.StartState = START_STORE;
                CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_STORE);
            }

            Grabbing = FALSE;
            ReleaseCapture();
            SetCursor(LoadCursor(NULL, IDC_ARROW));
            DestroyCursor(hGrabCur);

			GetWindowRect(hDlg, &rect);
			InvalidateRect(hDlg, &rect, FALSE);
			UpdateWindow(hDlg);

            return (TRUE);
    }

    return FALSE;
}


BOOL FAR PASCAL MainOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
	static DOCK_STRUCT  *Dock;
    static int 		   OrigPosn;
    static int 		   OrigSlotCount, OrigSlotScale;
	extern FARPROC     lpfnAboutDockDlgProc;	// Defined in dockwin.c
	RECT			   rect;

    switch (iMessage) {

        case WM_INITDIALOG:
			Dock = (DOCK_STRUCT *)lParam;

            OrigPosn = Dock->Orientation + QX_HORZ;  // Save original position incase user cancels
            OrigSlotCount = Dock->SlotCount;
            OrigSlotScale = gOptions.SlotScale;

            CheckDlgButton(hDlg, QX_DOCK_LOCKED, gOptions.DockLocked);
            CheckDlgButton(hDlg, QX_MAILCHECK, gOptions.MailActive);
            CheckDlgButton(hDlg, QX_WINEXIT, gOptions.WinExitActive);
            CheckDlgButton(hDlg, QX_CLOCK, gOptions.ClockActive);
            CheckDlgButton(hDlg, QX_ON_TOP, gOptions.AlwaysOnTop);
            CheckDlgButton(hDlg, QX_MAX_VIEW, gOptions.MaxView);
            CheckDlgButton(hDlg, QX_SINGLECLICK, gOptions.SingleClickStart);
            CheckRadioButton(hDlg, QX_HORZ, QX_VERT, OrigPosn);
            SetDlgItemInt(hDlg, QX_SLOT_COUNT, Dock->SlotCount, TRUE);
            SetDlgItemInt(hDlg, QX_SLOT_SCALE, gOptions.SlotScale, TRUE);
            return (TRUE);

        case WM_HSCROLL:		// Scale Scroll Bar
            if (wParam == SB_LINEDOWN)
	            if ( gOptions.SlotScale < MAX_SLOT_SCALE){
     		         gOptions.SlotScale++;
            		 SetDlgItemInt(hDlg, QX_SLOT_SCALE, gOptions.SlotScale, TRUE);
           		}
	
   	        if (wParam == SB_LINEUP)
     	        if ( gOptions.SlotScale > MIN_SLOT_SCALE){
   	    	        gOptions.SlotScale--;
       	    	    SetDlgItemInt(hDlg, QX_SLOT_SCALE, gOptions.SlotScale, TRUE);
           	    }
			gOptions.SlotWidth = gOptions.SlotHeight = gOptions.SlotScale;

			SetDockWinPos( Dock, Dock->Orientation + QX_HORZ );
            GetClientRect( Dock->hwndDock, &rect );
            InvalidateRect( Dock->hwndDock , &rect, TRUE );
    	    return (TRUE);

        case WM_VSCROLL:		// Slot Count Scroll Bar
            if (wParam == SB_LINEUP)
	            if ( Dock->SlotCount < gOptions.MaxDockSize){
 		            Dock->SlotCount++;
       		        SetDlgItemInt(hDlg, QX_SLOT_COUNT, Dock->SlotCount, TRUE);
           		}
	
   	        if (wParam == SB_LINEDOWN)
    	        if ( Dock->SlotCount > 1){
   	    	        Dock->SlotCount--;
       	    	    SetDlgItemInt(hDlg, QX_SLOT_COUNT, Dock->SlotCount, TRUE);
           	    }
			SetDockWinPos( Dock, Dock->Orientation + QX_HORZ );
   	    return (TRUE);


        case WM_COMMAND:
            switch (wParam) {
                case QX_OK:
                    gOptions.MailActive = IsDlgButtonChecked(hDlg, QX_MAILCHECK);
                    gOptions.WinExitActive = IsDlgButtonChecked(hDlg, QX_WINEXIT);
                    gOptions.ClockActive = IsDlgButtonChecked(hDlg, QX_CLOCK);
                    gOptions.AlwaysOnTop = IsDlgButtonChecked(hDlg, QX_ON_TOP);
                    gOptions.MaxView = IsDlgButtonChecked(hDlg, QX_MAX_VIEW);
                    gOptions.SingleClickStart = IsDlgButtonChecked(hDlg, QX_SINGLECLICK);
                    gOptions.DockLocked = IsDlgButtonChecked(hDlg, QX_DOCK_LOCKED);

                    /**************************************************************
                        Handle Always On Top Option
                    **************************************************************/
                    if (gOptions.AlwaysOnTop) {
                        SetWindowPos(Dock->hwndDock, HWND_TOPMOST, 0, 0, 0, 0,
                            SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);
                    } 
                    else {
                        SetWindowPos(Dock->hwndDock, HWND_BOTTOM, 0, 0, 0, 0,
                            SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);
                    }

                    EndDialog(hDlg, TRUE);
                    break;

                    /****************************************************************
                        Handle Dock Position Options, for updating dock pos as the
                        user choses the radio button.
                    ****************************************************************/
                case QX_HORZ:
                case QX_VERT:
                    SetDockWinPos( Dock, wParam );
                    break;

                case QX_CANCEL:
                    Dock->SlotCount = OrigSlotCount ;
                    gOptions.SlotScale = OrigSlotScale ;
					gOptions.SlotWidth = gOptions.SlotHeight = gOptions.SlotScale;
					SetDockWinPos( Dock, OrigPosn ); // restore original position
            		GetClientRect( Dock->hwndDock, &rect );
           		    InvalidateRect( Dock->hwndDock , &rect, TRUE );
                    EndDialog(hDlg, FALSE);
                    break;

				case QX_ABOUT:
                    DialogBox(gOptions.hAppInst, "ABOUTDOCKDLG", hDlg, lpfnAboutDockDlgProc);
					break;
            }
            break;
    }

    return FALSE;
}


BOOL FAR PASCAL WinExitOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{

    switch (iMessage) {
        case WM_INITDIALOG:
            CheckDlgButton(hDlg, QX_WINEXITCONFIRM, gOptions.WinExitConfirm);
            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {
                case QX_OK:
                    gOptions.WinExitConfirm = IsDlgButtonChecked(hDlg, QX_WINEXITCONFIRM);
                    EndDialog(hDlg, TRUE);
                    break;

                case QX_CANCEL:
                    EndDialog(hDlg, FALSE);
                    break;
            }
            break;
    }
    return FALSE;
}


BOOL FAR PASCAL MailOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
    BOOL   				bTxlated;
	static DOCK_STRUCT *Dock;

    switch (iMessage) {
        case WM_INITDIALOG:
		    Dock = (DOCK_STRUCT *)lParam;
            SendDlgItemMessage(hDlg, QX_MAILPATH, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
            SetDlgItemText(hDlg, QX_MAILPATH, gOptions.MailPath);
            SetDlgItemInt(hDlg, QX_MAILFREQ, gOptions.MailFreq, TRUE);
            CheckDlgButton(hDlg, QX_MAILSOUND, gOptions.MailSound);
            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {
                case QX_OK:
                    GetDlgItemText(hDlg, QX_MAILPATH, gOptions.MailPath, MAX_FPATH_LEN);
                    gOptions.MailFreq = GetDlgItemInt(hDlg, QX_MAILFREQ, &bTxlated, FALSE);
                    if (!bTxlated) {
                        MessageBox(hDlg, "Not a valid integer!", "FreeDock", MB_OK | MB_ICONSTOP);
                        SendDlgItemMessage(hDlg, QX_MAILFREQ, EM_SETSEL, 0, MAKELONG(0, -1));
                        return (TRUE);
                    }
                    gOptions.MailSound = IsDlgButtonChecked(hDlg, QX_MAILSOUND);
                    KillTimer(Dock->hwndDock, MAIL_TIMER);
                    SetTimer(Dock->hwndDock, MAIL_TIMER, gOptions.MailFreq * 1000 * 60, NULL);
                    EndDialog(hDlg, TRUE);
                    break;

                case QX_CANCEL:
                    EndDialog(hDlg, FALSE);
                    break;
            }
            break;
    }
    return FALSE;
}


BOOL FAR PASCAL ClockOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
    switch (iMessage) {
        case WM_INITDIALOG:
            return (TRUE);

        case WM_COMMAND:
            if (wParam == QX_OK)
                EndDialog(hDlg, TRUE);
            break;
    }
    return FALSE;
}


BOOL FAR PASCAL AboutDockDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
	static HICON hIcon;

    switch (iMessage) {
        case WM_INITDIALOG:
            SetDlgItemText(hDlg, QX_VERSION, VersionText);
            SetDlgItemText(hDlg, QX_OS_BITS, OS_Bits);
#ifdef WIN32
			hIcon = LoadIcon(gOptions.hAppInst, "FREEDOCK32");
#else
			hIcon = LoadIcon(gOptions.hAppInst, "FREEDOCK16");
#endif
            SendDlgItemMessage(hDlg, QX_ICON1, STM_SETICON, (WPARAM)hIcon, 0L);
            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {

                case QX_OK:
                    EndDialog(hDlg, TRUE);
                    break;

				case QX_EXIT:
                    //            nItem = MessageBox(hDlg, "Do you really want to exit FreeDock ?",
                    //                               "FreeDock", MB_OKCANCEL | MB_ICONSTOP);

                    //            if (nItem == IDOK) {
                    WriteGlobalOptions();
                    PostQuitMessage(0);
                    //            }
                    return (TRUE);
            }
    }
    return FALSE;
}


BOOL FAR PASCAL BrowseIconFile( HWND hwnd, char *FileName )
{
    OPENFILENAME ofn;
    char DirName[256], FileTitle[256], TmpFileName[MAX_FPATH_LEN];
    char Filter[] = "Executables (*.EXE)\0*.EXE\0Icon Files (*.ICO)\0*.ICO\0DLL's (*.DLL)\0*.DLL\0\0";
    BOOL Status;
    char *TmpPtr;

    memset( &ofn, 0, sizeof(OPENFILENAME) );

    strcpy( TmpFileName, FileName );
    strcpy( DirName, FileName );
    TmpPtr = strrchr(DirName, '\\');
    if(TmpPtr != NULL){
        *TmpPtr = '\0';
    }

    ofn.lpstrTitle   = "Select an Icon File";
    ofn.lStructSize  = sizeof(OPENFILENAME);
    ofn.hwndOwner    = hwnd;
    ofn.lpstrFilter  = Filter;
    ofn.nFilterIndex = 1;
    ofn.lpstrFile = TmpFileName;
    ofn.nMaxFile = MAX_FPATH_LEN;
    ofn.lpstrFileTitle = FileTitle;
    ofn.nMaxFileTitle = sizeof(FileTitle);
    ofn.lpstrInitialDir = DirName;
    ofn.Flags = OFN_HIDEREADONLY | OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

    Status = GetOpenFileName(&ofn);

    // If no change made, this is the same as cancel
    if(!stricmp( FileName, TmpFileName ))
        return(FALSE);

    if(Status){
        strcpy( FileName, TmpFileName );
    }

    return(Status);
}


BOOL FAR PASCAL BrowseFileName( HWND hwnd, char *FileName )
{
    OPENFILENAME ofn;
    char DirName[256], FileTitle[256], TmpFileName[MAX_FPATH_LEN];
    char Filter[] = "Executables (*.EXE)\0*.EXE\0Batch Files (*.BAT)\0*.BAT\0PIF Files (*.PIF)\0*.PIF\0DOS .COM Files (*.COM)\0*.COM\0\0";
    BOOL Status;
    char *TmpPtr;

    memset( &ofn, 0, sizeof(OPENFILENAME) );

    strcpy( TmpFileName, FileName );
    strcpy( DirName, FileName );
    TmpPtr = strrchr(DirName, '\\');
    if(TmpPtr != NULL){
        *TmpPtr = '\0';
    }

    ofn.lpstrTitle   = "Select a File";
    ofn.lStructSize  = sizeof(OPENFILENAME);
    ofn.hwndOwner    = hwnd;
    ofn.lpstrFilter  = Filter;
    ofn.nFilterIndex = 1;
    ofn.lpstrFile = TmpFileName;
    ofn.nMaxFile = MAX_FPATH_LEN;
    ofn.lpstrFileTitle = FileTitle;
    ofn.nMaxFileTitle = sizeof(FileTitle);
    ofn.lpstrInitialDir = DirName;
    ofn.Flags = OFN_HIDEREADONLY | OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

    Status = GetOpenFileName(&ofn);

    // If no change made, this is the same as cancel
    if(!stricmp( FileName, TmpFileName ))
        return(FALSE);

    if(Status){
        strcpy( FileName, TmpFileName );
    }

    return(Status);
}


BOOL FAR PASCAL ChooseIconDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
    // Slot info passed in via global TmpSlot struct
    // CurIcon and CurPage are 0 relative
    static int NumIcons, CurIcon, NumPages, CurPage;
    static HANDLE *Icon;            // pointer to array of icon handles
	static SLOT_ENTRY *Slot;		// Pointer to Slot
	static DOCK_STRUCT *Dock;		// Pointer to parent dock
    RECT rect;
    DRAWITEMSTRUCT *ItemStruct;
    int i, ItemNum;

    switch (iMessage) {
        case WM_INITDIALOG:

		    // Recover the pointer to the Slot from lParam
			Slot = (SLOT_ENTRY *)lParam;

            // find number of icons & create array to hold ALL handles
            NumIcons = Slot->IconTotal;
            Icon = (HANDLE *)malloc( NumIcons * sizeof(HANDLE) );

            // Read all Icon handles
            for( i=0; i< NumIcons ; i++){
                Icon[i] = ExtractIcon(gOptions.hAppInst, Slot->IconFile, i);
            }

            CurIcon = Slot->IconPos;
            NumPages = (NumIcons / 50) + 1;
            CurPage = (CurPage / 50);

            SetDlgItemInt(hDlg, QX_ICONNUM, (CurIcon+1), TRUE);
            SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
            SetDlgItemInt(hDlg, QX_PAGENUM, (CurPage+1), TRUE);
            SetDlgItemInt(hDlg, QX_PAGETOTAL, NumPages, TRUE);
            SendDlgItemMessage(hDlg, QX_ICONBOX, STM_SETICON, (WPARAM)Icon[CurIcon], 0L);
            return (TRUE);

        case WM_DRAWITEM :
            ItemStruct = (DRAWITEMSTRUCT *)lParam;
            ItemNum = (ItemStruct->CtlID-QX_ICON1) + (CurPage*50);
            if(ItemNum < NumIcons){
                DrawIcon( ItemStruct->hDC, 0, 0, Icon[ItemNum]);
            }
            return (TRUE);

        case WM_HSCROLL:
            if (wParam == SB_LINEDOWN)
                if (CurPage < (NumPages-1)){
                    CurPage = CurPage + 1;
                    GetClientRect( hDlg, &rect );
                    for(i = QX_ICON1; i < QX_ICON1+50; i++){
                        InvalidateRect( GetDlgItem( hDlg, i) , &rect, TRUE );
                    }
                }

            if (wParam == SB_LINEUP)
                if (CurPage > 0){
                    CurPage = CurPage - 1;
                    GetClientRect( hDlg, &rect );
                    for(i = QX_ICON1; i < QX_ICON1+50; i++){
                        InvalidateRect( GetDlgItem( hDlg, i) , &rect, TRUE );
                    }
                }

            SetDlgItemInt(hDlg, QX_PAGENUM, CurPage + 1, TRUE);

            return (TRUE);

        case WM_COMMAND:
            if (wParam == QX_OK){
                Slot->IconPos = CurIcon;
                free( Icon );
                EndDialog(hDlg, TRUE);
            }
            else if (wParam == QX_CANCEL){
                free( Icon );
                EndDialog(hDlg, FALSE);
            } 
            else if( (wParam >= QX_ICON1) && (wParam <= QX_ICON1+50) ){
                CurIcon = min( (int)(NumIcons-1), (int)((wParam - QX_ICON1) + (CurPage*50)));
                SetDlgItemInt(hDlg, QX_ICONNUM, (CurIcon+1), TRUE);
                SendDlgItemMessage(hDlg, QX_ICONBOX, STM_SETICON, (WPARAM)Icon[CurIcon], 0L);
            }
            break;
    }
    return FALSE;
}

/**************************************************************************
    ExecSlot.

    This function handles the execution of a program associated
    with a slot, it takes the Slot index so it can find the program
    to execute, it also takes the command line to pass, to enable
    this function to be used to execute a program as a result of
    a drag and drop action.
    In the case of drag&drop, the caller must create the command line
    before calling this function.
**************************************************************************/

void ExecSlot( SLOT_ENTRY *Slot, char *CmdLine ){

    int                 StartState;
    LPARAM              hStartAppInst;
    UINT                nItem;
    BOOL                bStartedApp = FALSE;
	DOCK_STRUCT			*Dock;

#ifdef WIN32
	DWORD				fdwCreate = 0L;
	STARTUPINFO         StartupInfo;
    PROCESS_INFORMATION ProcInfo;
    char    TmpBuffer[MAX_FPATH_LEN + MAX_CMDLINE_LEN + 1];
#endif

	Dock = Slot->Dock;

    switch (Slot->StartState) {

        case START_MINIMUM:
            StartState = SW_SHOWMINIMIZED;
            break;

        case START_MAXIMUM:
            StartState = SW_SHOWMAXIMIZED;
            break;

        case START_NORMAL:
        case START_STORE :
            StartState = SW_SHOWNORMAL;
            break;

        default: return;
    }

    if(Slot->StartOnTop){
        StartState = StartState | SW_SHOWMAXIMIZED;
    }


	/**********************************************
		Check if the working directory exists
		if not, return.
	**********************************************/
#ifdef WIN32
	if( !SetCurrentDirectory( Slot->RunTimeDir ) ){
#else
	if( _chdir( Slot->RunTimeDir ) ){
#endif
        MessageBox( NULL, "The Runtime Directory for this application cannot be found.\r\r\n Cannot Start Application",
					"FreeDock Error",
                    MB_ICONSTOP | MB_OK); 
		return;
	}

    SetCursor(LoadCursor(NULL, IDC_WAIT));
    /**********************************************
        Need to start up the application in a
        diferent manner under NT so we can 
        keep track of it and find the window
        it creates inorder to re-position
        it or set it on top
    *********************************************/
#ifdef WIN32

    memset( &StartupInfo, '\0', sizeof(STARTUPINFO) );
    memset( &ProcInfo,    '\0', sizeof(PROCESS_INFORMATION) );
    StartupInfo.cb = sizeof(StartupInfo);

	/***********************************************************
		Setup the window Pos, size at startup time
		if required (only WIN32 supports this option)
		Can't specify that a window is always on top though,
		so we still use the EnumWindows function for this,
	***********************************************************/    
    if (Slot->StartState == START_STORE){
		StartupInfo.dwX		= Slot->WinX;
		StartupInfo.dwY		= Slot->WinY;
		StartupInfo.dwXSize	= Slot->WinWidth;
		StartupInfo.dwYSize	= Slot->WinHeight;
		StartupInfo.dwFlags = STARTF_USEPOSITION | STARTF_USESIZE;
	}
	else if (Slot->StartState == START_MINIMUM){
		StartupInfo.wShowWindow	= SW_SHOWMINIMIZED;
		StartupInfo.dwFlags = STARTF_USESHOWWINDOW;
	}
	else if (Slot->StartState == START_MAXIMUM){
		StartupInfo.wShowWindow	= SW_SHOWMAXIMIZED;
		StartupInfo.dwFlags = STARTF_USESHOWWINDOW;
	}

	/*****************************************************************
		Are we trying to start this app in separate address space ?
	*****************************************************************/
	if (Slot->StartSeparate == TRUE){
	 	fdwCreate = CREATE_SEPARATE_WOW_VDM;
	}
    // Create Full Command Line
    sprintf( TmpBuffer, "%s %s", Slot->AppName, CmdLine );
	// Start process.
    bStartedApp = CreateProcess( NULL, TmpBuffer,
                                 NULL,                 // default process security
                                 NULL,                 // default thread security
                                 FALSE,                // don't inherit handles
                                 fdwCreate,            // startup flags
                                 NULL,                 // Inherit the ENVIRONMENT
                                 Slot->RunTimeDir,     // Current Dir
                                 &StartupInfo,         // Process Startup Info
                                 &ProcInfo );          // Recieves Process data
    /** Save the process ID to pass into the EnumWindows func later if reqd **/
    hStartAppInst = (LPARAM)ProcInfo.dwProcessId;
    
#else
    hStartAppInst = (LPARAM)ShellExecute(GetDesktopWindow(),
        "open",
        Slot->AppName,
        CmdLine,
        Slot->RunTimeDir,
        StartState);

    if (hStartAppInst > 33){
        bStartedApp = TRUE;
    }
#endif

    /** No Sleep() in 16bit windows, and it does not seem to be reqd **/
#ifdef WIN32
    /** Have a kip to give the window a chance to appear **/
    Sleep(gOptions.SleepTime*500);
#endif

    /** Check for an error starting the application **/
    if( !bStartedApp ){
        MessageBox( Dock->hwndDock, "Error starting application.",
                    "FreeDock Error", MB_OK | MB_ICONSTOP);
        SetCursor(LoadCursor(NULL, IDC_ARROW));
        return;
    }
    
	/** No need to try to reposition window in NT, it's done already **/
#ifdef WIN32
    if( Slot->StartOnTop ){               
#else
    if( (Slot->StartState == START_STORE) || 
        (Slot->StartOnTop)){               
#endif   
        /** Copy Slot into TmpSlot to pass to callback fuction **/
        memcpy( &TmpSlot, Slot, sizeof(SLOT_ENTRY) );

        gOptions.bFoundWindow = FALSE;	// Set Call back function success flag to fail

        do{
            nItem = IDCANCEL;
               EnumWindows( UtilSetupStartedWin, (LPARAM)hStartAppInst );
               if(!gOptions.bFoundWindow){
                    nItem = MessageBox(Dock->hwndDock, "Could not locate window for started application. Retry ?",
                            "Freedock Error", MB_RETRYCANCEL | MB_ICONSTOP);
               }
        }while( nItem != IDCANCEL );
    }

    SetCursor(LoadCursor(NULL, IDC_ARROW));

    return;
}


void SetDockWinPos( DOCK_STRUCT *Dock, int DockPos )
{

	// This fucntion correctly positions the dock on the desktop, 
	// allowing for the current orientation

	// Limit dock position to actual screen
	Dock->DockLeft = max(Dock->DockLeft, 0);
	Dock->DockTop = max(Dock->DockTop, 0);

	Dock->DockLeft = min(Dock->DockLeft, (gMetrics.ScreenWidth - gOptions.SlotWidth));
	Dock->DockTop = min(Dock->DockTop, (gMetrics.ScreenHeight - gOptions.SlotHeight));

    switch( DockPos ){

        case QX_HORZ:
            Dock->Orientation = DOCK_HORZ;
            SetWindowPos(Dock->hwndDock,
                HWND_TOP,
                Dock->DockLeft,
                Dock->DockTop,
                Dock->SlotCount * gOptions.SlotWidth + 2,
                gOptions.SlotHeight + 2,
                SWP_NOACTIVATE | SWP_SHOWWINDOW);
            break;

        default :
            Dock->Orientation = DOCK_VERT;
            SetWindowPos(Dock->hwndDock,
                HWND_TOP,
                Dock->DockLeft,
                Dock->DockTop,
                gOptions.SlotWidth + 2,
                Dock->SlotCount * gOptions.SlotHeight + 2,
                SWP_NOACTIVATE | SWP_SHOWWINDOW);
            break;

    }
}


void UtilLoadIcon( DOCK_STRUCT *Dock, SLOT_ENTRY *Slot ){

    RECT	rect;
    HICON   IconHandle;
    static HBITMAP  hBlank = NULL;

    // Load up the blank image if we don't already have it
    if( hBlank == NULL ){
        hBlank = LoadBitmap(gOptions.hAppInst, "SlotButton");
    }
    
    // If the Slot parameter is NULL this means we need to delete
    // the internal bitmap handle.
    if( Slot == NULL ){
        DeleteObject( hBlank );
        return;
    }
                                      
    rect.top = 0;
    rect.left = Slot->IconIndex;
    rect.bottom = gOptions.SlotHeight;
    rect.right = Slot->IconIndex + gOptions.SlotWidth;

    Slot->IconTotal = 0;
    
    switch( Slot->SlotType ){

        case SLOT_SPECIAL_TITLE :
#ifdef WIN32
            IconHandle = LoadIcon(gOptions.hAppInst, "FREEDOCK32");
#else
            IconHandle = LoadIcon(gOptions.hAppInst, "FREEDOCK16");
#endif
            break;

        case SLOT_SPECIAL_MAIL :
        	if( gOptions.bMailInBox ){
	            IconHandle = LoadIcon(gOptions.hAppInst, "MAILICON");
        	}
        	else{
            	IconHandle = LoadIcon(gOptions.hAppInst, "NOMAILICON");
            }
            break;

        case SLOT_SPECIAL_CLOCK :
            IconHandle = LoadIcon(gOptions.hAppInst, "CLOCKICON");
            break;

        case SLOT_SPECIAL_EXIT :
            IconHandle = LoadIcon(gOptions.hAppInst, "EXITICON");
            break;
    
        default:
            Slot->IconTotal  = (int)ExtractIcon(gOptions.hAppInst, Slot->IconFile, -1);
            IconHandle = (HICON)ExtractIcon(gOptions.hAppInst, Slot->IconFile, Slot->IconPos);
            break;
    }

	FillRect( Dock->hdcIconCache, &rect, GetStockObject(LTGRAY_BRUSH) );
	/** Draw the blank slot first **/                                      
    DrawBitmap(Dock->hdcIconCache, hBlank, Slot->IconIndex, 0);
    DrawIcon( Dock->hdcIconCache, Slot->IconIndex+3, 3, IconHandle );
}


/**********************************************************************
	Function to set all the variable fields in an emptied slot to
	their default values. Does not change ID or IconIndex fields
	since they are assigned at startup and cannot be changed.

	Also erases the Icon in the IconCache.
**********************************************************************/

void UtilEmptySlot( SLOT_ENTRY *Slot ){

    RECT	rect;
	DOCK_STRUCT *Dock;

	Dock = Slot->Dock;

    rect.top = 0;
    rect.left = Slot->IconIndex;
    rect.bottom = gMetrics.IconHeight;
    rect.right = Slot->IconIndex + gMetrics.IconWidth;
 
    Slot->SlotType		= SLOT_FREE;
    Slot->AppName[0]	= '\0';
    Slot->CmdLine[0]	= '\0';
    Slot->RunTimeDir[0]	= '\0';
    Slot->StartState	= START_NORMAL;
    Slot->StartOnTop	= FALSE;
    Slot->StartSeparate	= FALSE;
    Slot->WinX			= DEF_STORE_X;
    Slot->WinY			= DEF_STORE_Y;
    Slot->WinWidth		= DEF_STORE_W;
    Slot->WinHeight		= DEF_STORE_H;
    Slot->IconFile[0]	= '\0';
    Slot->IconPos		= 0;
    Slot->IconTotal		= 0;

	FillRect( Dock->hdcIconCache, &rect, GetStockObject(LTGRAY_BRUSH) );
}


BOOL CALLBACK UtilSetupStartedWin( HWND hwnd, LPARAM lParam ){


	/****************************************************
		Because Application Instances are not unique
		under WIN32 / Win NT we must locate the app
		just started in a diferent manner depending on
		the version of Windows in use
	****************************************************/

#ifdef WIN32
	{
    DWORD     ThreadId;
    DWORD     ProcessId = 1L;       // Must be non-NULL to recieve Process ID

    ThreadId = GetWindowThreadProcessId( hwnd, &ProcessId );
    if( ProcessId == (DWORD)lParam ){
        gOptions.bFoundWindow = TRUE;
    }
    }
#else
    if( GetWindowWord( hwnd, GWW_HINSTANCE ) == (HINSTANCE)lParam ){
        gOptions.bFoundWindow = TRUE;
    }
#endif

    if( gOptions.bFoundWindow ){

		// Window position is setup by CreateProcess if requried
        if(TmpSlot.StartState == START_STORE){
            /*****************************
            	Re-position the window 
            *****************************/
            SetWindowPos(hwnd, HWND_NOTOPMOST, TmpSlot.WinX, TmpSlot.WinY,
                    TmpSlot.WinWidth, TmpSlot.WinHeight, SWP_NOZORDER);
        }

        if( TmpSlot.StartOnTop ){
            /*****************************
                Put the window on top 
            *****************************/
            SetWindowPos(hwnd, HWND_TOPMOST, 0, 0, 0, 0, 
                         SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);
        }
    }
    // returning FALSE keeps enumeration going, TRUE stops it
    return (!gOptions.bFoundWindow);
}



BOOL FAR PASCAL PreviewerDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{

	static RECT rect;
	static SLOT_ENTRY *Slot;

    switch (iMessage) {
        case WM_INITDIALOG:

		    Slot = (SLOT_ENTRY *)lParam;

			SetWindowPos(hDlg, 0L, 
						 Slot->WinX,
						 Slot->WinY,
						 Slot->WinWidth,
						 Slot->WinHeight,
						 SWP_SHOWWINDOW | SWP_NOZORDER);
            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {
                case QX_OK:
		            GetWindowRect(hDlg, &rect);
    	        	Slot->WinX = rect.left;
        		    Slot->WinY = rect.top;
        	    	Slot->WinWidth = rect.right - rect.left;
		            Slot->WinHeight = rect.bottom - rect.top;
                    EndDialog(hDlg, TRUE);
                    break;

                case QX_CANCEL:
                    EndDialog(hDlg, FALSE);
                    break;
            }
            break;
    }
    return FALSE;
}


/***************************************
	Swaps the contents of two slots
	(Only swaps non-variable fields)
	not ID
***************************************/
void SwapSlots( SLOT_ENTRY *Slot1, SLOT_ENTRY *Slot2 ){

	int 		Slot1_ID, Slot2_ID;
	SLOT_ENTRY  WorkingSlot;

	/****************************************
		Store original Slot IDs for later
		These are important & cannot be moved.
	****************************************/
	Slot1_ID = Slot1->ID;
	Slot2_ID = Slot2->ID;

	/*********************************
		Copy Slot 1 to Working Slot
	*********************************/
	memcpy( &WorkingSlot, Slot1, sizeof(SLOT_ENTRY) );

	/*********************************
		Copy Slot 2 to  Slot1
	*********************************/
    memcpy( Slot1, Slot2, sizeof(SLOT_ENTRY) );

	/*********************************
		Copy Working Slot to Slot 2
	*********************************/
    memcpy( Slot2, &WorkingSlot, sizeof(SLOT_ENTRY) );

	/***************************************
		Now restore the correct values
		to the ID fields in the Slots
	***************************************/
	Slot1->ID = Slot1_ID;
	Slot2->ID = Slot2_ID;

	return;
}
